/******************************************************************************
* 
* Copyright (c) 2008 Freescale Semiconductor;
* All Rights Reserved                       
*
*******************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
***************************************************************************//*!
*
* @file      freemaster_HC08.c
*
* @author    r50233
* 
* @version   1.0.1.0
* 
* @date      Jul-7-2009
* 
* @brief     FreeMASTER Driver HC08-hardware dependent stuff 
*
*******************************************************************************/

#include "FreeMASTER/freemaster.h"
#include "FreeMASTER/freemaster_private.h"
#include "FreeMASTER/freemaster_HC08.h"

/*******************************************************************************
*
* @brief    API: Main SCI Interrupt handler call
*
* This Interrupt Service Routine handles the SCI interrupts for the FreeMASTER 
* driver. In case you want to handle the interrupt in the application yourselves,
* call the FMSTR_ProcessSCI function which does the same job but is not compiled
* as an Interrupt Service Routine.
*
* In poll-driven mode (FMSTR_POLL_DRIVEN) this function does nothing.
*
*******************************************************************************/

/* HC08 interrupt routine declaration */
#if defined(FMSTR_SCI_RX_INTERRUPT) && defined(FMSTR_SCI_TX_INTERRUPT)
/* user may define RX interrupt number in the configuration */
interrupt FMSTR_SCI_RX_INTERRUPT
#else
/* or may want to use the PRM file to set the vector to FMSTR_Isr */
#pragma TRAP_PROC
#endif

void FMSTR_Isr(void)
{
#if FMSTR_LONG_INTR || FMSTR_SHORT_INTR

    /* process serial interface */
#if FMSTR_USE_SCI
    FMSTR_ProcessSCI(); 
#endif
    
#endif
}

/*******************************************************************************
*
* @brief    The 2nd FMSTR interrupt handler 
*
* In case the user wants to use automatic interrupt code genration, he may use
* FMSTR_SCI_RX_INTERRUPT and FMSTR_SCI_TX_INTERRUPT constants in the configuration
* file (freemaster_cfg.h). The service code is the same for both interrupts, and
* it would be definitelly better for the user to use two VECTOR directives in the 
* PRM file. 
*
* Use the FMSTR_SCI_RX_INTERRUPT and FMSTR_SCI_TX_INTERRUPT constants in case 
* you do not want or can not modify the PRM file.
*
*******************************************************************************/

/* HC08 interrupt routine declaration */
#if defined(FMSTR_SCI_RX_INTERRUPT) && defined(FMSTR_SCI_TX_INTERRUPT)
/* user may define RX interrupt number in the configuration */
interrupt FMSTR_SCI_TX_INTERRUPT

void FMSTR_Isr2(void)
{
#if FMSTR_LONG_INTR || FMSTR_SHORT_INTR

    /* process serial interface */
#if FMSTR_USE_SCI
    FMSTR_ProcessSCI(); 
#endif
    
#endif
}

#endif

/**************************************************************************//*!
*
* @brief    The "memcpy" used internally in FreeMASTER driver
*
* @param    nDestAddr - destination memory address
* @param    nSrcAddr  - source memory address
* @param    nSize     - memory size (always in bytes)
*
******************************************************************************/

void FMSTR_CopyMemory(FMSTR_ADDR nDestAddr, FMSTR_ADDR nSrcAddr, FMSTR_SIZE8 nSize)
{
    FMSTR_U8* ps = (FMSTR_U8*) nSrcAddr;
    FMSTR_U8* pd = (FMSTR_U8*) nDestAddr;
    
    while(nSize--)
        *pd++ = *ps++;
}

/**************************************************************************//*!
*
* @brief  Write-into the communication buffer memory
*
* @param  pDestBuff - pointer to destination memory in communication buffer
* @param  nSrcAddr  - source memory address
* @param  nSize     - buffer size (always in bytes)
*
* @return This function returns a pointer to next byte in comm. buffer
*
******************************************************************************/

FMSTR_BPTR FMSTR_CopyToBuffer(FMSTR_BPTR pDestBuff, FMSTR_ADDR nSrcAddr, FMSTR_SIZE8 nSize)
{
    FMSTR_U8* ps = (FMSTR_U8*) nSrcAddr;
    FMSTR_U8* pd = (FMSTR_U8*) pDestBuff;
    
    while(nSize--)
        *pd++ = *ps++;
        
    return (FMSTR_BPTR) pd;
}

/**************************************************************************//*!
*
* @brief  Read-out memory from communication buffer
*
* @param  nDestAddr - destination memory address
* @param  pSrcBuff  - pointer to source memory in communication buffer
* @param  nSize     - buffer size (always in bytes)
*
* @return This function returns a pointer to next byte in comm. buffer
*
******************************************************************************/

FMSTR_BPTR FMSTR_CopyFromBuffer(FMSTR_ADDR nDestAddr, FMSTR_BPTR pSrcBuff, FMSTR_SIZE8 nSize)
{
    FMSTR_U8* ps = (FMSTR_U8*) pSrcBuff;
    FMSTR_U8* pd = (FMSTR_U8*) nDestAddr;
    
    while(nSize--)
        *pd++ = *ps++;
        
    return (FMSTR_BPTR) ps;
}


/**************************************************************************//*!
*
* @brief  Read-out memory from communication buffer, perform AND-masking
*
* @param  nDestAddr - destination memory address
* @param  pSrcBuff  - source memory in communication buffer, mask follows data
* @param  nSize     - buffer size (always in bytes)
*
******************************************************************************/

void FMSTR_CopyFromBufferWithMask(FMSTR_ADDR nDestAddr, FMSTR_BPTR pSrcBuff, FMSTR_SIZE8 nSize)
{
    FMSTR_U8* ps = (FMSTR_U8*) pSrcBuff;
    FMSTR_U8* pd = (FMSTR_U8*) nDestAddr;
    FMSTR_U8* pm = ps + nSize;
    FMSTR_U8 mask, stmp, dtmp;
    
    while(nSize--) 
    {
        mask = *pm++;
        stmp = *ps++;
        dtmp = *pd;
        
        /* perform AND-masking */
        stmp = (stmp & mask) | (dtmp & ~mask);

        /* put the result back */
        *pd++ = stmp;
    }
}

