/******************************************************************************
* 
* Copyright (c) 2009 Freescale Semiconductor;
* All Rights Reserved                       
*
*******************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
***************************************************************************//*!
*
* $File Name:	control.c$
*
* $Author:		r30322$
* 
* $Version:		1.0.10.0$
* 
* $Date:		Jan-24-2010$
* 
* $Brief: 		control algorithms
*
* Target:		MC9S08MP16 device
*
*******************************************************************************/
#include "intrinsic_math.h" /* include arithmetics header file */
#include "types.h"          /* include type definitions */
#include "control.h"        /* include control algorithms definitions */


signed int CTR_ControllerPI16 (signed int error, CTR_sPIparams * pHandle)
{ 
    signed char proportionalGainShift,integralGainShift;
    unsigned char proportionalGain,integralGain;
    signed int lowerLimit,upperLimit;
    signed int temp, proportionalPortion;
    signed long integralPortion;
    SWord32HL16 integralPortionK_1;

    proportionalGain = pHandle -> propGain;
    proportionalGainShift = pHandle -> propGainShift;
    integralGain = pHandle -> intGain;
    integralGainShift = pHandle -> intGainShift;
    integralPortionK_1.HL_W16.H = pHandle -> integralPortionK_1_32.HL_W16.H;
    integralPortionK_1.HL_W16.L = pHandle -> integralPortionK_1_32.HL_W16.L;
    lowerLimit = pHandle -> lowerLimit;
    upperLimit = pHandle -> upperLimit;
    
    if (error >= 0)
    {
        /* Positive difference */
        /* Proportional portion */
        proportionalPortion = (signed int)umul16_8to16((unsigned int)error,proportionalGain);        
        if (proportionalGainShift >= 0)
        {
            proportionalPortion = shl16_u8to16(proportionalPortion,proportionalGainShift);
        } 
        else 
        { 
            proportionalPortion = shr16_u8to16(proportionalPortion, (unsigned char)-proportionalGainShift); 
        }
        /* Integral portion */
        integralPortion = (signed long)umul16_8to32((unsigned int)error,integralGain);
        
        if (integralGainShift > 0)
        {
            integralPortion = shl32_u8to32(integralPortion,integralGainShift);
        } 
        else 
        { 
            if(integralGainShift != 0)
            {
                integralPortion = shr32_u8to32(integralPortion, (unsigned char)(-integralGainShift));
            }
        }
        integralPortionK_1.S32 = add32(integralPortionK_1.S32, integralPortion);        
    }
    else  
    {
        /* Negative difference */
        error   = neg16(error);
        
                /* Proportional portion */
        proportionalPortion = (signed int)umul16_8to16((unsigned int)error,proportionalGain);
        if (proportionalGainShift >= 0)
        {
            proportionalPortion = - shl16_u8to16(proportionalPortion,proportionalGainShift);
        } 
        else 
        { 
            proportionalPortion = - shr16_u8to16(proportionalPortion, (unsigned char)(-proportionalGainShift));
        }
        
                /* Integral portion */
        integralPortion = (signed long)umul16_8to32((unsigned int)error,integralGain);
        if (integralGainShift > 0)
        {
            integralPortion = shl32_u8to32(integralPortion,integralGainShift);
        } 
        else 
        {
            if(integralGainShift != 0)
            {
                integralPortion = shr32_u8to32(integralPortion, (unsigned char)(-integralGainShift));
            } 
        }
        integralPortionK_1.S32 = sub32(integralPortionK_1.S32, integralPortion);
    }
        
    /* Integral portion limitation */
    if(integralPortionK_1.HL_W16.H < lowerLimit) 
    {
        integralPortionK_1.HL_W16.H = lowerLimit;
        integralPortionK_1.HL_W16.L = 0; 
    } 
    else if(integralPortionK_1.HL_W16.H > upperLimit) 
    {
        integralPortionK_1.HL_W16.H = upperLimit;
        integralPortionK_1.HL_W16.L = 0;      
    }
    /* Store integralPortionK_1 to the structure */
    pHandle -> integralPortionK_1_32.HL_W16.H = integralPortionK_1.HL_W16.H;
    pHandle -> integralPortionK_1_32.HL_W16.L = integralPortionK_1.HL_W16.L;
    /* Calculate summation of integral portion and proportional portion */ 
    temp = add16(integralPortionK_1.HL_W16.H,proportionalPortion);
	

    /* Output limitation */
    if(temp > upperLimit) 
    {
        temp = upperLimit;    
    } 
    else if(temp < lowerLimit) 
    {
        temp = lowerLimit; 
    }
    return temp;  
}

signed int CTR_ControllerPI8 (signed char error, CTR_sPIparams8 * pHandle)
{ 
    signed char proportionalGainShift,integralGainShift;
    unsigned char proportionalGain,integralGain;
    signed int lowerLimit,upperLimit;
    signed int temp, proportionalPortion;
    signed int integralPortion, integralPortionK_1;

    proportionalGain = pHandle -> propGain;
    proportionalGainShift = pHandle -> propGainShift;
    integralGain = pHandle -> intGain;
    integralGainShift = pHandle -> intGainShift;
    integralPortionK_1 = pHandle -> integralPortionK_1;
    lowerLimit = pHandle -> lowerLimit;
    upperLimit = pHandle -> upperLimit;
    
    if (error >= 0)
    {
        /* Positive difference */
        /* Proportional portion */
        proportionalPortion = (signed int)umul8_8to16((unsigned char)error,proportionalGain);        
        if (proportionalGainShift >= 0)
        {
            proportionalPortion = shl16_u8to16(proportionalPortion,proportionalGainShift);
        } 
        else 
        { 
            proportionalPortion = shr16_u8to16(proportionalPortion, (unsigned char)-proportionalGainShift);
        }
        
        /* Integral portion */
        integralPortion = (signed int)umul8_8to16(error,integralGain);
        
        if (integralGainShift > 0)
        {
            integralPortion = shl16_u8to16(integralPortion,integralGainShift);
        } 
        else 
        { 
            if(integralGainShift != 0)
            {
                integralPortion = shr16_u8to16(integralPortion, (unsigned char)(-integralGainShift));
            }
        }     
    }
    else  
    {
        /* Negative difference */
        error   = neg8(error);
                /* Proportional portion */
        proportionalPortion = (signed int)umul16_8to16((unsigned char)error,proportionalGain);
        if (proportionalGainShift >= 0)
        {
            proportionalPortion = - shl16_u8to16(proportionalPortion,proportionalGainShift);
        } 
        else 
        { 
            proportionalPortion = - shr16_u8to16(proportionalPortion, (unsigned char)(-proportionalGainShift));
            //Debug 09.09.24 proportionalPortion = - ((signed int)proportionalPortion>>(unsigned char)(-proportionalGainShift));
        }
        
                /* Integral portion */
        integralPortion = (signed int)umul16_8to16((unsigned char)error,integralGain);
        
        if (integralGainShift > 0)
        {
            integralPortion = -shl16_u8to16(integralPortion,integralGainShift);
        } 
        else 
        {
            if(integralGainShift != 0)
            {
                integralPortion = - shr16_u8to16(integralPortion, (unsigned char)(-integralGainShift));
            } 
            else
            {
                integralPortion = - integralPortion;
            }
        }
    }
    integralPortionK_1 = add16(integralPortionK_1, integralPortion);
        
    /* Integral portion limitation */
    if(integralPortionK_1 > upperLimit) 
    {
        integralPortionK_1 = upperLimit;     
    } 
    else if(integralPortionK_1 < lowerLimit) 
    {
        integralPortionK_1 = lowerLimit;
    }    
    /* Store integralPortionK_1 to the structure */
    pHandle -> integralPortionK_1 = integralPortionK_1; 
        
    /* Calculate summation of integral portion and proportional portion */ 
    temp = add16(integralPortionK_1,proportionalPortion);

    /* Output limitation */
    if(temp > upperLimit) 
    {
        temp = upperLimit;    
    } 
    else if(temp < lowerLimit) 
    {
        temp = lowerLimit; 
    }
    
    return temp;  
}


