/*****************************************************************************
* BeeApp.c
*
* The wireless UART application is a serial cable replacement program. It
* allows applications which have traditionally communicated through RS232 to
* work over a ZigBee connection (with a few caveats).
*
* This application can operated at 1200,2400,4800,9600,19200,38400 baud.
*
* This application uses the common ASL user interface to
*
* Copyright (c) 2008, Freescale, Inc. All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
* USER INTERFACE
* --------------
*
* Like most BeeStack sample applications, this application uses the the common
* ASL user interface. The ASL interface has two "modes" to allow the keys, LEDs
* and (optional on NCB) the LCD display to be used for multiple purposes.
* Configuration mode includes commands such as starting and leaving the
* network, whereas application mode includes application specific functions
* and display.
*
* Each key (aka switch) can be pressed for a short duration (short press) or
* long duration of about 2+ seconds (long press). Examples include SW1 or LSW3.
*
* Application specific code can be found here. Code common among applications
* can be found in ASL_UserInterface.c
*
* Config Mode:
* SW1  - Form (Zc) / join (Zr / Zed) network with previous configuration
* SW2  - Toggle Permit Join (ZC/ZR only)
* SW3  - Use end-device-bind to bind to another node (e.g. switch to light)
* SW4  - Choose channel (default is 25). Only functional when NOT on network.
* LSW1 - Toggle display/keyboard mode (Config and Application)
* LSW2 - Leave network
* LSW3 - Remove all binding
* LSW4 - Form (Zc) / join (Zr / Zed) network with new configuration
*
* WirelessUART Application Mode:
* SW1  - Toggle char vs block mode
* SW2  - Toggle LoopBack mode on/off
* SW3  - (debug only) Toggle reliable vs. unreliable mode
* SW4  - Toggle through baud rates 1200,2400,9600,19200,38400
* LSW1 - Toggle display/keyboard mode (Config and Application)
* LSW2 - Not used
* LSW3 - Add Group
* LSW4 - Not used
*
* In run mode, the four LEDs are usually used to display status information:
* LED1 - On if in char mode; off if in block mode
* LED2 - On if in LoopBack mode, off otherwise
* LED3 - Toggles when data is sent to the remote ZigBee node
* LED4 - Toggles when data is sent to the UART
*
* When SW4 is used to select a baud rate, the LEDs will temporarily change
* to display the new baud rate as a binary integer, 1..N.
*
*****************************************************************************/

#include "BeeStack_Globals.h"
#include "BeeStackConfiguration.h"
#include "BeeStackParameters.h"
#include "AppZdoInterface.h"
#include "TS_Interface.h"
#include "TMR_Interface.h"
#include "AppAfInterface.h"
#include "FunctionLib.h"
#include "PublicConst.h"
#include "keyboard.h"
#include "Display.h"
#include "led.h"
#include "EndPointConfig.h"
#include "BeeApp.h"
#include "ZdoApsInterface.h"
#include "BeeAppInit.h"
#include "NVM_Interface.h"
#include "BeeStackInterface.h"
#include "HaProfile.h"
#include "ZdpManager.h"
#include "ASL_ZdpInterface.h"
#include "ASL_UserInterface.h"
#include "ASL_ZCLInterface.h"
#include "UART_Interface.h"

/******************************************************************************
*******************************************************************************
* Public memory declarations
*******************************************************************************
******************************************************************************/

extern anchor_t gAppDataConfirmQueue;
extern anchor_t gAppDataIndicationQueue;

/******************************************************************************
*******************************************************************************
* Private macros
*******************************************************************************
******************************************************************************/

#ifndef NumberOfElements
/* Number of elements in an array. */
#define NumberOfElements(array)     ((sizeof(array) / (sizeof(array[0]))))
#endif

/* Maximum size of a datagram sent from one side of the wireless uart to the */
/* other side. */
//#define mMaxDatagramLen_c   ApsmeGetMaxAsduLength(0)
#define mMaxDatagramLen_c   90

#define mDataClusterId_c    0x1200      /* Data. */
#define mAckClusterId_c     0x1201      /* Acknowledgements. */

/* Duration of the baud rate display. */
#define mBaudRateDisplayDuration_c          TmrMilliseconds(2000)

/* When a datagram is sent OTA, there's a chance that the destination may */
/* never respond. In reliable mode, a timer is used to notice the lack of */
/* response and to retry. It may or may not make sense to limit the number */
/* of retries. For now, keep it simple, and retry forever. */
/* Since polling end devices may be on a 4 second cycle, set the timer to */
/* something longer than 4 seconds. */
#define mTxZigBeeDataRetryTimerDuration_c   TmrSeconds(5)

/* In block mode, the last block will usually not be a full datagram. */
/* Whenever a byte is received from the UART, a timer is started. */
/* If it expires before it is reset by the arrival of another byte, the */
/* datagram is sent, even if it is not full. Note that this must be long */
/* enough to allow for a full byte transmission time at the slowest baud */
/* rate. */
#define mUartRxTimeout_c                    TmrMilliseconds(10)
//#define mUartRxTimeout_c                    TmrSeconds(3)

#define mTxZigBeeThrottleDuration_c         TmrMilliseconds(5)

/******************************************************************************
*******************************************************************************
* Private type definitions
*******************************************************************************
******************************************************************************/

/* Events for the user application, to be used for the BeeAppUpdateDevice */
enum {
	gOnEvent_c,
	gOffEvent_c,
	gToggleEvent_c
};

/* Data sent from one side ot the wireless uart to the other side. */
typedef uint8_t sequenceNumber_t;

#ifdef __IAR_SYSTEMS_ICC__
#pragma pack(1)
#endif
typedef struct structuredDatagram_tag {
  sequenceNumber_t sequenceNumber;
  unsigned char payload[mMaxDatagramLen_c - sizeof(sequenceNumber_t)];
} structuredDatagram_t;

/* A full sized datagram. */
typedef union datagram_tag {
  unsigned char raw[sizeof(structuredDatagram_t)];
  structuredDatagram_t structured;
} datagram_t;

typedef enum {
  mBufStatusFree_c = 0,                 /* Not being used. Must be == 0. */
  mBufStatusInProgress_c,               /* Being filled/emptied. */
  mBufStatusReadyToSend_c,
  mBufStatusBusy_c                      /* Waiting for an Rx/Tx to complete. */
} bufferStatus_t;

typedef struct buffer_tag {
  bufferStatus_t status;
  index_t len;
  datagram_t data;
} buffer_t;

typedef struct structuredAck_tag {
  sequenceNumber_t sequenceNumber;
  bool_t ack;                           /* T = ack, F = nak. */
} structuredAck_t;

/* A very small datagram, used only for ACKs. */
typedef union ackDatagram_tag {
  unsigned char raw[sizeof(structuredAck_t)];
  structuredAck_t structured;
} ackDatagram_t;

/* ACKs are small, and fixed size. */
typedef struct ackBuffer_tag {
  bufferStatus_t status;
  ackDatagram_t data;
} ackBuffer_t;
#ifdef __IAR_SYSTEMS_ICC__
#pragma pack()
#endif

/******************************************************************************
*******************************************************************************
* Private memory declarations
*******************************************************************************
******************************************************************************/

/* Debug and performance instrumentation. */

#if gBeeAppDebug_d
static bool_t mSendDebugPacketsFlag = FALSE;
static uint16_t mSendDebugPacketsCount = 0;

static bool_t mBreakOnCount = FALSE;

static uint16_t mRxZigBeeACKCount = 0;
static uint16_t mRxZigBeeACKBreak = 0;

static uint16_t mRxZigBeeNAKCount = 0;
static uint16_t mRxZigBeeNAKBreak = 0;

static uint16_t mRxZigBeeDataCount = 0;
static uint16_t mRxZigBeeDataBreak = 0;

static uint16_t mRxZigBeeAcceptedCount = 0;
static uint16_t mRxZigBeeAcceptedBreak = 0;

static uint16_t mRxZigBeeNotAcceptedCount = 0;
static uint16_t mRxZigBeeNotAcceptedBreak = 0;

static uint16_t mTxZigBeeACKCount = 0;
static uint16_t mTxZigBeeACKBreak = 0;

static uint16_t mTxZigBeeDataCount = 0;
static uint16_t mTxZigBeeDataBreak = 0;

/* Discard UART data without sending it to the UART if this is FALSE. */
static bool_t mTxUartDataFlag = TRUE;
#endif                                  /* #if gBeeAppDebug_d */

/**************************************/

/* Timer IDs. */

static tmrTimerID_t mUartRxTimerID;
//static tmrTimerID_t mBaudRateDisplayTimerID;
static tmrTimerID_t mTxZigBeeDataRetryTimerID;
static tmrTimerID_t mTxZigBeeThrottleTimerID;

/* This flag will be TRUE when it has been at least a minimum time since the */
/* last ZigBee Tx. */
static bool_t mTxZigBeeThrottleFlag = TRUE;

/**************************************/

/* Buffers. There are four data transfer points: to and from the UART, and */
/* to and from the remote node via ZigBee. The UART driver already has a */
/* buffer for data received from the serial port, so we only need buffers */
/* for three of the four transfer types. */

/* Data to be sent to the remote ZigBee node. */
static sequenceNumber_t curZbTxSequenceNumber = 0;
static buffer_t zbTxDataBuffer;

/* ACK or NAK to be sent to the remote ZigBee node. */
static ackBuffer_t zbTxAckBuffer;

/* Data to be sent to the UART. */
static buffer_t uartTxDataBuffer;

/**************************************/

/* There are three different mode settings:
 *      char vs. block mode,
 *      reliable vs. unreliable mode, and
 *      loopback mode.
 *
 * Char mode sends data received from the serial port immediately, without
 * waiting to accumulate a full datagram. Char mode is intended mostly for
 * keyboard repeaters.
 *
 * Block mode sends full datagrams, with a timeout to send a final, partial
 * datagram. Block mode is intended for file transfers.
 *
 * Reliable mode checks for ACKs from the remote node, and resends datagrams
 * if necessary. Reliable mode is intended for any data transfer where
 * reliablity is important.
 *
 * Unreliable mode ignores ACKs from the remote node, and never resends
 * datagrams. Unreliable mode is intended for data transfers like encoded
 * voice, where it is better to drop data than to delay it with retries.
 *
 * LoopBack mode sends data received from the UART back out the UART, and
 * sends data received OTA back out OTA.
 */

static bool_t mInCharMode = FALSE;       /* T = char mode, F = block mode. */
static bool_t mInReliableMode = FALSE;   /* T = reliable, F = unreliable. */
static bool_t mInLoopBackMode = FALSE;  /* T = loopback mode, F otherwise. */

/**************************************/

/* Table of supported baud rates. The default is maBaudRateTable[0]. */
//static index_t mCurBaudRateIndex = 5;
//static UartBaudRate_t const maBaudRateTable[] = {
//  gUARTBaudRate1200_c,                  /* 0 */
//  gUARTBaudRate2400_c,                  /* 1 */
//  gUARTBaudRate4800_c,                  /* 2 */
//  gUARTBaudRate9600_c,                  /* 3 */
//  gUARTBaudRate19200_c,                 /* 4 */
//  gUARTBaudRate38400_c                  /* 5 */
//};

/**************************************/

/* The UI sometimes overrides the normal LED display with another value for */
/* a short time, and then returns to the previous display. The normal state */
/* of the LEDs is kept in shadow registers. These can be changed at any time, */
/* even if when the normal display is overridden, without losing their state. */
static bool_t mIsIntegerDisplayedOnLEDs = FALSE;
static struct ledState_tag {
  LED_t number;
  LedState_t state;
} maLedShadowStateTable[] = {
  {LED1, gLedOff_c},
  {LED2, gLedOff_c},
  {LED3, gLedOff_c},
  {LED4, gLedOff_c},
};

/******************************************************************************
*******************************************************************************
* Private prototypes
*******************************************************************************
******************************************************************************/

/* ProcessConfirmData() is only needed if if APS confirms are used. */
static void ProcessConfirmData(void);
#if gInterPanCommunicationEnabled_c
void BeeAppInterPanDataConfirm(void);
void BeeAppInterPanDataIndication(void);
#endif 

//static void DisplayLedInteger(index_t value);
//static void FlushAllBuffers(void);
//static void RestoreLedShadowState(tmrTimerID_t timerID);
static void RxUartData(void);
static void RxZigBeeData(void);
//static void SetAllLeds(index_t value);
static void SetOneLed(LED_t ledNumber, LedState_t state);
static bool_t SendUartDataToUart(unsigned char byte);
static bool_t SendUartDataToZigBee(unsigned char byte);
static bool_t SendZigBeeDataToUart(apsdeToAfMessage_t *pMsg);
static void UartRxCallBack(void);
static void UartRxTimeout(tmrTimerID_t timerID);
static void UartTxCallBack(unsigned char const *pBuf);
static void TxUartData(void);
static void TxZigBeeAck(void);
static void TxZigBeeData(void);
static void TxZigBeeDataRetryTimerCallBack(tmrTimerID_t timerID);
static void TxZigBeeThrottleTimerCallBack(tmrTimerID_t timerID);

#if gBeeAppDebug_d
static void TxZigBeeDebugPacket(void);
static void DebugBreakpointTarget(void);
#endif

#if gFrequencyAgilityCapability_d && (!gEndDevCapability_d)
void BeeAppZdpCallBack(zdpToAppMessage_t *pMsg, zbCounter_t counter);
#endif


/******************************************************************************
*******************************************************************************
* Public memory definitions
*******************************************************************************
******************************************************************************/

zbEndPoint_t appEndPoint;

/* This data set contains app layer variables to be preserved across resets */
NvDataItemDescription_t const gaNvAppDataSet[] = {
  gAPS_DATA_SET_FOR_NVM,    /* APS layer NVM data */
  {&gZclCommonAttr,         sizeof(zclCommonAttr_t)},       /* scenes, location, etc... */
  {&gAslData,               sizeof(ASL_Data_t)},            /* state of ASL */
  /* insert any user data for NVM here.... */
  {NULL, 0}       /* Required end-of-table marker. */
};

/******************************************************************************
*******************************************************************************
* Public functions
*******************************************************************************
******************************************************************************/

/* Initialize the application. */
void BeeAppInit(void) {
  index_t i;

  /* Register the application endpoint(s), so we receive callbacks. */
  for (i = 0; i < gNum_EndPoints_c; ++i) {
    (void) AF_RegisterEndPoint(endPointList[i].pEndpointDesc);
  }

  /* Start with all LEDs off */
  ASL_InitUserInterface("Wireless UART");
  SetOneLed(LED1, (mInCharMode ? gLedOn_c : gLedOff_c));
  SetOneLed(LED2, (mInLoopBackMode ? gLedOn_c : gLedOff_c));

  //mBaudRateDisplayTimerID = TMR_AllocateTimer();
  mUartRxTimerID = TMR_AllocateTimer();
  mTxZigBeeDataRetryTimerID = TMR_AllocateTimer();
  mTxZigBeeThrottleTimerID = TMR_AllocateTimer();

  gSendingNwkData.gAddressMode = gZbAddrModeIndirect_c;

  /* Only one endpoint is needed for this application. */
  appEndPoint = endPointList[0].pEndpointDesc->pSimpleDesc->endPoint;

#ifdef gHostApp_d
  Uart1_SetRxCallBack(UartRxCallBack);
#else
  UartX_SetRxCallBack(UartRxCallBack);
#endif

#if gFrequencyAgilityCapability_d && (!gEndDevCapability_d)
  /* register to get ZDP responses */
  Zdp_AppRegisterCallBack(BeeAppZdpCallBack);
  /* Get the statemachine for FA into start state. */
  FA_StateMachineInit();
#endif
  
  BeeAppHandleKeys(gKBD_EventSW1_c);  // Create the network after the startup
}                                       /* BeeAppInit() */

/****************************************************************************/

/* Handles all key events for this device. The UI has two modes:
 *  config-mode, for setting up the network, joining, forming, etc., and
 *  run-mode, for controlling the application.
 */
void BeeAppHandleKeys(key_event_t events) {

  if (gmUserInterfaceMode == gConfigureMode_c) {
    /* Use the standard handlers for config mode keys. */
  	ASL_HandleKeys(events);
    return;
  }

  /* Handle keys in Application Mode. */

  switch (events) {

    /* Toggle char vs block mode. */
//    case gKBD_EventSW1_c:
//      mInCharMode = !mInCharMode;
//      SetOneLed(LED1, mInCharMode ? gLedOn_c : gLedOff_c);
//      break;
//
//      /* Toggle LoopBack mode. */
//    case gKBD_EventSW2_c:
//      mInLoopBackMode = !mInLoopBackMode;
//      SetOneLed(LED2, mInLoopBackMode ? gLedOn_c : gLedOff_c);
//
//      /* Whenever LoopBack mode changes, flush all of the in-progress buffers. */
//      FlushAllBuffers();
//      break;

//#if gBeeAppDebug_d
//    case gKBD_EventSW3_c:
//      mInReliableMode = !mInReliableMode;
//      DisplayLedInteger(mInReliableMode ? 0x0F : 0x0B);
//      break;
//#endif
//
//      /* Toggle through baud rates 1200, 2400, 9600, 19200, 38400. */
//    case gKBD_EventSW4_c:

//      if (++mCurBaudRateIndex >= NumberOfElements(maBaudRateTable)) {
//        mCurBaudRateIndex = 0;
//      }

//#ifdef gHostApp_d
//      Uart1_SetBaud(maBaudRateTable[mCurBaudRateIndex]);
//#else
//      UartX_SetBaud(maBaudRateTable[mCurBaudRateIndex]);
//#endif
//      DisplayLedInteger(mCurBaudRateIndex + 1);
//      break;

    default:
      ASL_HandleKeys(events);
      break;
  }                                     /* switch (events) */
}                                       /* BeeAppHandleKeys() */

/****************************************************************************/

/* Application task. Responds to events for the application. */
void BeeAppTask(event_t events) {
  /* The retry events should be handled first, to improve the chances of */
  /* freeing a buffer before the Rx/Tx events need a buffer. */
  if (events & mAppRetryZigBeeAckTx_c)      { TxZigBeeAck(); }
  if (events & mAppRetryUartDataTx_c)       { TxUartData(); }
  if (events & mAppRetryZigBeeDataTx_c)     { TxZigBeeData(); }

  if (events & gAppEvtDataConfirm_c)        { ProcessConfirmData(); }

  if (events & gAppEvtDataIndication_c)     { RxZigBeeData(); }
  if (events & mAppRxFromUart_c)            { RxUartData(); }

  if (events & gAppEvtAddGroup_c)           { ASL_ZclAddGroupHandler(); }
  if (events & gAppEvtSyncReq_c)            { ASL_Nlme_Sync_req(FALSE); }
#if gInterPanCommunicationEnabled_c
    /* received one or more data confirms */
  if(events & gInterPanAppEvtDataConfirm_c)
    BeeAppInterPanDataConfirm();

  /* received one or more data indications */
  if(events & gInterPanAppEvtDataIndication_c)
    BeeAppInterPanDataIndication();
#endif 

#if gBeeAppDebug_d
  if (events & mAppSendZigBeeDebugPacket_c) { TxZigBeeDebugPacket(); }
#endif
}                                       /* BeeAppTask() */

/****************************************************************************/

/* Update device in a Application Mode. */
/* The events on this functions are from the user application. */
/* Note: There should be events marking: */
/*  successful binding with a remote device, and */
/*  unbinding initiated by a remote device, */
/* The wireless uart app initialzes the uart driver when there is a remote */
/* device to exchange data with, and stops the uart driver when there is no */
/* longer a far end. Note also that there needs to be some timeout logic to */
/* decide that the far end has disappeared without notice. */

void BeeAppUpdateDevice
  (
  zbEndPoint_t endPoint,    /* IN: endpoint update happend on */
  zclUIEvent_t event,        /* IN: state to update */
  zclAttrId_t attrId, 
  zbClusterId_t clusterId, 
  void *pData
  )
{
 (void) attrId;
 (void) clusterId;
 (void) pData;
  ASL_UpdateDevice(endPoint, event);
}                                       /* BeeAppUpdateDevice() */

/******************************************************************************
*******************************************************************************
* Private functions
*******************************************************************************
******************************************************************************/

#if gBeeAppDebug_d
static void DebugBreakpointTarget(void) {
  mBreakOnCount = FALSE;
}                                       /* DebugBreakpointTarget() */
#endif

/****************************************************************************/

/* The LEDs normally display status information, but sometimes the UI wants
 * to display something in response to a keypress, like a baud rate change.
 * This display takes precedence over the status indications for a short while.
 *
 * Note that calling this before the timer has expired from a previous call
 * will harmlessly overwrite the previous display.
 * The value is assumed to be 0..15. Displaying 0 (all lights off) may
 * be confusing to the user. If the maximum value is < 15, it is better
 * to pass in an integer in the range 1..15.
 */
//static void DisplayLedInteger(index_t value) {
//  mIsIntegerDisplayedOnLEDs = TRUE;
//  SetAllLeds(value);
//  TMR_StartSingleShotTimer(mBaudRateDisplayTimerID,
//                           mBaudRateDisplayDuration_c,
//                           RestoreLedShadowState);
//}                                       /* DisplayLedInteger() */

/****************************************************************************/

/* Transmit all buffers that are marked as in progress. */
//static void FlushAllBuffers(void) {
//  if (uartTxDataBuffer.status == mBufStatusInProgress_c) {
//    uartTxDataBuffer.status = mBufStatusReadyToSend_c;
//  }
//  TxUartData();
//
//  if (zbTxAckBuffer.status == mBufStatusInProgress_c) {
//    zbTxAckBuffer.status = mBufStatusReadyToSend_c;
//    }
//  TxZigBeeAck();
//
//  if (zbTxDataBuffer.status == mBufStatusInProgress_c) {
//    zbTxDataBuffer.status = mBufStatusReadyToSend_c;
//  }
//  TxZigBeeData();
//}                                       /* FlushAllBuffers() */

/****************************************************************************/

/* Process incoming ZigBee over-the-air data confirms. */
/* The Wireless UART uses application-level ACKs and NAKs, not APS confirms. */
/* Any APS confirm that reaches this routine is not ours. */
static void ProcessConfirmData(void) {
  apsdeToAfMessage_t *pMsg;
  zbApsdeDataConfirm_t *pConfirm;

  while (MSG_Pending(&gAppDataConfirmQueue)) {

    pMsg = MSG_DeQueue(&gAppDataConfirmQueue);
    pConfirm = &(pMsg->msgData.dataConfirm);

    /* Action taken when confirmation is received. */
    if (pConfirm->status == gZbSuccess_c) {
      BeeAppUpdateDevice(appEndPoint, gToggleEvent_c,0,0,NULL);
    }

    /* Free memory allocated in Call Back function */
    MSG_Free(pMsg);
  }
}                                       /* ProcessConfirmData() */

/****************************************************************************/

/* Revert the LEDs to displaying their normal (shadow) state. */
//static void RestoreLedShadowState(tmrTimerID_t timerID) {
//  index_t i;
//
//  (void) timerID;
//
//  for (i = 0; i < NumberOfElements(maLedShadowStateTable); ++i) {
//    ASL_SetLed(maLedShadowStateTable[i].number, maLedShadowStateTable[i].state);
//  }
//
//  mIsIntegerDisplayedOnLEDs = FALSE;
//}                                       /* RestoreLedShadowState() */

/****************************************************************************/

/* Handle bytes arriving from the UART driver. */
static void RxUartData(void) {
  unsigned char byte;

  /* Get bytes from the UART driver until it doesn't have any more. */
#ifdef gHostApp_d
  while (Uart1_GetByteFromRxBuffer(&byte)) {
#else
  while (UartX_GetByteFromRxBuffer(&byte)) {
#endif
    if (mInLoopBackMode) {
      if (!SendUartDataToUart(byte)) {
#ifdef gHostApp_d
        Uart1_UngetByte(byte);          /* ReQueue the last char. */
#else
        UartX_UngetByte(byte);          /* ReQueue the last char. */
#endif
        break;
      }
    } else {
      if (!SendUartDataToZigBee(byte)) {
#ifdef gHostApp_d
        Uart1_UngetByte(byte);          /* ReQueue the last char. */
#else
        UartX_UngetByte(byte);          /* ReQueue the last char. */
#endif
        break;
      }
    }
  }                                     /* while ... */

  /* The current datagram may contain unsent data. Start a timer. If */
  /* the timer expires before another byte is received, transmit the */
  /* datagram to the remote node, even if it isn't full. */
  TMR_StartSingleShotTimer(mUartRxTimerID, mUartRxTimeout_c, UartRxTimeout);
}                                       /* RxUartData() */

/****************************************************************************/

/* Process incoming ZigBee over-the-air messages. */
/* Datagrams arrive here with cluster id == mDataClusterId_c. */
/* ACKs/NAKs from the remote node also arrive here, with cluster id == */
/* mAckClusterId_c. */
static void RxZigBeeData(void) {
  ackDatagram_t *pAckDatagram;
  datagram_t *pDatagram;
  zbApsdeDataIndication_t *pIndication;
  apsdeToAfMessage_t *pMsg;
  bool_t messageAccepted;
  sequenceNumber_t sequenceNumber;

  while (MSG_Pending(&gAppDataIndicationQueue)) {
    pMsg = MSG_DeQueue(&gAppDataIndicationQueue);

    pIndication = &(pMsg->msgData.dataIndication);

    if (   (pIndication->aClusterId[0] == (mDataClusterId_c & 0xFF))
        && (pIndication->aClusterId[1] == (mDataClusterId_c >> 8))) {

      /* Received data from the remote node. */
      /* If there is no buffer available to store it, discard it. There */
      /* are not enough message buffers in BeeStack to keep them around */
      /* for later. */
      pDatagram = (datagram_t *) pIndication->pAsdu;
      sequenceNumber = pDatagram->structured.sequenceNumber;
      messageAccepted = SendZigBeeDataToUart(pMsg);

#if gBeeAppDebug_d
    ++mRxZigBeeDataCount;
    if (messageAccepted) {
      ++mRxZigBeeAcceptedCount;
    } else {
      ++mRxZigBeeNotAcceptedCount;
    }

    if (mBreakOnCount) {
      if (   (mRxZigBeeDataBreak        && (mRxZigBeeDataCount        == mRxZigBeeDataBreak))
          || (mRxZigBeeAcceptedBreak    && (mRxZigBeeAcceptedCount    == mRxZigBeeAcceptedBreak))
          || (mRxZigBeeNotAcceptedBreak && (mRxZigBeeNotAcceptedCount == mRxZigBeeNotAcceptedBreak))) {
        DebugBreakpointTarget();
      }
    }
#endif

      /* Free the indication message buffer. Do this before trying to */
      /* send an acknoledgement, to insure that there is a message buffer */
      /* available for the ACK/NAK. */
      AF_FreeDataIndicationMsg(pMsg);

      /* Acknowledge receipt, ACK or NAK. */
      if (mInReliableMode) {
        zbTxAckBuffer.data.structured.sequenceNumber = sequenceNumber;
        zbTxAckBuffer.data.structured.ack = messageAccepted;
        TxZigBeeAck();
      }
    } else if (   (pIndication->aClusterId[0] == (mAckClusterId_c & 0xFF))
               && (pIndication->aClusterId[1] == (mAckClusterId_c >> 8))) {

#if gBeeAppDebug_d
      if (((ackDatagram_t *) pIndication->pAsdu)->structured.ack) {
        ++mRxZigBeeACKCount;
      } else {
        ++mRxZigBeeNAKCount;
      }

      if (mBreakOnCount) {
        if (   (mRxZigBeeACKBreak && (mRxZigBeeACKCount == mRxZigBeeACKBreak))
               || (mRxZigBeeNAKBreak && (mRxZigBeeNAKCount == mRxZigBeeNAKBreak))) {
          DebugBreakpointTarget();
        }
      }
#endif

      /* Received an ACK or NAK from the remote node. Ignore it if we're not */
      /* in reliable mode. */
      if (mInReliableMode) {
        /* The remote node responded, so we don't need a no-response timer. */
        TMR_StopTimer(mTxZigBeeDataRetryTimerID);

        /* If it was an ACK, mark the buffer as available. If it was a NAK, */
        /* send it again. */
        pAckDatagram = (ackDatagram_t *) pIndication->pAsdu;
        if (pAckDatagram->structured.ack) {
          zbTxDataBuffer.status = mBufStatusFree_c;
          zbTxDataBuffer.len = 0;
        } else {
          zbTxDataBuffer.status = mBufStatusReadyToSend_c;
          TxZigBeeData();
        }                               /* if (pAckDatagram-> ... else */

        /* Free the indication message. */
        AF_FreeDataIndicationMsg(pMsg);
      }                                 /* if (mInReliableMode) */
    } else {                            /* if (IsEqual2BytesInt( ... */
      /* Free the indication message. */
      AF_FreeDataIndicationMsg(pMsg);
    }
  }                                     /* while (MSG_Pending( ... */
}                                       /* RxZigBeeData() */

/****************************************************************************/

/* Send data received from the UART back out via the UART (loop back). */
/* Note that UART buffers don't need or want sequence numbers. */
/* Return TRUE if successful. */
static bool_t SendUartDataToUart(unsigned char byte) {
  if (   (uartTxDataBuffer.status != mBufStatusInProgress_c)
      && (uartTxDataBuffer.status != mBufStatusFree_c)) {
    return FALSE;
  }

  uartTxDataBuffer.status = mBufStatusInProgress_c;
  uartTxDataBuffer.data.structured.payload[uartTxDataBuffer.len++] = byte;
  if (mInCharMode
      || (uartTxDataBuffer.len >= sizeof(uartTxDataBuffer.data.structured.payload))) {
    uartTxDataBuffer.status = mBufStatusReadyToSend_c;
    TxUartData();
  }

  return TRUE;
}                                       /* SendUartDataToUart() */

/****************************************************************************/

/* Send data received from the UART out OTA. */
/* Return TRUE if successful. */
static bool_t SendUartDataToZigBee(unsigned char byte) {
  if (   (zbTxDataBuffer.status != mBufStatusInProgress_c)
      && (zbTxDataBuffer.status != mBufStatusFree_c)) {
    return FALSE;
  }

  zbTxDataBuffer.status = mBufStatusInProgress_c;
  zbTxDataBuffer.data.structured.payload[zbTxDataBuffer.len++] = byte;
  if (mInCharMode
      || (zbTxDataBuffer.len >= sizeof(zbTxDataBuffer.data.structured.payload))) {
    zbTxDataBuffer.status = mBufStatusReadyToSend_c;
    zbTxDataBuffer.data.structured.sequenceNumber = curZbTxSequenceNumber++;
    ++zbTxDataBuffer.len;                 /* Count the sequence number. */
    TxZigBeeData();
  }

  return TRUE;
}                                       /* SendUartDataToZigBee() */

/****************************************************************************/

/* Send data received OTA out through the UART. */
/* Note that the entire buffer received OTA is sent to the UART, all at */
/* once, with no blocking or deblocking. This is simpler: one ZigBee */
/* buffer == one UART buffer. */
/* Return TRUE if successful. */
static bool_t SendZigBeeDataToUart(apsdeToAfMessage_t *pMsg) {
  zbApsdeDataIndication_t *pIndication;

  if (uartTxDataBuffer.status != mBufStatusFree_c) {
    /* Double check as the UART might not always call the TX callback */
#ifdef gHostApp_d
    if (Uart1_IsTxActive())
#else
    if (UartX_IsTxActive())
#endif
      return FALSE;
    else
      uartTxDataBuffer.status = mBufStatusFree_c;
  }

  pIndication = &(pMsg->msgData.dataIndication);
  uartTxDataBuffer.status = mBufStatusReadyToSend_c;
  uartTxDataBuffer.len = pIndication->asduLength - sizeof(sequenceNumber_t);

  /* Don't send, or count, the sequence number we received. */
  FLib_MemCpy(uartTxDataBuffer.data.structured.payload,
              pIndication->pAsdu + sizeof(sequenceNumber_t),
              uartTxDataBuffer.len);

  TxUartData();
  return TRUE;
}                                       /* SendZigBeeDataToUart() */


/****************************************************************************/

/* Given a binary integer, set all of the LEDs to on or off accordingly. */
/* Does not handle toggling or flashing. Sets the actual LED state, not the */
/* shadow copy. */
//static void SetAllLeds(index_t value) {
//  ASL_SetLed(LED1, ((value & 0x01) ? gLedOn_c : gLedOff_c));
//  ASL_SetLed(LED2, ((value & 0x02) ? gLedOn_c : gLedOff_c));
//  ASL_SetLed(LED3, ((value & 0x04) ? gLedOn_c : gLedOff_c));
//  ASL_SetLed(LED4, ((value & 0x08) ? gLedOn_c : gLedOff_c));
//}                                       /* SetAllLeds() */

/****************************************************************************/

/* Change the shadow copy of the LED state. If an integer is being */
/* displayed, it takes priority, so this change will not be written to the */
/* LEDs yet. */
/* Otherwise, display the change on the LED. */
static void SetOneLed(LED_t ledNumber, LedState_t state) {
  index_t i;

  for (i = 0; i < NumberOfElements(maLedShadowStateTable); ++i) {
    if (maLedShadowStateTable[i].number == ledNumber) {
      maLedShadowStateTable[i].state = state;

      if (!mIsIntegerDisplayedOnLEDs) {
        ASL_SetLed(ledNumber, state);
      }
    }
  }
}                                       /* SetOneLed() */

/****************************************************************************/

/* Send the contents of the UART transmit buffer out via the UART. */
static void TxUartData(void) {

#if gBeeAppDebug_d
  /* Discard the buffer without transmitting it if FALSE. */
  if (!mTxUartDataFlag) {
    uartTxDataBuffer.status = mBufStatusFree_c;
  }
#endif

  if (uartTxDataBuffer.status != mBufStatusReadyToSend_c) {
    return;
  }

  /* Status will be set back to mBufStatusFree_c by UartTxCallBack(). */
  uartTxDataBuffer.status = mBufStatusBusy_c;

#ifdef gHostApp_d   
  if (!Uart1_Transmit(uartTxDataBuffer.data.structured.payload,
                      uartTxDataBuffer.len,
                      UartTxCallBack)) {
#else
  if (!UartX_Transmit(uartTxDataBuffer.data.structured.payload,
                      uartTxDataBuffer.len,
                      UartTxCallBack)) {
#endif

    /* The UART driver is out of buffer slots. Set an event to insure */
    /* this code gets run again next time the application task is called, */
    /* and give up for now. */
    uartTxDataBuffer.status = mBufStatusReadyToSend_c;
    TS_SendEvent(gAppTaskID, mAppRetryUartDataTx_c);
    return;
  }                                     /* if (!UartX_Transmit( ... */

  /* The send succeeded, or at least the UART driver accepted */
  /* the buffer to be sent. */
  SetOneLed(LED4, gLedToggle_c);
}                                       /* TxUartData() */

/****************************************************************************/

/* Let the other side know that a datagram has been received. */
static void TxZigBeeAck(void) {
  afAddrInfo_t afAddrInfo;

  if (!mTxZigBeeThrottleFlag) {
    TS_SendEvent(gAppTaskID, mAppRetryZigBeeAckTx_c);
    return;
  }

  BeeUtilZeroMemory(&afAddrInfo, sizeof(afAddrInfo));

  /* Don't care about afAddrInfo.dstAddr or afAddrInfo.dstEndPoint in */
  /* indirect mode. */
  /* For direct mode, dstAddr.aNwkAddr = 0 for Zc, 1 for Zr (temporary, */
  /* for testing). */
  /* afAddrInfo.dstAddr.aNwkAddr = 0; */
  /* afAddrInfo.dstEndPoint = 0; */

  afAddrInfo.aClusterId[0] = (mAckClusterId_c & 0xFF);
  afAddrInfo.aClusterId[1] = (mAckClusterId_c >> 8);
  afAddrInfo.srcEndPoint = appEndPoint;
  afAddrInfo.dstAddrMode = gZbAddrModeIndirect_c;

  /* Turn on security and leave it on. Harmless In a non-secure network. */
  afAddrInfo.txOptions = 0;
  afAddrInfo.radiusCounter = afDefaultRadius_c;

  if (AF_DataRequest(&afAddrInfo,
                     sizeof(zbTxAckBuffer.data.raw),
                     zbTxAckBuffer.data.raw,
                     NULL) != gZbSuccess_c) {
    /* The send failed. Set an event to insure this code gets */
    /* run next time the application task is called, and give */
    /* up for now. */
    TS_SendEvent(gAppTaskID, mAppRetryZigBeeAckTx_c);
    return;
  }                                     /* if (AF_DataRequest( ... */

#if gBeeAppDebug_d
  ++mTxZigBeeACKCount;
    if (mBreakOnCount) {
      if (mTxZigBeeACKBreak && (mTxZigBeeACKCount == mTxZigBeeACKBreak)) {
        DebugBreakpointTarget();
      }
    }
#endif

  /* Don't try to send too often. */
  mTxZigBeeThrottleFlag = FALSE;
  TMR_StartSingleShotTimer(mTxZigBeeThrottleTimerID,
                           mTxZigBeeThrottleDuration_c,
                           TxZigBeeThrottleTimerCallBack);
}                                       /* TxZigBeeAck() */

/****************************************************************************/

/* Send the contents of the ZigBee transmit buffer OTA to the remote node. */
void TxZigBeeData(void) {
  afAddrInfo_t afAddrInfo;

  if (!mTxZigBeeThrottleFlag) {
    TS_SendEvent(gAppTaskID, mAppRetryZigBeeDataTx_c);
    return;
  }

  /* Don't need to be interrupted just to come back here again. */
  TMR_StopTimer(mTxZigBeeDataRetryTimerID);

  if (zbTxDataBuffer.status != mBufStatusReadyToSend_c) {
    return;
  }

  BeeUtilZeroMemory(&afAddrInfo, sizeof(afAddrInfo));

  /* Don't care about afAddrInfo.dstAddr or afAddrInfo.dstEndPoint in */
  /* indirect mode. */
  /* For direct mode, dstAddr.aNwkAddr = 0 for Zc, 1 for Zr (temporary, */
  /* for testing). */
  /* afAddrInfo.dstAddr.aNwkAddr = 0; */

  afAddrInfo.aClusterId[0] = (mDataClusterId_c & 0xFF);
  afAddrInfo.aClusterId[1] = (mDataClusterId_c >> 8);
  afAddrInfo.srcEndPoint = appEndPoint;
  afAddrInfo.dstAddrMode = gZbAddrModeIndirect_c;

  /* Turn on security and leave it on. Harmless In a non-secure network. */
  afAddrInfo.txOptions = 0;
  afAddrInfo.radiusCounter = afDefaultRadius_c;

  /* The Wireless UART application on the remote node will ack or nak this */
  /* datagram (regardless of the reliability mode setting), so a confirm id */
  /* is not needed here. */
  if (AF_DataRequest(&afAddrInfo,
                     zbTxDataBuffer.len,
                     zbTxDataBuffer.data.raw,
                     NULL) != gZbSuccess_c) {
    /* The send failed. Set an event to insure this code gets */
    /* run next time the application task is called, and give */
    /* up for now. Note that status is still == mBufStatusReadyToSend_c. */
    TS_SendEvent(gAppTaskID, mAppRetryZigBeeDataTx_c);
    return;
  }                                     /* if (AF_DataRequest( ... */

  /* The send succeeded, at least to the point where the local */
  /* stack has accepted the data to be transmitted. */
  if (mInReliableMode) {
    /* Set status busy until we receive either an ack or a nak from the */
    /* remote node. Also set a retry timer, just in case we don't receive */
    /* any response from the remote node. */
    zbTxDataBuffer.status = mBufStatusBusy_c;
    TMR_StartSingleShotTimer(mTxZigBeeDataRetryTimerID,
                             mTxZigBeeDataRetryTimerDuration_c,
                             TxZigBeeDataRetryTimerCallBack);
  } else {
    zbTxDataBuffer.status = mBufStatusFree_c;
    zbTxDataBuffer.len = 0;
  }

#if gBeeAppDebug_d
  ++mTxZigBeeDataCount;
    if (mBreakOnCount) {
      if (mTxZigBeeDataBreak && (mTxZigBeeDataCount == mTxZigBeeDataBreak)) {
        DebugBreakpointTarget();
      }
    }
#endif

  /* Toggle the ZigBee activity light. */
  SetOneLed(LED3, gLedToggle_c);

  /* Don't try to send too often. */
  mTxZigBeeThrottleFlag = FALSE;
  TMR_StartSingleShotTimer(mTxZigBeeThrottleTimerID,
                           mTxZigBeeThrottleDuration_c,
                           TxZigBeeThrottleTimerCallBack);
}                                       /* TxZigBeeData() */

/****************************************************************************/

#if gBeeAppDebug_d
static void TxZigBeeDebugPacket(void) {
  index_t i;
  unsigned char byte;

  if (!mSendDebugPacketsFlag) {
    mSendDebugPacketsFlag = TRUE;
    mSendDebugPacketsCount = mNumberOfTestPacketsToSend_c;
  }

  if (zbTxDataBuffer.status == mBufStatusFree_c) {
    for (i = 0; i < sizeof(zbTxDataBuffer.data.structured.payload); ++i) {
      byte = (i & 0x01) ? mSendDebugPacketsCount >> 8
                        : mSendDebugPacketsCount & 0xFF;
      zbTxDataBuffer.data.structured.payload[i] = byte;
    }
    zbTxDataBuffer.status = mBufStatusReadyToSend_c;
    zbTxDataBuffer.data.structured.sequenceNumber = curZbTxSequenceNumber++;
    zbTxDataBuffer.len = sizeof(zbTxDataBuffer.data.structured.payload)
                       + sizeof(zbTxDataBuffer.data.structured.sequenceNumber);
    TxZigBeeData();
    --mSendDebugPacketsCount;
  }

  if (mSendDebugPacketsCount) {
    TS_SendEvent(gAppTaskID, mAppSendZigBeeDebugPacket_c);
  } else {
    /* Put a breakpoint here to know when all of the packets have been sent. */
    mSendDebugPacketsFlag = FALSE;
  }
}                                       /* TxZigBeeDebugPacket() */
#endif                                  /* #if gBeeAppDebug_d */

/****************************************************************************/

static void TxZigBeeThrottleTimerCallBack(tmrTimerID_t timerID) {
  (void) timerID;
  mTxZigBeeThrottleFlag = TRUE;
}                                       /* TzZigBeeThrottleTimerCallBack() */

/****************************************************************************/

/* When a datagram is sent OTA in reliable mode, a timer is started. */
/* When an ack or nak is received for that datagram, the timer is stopped. */
/* If the timer expires, the remote node apparently has not received the */
/* datagram, so set an event for ourselves to send it again. */
static void TxZigBeeDataRetryTimerCallBack(tmrTimerID_t timerID) {
  (void) timerID;
  zbTxDataBuffer.status = mBufStatusReadyToSend_c;
  TS_SendEvent(gAppTaskID, mAppRetryZigBeeDataTx_c);
}                                       /* TxZigBeeDataRetryTimerCallBack() */

/****************************************************************************/

/* A byte is available from the serial port. */
static void UartRxCallBack(void) {
  TS_SendEvent(gAppTaskID, mAppRxFromUart_c);
}                                       /* UartRxCallBack() */

/****************************************************************************/

/* The last byte received from the UART was long enough ago that it is now */
/* time to send any outstanding data, even if the buffer(s) are not full. */
static void UartRxTimeout(tmrTimerID_t timerID) {
  (void) timerID;

  if (mInLoopBackMode) {
    if (uartTxDataBuffer.status != mBufStatusInProgress_c) {
      return;
    }

    uartTxDataBuffer.status = mBufStatusReadyToSend_c;
    TxUartData();
  } else {
    if (zbTxDataBuffer.status != mBufStatusInProgress_c) {
      return;
    }

    zbTxDataBuffer.status = mBufStatusReadyToSend_c;
    zbTxDataBuffer.data.structured.sequenceNumber = curZbTxSequenceNumber++;
    ++zbTxDataBuffer.len;                 /* Count the sequence number. */
    TxZigBeeData();
  }
}                                       /* UartRxTimeout() */

/****************************************************************************/

/* The UART has finished transmitting a buffer out through the serial port. */
/* pBuf is the start of the data that the UART driver sent, not the address */
/* of the buffer_t that contains it. */
static void UartTxCallBack(unsigned char const *pBuf) {
  (void) pBuf;
  uartTxDataBuffer.status = mBufStatusFree_c;
  uartTxDataBuffer.len = 0;
}                                       /* UartTxCallBack() */

/****************************************************************************
* BeeAppZdpCallBack
*
* ZDP calls this function when it receives a response to a request.
* For example, ASL_MatchDescriptor_req will return whether it worked or not.
*****************************************************************************/
#if gFrequencyAgilityCapability_d && (!gEndDevCapability_d)
void BeeAppZdpCallBack
  (
  zdpToAppMessage_t *pMsg,
  zbCounter_t counter
  )
{
  uint8_t event;
  (void)counter;

  /* get the event from ZDP */
  event = pMsg->msgType;
  if(event == gzdo2AppEventInd_c) /* many possible application events */
    event = pMsg->msgData.zdo2AppEvent;
ZclApsAckConfirm(pConfirm->dstAddr.aNwkAddr, pConfirm->status);
  /* got a response to match descriptor */
  switch(event) {

    /* network has been started */
    case gZDOToAppMgmtZCRunning_c:
    case gZDOToAppMgmtZRRunning_c:
    case gZDOToAppMgmtZEDRunning_c:
      if (appState == mStateIdle_c) {

        appState = mStateZDO_device_running_c;

        /* stop the flashing and indicate the device is running (has joined the network) */
        LED_SetLed(LED1, gLedOn_c);
        LCD_WriteString(1,(uint8_t *)"Running Device");
        }
      break;

    case gNlmeTxReport_c:
      FA_ProcessNlmeTxReport(&pMsg->msgData.nlmeNwkTxReport);
      break;

    case gMgmt_NWK_Update_notify_c:
      /* Already handle in ZDP. */
      break;

    case gNlmeEnergyScanConfirm_c:
      FA_ProcessEnergyScanCnf(&pMsg->msgData.energyScanConf);
      break;

    case gChannelMasterReport_c:
      FA_SelectChannelAndChange();
      break;
  }
  if (pMsg->msgType == gNlmeEnergyScanConfirm_c)
    MSG_Free(pMsg->msgData.energyScanConf.resList.pEnergyDetectList);

  /* free the message from ZDP */
  MSG_Free(pMsg);
}
#endif

#if gInterPanCommunicationEnabled_c

/*****************************************************************************
  BeeAppInterPanDataIndication

  Process InterPan incoming ZigBee over-the-air messages.
*****************************************************************************/
void BeeAppInterPanDataIndication(void)
{
  InterPanMessage_t *pMsg;
  zbInterPanDataIndication_t *pIndication;
  zbStatus_t status = gZclMfgSpecific_c;
  
  while(MSG_Pending(&gInterPanAppDataIndicationQueue))
  {
    /* Get a message from a queue */
    pMsg = MSG_DeQueue( &gInterPanAppDataIndicationQueue );

    /* ask ZCL to handle the frame */
    pIndication = &(pMsg->msgData.InterPandataIndication );
      /* insert manufacturer specific code here... */

    /* Free memory allocated by data indication */
    MSG_Free(pMsg);
  }
/*remove compiler warnings */
  (void) pIndication;
  (void) status;  
}


/*****************************************************************************
  BeeAppDataConfirm

  Process InterPan incoming ZigBee over-the-air data confirms.
*****************************************************************************/
void BeeAppInterPanDataConfirm
(
void
)
{
  InterPanMessage_t *pMsg;
  zbInterPanDataConfirm_t *pConfirm;
  
  while(MSG_Pending(&gInterPanAppDataConfirmQueue))
  {
    /* Get a message from a queue */
    pMsg = MSG_DeQueue( &gInterPanAppDataConfirmQueue );
    pConfirm = &(pMsg->msgData.InterPandataConf);
    
    /* Action taken when confirmation is received. */
    if( pConfirm->status != gZbSuccess_c )
    {
      /* The data wasn't delivered -- Handle error code here */
    }
    
    /* Free memory allocated in Call Back function */
    MSG_Free(pMsg);
  }
}
#endif 