// --------------------------------------------------------------------------
// COPYRIGHT (c) Freescale 2009
// Freescale Confidential Proprietary
// All Rights Reserved
// --------------------------------------------------------------------------
//!\file    serial.c
//!\brief   This file contains the functions for initalizing linflex in uart mode
//!\brief   and sending and receiving data on serial port
//
// TYPE: C source file
// DEPARTMENT: MSG
// CREATED BY: Ankur Sharma  (B17102)
// --------------------------------------------------------------------------
// PURPOSE:  Used to make settings for serial communication
// --------------------------------------------------------------------------
// $Log: serial.c.rca $
// 
//  Revision: 1.2 Tue Jul  3 10:52:48 2012 B37394
//  remove linflex2
// 
//  Revision: 1.1 Mon May  7 07:28:48 2012 B17104
//  *** empty comment string ***
// 
//  Revision: 1.15 Tue Mar  6 23:36:39 2012 MadhusudanAggarwalFSL
//  *** empty comment string ***
// 
//  Revision: 1.14 Tue Mar  6 22:55:37 2012 ArunMishraFSL
//  Correct programming of MSCR for Tx pads
// 
//  Revision: 1.12 Thu Mar  1 09:29:16 2012 ArunMishraFSL
//  Automatic calculation of fbr
// 
//  Revision: 1.11 Thu Mar  1 00:20:26 2012 ArunMishraFSL
//  With working serial configuration on Silicon
// 
//  Revision: 1.9 Thu Jul 14 01:30:19 2011 PankajKushwahaFSL
//  Changed MSCOR to MSCR for SIUL
// 
//  Revision: 1.8 Thu Jul 14 01:18:40 2011 PankajKushwahaFSL
//  Checked in as per latest CGM module
// 
//  Revision: 1.7 Sat Jun 25 18:48:57 2011 ArunMishraFSL
//  With working configuration of MSCR
// 
//  Revision: 1.6 Thu Jun  2 02:00:53 2011 AijazFatimaFSL
//  Modified wrt new header file
// 
//  Revision: 1.5 Tue May 10 01:28:02 2011 YaqubMohamedFSL
//  *** empty comment string ***
// 
//  Revision: 1.4 Tue May 10 01:12:01 2011 ArunMishraFSL
//  Changed the MSCR for Rx configuration
// 
//  Revision: 1.3 Tue May  3 04:54:28 2011 ArunMishraFSL
//  Added the baud configurations for LTB2 and 3 also
// 
//  Revision: 1.2 Tue May  3 03:47:28 2011 ArunMishraFSL
//  First version working on McKinley
// 
//  Revision: 1.1 Tue Apr 19 04:09:35 2011 AijazFatimaFSL
//  Initial version
// 
//  Revision: 1.6 Tue Apr 27 11:06:48 2010 PankajKushwahaFSL
//  Added support to sent start characters once sci is initialized
// 
//  Revision: 1.5 Tue Apr 27 06:18:31 2010 ArunMishraFSL
//  SCI setting as per 90MHz
// 
//  Revision: 1.4 Sun Apr 25 16:46:22 2010 AnkurSharmaFSL
//  *** empty comment string ***
// 
//  Revision: 1.3 Mon Apr 19 10:57:10 2010 MishraArunFSL
//  Changed as per SoC.
// 
//  Revision: 1.2 Wed Apr  7 13:35:46 2010 AijazFatimaFSL
//  Modified code as per latest header file changes
// 
//  Revision: 1.1 Wed Jan 27 12:45:02 2010 PankajKushwahaFSL
//  linflex for Komodo
// 
//  Revision: 1.3 Mon Mar 30 15:31:28 2009 AmitGoelFSL
//  Modified the file in so that Print can be taken on both LINFLEX instances.
// 
//  Revision: 1.2 Wed Mar 18 16:55:29 2009 MadhusudanAggarwalFSL
//  Linflex function generalised for any frequency.
//
//  Revision: 1.1 Thu Mar  5 17:57:04 2009 AnkurSharmaFSL
//  Initial Check In
// -------------------------------------------------------------------------

#include "common.h"
#include "serial.h"
extern const unsigned char start_char;

void ConfigureSciPins(void);
void Linflex_SetSpeed(uint32_t,uint32_t);
void SetUpSci (UINT8 module, UINT16 sbr, UINT8 m, UINT8 pe, UINT8 pt);

#define LIN0 /*!< Define if Linflex 0 is being used for serial communication */
//#define LIN1 /*!< Define if Linflex 1 is being used for serial communication */
//#define LIN2 /*!< Define if Linflex 2 is being used for serial communication */

/**
 *\brief configures the linflex in UART mode
 *
 *configures the linflex in UART mode and sets the Linflex speed according to desired baud rate
 */

void sci_init(uint32_t freq)
{
    int i;
    
//    MC_CGM.AC[0].AC_DC0.B.DE=1;                 //Enabling Aux clock divider
    
    ConfigureSciPins();

#ifdef	LIN0
    LINFlexD_0.LINCR1.R = 0x1;

   LINFlexD_0.UARTCR.B.UART = 0x1;  //Linflex working in UART mode

   LINFlexD_0.UARTCR.B.TxEn = 0x1;    // Enable transmission of data now
   LINFlexD_0.UARTCR.B.RxEn = 0x1;    //Receiver enabled
   LINFlexD_0.UARTCR.B.WL0   = 0x1;    //8 bit data
   LINFlexD_0.UARTCR.B.PCE  = 0x0;    //No parity

  Linflex_SetSpeed(19200,freq);
   
   // Leave initialization mode by clearing INIT bit
   LINFlexD_0.LINCR1.R = 0x0;
#endif

#ifdef	LIN1
   // Enter initialization mode by setting INIT bit
   LINFlexD_1.LINCR1.R = 0x1;

   LINFlexD_1.UARTCR.B.UART = 0x1;  //Linflex working in UART mode

   LINFlexD_1.UARTCR.B.TxEn = 0x1;    // Enable transmission of data now
   LINFlexD_1.UARTCR.B.RxEn = 0x1;    //Receiver enabled
   LINFlexD_1.UARTCR.B.WL0   = 0x1;    //8 bit data
   LINFlexD_1.UARTCR.B.PCE  = 0x0;    //No parity

   Linflex_SetSpeed(19200,freq);

   
   // Leave initialization mode by clearing INIT bit
   LINFlexD_1.LINCR1.R = 0x0;
#endif

#ifdef	LIN2
   // Enter initialization mode by setting INIT bit
   LINFlexD_2.LINCR1.R = 0x1;

   LINFlexD_2.UARTCR.B.UART = 0x1;  //Linflex working in UART mode

   LINFlexD_2.UARTCR.B.TxEn = 0x1;    // Enable transmission of data now
   LINFlexD_2.UARTCR.B.RxEn = 0x1;    //Receiver enabled
   LINFlexD_2.UARTCR.B.WL0   = 0x1;    //8 bit data
   LINFlexD_2.UARTCR.B.PCE  = 0x0;    //No parity

   Linflex_SetSpeed(19200,freq);
   
   // Leave initialization mode by clearing INIT bit
   LINFlexD_2.LINCR1.R = 0x0;
#endif
//    while(1)sci_a_putchar('A');
}


/**
 *\brief Function to get a character from the serial interface
 *
 *waits for the character to be received on serial interface and returns it
 *\return The character received from the serial interface
 */
unsigned char sci_a_getchar (void)
{
    unsigned char cDummy;

#ifdef LIN0
    {
	// Wait for the reception to complete
	while(LINFlexD_0.UARTSR.B.DRFRFE == 0)
	{
	    asm("nop");
	}

	cDummy = (uint8_t) LINFlexD_0.BDRM.R;  // Retrieve data received from LinFlex

	   LINFlexD_0.UARTSR.B.RMB = 1;
	   LINFlexD_0.UARTSR.B.DRFRFE = 1;
    }
    
    return cDummy;
#endif

#ifdef LIN1
    {
	// Wait for the reception to complete
	while(LINFlexD_1.UARTSR.B.DRFRFE == 0)
	{
	    asm("nop");
	}

	cDummy = (uint8_t) LINFlexD_1.BDRM.R;  // Retrieve data received from LinFlex

	LINFlexD_1.UARTSR.B.RMB = 1;
	LINFlexD_1.UARTSR.B.DRFRFE = 1;
    }
    
    return cDummy;
#endif

#ifdef LIN2
    {
	// Wait for the reception to complete
	while(LINFlexD_2.UARTSR.B.DRFRFE == 0)
	{
	    asm("nop");
	}

	cDummy = (uint8_t) LINFlexD_2.BDRM.R;  // Retrieve data received from LinFlex

	LINFlexD_2.UARTSR.B.RMB = 1;
	LINFlexD_2.UARTSR.B.DRFRFE = 1;
    }

    return cDummy;
#endif
}

/**
 *\brief Function to put a character on the serial interface
 *
 *receives a character from user and puts it on the serial interface
 *\param ch The character received from user to be send on the serial interface
 */
void sci_a_putchar (char ch)
{
#ifdef LIN0
    {
	LINFlexD_0.UARTCR.B.TxEn = 1;
	LINFlexD_0.BDRL.R = (uint32_t) ch; // Start transmission by writing the data

	// Wait for the transmission to complete
	while (LINFlexD_0.UARTSR.B.DTFTFF == 0)
	{
	    asm("nop");
	}

	LINFlexD_0.UARTSR.B.DTFTFF = 1;		//clear the trasmission completed flag

	LINFlexD_0.UARTCR.B.TxEn = 0;	// Disable transmission
    }
#endif

#ifdef LIN1
    {
	LINFlexD_1.UARTCR.B.TxEn = 1;
	LINFlexD_1.BDRL.R = (uint32_t) ch; // Start transmission by writing the data

	// Wait for the transmission to complete
	while (LINFlexD_1.UARTSR.B.DTFTFF == 0)
	{
	    asm("nop");
	}

	LINFlexD_1.UARTSR.B.DTFTFF = 1;		//clear the trasmission completed flag

	LINFlexD_1.UARTCR.B.TxEn = 0;	// Disable transmission
    }
#endif

#ifdef LIN2
    {
	LINFlexD_2.UARTCR.B.TxEn = 1;
	LINFlexD_2.BDRL.R = (uint32_t) ch; // Start transmission by writing the data

	// Wait for the transmission to complete
	while (LINFlexD_2.UARTSR.B.DTFTFF == 0)
	{
	    asm("nop");
	}

	LINFlexD_2.UARTSR.B.DTFTFF = 1;		//clear the trasmission completed flag

	LINFlexD_2.UARTCR.B.TxEn = 0;	// Disable transmission
    }
#endif
    
}

/**
 *\brief Sets up the speed of linflex
 *
 *Receives the system frequency and desired baud rate from user and configures linflex accordingly
 *\param baudrate The desired baudrate of linflex
 *\param Frequency The system frequency
 */
void Linflex_SetSpeed(uint32_t Baudrate,uint32_t Frequency)
{
	/* This will be extended in future revision */
	 uint32_t ibr,fbr;
    uint32_t dividr,divisr,temp;

    divisr = (Frequency * 1000000);
    dividr = (Baudrate * 16);

    ibr = divisr /dividr;
	
#ifdef LIN0
    LINFlexD_0.LINIBRR.R = ibr;
#endif
#ifdef LIN1
    LINFlexD_1.LINIBRR.R = ibr;
#endif
#ifdef LIN2
    LINFlexD_2.LINIBRR.R = ibr;
#endif
    temp = (divisr - ibr*dividr);
    fbr = (temp)/Baudrate;


#ifdef LIN0
    LINFlexD_0.LINFBRR.R = fbr;
#endif
#ifdef LIN1
    LINFlexD_1.LINFBRR.R = fbr;
#endif
#ifdef LIN2
    LINFlexD_2.LINFBRR.R = fbr;
#endif
   
}


/**
 *\brief Configures the Linflex pins
 *
 *Configures the Linflex pins
 */
void ConfigureSciPins( void )
{
#ifdef LIN0
    SIUL2.MSCR[10].R = 0x32810002;	// PA[10] as TXD
    SIUL2.MSCR[848].R = 0x2;		// PA[11] as RXD
#endif
#ifdef LIN1
    SIUL2.MSCR[2].R = 0x32810002;	//PA[2] as TXD
    SIUL2.MSCR[849].R = 0x2;		// PA[15] as RXD
#endif
#ifdef LIN2
    SIUL2.MSCR[62].R = 0x32810002;     	// PD[14] as TXD
    SIUL2.MSCR[850].R = 0x1;	       	// PD[15] as RXD
#endif
    
}


