#include "drivers.h"
/*******************************************************************************
*  Global constants
*******************************************************************************/
// Range of requested value for output current regulator
#define REQ_VAL_RANGE           1024 // this means that the range 4-20mA is divided to 1024 steps

// By comment/uncomment this line is switched type of application 
//#define TYPE_OF_APPLICATION_MPXV61115   0

/*******************************************************************************
*  Function prototypes
*******************************************************************************/
void AFE_ShuntResistorCB(AFE_CH_CALLBACK_TYPE type, int32 result);

#if TYPE_OF_APPLICATION_MPXV61115
  void AFE_MPXV6115_CB(AFE_CH_CALLBACK_TYPE type, int32 result);
#else
  void AFE_MPX53_CB(AFE_CH_CALLBACK_TYPE type, int32 result);
#endif
  
static void TMR_ShuntLoadSetPWM(uint16 load);

/*******************************************************************************
*  Global variables
*******************************************************************************/
volatile uint32  requsted_val = 0;   // range 1024
uint32  pwm_val = 0;   // range 1024

/*******************************************************************************
*  Functions bodies
*******************************************************************************/

/***************************************************************************//*!
 * @brief   Sets the PWM output value.
 * @details This function set up both compare register of Quad Timer to generate PWM. 
 * @param   load - Innput parameter of PWM, range [0..PWM_STEPS]
 * @return  NA
 * @note    Static local help function.
 ******************************************************************************/
static void TMR_ShuntLoadSetPWM(uint16 load)
{
  if(load >= REQ_VAL_RANGE)
    load = REQ_VAL_RANGE - 1;

  pwm_val = load;
  TMRCH0_SetComp2Val(pwm_val);
  TMRCH0_SetComp1Val(REQ_VAL_RANGE - pwm_val - 1);
}

/***************************************************************************//*!
 * @brief   Call Back function to handle board ouput current regulator.
 * @details In this short callback fuction from ADC (AFE) is algorithm of simple
 *          proportional regulator of board output current, where input value is 
 *          measured voltage on board shunt resistor and the actuator is load
 *          resistor controlled by FET transistor connected to PWM through 2nd order
 *          low pass filter. 
 * @param   type - type of callback
 * @param   result - result of ADC reading
 * @return  NA
 * @note    Call back function from interrupt from ADC (AFE).
 ******************************************************************************/
void AFE_ShuntResistorCB(AFE_CH_CALLBACK_TYPE type, int32 result)
{
#define AFE1_OFFSET     -11449

#define START_OF_RANGE          (0.004)         // Start of Range of Output Current In Ampers
#define END_OF_RANGE            (0.020)         // End of Range of Output Current In Ampers        
#define CURRENT_SHUNT_R         (6.8)           // Size of Shunt resistor in Ohms
  
  // Recompute intput values to AFE LSB
#define ADC_START_OF_RANGE      ((START_OF_RANGE * CURRENT_SHUNT_R) / (0.5 / (1<<23)))  
#define ADC_END_OF_RANGE        ((END_OF_RANGE * CURRENT_SHUNT_R) / (0.5 / (1<<23)))

#define ADC_RANGE               (ADC_END_OF_RANGE - ADC_START_OF_RANGE)  
  
  int32 action_diff;
  static int32 last_result;
  
  // Prepare the requated value on shunt in ADC value range
  int32 adc_requested_value = (int32)(ADC_START_OF_RANGE + (requsted_val * (ADC_RANGE / REQ_VAL_RANGE)));
  
  // Check if the reading is valid
  if(type == COC_CALLBACK)
  {
    // Do very simple additional filter with last readed value
    result += last_result;
    result /= 2;
    last_result = result;
    
    // Correction of AFE offset on 0V  (shorted diffrencial inputs of AFE)
    result -= AFE1_OFFSET;
    
    // Compute  regulation deviation to regulator
    action_diff = adc_requested_value - result;
    
    // Rescale the ADC values to axceptable for PWM range, in fact aply P part of regulator
    action_diff /= 32768;
    
    // Check if is needed update the PWM output
    if(action_diff)
    {
      // Compute absolute value of output PWM
      action_diff = pwm_val + action_diff;
      // Update the PWM outputs
      TMR_ShuntLoadSetPWM(action_diff);
    }    
  }
}

#if TYPE_OF_APPLICATION_MPXV61115
    /***************************************************************************//*!
   * @brief   Call Back function to handle board ouput current regulator. MPXV6115 version
   * @details In this short callback fuction from ADC (AFE) is algorithm to compute 
   *          the requested value in REQ_VAL_RANGE as a input to regulator of board current.
   *          
   * @param   type - type of callback
   * @param   result - result of ADC reading
   * @return  NA
   * @note    Call back function from interrupt from ADC (AFE).
   ******************************************************************************/
  void AFE_MPXV6115_CB(AFE_CH_CALLBACK_TYPE type, int32 result)
  {

  // The ouput of this sensor is coonected through the voltage divider,
  // so there is constants  to compute the final voltage on the input of AFE
  #define MPXV6115_VOLTGDIV_R1             150000       
  #define MPXV6115_VOLTGDIV_R2             24000
  
  // Equation of voltage divider of U2 from U1 with resistors R1 and R2  
  #define COMPUTE_VOLTGDIV_U2(U, R1, R2)  (((U) * (R2))/((R1)+(R2)))  

  #define MPXV6115_START_OF_RANGE          (COMPUTE_VOLTGDIV_U2(0.20, MPXV6115_VOLTGDIV_R1, MPXV6115_VOLTGDIV_R2))      // Start of Range of Output Current In Ampers
  #define MPXV6115_END_OF_RANGE            (MPXV6115_START_OF_RANGE + (COMPUTE_VOLTGDIV_U2(4.40, MPXV6115_VOLTGDIV_R1, MPXV6115_VOLTGDIV_R2))) // End of Range of Output Current In Ampers

    // Recompute intput values to AFE LSB
  #define MPXV6115_ADC_START_OF_RANGE      (MPXV6115_START_OF_RANGE / (0.5 / (1<<23)))
  #define MPXV6115_ADC_END_OF_RANGE        (MPXV6115_END_OF_RANGE / (0.5 / (1<<23)))

  #define MPXV6115_ADC_RANGE               (MPXV6115_ADC_END_OF_RANGE - MPXV6115_ADC_START_OF_RANGE)  

  #define AFE2_OFFSET     -188886         // Offset of DC or AFE

    int32 action_result;

    if(type == COC_CALLBACK)
    {
      // Correction of AFE offset on 0V  (shorted diffrencial inputs of AFE)
      result -= AFE2_OFFSET;
     
      // Remove the offset of sensor output value
      action_result = result - (uint32)MPXV6115_ADC_START_OF_RANGE; // storno offset
     
      // Recalculate the output to 1024 range that is able the current output able
      action_result = (REQ_VAL_RANGE * action_result)/ (int32)MPXV6115_ADC_RANGE;

      // Check the maximal limit
      if(action_result > REQ_VAL_RANGE)
        action_result = REQ_VAL_RANGE;
      
      // Check the minimal limit
      if(action_result < 0)
        action_result = 0;
      
      // Copy the local computed value to global requested value variable
      requsted_val = action_result;
    }
  }

#else
    /***************************************************************************//*!
   * @brief   Call Back function to handle board ouput current regulator. MPX53 version
   * @details In this short callback fuction from ADC (AFE) is algorithm to compute 
   *          the requested value in REQ_VAL_RANGE as a input to regulator of board current.
   *          
   * @param   type - type of callback
   * @param   result - result of ADC reading
   * @return  NA
   * @note    Call back function from interrupt from ADC (AFE).
   ******************************************************************************/
  void AFE_MPX53_CB(AFE_CH_CALLBACK_TYPE type, int32 result)
  {
  // The ouput of this sensor is coonected through the voltage divider,
  // so there is constants  to compute the final voltage on the input of AFE
  #define MPX53_VOLTGDIV_R1             10000
  #define MPX53_VOLTGDIV_R2             6800
  // Equation of voltage divider of U2 from U1 with resistors R1 and R2  
  #define COMPUTE_VOLTGDIV_U2(U, R1, R2)  (((U) * (R2))/((R1)+(R2)))  

  #define MPX53_START_OF_RANGE          (COMPUTE_VOLTGDIV_U2(0.020, MPX53_VOLTGDIV_R1, MPX53_VOLTGDIV_R2))
  #define MPX53_END_OF_RANGE            (MPX53_START_OF_RANGE + (COMPUTE_VOLTGDIV_U2(0.060, MPX53_VOLTGDIV_R1, MPX53_VOLTGDIV_R2)))

  #define MPX53_ADC_START_OF_RANGE      (MPX53_START_OF_RANGE / (0.5 / (1<<23)))
  #define MPX53_ADC_END_OF_RANGE        (MPX53_END_OF_RANGE / (0.5 / (1<<23)))

  #define MPX53_ADC_RANGE               (MPX53_ADC_END_OF_RANGE - MPX53_ADC_START_OF_RANGE)  

  #define AFE0_OFFSET     -188886         // Offset moved for common voltage 0.935V
    
    int32 action_result;

    if(type == COC_CALLBACK)
    {
      // Correction of AFE offset on 0V  (shorted diffrencial inputs of AFE)
      result -= AFE0_OFFSET;
      
      // Remove the offset of sensor output value
      action_result = result - (uint32)MPX53_ADC_START_OF_RANGE; // storno offset
      
      // Recalculate the output to 1024 range that is able the current output able
      action_result = (REQ_VAL_RANGE * action_result)/ (int32)MPX53_ADC_RANGE;

      // Check the maximal limit
      if(action_result > REQ_VAL_RANGE)
        action_result = REQ_VAL_RANGE;
      
      // Check the minimal limit
      if(action_result < 0)
        action_result = 0;
      
      // Copy the local computed value to global requested value variable
      requsted_val = action_result;
    }
  }
#endif

/***************************************************************************//*!
* @brief   Main function of the Project
* @details In this function is only initilized the used peripherals and kick of the measuring of AFE
* @return  NA
* @note    Ctandard main function of "C" code
******************************************************************************/
void main (void)
{
  /* enable clocks to all used chip peripherals                             */
  SIM_Init (SIM_MODULE_PRSR_SNSR_CONFIG);\
  
  /* clock mode 1:1:1, ~21MHz                                                */
  SIM_SetClkMode (SYSCLK_MODE0); 
  SIM_SetClkDiv  (SYSCLK_DIV1);
  FLL_Init       (FLL_MODULE_FEI_20_25MHZ_CONFIG);

  /* VREF module must be initialized after SIM module                       */
  VREF_Init (VREF_CHOP_1V75_0V4_HIGHPOWER_CONFIG, VREF_SWITCH_S1_L_S2_L_S3_L);
  VREFH_Trim(50);                               /* VREFH trimmed to 1.193 V */
  VREFL_Trim(3);                                /* VREFL trimmed to 0.397 V */
  

  // Initialization of Quad Timer to generate PWM signal with frequency ~20Khz
  TMR_Init(CH0, TMR_CH_OUTCMP_CONFIG(COUNT_RISING_EDGE, BUS_CLK_DIV1, SEC_CNTR0_INP, TGL_ON_ALT_COMP), REQ_VAL_RANGE, 0, 0, 0, 0, 0, NULL);
  // Enable the used TMR0 signal for generating of output PWM signal
  TMR_Enable(CH0);
  // Eneble of output the timer channel to MCU pin
  TMR_CtrlOEN(CH0, TRUE);
  

  // Output pin configuration to alternative 2 where is the TMR0 function             
  PORT_Init (PORTF, PORT_MODULE_ALT2_MODE, PIN1);    
  
  // Chosse FLL output to use by AFE
  SIM_SelAfePllClkSrc (SIM_MCG_FLL_CLK);
  
  // Initilize the AFE channel where is connected the pressure sensor
  #if TYPE_OF_APPLICATION_MPXV61115
    AFE_ChanInit(CH2, AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(DEC_OSR2048), 0, 4, AFE_MPXV6115_CB); 
  #else
    AFE_ChanInit(CH0, AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(DEC_OSR2048), 0, 4, AFE_MPX53_CB); 
  #endif
    
  // Initilize the AFE channel where is connected the board current shunt resistor  
  AFE_ChanInit(CH1, AFE_CH_SWTRG_CCM_PGAOFF_CONFIG(DEC_OSR2048), 0, 3, AFE_ShuntResistorCB); 
  
  // Initilize the AFE peripheral itself
  AFE_Init (AFE_MODULE_RJFORMAT_CONFIG(AFE_PLL_CLK, AFE_DIV16, (32768*640)));
  
  // Kick off the AFE measuring
#if TYPE_OF_APPLICATION_MPXV61115
  AFE_SwTrigger (CH0|CH2);
#else
  AFE_SwTrigger (CH0|CH1);
#endif
  
  // Enable interrupts to allow run all the interrupts routines
  EnableInterrupts();
  
  // And loop forever - rest of application is running from interrupts
  while(1);
}
