/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
*******************************************************************************
*
* @file crc.c
*
* @author Freescale
*
* @version 0.0.1
*
* @date Jun-6-2013
*
* @brief EEPROM simulation
*
*******************************************************************************/

#include "common.h"
#include "ee_emulation.h"
#include "flash.h"
/******************************************************************************
* Global variables
******************************************************************************/

/******************************************************************************
* Constants and macros
******************************************************************************/

/******************************************************************************
* Local types
******************************************************************************/

/******************************************************************************
* Local function prototypes
******************************************************************************/

/******************************************************************************
* Local variables
******************************************************************************/

/******************************************************************************
* Local functions
******************************************************************************/
 uint8_t EE_WriteItemToAddress(EE_ItemInfoPtr pWrItemInfo,uint32_t u32CurrentAddress);
 uint8_t EE_CheckItemStatus(EE_ItemInfoPtr pItemInfo);

/******************************************************************************
* Global functions
******************************************************************************/


/*******************************************************************************
 *
 * Description:  Initialize EEPROM  
 *
 * Returns:         work mode
 *
 * Notes:
 *
 *******************************************************************************/

 uint8_t EE_Init( uint32_t *pCurrentAddress,uint32_t u32BusClock )
 {
 	/* initization flash */
 	FLASH_Init(u32BusClock);
    /* Serch index */
	return EE_SearchIndex(pCurrentAddress);
 }
 
 /*******************************************************************************
 *
 * Description:  find the valid item index   
 *
 * Returns:         work mode
 *
 * Notes:
 *
 *******************************************************************************/

 uint8_t EE_SearchIndex(uint32_t *pCurrentAddress)
 {
     uint32_t i,j;
     uint32_t u32PageEndAddress;
     uint8_t eeItemStatus;
     EE_ItemInfoType sItemInfo;
     EE_ItemInfoPtr pItemInfo = &sItemInfo;
	
   	for(i=EE_START_ADDRESS;i<EE_END_ADDRESS;i+=sizeof(EE_ItemInfoType))
	{
		pItemInfo = (EE_ItemInfoPtr)i;
		switch(pItemInfo->u8Flag)
		{
		case EE_ITEM_INFO_NULL:
			*pCurrentAddress = i;
            if( EE_END_ADDRESS < (i+sizeof(EE_ItemInfoType)))
            {
                /* erase flash start secor, and set the index to starter */
                if( FLASH_EraseSector(EE_START_ADDRESS) != FLASH_ERR_ACCESS )
                {
                    *pCurrentAddress = EE_START_ADDRESS;
                    return TRUE;
                }
                else
                {
                    return FALSE;
                }
            }
            /* check all content in this sector */
            for(j=i;j<(FLASH_PAGE_SIZE-(i%FLASH_PAGE_SIZE));j++)
            {
                if(*(uint8_t *)j != 0xff)
                {
                    /* erase flash start secor, and set the index to starter */
                    if( FLASH_EraseSector(EE_START_ADDRESS) != FLASH_ERR_ACCESS )
                    {
                        *pCurrentAddress = EE_START_ADDRESS;
                        return TRUE;
                    }
                    else
                    {
                        return FALSE;
                    }
                }
            }
 
            /* rest all are 0xff */
            return TRUE;
          
		case EE_ITEM_INFO_VALID:
			*pCurrentAddress = i;
            if( EE_END_ADDRESS < ((*pCurrentAddress)+sizeof(EE_ItemInfoType)))
            {
                /* Currently item is invalid */
                /* erase flash start secor, and set the index to starter */
                if( FLASH_EraseSector(EE_START_ADDRESS) != FLASH_ERR_ACCESS )
                {
                    *pCurrentAddress = EE_START_ADDRESS;
                    return TRUE;
                }
                else
                {
                    return FALSE;
                }
            }
            eeItemStatus = EE_CheckItemStatus((EE_ItemInfoPtr)((*pCurrentAddress)+sizeof(EE_ItemInfoType)));
            if( (eeItemStatus == EE_ITEM_INFO_NULL)||
                (EE_END_ADDRESS < ((*pCurrentAddress)+2*sizeof(EE_ItemInfoType)))
               )
            {
                /* the next item is NULL, or current item is last one, then current item is valid */
                return TRUE;
            }
            else if(eeItemStatus == EE_ITEM_INFO_VALID)
            {
                /* next item is also valid, then next item is latest. */ 
                break;
            }
            else 
            {
                /* the next item is invalid,(processing or others */
                /* read current item, then write it to valid address */
                EE_Read(&sItemInfo,(*pCurrentAddress));
                sItemInfo.u8Flag = EE_ITEM_INFO_INVALID;
                FLASH_Program((*pCurrentAddress),
                    (uint8_t *)&sItemInfo,4);
                /* check all content in this sector */
                (*pCurrentAddress) += sizeof(EE_ItemInfoType);
                i = (*pCurrentAddress)+sizeof(EE_ItemInfoType);
                u32PageEndAddress = i/FLASH_PAGE_SIZE*FLASH_PAGE_SIZE+FLASH_PAGE_SIZE;
                for(j=i;j<u32PageEndAddress;j++)
                {
                    if(*(uint8_t *)j != 0xff)
                    {
                        /* erase flash start secor, and set the index to starter */
                        if( FLASH_EraseSector(EE_START_ADDRESS) != FLASH_ERR_ACCESS )
                        {
                            *pCurrentAddress = EE_START_ADDRESS;
                        }
                        else
                        {
                            return FALSE;
                        }
                    }
                }
                
                sItemInfo.u8Flag = EE_ITEM_INFO_VALID;
                return EE_Write(&sItemInfo,pCurrentAddress);
            }
            break;
		case EE_ITEM_INFO_INVALID:
			break;
		default:
            break;
		}
	}
    
    /* if don't find the valid flag, erase the flash starter addreee */
	if( FLASH_EraseSector(EE_START_ADDRESS) != FLASH_ERR_ACCESS )
	{
		*pCurrentAddress = EE_START_ADDRESS;
		return TRUE;
	}
	else
	{
		return FALSE;
	}
 }
/*******************************************************************************
 *
 * Description:  check current index status  
 *
 * Returns:      item status
 *
 * Notes:
 *
 *******************************************************************************/

 uint8_t EE_CheckItemStatus(EE_ItemInfoPtr pItemInfo)
 {
    return pItemInfo->u8Flag;
 }

 /*******************************************************************************
 *
 * Description:  Write a item to specified address  
 *
 * Returns:      fail or success
 *
 * Notes:
 *
 *******************************************************************************/

 uint8_t EE_WriteItemToAddress(EE_ItemInfoPtr pWrItemInfo,uint32_t u32CurrentAddress)
 {
    uint8_t u8ErrorStatus;
    u8ErrorStatus = EE_ERROR_SUCCESS;
    pWrItemInfo->u8Flag = EE_ITEM_INFO_PROCESSING;
    if(FLASH_Program(u32CurrentAddress,
                    (uint8_t *)pWrItemInfo,sizeof(EE_ItemInfoType)) !=
                                    FLASH_ERR_SUCCESS )
    {
        u8ErrorStatus |= EE_ERROR_PROGRAM_FAIL;
    }
    pWrItemInfo->u8Flag = EE_ITEM_INFO_VALID;
    if(FLASH_Program(u32CurrentAddress,
                    (uint8_t *)pWrItemInfo,4) !=
                                    FLASH_ERR_SUCCESS )
    {
        u8ErrorStatus |= EE_ERROR_PROGRAM_FAIL;
    }
    
    return u8ErrorStatus;
 }
 /*******************************************************************************
 *
 * Description:    write item information
 *
 * Returns:         result  1 - success
 *                                0 - fail
 *
 * Notes:
 *
 *******************************************************************************/
 uint8_t EE_Write(EE_ItemInfoPtr pWrItemInfo,uint32_t *p32CurrentAddress)
 {
	EE_ItemInfoType sCurrentItemInfo;
    uint32_t u32CurrentAddress;
    uint8_t u8ErrorStatus;
    
    u8ErrorStatus = EE_ERROR_SUCCESS;
    u32CurrentAddress = *p32CurrentAddress;
    EE_Read( &sCurrentItemInfo,u32CurrentAddress);

	if( sCurrentItemInfo.u8Flag == EE_ITEM_INFO_NULL )
	{
		EE_WriteItemToAddress(pWrItemInfo,u32CurrentAddress);
  	}
	else if( (sCurrentItemInfo.u8Flag == EE_ITEM_INFO_VALID )||
            (sCurrentItemInfo.u8Flag == EE_ITEM_INFO_PROCESSING))
	{
			
		u32CurrentAddress += sizeof(EE_ItemInfoType);	
		if( (FLASH_PAGE_SIZE - (u32CurrentAddress%FLASH_PAGE_SIZE))>sizeof(EE_ItemInfoType))
		{
            EE_WriteItemToAddress(pWrItemInfo,u32CurrentAddress);
		}
		else
		{
			/* erase start page */
			if( EE_END_ADDRESS < (u32CurrentAddress + sizeof(EE_ItemInfoType)) )
			{
				printf("erase the start sector!\n");
				FLASH_EraseSector(EE_START_ADDRESS);
				u32CurrentAddress = EE_START_ADDRESS;
				EE_WriteItemToAddress(pWrItemInfo,u32CurrentAddress);
			}
			else
			{
				/* erase next page */
				printf("erase the next sector!\n");
				FLASH_EraseSector(u32CurrentAddress + sizeof(EE_ItemInfoType));

				/* write new item */
				EE_WriteItemToAddress(pWrItemInfo,u32CurrentAddress);
			}
		}
        
        /* write index item to invalid */
        if((EE_START_ADDRESS != u32CurrentAddress)||
           (EE_PAGE_NUMBER != 1))
        {
            sCurrentItemInfo.u8Flag = EE_ITEM_INFO_INVALID;
            if( FLASH_Program((*p32CurrentAddress),
                            (uint8_t *)&sCurrentItemInfo,4)!=
                                        FLASH_ERR_SUCCESS )
            {
                u8ErrorStatus |= EE_ERROR_PROGRAM_FAIL;
            }
        }
		
        /* updated the valid address */
        *p32CurrentAddress = u32CurrentAddress;
	}
    else
    {
        printf("current index is invalid,re-initialize EE!\n");
        u8ErrorStatus |= EE_ERROR_CONTENT_INVALID;
    }
    
    /* if error status is not success, erase the first sector and write new item */
    if( u8ErrorStatus != EE_ERROR_SUCCESS )
    {
        /* flash program fail */
        printf("erase the first sector, error status 0x%x!\n",u8ErrorStatus);
        FLASH_EraseSector(EE_START_ADDRESS);
        (*p32CurrentAddress) = EE_START_ADDRESS;
        EE_WriteItemToAddress(pWrItemInfo,(*p32CurrentAddress));
    }
    return TRUE;
 }

  /*******************************************************************************
 * Function:        EE_Read
 *
 * Description:    read current item inforamtion
 *
 * Returns:         result  1 - success
 *                          0 - fail
 *
 * Notes:
 *
 *******************************************************************************/
 uint8_t EE_Read(EE_ItemInfoPtr pRdItemInfo,uint32_t u32CurrentAddress)
 {
	EE_ItemInfoPtr pTemp;
    pTemp = (EE_ItemInfoPtr)u32CurrentAddress;
	if( pTemp->u8Flag == EE_ITEM_INFO_INVALID )
	{
		return FALSE;
	}
	memcpy((void *)pRdItemInfo,(uint8_t *)u32CurrentAddress,sizeof(EE_ItemInfoType));
	return TRUE;
 }


