/******************************************************************************
 *
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2013 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
 ******************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************//*!
 *
 * @file dac.h
 *
 * @author  
 *
 * @version 1.0.1
 *
 * @date Feb-22-2013
 *
 * @brief DAC functions for Kinetis microcontrollers
 *
 *****************************************************************************/

#ifndef DAC_KINETIS_H_
#define DAC_KINETIS_H_

#include "derivative.h"
#include "fsl_types.h"

#if (!defined(MCU_MK51D7))
#define DAC1_PRESENT
#endif

/*****************************************************************************
 * Globals
 ****************************************************************************/
extern unsigned char gu8DacIrqFlag;

/*****************************************************************************
 * Type definitions
 ****************************************************************************/
typedef struct{
	DAC_MemMapPtr DacModule;					//DAC module to use (i.e. DAC0)
	unsigned char u8DacRefvTrigPowermode;		//DAC Vref, Trigger and power mode (see lists below)
	unsigned char u8DacIntEn;					//Enable selected interrupts (See interrupts available below. Separate interrupts with OR "|")
	unsigned char u8DacDmaBufferenBuffermode;   //Enable DMA, buffer and select buffer mode
	unsigned char u8DacBufferWms;				//Watermark select for buffer mode (see list below)
	unsigned char u8DacBufferUp;				//Upper limit value for buffer (see table below)
}tDAC;

typedef enum{
	DAC_BFRUP_NOLIMIT,
	DAC_BFRUP_LIMIT1,
	DAC_BFRUP_LIMIT2,
	DAC_BFRUP_LIMIT3,
	DAC_BFRUP_LIMIT4,
	DAC_BFRUP_LIMIT5,
	DAC_BFRUP_LIMIT6,
	DAC_BFRUP_LIMIT7,
	DAC_BFRUP_LIMIT8,
	DAC_BFRUP_LIMIT9,
	DAC_BFRUP_LIMIT10,
	DAC_BFRUP_LIMIT11,
	DAC_BFRUP_LIMIT12,
	DAC_BFRUP_LIMIT13,
	DAC_BFRUP_LIMIT14,
	DAC_BFRUP_LIMIT15
}DAC_BUFFER_UPPER_LIMITS;

/*****************************************************************************
 * User definitions
 ****************************************************************************/
/********************** General **********************/
#define DAC_OK	0
#define DAC_ERR 1

/********************* IRQ Flags *********************/
/* DAC0 Buffer Read Pointer Bottom Position*/
#define DAC0_BFRPB			COMPARE(gu8DacIrqFlag,0)
#define DAC0_BFRPB_EVENT	EVENT(gu8DacIrqFlag,0)
#define DAC0_BFRPB_CLEAR	CLEAR(gu8DacIrqFlag,0)

/* DAC0 Buffer Read Pointer Top Position*/
#define DAC0_BFRPT			COMPARE(gu8DacIrqFlag,1)
#define DAC0_BFRPT_EVENT	EVENT(gu8DacIrqFlag,1)
#define DAC0_BFRPT_CLEAR	CLEAR(gu8DacIrqFlag,1)

/* DAC0 Buffer Watermark*/
#define DAC0_BFWM			COMPARE(gu8DacIrqFlag,2)
#define DAC0_BFWM_EVENT		EVENT(gu8DacIrqFlag,2)
#define DAC0_BFWM_CLEAR		CLEAR(gu8DacIrqFlag,2)

/* DAC1 Buffer Read Pointer Bottom Position*/
#define DAC1_BFRPB			COMPARE(gu8DacIrqFlag,3)
#define DAC1_BFRPB_EVENT	EVENT(gu8DacIrqFlag,3)
#define DAC1_BFRPB_CLEAR	CLEAR(gu8DacIrqFlag,3)

/* DAC1 Buffer Read Pointer Top Position*/
#define DAC1_BFRPT			COMPARE(gu8DacIrqFlag,4)
#define DAC1_BFRPT_EVENT	EVENT(gu8DacIrqFlag,4)
#define DAC1_BFRPT_CLEAR	CLEAR(gu8DacIrqFlag,4)

/* DAC1 Buffer Watermark*/
#define DAC1_BFWM			COMPARE(gu8DacIrqFlag,5)
#define DAC1_BFWM_EVENT		EVENT(gu8DacIrqFlag,5)
#define DAC1_BFWM_CLEAR		CLEAR(gu8DacIrqFlag,5)

/********************** DacModule Options *********************/ 
//DAC modules
#define DAC0	DAC0_BASE_PTR
#define DAC1	DAC1_BASE_PTR

/********************** u8DacRefvTrigPowermode Options *********************/ 
//Reference voltages
#define DACREF_1	(0<<DAC_C0_DACRFS_SHIFT)
#define DACREF_2	(1<<DAC_C0_DACRFS_SHIFT)

//Trigger options
#define DAC_HW_TRIGGER	(0<<DAC_C0_DACTRGSEL_SHIFT)
#define DAC_SW_TRIGGER	(1<<DAC_C0_DACTRGSEL_SHIFT)

//Power modes
#define DAC_HIGH_POWER	(0<<DAC_C0_LPEN_SHIFT)
#define DAC_LOW_POWER	(1<<DAC_C0_LPEN_SHIFT)

/********************** u8DacIntEn Options *********************/ 
//Interrupts available
#define DAC_DISABLE_INT			(0x00u)
#define DAC_WATERMARK_INT		(1<<DAC_C0_DACBWIEN_SHIFT)
#define DAC_POINTER_TOP_INT		(1<<DAC_C0_DACBTIEN_SHIFT)
#define DAC_POINTER_BOT_INT		(1<<DAC_C0_DACBBIEN_SHIFT)

/********************** u8DacDmaBufferenBuffermode Options *********************/ 
//DMA
#define DAC_DMA_DIS		(0<<DAC_C1_DMAEN_SHIFT)
#define DAC_DMA_EN		(1<<DAC_C1_DMAEN_SHIFT)

//Buffer
#define DAC_BUFFER_DIS		(0<<DAC_C1_DACBFEN_SHIFT)
#define DAC_BUFFER_EN		(1<<DAC_C1_DACBFEN_SHIFT)

//Buffer modes
#define DAC_BUFFER_NORMAL		(DAC_C1_DACBFMD(0))
#define DAC_BUFFER_SWING		(DAC_C1_DACBFMD(1))
#define DAC_BUFFER_ONETIMESCAN	(DAC_C1_DACBFMD(2))

/********************** u8DacBufferWms Options *********************/ 
//Watermark options
#define DAC_WTRM_1WORD		(DAC_C1_DACBFWM(0))
#define DAC_WTRM_2WORD		(DAC_C1_DACBFWM(1))
#define DAC_WTRM_3WORD		(DAC_C1_DACBFWM(2))
#define DAC_WTRM_4WORD		(DAC_C1_DACBFWM(3))

/*****************************************************************************
 * Function prototypes
 ****************************************************************************/
void vfnDacInit(tDAC DacCfg);
unsigned char u8DacLoadVal(DAC_MemMapPtr Module, unsigned short u16Value, unsigned char u8Index);
unsigned char u8DacLoadBuff (DAC_MemMapPtr Module, unsigned short* pBuffer, unsigned char u8Size);
void vfnDacSwTrigger(DAC_MemMapPtr Module);
void vfnDacDeinit(DAC_MemMapPtr Module);

/***************** DAC ISR ****************************/
void vfnDac0Isr (void);
void vfnDac1Isr (void);


#endif /* DAC_KINETIS_H_ */
