/******************************************************************************
 *
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2013 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
 ******************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************//*!
 *
 * @file pdb.c
 *
 * @author  
 *
 * @version 2.0
 *
 * @date May-02-2013
 *
 * @brief Functions to enable PDB on Kinetis
 *
 *****************************************************************************/

/*******************************************************************************
* Header files
*******************************************************************************/
#include "pdb.h"
#include "nvic.h"

/*******************************************************************************
* Globals
*******************************************************************************/
unsigned char gu8Pdb0Status = 0;

/*******************************************************************************
* Functions
*******************************************************************************/

/*****************************************************************************
* function: u8PdbInit
*
* description: Initializes PDB module
*
* input:
*       -InitParam: tPDB struct containing the configuration parameters
*
* output
*       -unsigned char: Error status
*               -PDB_OK: OK
*               -PDB_ERR: Error occurred
*
*****************************************************************************/
unsigned char u8PdbInit (tPDB InitParam)
{
  unsigned int u32Temp;
  
  SIM_SCGC6 |= SIM_SCGC6_PDB_MASK;
  
  PDB0_CH0C1 = 0x00u;
  PDB0_CH1C1 = 0x00u;
  PDB0_SC = 0x01u;              //Reset PDB module
  
  // Load timing values 
  PDB0_MOD = InitParam.u16PdbModulo;
  
  if(InitParam.u16Adc0ADelay != 0x00u)
  {
    PDB0_CH0C1 |= PDB_C1_TOS(1) | PDB_C1_EN(1);
    PDB0_CH0DLY0 = PDB_DLY_DLY(InitParam.u16Adc0ADelay);
  }
  
  if(InitParam.u16Adc0BDelay != 0x00u)
  {
    PDB0_CH0C1 |= PDB_C1_TOS(2) | PDB_C1_EN(2);
    PDB0_CH0DLY1 = PDB_DLY_DLY(InitParam.u16Adc0BDelay);
  }
  
  if(InitParam.u16Adc1ADelay != 0x00u)
  {
    PDB0_CH1C1 |= PDB_C1_TOS(1) | PDB_C1_EN(1);
    PDB0_CH1DLY0 = PDB_DLY_DLY(InitParam.u16Adc1ADelay);
  }
  
  if(InitParam.u16Adc1BDelay != 0x00u)
  {
    PDB0_CH1C1 |= PDB_C1_TOS(2) | PDB_C1_EN(2);
    PDB0_CH1DLY1 = PDB_DLY_DLY(InitParam.u16Adc1BDelay);
  }
  
  if(InitParam.u16Dac0Delay != 0x00u)
  {
    PDB0_DACINTC0 = PDB_INTC_TOE_MASK;
    PDB0_DACINT0 = PDB_INT_INT(InitParam.u16Dac0Delay);
  }
  
  #if (!defined(MCU_MK51D7))
  if(InitParam.u16Dac1Delay != 0x00u)
  {
    PDB0_DACINTC1 = PDB_INTC_TOE_MASK;
    PDB0_DACINT1 = PDB_INT_INT(InitParam.u16Dac1Delay);
  }
  #endif
  
  if(InitParam.u32IrqContinuousDma & (PDB_SC_PDBEIE_MASK | PDB_SC_PDBIE_MASK))
  {
    PDB0_IDLY = PDB_IDLY_IDLY(InitParam.u16PdbIdly);
    if(u8NvicEnableIrq(INT_PDB0 - 16))
      return PDB_ERR;
  }
  
  //Start PDB
  u32Temp = InitParam.u32PreescalerTrigger | InitParam.u32IrqContinuousDma;
  PDB0_SC = u32Temp | PDB_SC_LDOK_MASK;
  PDB0_SC |= PDB_SC_PDBEN_MASK;
  
  return PDB_OK;
}

/*****************************************************************************
* function: u8PdbDeinit
*
* description: Deinitializes PDB module
*
* input: (none)
*
* output
*       -unsigned char: Error status
*               -PDB_OK: OK
*               -PDB_ERR: Error occurred
*
*****************************************************************************/
unsigned char u8PdbDeinit (void)
{
  PDB0_CH0C1 = 0x00u;
  PDB0_CH1C1 = 0x00u;
  PDB0_SC = 0x01u;              //Reset PDB module
  
  SIM_SCGC6 &= ~SIM_SCGC6_PDB_MASK;
  
  if(u8NvicDisableIrq(INT_PDB0 - 16))
    return PDB_ERR;
  
  return PDB_OK;
}


/*****************************************************************************
* function: vfnPdbIsr
*
* description: Interrupt service routine for PDB module
*
* input: (none)
*
* output: (none)
*
* warning: This functions must be declared in the interrupt vector table prior
-          use
*****************************************************************************/
void vfnPdbIsr (void)
{
  PDB0_SC &= ~PDB_SC_PDBIF_MASK;
  
  PDB_INTERRUPT_EVENT;
}
