/*****************************************************************************
*
* Motorola Inc.
* (c) Copyright 2001,2002 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
******************************************************************************
*
*
* File Name:         jtag.c
*
* Description:       Jtag and OnCE access
*
* Modules Included:
*
* Author: William Jiang (a13984@freescale.com)
*
****************************************************************************/

#define START_APPLICATION	// reset the part after programming the chip

#undef DEBUG
//#define DEBUG

#include "hw_access.h"
#include "flash_over_jtag.h"
#include "flash.h"
#include "jtag.h"
#include "report.h"
#include "exit_codes.h"


#if port_IO_lib == ON

  #include "stdio.h"
  #ifdef WIN32
      #include <windows.h>
  #endif
  #ifdef MSDOS
      #include <time.h>		/* use time instead of GetTickCount under DOS */
  #endif
  unsigned int jtag_port=JTAG_PORT_DEFAULT;		/* contains printer port port number*/

#endif

unsigned int pport_data=0;						/* mirror of output port to save accesses */

unsigned char wait_for_DSP=0;					/* 1: wait for DSP to come out of reset (external reset circuit or power down/up for 801 bootloader erasure) */

unsigned char fast_port=0;						/* 1: adjust for fast printer port by checking ststus of the EOnCE fifo */

unsigned char eonce_base=0xff;					/* top 8 bits of eonce address in the target address space */

int data_pl;		/* lengths of JTAG paths */
int instr_pl;
int data_pp=0;		/* position of the part in the JTAG chain, 0=beginning */
//int instr_pp=0;	// Ethan: this parameter is always = 0

unsigned char exit_mode=0;	/* ==0 - reset the target, !=0 - leave in debug mode */

/* port number */
#if port_IO_lib == ON
void set_port(unsigned int port) {
	jtag_port=port;
}
#endif

/* Keivn, Flash read function */
UINT16 Flash_Read_2Bytes(UINT16 addr)
{
    UINT16 content;

    content = REG_READ(addr+0x4000);
    content = (content>>8) | (content<<8);

    return content;
}

/* routines for handling data path length variables */
int get_data_pl(void) {
	return(data_pl);
}
int get_instr_pl(void) {
	return(instr_pl);
}
int get_data_pp(void) {
	return(data_pp);
}
/*
int get_instr_pp(void) {
	return(instr_pp);
}
*/
void set_data_pp(int length) {
	data_pp=length;
}
/*
void set_instr_pp(int length) {
	instr_pp=length;
}
*/
void set_DSP_wait(char wait) {
	wait_for_DSP=wait;
}

void set_fast_port(char state) {
	fast_port=state;
}

void set_exit_mode(unsigned char mode) {
	exit_mode=mode;
}

/* measures data and instruction JTAG path lengths */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
/* returns 0 in case measurement has overflown, 1 in case measurement is OK */
int jtag_measure_paths(void) {
    int i;

    JTAG_TMS_SET;	/* Go to Select-DR-Scan */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;

    //	JTAG_TMS_SET;	/* Go to Select-IR-Scan */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;
    JTAG_TMS_RESET;	/* Go to Capture-IR */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;
    JTAG_TCK_RESET;
    JTAG_TCK_SET;	/* Go to Shift-IR */ /* Now the Jtag is in the Shift-IR state */
    JTAG_TDI_ASSIGN(1);
    for (i=0;i<JTAG_PATH_LEN_MAX;i++) {
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
    }
    JTAG_TDI_ASSIGN(0); /* shift 0 into beginning of the IR path */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;
    JTAG_TDI_ASSIGN(1);
    i=0;
    while ((JTAG_TDO_VALUE)&&(i<JTAG_PATH_LEN_MAX)) { 
	/* wait for the 0 to appear at the path end */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
	i++;
    }
    JTAG_TCK_RESET;
    JTAG_TCK_SET;	/* the whole IR path now contains 1 = BYPASS command */
    JTAG_TMS_SET;
    JTAG_TCK_RESET;	/* Go to Exit1-IR */
    JTAG_TCK_SET;
    JTAG_TCK_RESET;	/* Go to Update-IR */
    JTAG_TCK_SET;
    JTAG_TCK_RESET;	/* Go to Select-DR-Scan */
    JTAG_TCK_SET;
    instr_pl=i;

    JTAG_TMS_RESET;	/* Go to Capture-DR */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;	/* Go to Shift-DR */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;	/* Now the Jtag is in the Shift-DR state */
    JTAG_TDI_ASSIGN(1);
    for (i=0;i<JTAG_PATH_LEN_MAX;i++) {
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
    }
    JTAG_TDI_ASSIGN(0); /* shift 0 into beginning of the DR path */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;
    JTAG_TDI_ASSIGN(1);
    i=0;
    while ((JTAG_TDO_VALUE)&&(i<JTAG_PATH_LEN_MAX)) { 
	/* wait for the 0 to appear at the path end */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
	i++;
    }
    JTAG_TCK_RESET;
    JTAG_TCK_SET;	/* the whole DR path now contains 1 */

    JTAG_TMS_SET;
    JTAG_TCK_RESET;	/* Go to Exit1-DR */
    JTAG_TCK_SET;

    JTAG_TCK_RESET;	/* Go to Update-DR */
    JTAG_TCK_SET;

    JTAG_TMS_RESET;								
    JTAG_TCK_RESET;	/* Go to Run-test/idle state */
    JTAG_TCK_SET;
    data_pl=i;

    if ((data_pl<JTAG_PATH_LEN_MAX)&&(instr_pl<JTAG_PATH_LEN_MAX)) 
	return(1); 
    else return(0);
}

/* initialises JTAG, but leaves the part int reset */
/* the DSP is brought out of reset in "init_target" routine */
/* after return, it is in run-test-idle state */
int jtag_init(void) {
    long int i;

#if env_type == FSL_internal_use
	JTAG_VCC_SET;   /* supply for the command converter */
#endif

    WAIT_100_NS;    /* wait for power to stabilise */
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;    /* wait for power to stabilise */
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;
    WAIT_100_NS;

    JTAG_TMS_SET;//JTAG_TRST_SET;
    JTAG_TDI_SET;   /* all JTAG signals are reset to known state */
    JTAG_TCK_SET;//JTAG_RESET_SET;
    

    WAIT_100_NS;WAIT_100_NS;WAIT_100_NS;

#if 0 //delete by kevin
    JTAG_ID_SET;
    if (!JTAG_ID_VALUE) return(-1);
    JTAG_ID_RESET;
    if (JTAG_ID_VALUE) return(-1);

    JTAG_TRST_RESET;	/* /TRST signal goes low */
    JTAG_RESET_RESET;	/* /RESET signal goes low */
    /* The TRST pin does not neccessarily have to be connected, in such case we expect
       the JTAG state machine to be already reset to Test-Logic-Reset state */
#endif


    /* reset JTAG on the target */
    for(i = 0; i< 4000; i++)
	WAIT_100_NS;		
	/* wait for  32 MSTR_OSC clocks + 64 system clocks = 32*1000/8+64*1000/4=20Kns.
	 * the data sheet says it is about 28us.
	 */

    for (i=0;i<10;i++) {
	JTAG_TCK_RESET;	/* TMS must be sampled as '1' at least 5 times after power-up */
	JTAG_TCK_SET;	/* plus 5 more times to bring the target to Test-Logic-Reset in case /TRST is not connected */
    }

    JTAG_TMS_RESET;
    JTAG_TCK_RESET;									/* Go to Run-Test-Idle */
    JTAG_TCK_SET;

    return(0);
}


// Changed by William Jiang
#define PC_START_ADDR	0x0000

/* Set all JTAG signals inactive and reset target DSP */
/* parameter is address of the SIM RESET register (to reset the chip) */
void jtag_disconnect(void) {
	long i;

#ifdef	START_APPLICATION
  #if env_type == FSL_internal_use
		JTAG_RESET_RESET;  /* /TRST & /RESET signals go low */
  #endif

		for(i=0;i<400;i++)/* wait a few seconds for reset to be stable */
		    WAIT_100_NS;

		    eonce_move_value_to_pc(PC_START_ADDR);	/* load PC */
		    /* execute from reset vector (in case /RESET is not connected) */
		    eonce_exit_debug_mode();

  #if (env_type==FSL_internal_use) && (print_inside_function==1)
		print("The target was reset, the application is running\r\n");
  #endif
		for(i=0;i<400;i++)	/* wait  for the RESET signal high */
		WAIT_100_NS;

  #if env_type == FSL_internal_use
		JTAG_RESET_SET; /* /RESET signals go high */
  #endif

#endif
		/* select master tap (just in case the jtag was not reset) */
		select_master_tap();  
		jtag_instruction_exec(0x2,8);  /* IDCODE */ //???

}

/* Executes Jtag command */
/* expects Run-Test/idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
/* the second parameter specifies instruction length in bits */
int jtag_instruction_exec(int instruction, int instr_len) {
    int i,status=0;


    JTAG_TMS_SET;   /* Go to Select-DR-Scan */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;

    JTAG_TMS_SET;   /* Go to Select-IR-Scan */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;

    JTAG_TMS_RESET; /* Go to Capture-IR */
    JTAG_TCK_RESET;
    JTAG_TCK_SET;

    JTAG_TCK_RESET;
    JTAG_TCK_SET;   /* Go to Shift-IR */ /* Now the Jtag is in the Shift-IR state */

    if (instr_pl -instr_len >0) 
	JTAG_TDI_ASSIGN(1);

    for(i=0;i<(instr_pl -instr_len);i++) {
	JTAG_TCK_RESET;JTAG_TCK_SET;
	
    }

    for (i=0;i<instr_len;i++) {
	JTAG_TDI_ASSIGN(instruction);instruction>>=1;
	if (i==(instr_len-1)) 
	    JTAG_TMS_SET;	/* Go to Exit1-IR */

	JTAG_TCK_RESET;JTAG_TCK_SET;
	
	status>>=1;  status|=JTAG_TDO_VALUE<<3;
    }
    JTAG_TCK_RESET; /* Go to Update-IR */
    JTAG_TCK_SET;

    JTAG_TMS_RESET;
    JTAG_TCK_RESET; /* Go to RUN-TEST-IDLE */
    JTAG_TCK_SET;
    return(status);
}

/* shifts up to 32 bits in and out of the jtag DR path */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
unsigned long int jtag_data_shift(unsigned long int data, int bit_count) {
	int i;
	unsigned long int result=0;

	JTAG_TMS_SET;	/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;	/* Go to Capture-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;	/* Go to Shift-DR */

	JTAG_TCK_RESET;
	JTAG_TCK_SET;	/* Now the Jtag is in the Shift-DR state */
	if (data_pl-1-data_pp>0) 
	    JTAG_TDI_ASSIGN(1);

	for(i=0;i<(data_pl-1-data_pp);i++) {
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}

	for (i=0;i<bit_count;i++) {
		JTAG_TDI_ASSIGN(data);data>>=1;
		
		if ((data_pp==0)&&(i==(bit_count-1))) 
		    JTAG_TMS_SET;	/* Go to Exit1-DR */

		JTAG_TCK_RESET;JTAG_TCK_SET;
		
		result>>=1;
		result|=((unsigned long int)JTAG_TDO_VALUE)<<(bit_count-1);
	}
	if (data_pp) 
	    JTAG_TDI_ASSIGN(1);
	for (i=0;i<data_pp;i++) {
		if (i==(data_pp-1)) 
		    JTAG_TMS_SET;			/* Go to Exit1-DR */

		JTAG_TCK_RESET;JTAG_TCK_SET;
	}
	JTAG_TCK_RESET;	/* Go to Update-DR */
	JTAG_TCK_SET;
	
	JTAG_TMS_RESET;
	JTAG_TCK_RESET;	/* Go to run-test-idle */
	JTAG_TCK_SET;
	return(result);
}


void OCCS_Init(void)
{
	unsigned short sim_val, occs_val;

	/*clrSetReg16Bits(OCCS_OSCTL1, 0x3FF, ((word)SIM_NVMOPT2H & 0x3FF)); //Set the trim osc freq with factory programmed value */
	eonce_move_long_to_r2(SIM_NVMOPT2H); 
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
	eonce_nop();
	eonce_move_at_r2_to_y0();
	eonce_move_y0_at_r0();
	sim_val = eonce_rx_upper_data();

	eonce_move_long_to_r2(OCCS_OSCTL1); 
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
	eonce_nop();
	eonce_move_at_r2_to_y0();
	eonce_move_y0_at_r0();
	occs_val = eonce_rx_upper_data();
	occs_val |= (sim_val & 0x3FF);
//	eonce_move_long_to_r2(OCCS_OSCTL1); 
	eonce_move_value_at_r2(occs_val);

	/* clrSetReg16Bits(OCCS_OSCTL1, 0xF400, (0x1400 & 0xF400)); //Set the trim osc freq with factory programmed value  */
	occs_val = (occs_val & ~(0xF400)) | (0x1400 & 0xF400);
	eonce_move_value_at_r2(occs_val);


	/*     temp = SIM_NVMOPT2H & 0x3C00;
	OCCS_OSCTL2 = (OCCS_OSCTL2 & ~0x3C00) | temp; // program temprature trim */
	eonce_move_long_to_r2(OCCS_OSCTL2); 
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
	eonce_nop();
	eonce_move_at_r2_to_y0();
	eonce_move_y0_at_r0();
	occs_val = eonce_rx_upper_data();
	occs_val = (occs_val & ~0x3C00) | (sim_val & 0x3C00);
	eonce_move_value_at_r2(occs_val);

	/* clrSetReg16Bits(OCCS_OSCTL2, 0xC200, 0xC000); // Set the trim osc freq with factory programmed value */
	occs_val = (occs_val & ~(0xC200)) | 0xC000 ;
	eonce_move_value_at_r2(occs_val);


#if (env_type==FSL_internal_use) && (print_inside_function==1)
	print("sim_val=0x%x   occs_val=0x%x \n", sim_val, occs_val);
#endif
	
}

/* brings target into Debug mode and enables the Once interface */
int init_target (dsc *target) {
	int status=0,i,k;
	long int j;
	unsigned long int result;
	
	jtag_measure_paths();				/* measure JTAG chain length */
	jtag_instruction_exec(0x02,8);		/* IDCODE */
	result=jtag_data_shift(0,32);		/* read IDcode */
	target->chipID   = result;
	target->instr_pl = instr_pl;
	target->data_pl  = data_pl;

#if (env_type==FSL_internal_use) && (print_inside_function==1)
	/* print JTAG path lengths */
	print("After reseting the part in programming, JTAG paths now are as follows:\n");
	print("JTAG IR path length: %d\r\n",get_instr_pl());		
	print("JTAG DR path length: %d (BYPASS)\r\n",get_data_pl());
#endif


	select_core_tap();
/* select core TAP interface and rescan the JTAG path 
 * since the path is changed for chip TAP to core TAP  */
							

	status=jtag_instruction_exec(0x2,4);	/* Core IDCODE */
	result=jtag_data_shift(0,32);			/* read core IDcode */
	target->coreID = result;
	target->instr_pl = instr_pl;
	target->data_pl  = data_pl;


#if (env_type==FSL_internal_use) && (print_inside_function==1)
	core_status(status);print("\n");
	print("After selecting the core TAP of the part in programming, \
	       JTAG paths are changed now!!!\n");
	print("JTAG IR path length: %d\r\n",get_instr_pl());		
	/* print JTAG path lengths */
	print("JTAG DR path length: %d (BYPASS)\r\n",get_data_pl());

#endif
	

	
	status=jtag_instruction_exec(0x7,4);		/*Debug Request*/
	

	// deleta by kevin
	//JTAG_RESET_SET;	/* /RESET signal goes high */


	/* wait for stable for about 28us */
	//for(i=0;i<4000;i++)
	for(i=0;i<350;i++)
	    WAIT_100_NS;

	status=jtag_instruction_exec(0x7,4);		/*Debug Request #2 */
	target->status = status;

#if (env_type==FSL_internal_use) && (print_inside_function==1)
	print("Debug Request status: %#x - ",status);
	core_status(status);print("\n");
#endif


	i=RETRY_DEBUG;
	do {
		status=jtag_instruction_exec(0x6,4);	/*Enable OnCE*/
		target->status = status;

#if (env_type==FSL_internal_use) && (print_inside_function==1)
		print("Enable OnCE status: %#x - ",status);
		core_status(status);print("\n");
#endif
		
		if (!(i--)) {
#if (env_type==FSL_internal_use) && (print_inside_function==1)
			print("Target chip refused to enter Debug mode!\r\n");
#endif
			target->status = status;
			return(1);
		}
	} while (status!=0xd);

#if (env_type==FSL_internal_use) && (print_inside_function==1)
	print("Enable OnCE successful, target chip is in Debug mode\r\n");
#endif


	return(0);
}


/* writes 8 bits to the JTAG DR path */
/* expects shift-dr state of the Jtag state machine state upon entry */
/* and leaves the Jtag in Shift-DR state on exit */
void jtag_data_write8_shift_dr(unsigned int data) {
	int i;

	for (i=0;i<8;i++) {
		JTAG_TDI_ASSIGN(data);
		data>>=1;
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}
	if (data_pp) JTAG_TDI_ASSIGN(1);
	for (i=0;i<data_pp;i++) {
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}

}


/* writes 7 bits to the JTAG DR path */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
void jtag_data_write7(unsigned int data) {
	int i;
	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;								/* Go to Capture-DR */
	JTAG_TCK_RESET;
	WAIT_100_NS;
	WAIT_100_NS;
	JTAG_TCK_SET;
	JTAG_TCK_RESET;								/* Go to Shift-DR */
	JTAG_TCK_SET;								/* Now the Jtag is in the Shift-DR state */

	for (i=0;i<7;i++) {
		JTAG_TDI_ASSIGN(data);
		data>>=1;
		if ((data_pp==0)&&(i==6)) JTAG_TMS_SET;		/* Go to Exit1-DR */
		JTAG_TCK_RESET;
		JTAG_TCK_SET;	
#if 0
		if ((data_pp==0)&&(i==6))
		{
			JTAG_TMS_SET;		/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;	
		}
#endif
	}
	if (data_pp) JTAG_TDI_ASSIGN(1);
	for (i=0;i<data_pp;i++) {
		if (i==(data_pp-1))
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
		}
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
#if 0
		if (i==(data_pp-1))
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif
	}
	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDle */
	JTAG_TCK_SET;
}


/* writes 8 bits to the JTAG DR path */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
void jtag_data_write8(unsigned int data) {
	int i;
	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;								/* Go to Capture-DR */
	JTAG_TCK_RESET;
	WAIT_100_NS;
	WAIT_100_NS;
	JTAG_TCK_SET;
	JTAG_TCK_RESET;								/* Go to Shift-DR */
	JTAG_TCK_SET;								/* Now the Jtag is in the Shift-DR state */

	for (i=0;i<8;i++) {
		JTAG_TDI_ASSIGN(data);
		data>>=1;
		if ((data_pp==0)&&(i==7)) JTAG_TMS_SET;		/* Go to Exit1-DR */
		JTAG_TCK_RESET;
		JTAG_TCK_SET;	
#if 0
		if ((data_pp==0)&&(i==7))
		{
			JTAG_TMS_SET;		/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;	
		}
#endif
	}
	if (data_pp) JTAG_TDI_ASSIGN(1);
	for (i=0;i<data_pp;i++) {
		if (i==(data_pp-1))
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
		}
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
#if 0
		if (i==(data_pp-1))
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif
	}
	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDle */
	JTAG_TCK_SET;
}

/* writes 16 bits to the JTAG DR path */
/* expects RUN-TEST-IDLE state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
void jtag_data_write16(unsigned int data) {
	int i;
	
	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;								/* Go to Capture-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
	JTAG_TCK_RESET;								/* Go to Shift-DR */
	JTAG_TCK_SET;								/* Now the Jtag is in the Shift-DR state */

	for (i=0;i<16;i++) {
		JTAG_TDI_ASSIGN(data);
		data>>=1;
		if ((data_pp==0)&&(i==15)) JTAG_TMS_SET;		/* Go to Exit1-DR */
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
#if 0
		if ((data_pp==0)&&(i==15)) 
		{
			JTAG_TMS_SET;		/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif		
	}
	if (data_pp) JTAG_TDI_ASSIGN(1);
	for (i=0;i<data_pp;i++) {
		if (i==(data_pp-1)) 
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
		}
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
#if 0
		if (i==(data_pp-1)) 
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif
	}
	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */
	JTAG_TCK_SET;
	
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */
	JTAG_TCK_SET;

	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */
	JTAG_TCK_SET;
	
}

/* writes three 16 bit words to the JTAG DR path */
/* the parameters are: N - number of words to write, datap - pointer to the actual data */
/* expects RUN-TEST-IDLE state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
void jtag_data_write_n_words(unsigned int n,unsigned int *datap) {
	int i;
	unsigned int data;

	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;
	
	JTAG_TMS_RESET;									/* Go to Capture-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

//	JTAG_TMS_RESET;									/* Go to Shift-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	while(n) {
		data=*datap;
		for (i=0;i<16;i++) {
			JTAG_TDI_ASSIGN(data);
			data>>=1;
			if ((n==1)&&(data_pp==0)&&(i==15)) JTAG_TMS_SET;		/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
		if (data_pp) JTAG_TDI_ASSIGN(1);
		n--;
		datap++;
	}
	for (i=0;i<data_pp;i++) {
		if (i==(data_pp-1)) 
		{
			JTAG_TMS_SET;			/* Go to Exit1-DR */
		}
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}
	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */
	JTAG_TCK_SET;
}

/* reads 16 bits from the jtag DR path */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
unsigned int jtag_data_read16(void) {
	int i;
	unsigned int result=0;

	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;								/* Go to Capture-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;								/* Go to Shift-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;								/* Now the Jtag is in the Shift-DR state */
	if (data_pl-1-data_pp>0) JTAG_TDI_ASSIGN(1);
	for(i=0;i<(data_pl-1-data_pp);i++) {
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}
	for (i=0;i<16;i++) {
		if (i==15) 
		{
			JTAG_TMS_SET;					/* Go to Exit1-DR */
		}
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
		result>>=1;
		result|=JTAG_TDO_VALUE_SHIFTED_15;
		//if (JTAG_TDO_VALUE_SHIFTED_15 == 0) 
		//    ;
		//else
		//    result |= (1<<15);


		
#if 0
		if (i==15) 
		{
			JTAG_TMS_SET;					/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif
	}

	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */

	JTAG_TCK_SET;
	return(result);
}

/* reads 8 bits from the jtag DR path */
/* expects run-test-idle state of the Jtag state machine state upon entry */
/* and leaves the Jtag in run-test-idle on exit */
unsigned int jtag_data_read8(void) {
	int i;
	unsigned int result=0;
	JTAG_TMS_SET;									/* Go to Select-DR-Scan */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;

	JTAG_TMS_RESET;								/* Go to Capture-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;								/* Go to Shift-DR */
	JTAG_TCK_RESET;
	JTAG_TCK_SET;								/* Now the Jtag is in the Shift-DR state */
	if (data_pl-1-data_pp>0) JTAG_TDI_ASSIGN(1);
	for(i=0;i<(data_pl-1-data_pp);i++) {
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
	}
	for (i=0;i<8;i++) {
		if (i==7) JTAG_TMS_SET;	/* Go to Exit1-DR */
		JTAG_TCK_RESET;
		JTAG_TCK_SET;
		result>>=1;
		result|=JTAG_TDO_VALUE_SHIFTED_15;
#if 0
		if (i==7) 
		{
			JTAG_TMS_SET;					/* Go to Exit1-DR */
			JTAG_TCK_RESET;
			JTAG_TCK_SET;
		}
#endif
	}
	result>>=8;


	JTAG_TCK_RESET;								/* Go to Update-DR */
	JTAG_TCK_SET;

	JTAG_TMS_RESET;
	JTAG_TCK_RESET;								/* Go to RUN-TEST-IDLE */

	JTAG_TCK_SET;
	return(result);
}

/* wrapper around jtag_data_write_n_words */
/* this enables use of eonce_execute_instruction macro */
void eonce_data_write(unsigned int words, unsigned int data0, unsigned int data1, unsigned int data2) {
 	unsigned int data[3];
	data[0]=data0;
	data[1]=data1;
	data[2]=data2;
	jtag_data_write_n_words(words,data);
}

/* selects the CORE TAP Controller by modifying the TLM register */
/* to be used when Master TAP is selected */
void select_core_tap(void) {
	jtag_instruction_exec(0x05,8);	/* select TLM */
	jtag_data_shift(0x02,4);		/* select core TAP */
	jtag_measure_paths();			/* rescan jtag chain (Core TAP and Master TAP have different intruction lengths) */
}

/* selects the Master TAP Controller by modifying the TLM register */
/* to be used when Core TAP is selected */
void select_master_tap(void) {
	jtag_instruction_exec(0x08,4);	/* select TLM */
	jtag_data_shift(0x01,4);		/* select master TAP */
	jtag_measure_paths();			/* rescan jtag chain (Core TAP and Master TAP have different intruction lengths) */

}

/* prints core status */
void core_status(int status) {
  #if env_type == FSL_internal_use
	switch (status) {
		case (0x01):	print("Normal/Exception/Reset\r"); break;
		case (0x05):	print("Stop/Wait             \r"); break;
		case (0x09):	print("Busy                  \r"); break;
		case (0x0d):	print("Debug                 \r"); break;
		default:		print("Invalid (no power?)   \r"); break;
	}
  #endif 
}

//kevin: program 4 byte with array content
unsigned int FlashProgram_4bytes_LongWord(unsigned int stAddr,unsigned char *data)
{
    unsigned char chVar;
    unsigned char command[8];

    command[0] = FTFx_PROGRAM_LONGWORD;
    command[1] = (UINT8)(  stAddr>>16);
    command[2] = (UINT8)( (stAddr>>8) & 0xff );
    command[3] = (UINT8)(  stAddr     & 0xff );
    for (chVar = 7; chVar>3; chVar--) 
	command[chVar] = data[7-chVar];

     
    return(  eonce_ftfa_execute_command(command, 7) );
}

//kevin: verify 4 byte with array content
unsigned int FlashCheck_4bytes_LongWord(unsigned int stAddr,unsigned char *data)
{
    unsigned char command[0xc];
    unsigned int flashSTATE;

    command[0] = FTFx_PROGRAM_CHECK;
    command[1] = (UINT8)(  stAddr>>16);
    command[2] = (UINT8)( (stAddr>>8) & 0xff );
    command[3] = (UINT8)(  stAddr     & 0xff );
    command[4] = (UINT8) 2; //  factory margin

    //command[8] = (UINT8) ( data[3]+GETINDEX(0) );
    //command[9] = (UINT8) ( data[2]+GETINDEX(1) );
    //command[0xa] = (UINT8) ( data[1]+GETINDEX(2) );
    //command[0xb] = (UINT8) ( data[0]+GETINDEX(3) );

    command[8] = (UINT8) ( data[3]);
    command[9] = (UINT8) ( data[2]);
    command[0xa] = (UINT8) ( data[1]);
    command[0xb] = (UINT8) ( data[0]);
    flashSTATE = eonce_ftfa_execute_command(command, 11);

#if env_type==FSL_internal_use
    if (flashSTATE) 
	printf("--ftfa execute fail!");
#endif

    //flashSTATE = REG_READ(FTFA_BASE_PTR);
    //return( flashSTATE&FTFx_SSD_FSTAT_MGSTAT0  );
    //
    return flashSTATE;
}


/* Ethan Add for 56F827xx */
/* returns FTFA_FSTAT after executing the command */
unsigned int eonce_ftfa_execute_command(unsigned char *pCommandArray, unsigned char index) 
{
    unsigned int ftfa_fstat_reg;
    unsigned char counter;
    unsigned short command; 

    eonce_move_long_to_r2(FTFA_BASE_PTR);/* load FTFA base to r2 */
    eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);	
    /* load OTx/ORx reg address */

    eonce_nop();

    do {
	eonce_move_at_r2_to_y0();   /* read FTFA_FSTAT */
	eonce_move_y0_at_r0();
	ftfa_fstat_reg=eonce_rx_upper_data();
    } while (!( ftfa_fstat_reg & FTFx_SSD_FSTAT_CCIF));
    /* wait until current command is complete (CCIF ==1)*/
    //	print(" ftfa_fstat_reg = 0x%x\n", ftfa_fstat_reg);
    
    /* clear RDCOLERR & ACCERR & FPVIOL flag in flash status register. Write 1 to clear*/
    eonce_move_value_at_r2(
	    FTFx_SSD_FSTAT_RDCOLERR|FTFx_SSD_FSTAT_ACCERR|FTFx_SSD_FSTAT_FPVIOL
	    );
    
    eonce_nop();

    /* load FCCOB registers */
    for(counter=0; counter<=index; counter++)
    {
	switch (counter)
	{
	    case 0: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB0_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB0_OFFSET);
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		//print("case 0 command  = 0x%x\n", command);
		break;
	    case 1: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB1_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB1_OFFSET);
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		//print("case 1 command = 0x%x\n", command);
		break;
	    case 2: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB2_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB2_OFFSET);	
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		break;
	    case 3: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB3_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB3_OFFSET);	
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		break;
	    case 4: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB4_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB4_OFFSET);	
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		break;
	    case 5: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB5_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB5_OFFSET);	
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		break;
	    case 6: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB6_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB6_OFFSET);	
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		break;
	    case 7: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB7_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB7_OFFSET);	
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		break;
	    case 8: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB8_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB8_OFFSET);	
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		break;
	    case 9: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOB9_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOB9_OFFSET);	
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		break;
	    case 10: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOBA_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOBA_OFFSET);	
		command = pCommandArray[counter] << 8 ;
		eonce_move_value_at_r2(command);
		break;
	    case 11: //REG_WRITE(PSSDConfig->ftfxRegBase + FTFx_SSD_FCCOBB_OFFSET, pCommandArray[counter]);
		eonce_move_long_to_r2(FTFx_SSD_FCCOBB_OFFSET);	
		command =( (pCommandArray[counter - 1]) << 8) | (pCommandArray[counter] & 0xFF);
		eonce_move_value_at_r2(command);
		break;

	    default: break;
	}
    }

    eonce_move_long_to_r2(FTFA_BASE_PTR);   /* load FTFA base to r2 */		
    eonce_move_value_at_r2(FTFx_SSD_FSTAT_CCIF);    /* clear CCIF to launch command */

    do {
	eonce_move_at_r2_to_y0();   /* read FTFA_FSTAT */
	ftfa_fstat_reg=eonce_rx_upper_data();
    } while (!( ftfa_fstat_reg & FTFx_SSD_FSTAT_CCIF));
    /* wait until current command is complete (CCIF ==1)*/

    //	print("FTFA command is complete !! ftfa_fstat_reg = 0x%x\n", ftfa_fstat_reg);




    /* checking access error */
    if(0 != (ftfa_fstat_reg & FTFx_SSD_FSTAT_ACCERR)) {
#if (env_type==FSL_internal_use) && (print_inside_function==1)
	print("error code FTFx_ERR_ACCERR! FTFA Command=0x%x \r\n", pCommandArray[0]);
#endif  /* return an error code FTFx_ERR_ACCERR */
	return(SYSTEM_ERROR);
    }
    else if(0 != (ftfa_fstat_reg & FTFx_SSD_FSTAT_FPVIOL)){/* checking protection error */
	/* return an error code FTFx_ERR_PVIOL */
#if irdr_dump_by_printf ==1
	print("error code FTFx_ERR_PVIOL!\r\n");
#endif
	return(SYSTEM_ERROR);
    }
    else if(0 != (ftfa_fstat_reg & FTFx_SSD_FSTAT_MGSTAT0)){/* checking MGSTAT0 non-correctable error */
#if (env_type==FSL_internal_use) && (print_inside_function==1)
	/* return an error code FTFx_ERR_MGSTAT0 */
	print("error code FTFx_ERR_MGSTAT0!\r\n");
#endif
	return(SYSTEM_ERROR);
    }

    return(SUCCESS);
}

void tog_led(UINT8 ledDR)  //Ethan, just for test
{
    int i ,j =0;

#if 1
    eonce_move_long_to_r2(0xE40C); 
    eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
    eonce_nop();
    eonce_move_at_r2_disp_to_y0(0); // offset = 0
    eonce_move_y0_at_r0();
    i=eonce_rx_upper_data();		// read Trim value to I
    i |= 0x04U;

    //	eonce_move_long_to_r2(FTFx_REG_BASE);	
    eonce_move_value_at_r2(i);

    eonce_move_long_to_r2(0xE243);	
    eonce_move_value_at_r2(0x0000);

    eonce_move_long_to_r2(0xE242);	
    eonce_move_value_at_r2(0x00FF);

    eonce_move_long_to_r2(0xE241);	
    eonce_move_value_at_r2(0xFFFF);

    {
	eonce_move_long_to_r2(0xE241);	
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
	eonce_nop();
	eonce_move_at_r2_disp_to_y0(0); // offset = 0
	eonce_move_y0_at_r0();
	i=eonce_rx_upper_data();		// read Trim value to I

	    eonce_move_value_at_r2(ledDR);

	//print("tog_led value=0x%x \n", i);
	//system("pause");
    }

#else  
void tog_led(void)  //original version from Ethan
    eonce_move_long_to_r2(0xE40C); 
    eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
    eonce_nop();
    eonce_move_at_r2_disp_to_y0(0); // offset = 0
    eonce_move_y0_at_r0();
    i=eonce_rx_upper_data();		// read Trim value to I
    i |= 0x04U;

    //	eonce_move_long_to_r2(FTFx_REG_BASE);	
    eonce_move_value_at_r2(i);

    eonce_move_long_to_r2(0xE243);	
    eonce_move_value_at_r2(0x0000);

    eonce_move_long_to_r2(0xE242);	
    eonce_move_value_at_r2(0x00FF);

    eonce_move_long_to_r2(0xE241);	
    eonce_move_value_at_r2(0xFFFF);

    while(1){
	eonce_move_long_to_r2(0xE241);	
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);
	eonce_nop();
	eonce_move_at_r2_disp_to_y0(0); // offset = 0
	eonce_move_y0_at_r0();
	i=eonce_rx_upper_data();		// read Trim value to I
	if( j%2 ==0){
	    eonce_move_value_at_r2(0x00);
	    j ++;
	}
	else if(j <= 3){
	    eonce_move_value_at_r2(0xFF);
	    j ++;
	}
	else{ 
	    print("break \n", i);
	    break;
	}
	print("tog_led value=0x%x \n", i);
	system("pause");
    }


#endif
}

unsigned short REG_READ(unsigned short reg_addr)
{
	unsigned short reg_val;
	
	eonce_move_long_to_r2(reg_addr);						/* load reg_addr to r2 */
	eonce_move_long_to_r0(((unsigned long)(eonce_base)<<16)+0xffff);	/* load OTx/ORx reg address */
	eonce_nop();

	eonce_move_at_r2_to_y0();			/* read FTFA_FSTAT */
	eonce_move_y0_at_r0();
	reg_val=eonce_rx_upper_data();

	return reg_val;
}
unsigned short REG_WRITE(unsigned short reg_addr, unsigned short reg_val)
{
	eonce_move_long_to_r2(reg_addr);	
	eonce_move_value_at_r2(reg_val);

	return reg_val;
}

UINT32 FlashEraseAllBlock(void){
	unsigned char commandArray[1];     /* command sequence array */
	unsigned int returnCode;         /* return code variable */

	 commandArray[0] = FTFx_ERASE_ALL_BLOCK;
	 returnCode = eonce_ftfa_execute_command(commandArray, 0);

	 return(returnCode);
}

UINT32 FlashVerifyAllBlock(unsigned char marginLevel){ //Margin Level: normal=0x0, user=0x01
	unsigned char commandArray[2];     /* command sequence array */
	unsigned int returnCode;         /* return code variable */

    /* preparing passing parameter to verify all block command*/
    commandArray[0] = FTFx_VERIFY_ALL_BLOCK;
    commandArray[1] = marginLevel;

    /* calling flash command sequence function to execute the command */
    returnCode = eonce_ftfa_execute_command(commandArray, 1);

    return(returnCode);
}

/*
UINT32 FlashProgramLongword(PFLASH_SSD_CONFIG PSSDConfig, unsigned int destination, \
	                    unsigned int size, unsigned char* pData)
*/
UINT32 FlashProgramLongword(PFLASH_SSD_CONFIG PSSDConfig, unsigned long destination, \
	                    unsigned int size, unsigned char* pData)
{
    unsigned char pCommandArray[8]; /* command sequence array */
    unsigned int returnCode = 0;      /* return code variable */
    unsigned int endAddress;      /* storing end address */
    unsigned char i;

	/* convert to byte addr */
	destination = WORD2BYTE(destination);
    /* calculating Flash end address */
	endAddress = destination + size;
        
    /* check if the destination is Longword aligned or not */
	if (destination & (FTFx_LONGWORD_SIZE - 1))
    {
        /* return an error code FTFx_ERR_ADDR */
		return(SYSTEM_ERROR);
    }  
    /* check if the size is Longword alignment or not */
    if(size & (FTFx_LONGWORD_SIZE-1))
    {
        /* return an error code FTFx_ERR_SIZE */
		return(SYSTEM_ERROR);
    }
    
    /* check for valid range of the target addresses */
    if ((destination < WORD2BYTE(PSSDConfig->PFlashBlockBase)) || 
		(endAddress > (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
    {
        	/* return an error code FTFx_ERR_RANGE */
			return(SYSTEM_ERROR);
    }
    else
    {
        /* Convert System memory address to FTFx internal memory address */ 
		destination -= WORD2BYTE(PSSDConfig->PFlashBlockBase);
#if env_type==FSL_internal_use
		//print("FlashProgramLongword destination=0x%x \n", destination);
		print("FlashProgramLongword destination=0x%x ", destination);
#endif
    }
	
    /* check for error return code */
	while(size > 0)
	{
		/* preparing passing parameter to program the flash block */
		pCommandArray[0] = FTFx_PROGRAM_LONGWORD;
		pCommandArray[1] = (UINT8)(destination >> 16);
		pCommandArray[2] = (UINT8)((destination >> 8) & 0xFF);
		pCommandArray[3] = (UINT8)(destination & 0xFF);
#if env_type==FSL_internal_use
		printf("CommandArray[1:3] = 0x %x.%x.%x.  ", pCommandArray[1], \
			pCommandArray[2], pCommandArray[3]);
#endif

		for (i = 4; i < 8; i ++)
		{
			//pCommandArray[i] = READ8(source + GETINDEX(i - 4));
			pCommandArray[i] = *(pData + GETINDEX(i - 4));
		}
#if env_type==FSL_internal_use
		printf("CommandArray[4:7] = 0x %x.%x.%x.%x\n", pCommandArray[4], \
			pCommandArray[5], pCommandArray[6],pCommandArray[7]);
#endif
		
		/* calling flash command sequence function to execute the command */
		returnCode = eonce_ftfa_execute_command(pCommandArray, 7);
	  
		/* checking for the success of command execution */
		if(SUCCESS != returnCode)
		{
			return(SYSTEM_ERROR);
			break;
		}
		else
		{
			/* update destination address for next iteration */
			destination += FTFx_LONGWORD_SIZE;
			/* update size for next iteration */
			size -= FTFx_LONGWORD_SIZE;
					/* increment the source adress by 1 */
			pData += FTFx_LONGWORD_SIZE;
		}
	}
	return(returnCode);
}

#if 0 //Ethan, this is useless
UINT32 FlashCheckSum(PFLASH_SSD_CONFIG PSSDConfig, UINT32 destination, UINT32 size,  UINT32* PSum){
    UINT32 data;             /* Data read from Flash address */
    UINT32 returnCode = FTFx_OK;       /* Return code variable */
    UINT32 endAddress;       /* P Flash end address */

    /* convert to byte address */
	destination = WORD2BYTE(destination);
    /* calculating Flash end address */
    endAddress = destination + size;
    
    /* check for valid range of the target addresses */
    if ((destination < WORD2BYTE(PSSDConfig->PFlashBlockBase)) || 
		(endAddress > (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
    {
        /* return an error code FTFx_ERR_RANGE */
        return(SYSTEM_ERROR);
    }
	*PSum = 0;
	/* doing sum operation */
	while(size > 0)
	{          
		data = READ8(destination);  
		*PSum += (UINT32)data;
		destination += 1;
		size -= 1;
	}

	return(SUCCESS);
}
#endif
UINT32 FlashProgramCheck(PFLASH_SSD_CONFIG PSSDConfig,  UINT32  destination, UINT32  size, \
								UINT8*  pExpectedData,UINT32* pFailAddr, UINT8   marginLevel)
{
    UINT8 pCommandArray[12]; /* command sequence array */
    UINT32 returnCode;      /* return code variable */
	UINT32 offsetAddr; /* offset addr to convert to internal memory addr */
	UINT32 endAddress;

    /* convert to byte address */
	destination = WORD2BYTE(destination);
	endAddress = destination + size;
    /* check if the destination is Longword aligned or not */
    if ((destination & (PGMCHK_ALIGN_SIZE - 1)))
    {
        return(SYSTEM_ERROR);
    }  
    /* check if the size is Longword aligned or not */
	if (size & (PGMCHK_ALIGN_SIZE - 1))	
    {
        return(SYSTEM_ERROR);
    }
    /* check for valid range of the target addresses */    
    if ((destination < WORD2BYTE(PSSDConfig->PFlashBlockBase)) || 
		(endAddress > (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
	{
            /* return an error code FTFx_ERR_RANGE */
            return(SYSTEM_ERROR);
    }
    else
    {
		offsetAddr = WORD2BYTE(PSSDConfig->PFlashBlockBase);
    }

	/* convert system memory addr to FTFx internal memory addr*/
	destination = destination - offsetAddr;
    while (size)
    {
		/* preparing passing parameter to program check the flash block */
		pCommandArray[0] = FTFx_PROGRAM_CHECK;
		pCommandArray[1] = (UINT8)(destination >> 16);
		pCommandArray[2] = (UINT8)((destination >> 8) & 0xFF);
		pCommandArray[3] = (UINT8)(destination & 0xFF);
		pCommandArray[4] = marginLevel;        

		pCommandArray[8] = *(pExpectedData + GETINDEX(0));
		pCommandArray[9] = *(pExpectedData + GETINDEX(1));
		pCommandArray[10] = *(pExpectedData + GETINDEX(2));
		pCommandArray[11] = *(pExpectedData + GETINDEX(3));

		/* calling flash command sequence function to execute the command */
		returnCode = eonce_ftfa_execute_command(pCommandArray, 11);
		
		/* checking for the success of command execution */
		if(SUCCESS != returnCode)
		{
			return(SYSTEM_ERROR);
		}
		
		size-=PGMCHK_ALIGN_SIZE;
		pExpectedData+=PGMCHK_ALIGN_SIZE;
		destination+= PGMCHK_ALIGN_SIZE;
    }

	return(returnCode);
}

UINT32 FlashReadResource(PFLASH_SSD_CONFIG PSSDConfig, \
                                                UINT32 destination, \
                                                UINT8* pDataArray, \
                                                UINT8  resourceSelectCode)
{
    UINT8  pCommandArray[RDRSRC_LAST_IDX + 1]; /* command sequence array */
    UINT8 i;
    UINT32 returnCode;       /* return code variable */
    /* convert to byte addr */
	destination = WORD2BYTE(destination);
	
    /* check if the destination is longword aligned or not */
    if (destination & (RDRSRC_ALIGN_SIZE-1))
    {
        /* return an error code FTFx_ERR_ADDR */
        return(SYSTEM_ERROR);
    }
    /* check for valid range of the destination */
    if((destination >= WORD2BYTE(PSSDConfig->PFlashBlockBase)) && \
        (destination < (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
    {
        /* Convert System memory address to FTFx internal memory address */
		destination -= WORD2BYTE(PSSDConfig->PFlashBlockBase);
    }
    /* preparing passing parameter for read resource command */
    /* 1st element for the FCCOB register */
    pCommandArray[0] = FTFx_READ_RESOURCE;
    pCommandArray[1] = (UINT8)(destination >> 16);
    pCommandArray[2] = (UINT8)((destination >> 8) & 0xFF);
    pCommandArray[3] = (UINT8)(destination & 0xFF);

	pCommandArray[RSRC_CODE_IDX] = resourceSelectCode;
	returnCode = eonce_ftfa_execute_command(pCommandArray, RDRSRC_LAST_IDX);
	   
    if (SUCCESS == returnCode)
	{
		/* Read the data from the FCCOB registers into the pDataArray */
		for (i = 4; i < RDRSRC_SIZE + 4; i ++)
		{
			// Ethan : need to implement
                  //pDataArray[GETINDEX(i - 4)] = REG_READ(PSSDConfig->ftfxRegBase + GETINDEX(i + 4));
		}
	}

    return(returnCode);
}

UINT32 FlashGetSecurityState(PFLASH_SSD_CONFIG PSSDConfig, UINT8* securityState)
{
    /* store data read from flash register */
    unsigned short  registerValue;

    /*Get flash security register value */
    registerValue = REG_READ(0xE3C1); //read Flash Security Register (FTFA_FSEC)

    /* check the status of the flash security bits in the security register */
    if(0x02 == (registerValue & FTFx_SSD_FSEC_SEC))
    {
        /* Flash in unsecured state */
        *securityState = FLASH_NOT_SECURE;
#if env_type == FSL_internal_use
	print("Flash in unsecured state \r\n");
#endif
    }
    else
    {
        /* Flash in secured state */
        /* check for backdoor key security enable bit */
        if(0x80 == (registerValue & FTFx_SSD_FSEC_KEYEN))
        {
            /* Backdoor key security enabled */
            *securityState = FLASH_SECURE_BACKDOOR_ENABLED;
#if env_type == FSL_internal_use
		print("Backdoor key security enabled \r\n");
#endif
        }
        else
        {
            /* Backdoor key security disabled */
            *securityState = FLASH_SECURE_BACKDOOR_DISABLED;
#if env_type == FSL_internal_use
		print("Backdoor key security disabled \r\n");
#endif
        }
    }

    return(SUCCESS);
}

UINT32 FlashEraseSector(PFLASH_SSD_CONFIG PSSDConfig, UINT32 destination, UINT32 size)
{
    UINT8 pCommandArray[4]; /* command sequence array */
    UINT32 returnCode = SUCCESS;      /* return code variable */
    UINT32 sectorSize;        /* size of one sector */
    UINT32 addrAlign;    /* storing the aligned size condition of destination  */
	UINT32 endBlock;    /* store the fist out of range addr of block */
	UINT32 offSetAddr;	/* offset addr to convert to internal memory addr */
	
    /* convert to byte address */
	destination = WORD2BYTE(destination);
	/* check range for destination*/
    if((destination < WORD2BYTE(PSSDConfig->PFlashBlockBase)) || \
        (destination >= (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
	{
        	/* return an error code FTFx_ERR_RANGE */
            return(SYSTEM_ERROR);
    }
    else
    {
		sectorSize = FTFx_PSECTOR_SIZE;
		addrAlign = PERSSEC_ALIGN_SIZE;
		endBlock = WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize;
		offSetAddr = WORD2BYTE(PSSDConfig->PFlashBlockBase);
    } 
	/* check for aligment of destination */
	if (destination & (addrAlign-1))
    {
        /* return an error code FTFx_ERR_ADDR */
        return(SYSTEM_ERROR);
    }
    
    /* check if the size is sector alignment or not */
    if(size & (sectorSize-1))
    {
        /* return an error code FTFx_ERR_SIZE */
        return(SYSTEM_ERROR);
    }
	/* calculating Flash end address. need to move destination to the beginning of each sector */
	//endAddress = (destination/sectorSize)*sectorSize + size;
	/* check range for endaddr */
	if (((destination/sectorSize)*sectorSize + size) <= endBlock)
	{
		/* Convert System memory address to FTFx internal memory address */ 
		destination -= offSetAddr;
	}
	else
	{
		return(SYSTEM_ERROR);
	}

	while(size > 0)
	{
		/* preparing passing parameter to erase a flash block */
		pCommandArray[0] = FTFx_ERASE_SECTOR;
		pCommandArray[1] = (UINT8)(destination >> 16);
		pCommandArray[2] = (UINT8)((destination >> 8) & 0xFF);
		pCommandArray[3] = (UINT8)(destination & 0xFF);
		  
		/* calling flash command sequence function to execute the command */
		returnCode = eonce_ftfa_execute_command(pCommandArray, 3);
		
		/* checking the success of command execution */
		if(SUCCESS != returnCode)
		{
			break;
		}
		else
		{
			/* update size and destination address */
			size -= sectorSize;
			destination += sectorSize;
		}
	}

    return(returnCode);
}	

UINT32 FlashVerifySection(PFLASH_SSD_CONFIG PSSDConfig, UINT32 destination, UINT16 Number, UINT8 marginLevel)
{
    UINT8 pCommandArray[7]; /* command sequence array */
    UINT32 returnCode;      /* return code variable */
    UINT32 endAddress;      /* storing end address. This is the starting point of the next block */   
    UINT32 sectionAlign;
    /* convert to byte addr */
	destination = WORD2BYTE(destination);
    /* check for valid range of the destination */
	if((destination >= WORD2BYTE(PSSDConfig->PFlashBlockBase)) && \
        (destination < (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
    {
		sectionAlign = PRD1SEC_ALIGN_SIZE;
    }
    else
    {
        /* return an error code FTFx_ERR_RANGE */
        return(SYSTEM_ERROR);
    }
     /* calculating Flash end address */
	endAddress = destination + Number * sectionAlign;       
    /* check for valid range of end address */
	if ((endAddress >= WORD2BYTE(PSSDConfig->PFlashBlockBase)) &&
		(endAddress <= (WORD2BYTE(PSSDConfig->PFlashBlockBase) + PSSDConfig->PFlashBlockSize)))
    {
        destination -= WORD2BYTE(PSSDConfig->PFlashBlockBase);
		
    }
    else
    {
        /* return an error code FTFx_ERR_RANGE */
        return(SYSTEM_ERROR);
    }
    
    /* check if the destination is aligned or not */
	if (destination & (sectionAlign-1))
    {
        /* return an error code FTFx_ERR_ADDR */
        return(SYSTEM_ERROR);
    }
        
    /* preparing passing parameter to verify section command */
    pCommandArray[0] = FTFx_VERIFY_SECTION;
    pCommandArray[1] = (UINT8)(destination >> 16);
    pCommandArray[2] = (UINT8)((destination >> 8) & 0xFF);
    pCommandArray[3] = (UINT8)(destination & 0xFF);
    pCommandArray[4] = (UINT8)(Number >> 8);
    pCommandArray[5] = (UINT8)(Number & 0xFF);
    pCommandArray[6] = marginLevel;

    /* calling flash command sequence function to execute the command */
    returnCode = eonce_ftfa_execute_command(pCommandArray, 6);

    return(returnCode);
}	

UINT32 PFlashGetProtection(PFLASH_SSD_CONFIG PSSDConfig, UINT32* protectStatus)
{
    UINT16 registerValue01;
    UINT16 registerValue23;
    
    registerValue01=REG_READ(FTFx_SSD_FPROT0_OFFSET);
    registerValue23=REG_READ(FTFx_SSD_FPROT2_OFFSET);
    
    *protectStatus = (((registerValue01 & 0xFF00) << 16 ) | \
                     ((registerValue01 & 0xFF) << 16 ) | \
                     (registerValue23 & 0xFF00 ) | \
                      (registerValue23 & 0xFF));
#if env_type == FSL_internal_use
	print("read protectStatus = 0x%x \n", *protectStatus);
#endif
                     
  return(SUCCESS);
}

UINT32 PFlashSetProtection(PFLASH_SSD_CONFIG PSSDConfig, UINT32 protectStatus)
{
    UINT32 registerValue;    /* store data read from flash register */
    UINT32 returnCode;       /* return code variable */
    UINT32 temp;
    UINT16 registerValue01;
    UINT16 registerValue23;
  
    /* set the default return as FTFx_OK */
    returnCode = SUCCESS;

    /* Read the value of FPPROT registers */
    registerValue01=REG_READ(FTFx_SSD_FPROT0_OFFSET);
    registerValue23=REG_READ(FTFx_SSD_FPROT2_OFFSET);
    
    registerValue = (((registerValue01 & 0xFF00) << 16 ) | \
                     ((registerValue01 & 0xFF) << 16 ) | \
                     (registerValue23 & 0xFF00) | \
                      (registerValue23 & 0xFF)); 
    
    /* Check if normal mode */
    REG_WRITE(FTFx_SSD_FPROT0_OFFSET, 0xFFFF);
    REG_WRITE(FTFx_SSD_FPROT2_OFFSET, 0xFFFF);

  
    /* Read the value of FPPROT registers */
    registerValue01=REG_READ(FTFx_SSD_FPROT0_OFFSET);
    registerValue23=REG_READ(FTFx_SSD_FPROT2_OFFSET);
    
    if((registerValue01 != 0xFFFF) || (registerValue23 != 0xFFFF))
    {
        /* if normal mode */
        temp = protectStatus ^ registerValue;
        if(temp & protectStatus)
        {
        	return(SYSTEM_ERROR);
    	}
    }   
    /* if unprotected or special mode */
	registerValue01 = ((protectStatus & 0xFF000000) >> 16) |((protectStatus & 0x00FF0000) >> 16);
	registerValue23 = (protectStatus & 0x0000FF00) |(protectStatus & 0x000000FF) ;
#if env_type == FSL_internal_use
	print("Set registerValue01 = 0x%x , registerValue23 = 0x%x \n", registerValue01, registerValue23);
#endif
    REG_WRITE(FTFx_SSD_FPROT0_OFFSET, registerValue01);
    REG_WRITE(FTFx_SSD_FPROT2_OFFSET, registerValue23);
    
    return(SUCCESS);
}	
