/***************************************************************************************/
/* FILE NAME: SD_ADC.c                    COPYRIGHT (c) Freescale 2015    */
/*                                                      All Rights Reserved  						 */
/* DESCRIPTION: SD_ADC program    for LSP/SDADC Example                */
/*                                                                                             				 */
/***************************************************************************************/	
/* REV      AUTHOR        DATE        DESCRIPTION OF CHANGE 			 */
/* ---   -----------    ----------    ---------------------                                     		     */
/* 1.0	  A Turner   			April 2015   Initial Public Release        		     */
/***************************************************************************************/

#define maxSDADCcurrent

#include "../headers/project.h"



void SDADC_INIT(void); 

extern vint16_t SDADC1_RESULTS[];
uint8_t FIFO_DEPTH = 16;

/***************************************************/
/* enable_SDADCs: Use to enable SDADCs             */
/* 												   */
/***************************************************/




void enable_SDADC1 (void)                       
{
    SDADC_1.MCR.B.EN = 1;			 //Power up SDADC 1
}
/***************************************************/
/* enable_FRZ: Use to enable SDADC Freeze in debug */
/* 												   */
/***************************************************/

void enable_FRZ (void)                       
{
    // SDADC_1.MCR.B.FRZ = 1;			 //Power up SDADC 0
    // SDADC_1.MCR.B.FRZ = 1;			 //Power up SDADC 1
     //SDADC_2.MCR.B.FRZ = 1;			 //Power up SDADC 2
    // SDADC_3.MCR.B.FRZ = 1;			 //Power up SDADC 3
    // SDADC_4.MCR.B.FRZ = 1;			 //Power up SDADC 4
    // SDADC_5.MCR.B.FRZ = 1;			 //Power up SDADC 5
}

/***************************************************/
/* set_MCR : Edit to configure SDADC1 conversion   */
/* 			 in Single Ended Mode				   */
/***************************************************/
void set_MCR_SDADC1_SE (void)
{
    SDADC_1.MCR.B.PDR = 0xb;			 //configure OSR 4MHz /80 = 50KHz. This Gives Cut_Off frequency of 0.333 * 50KHz = 16650 Hz
	SDADC_1.MCR.B.PGAN = 0x0;			 //configure GAIN
	SDADC_1.MCR.B.EMSEL = 0x0;			 //configure EXT/INT Modulator
	SDADC_1.MCR.B.HPFEN = 0x0;			 //configure Hi Pass filter
	SDADC_1.MCR.B.GECEN = 0x0;			 //Gain calibration mode disabled
	SDADC_1.MCR.B.VCOMSEL = 0x1;		 // 0x1 - VrefN is Single ended reference, 0x0 = VrefP/2 is single ended reference
	SDADC_1.MCR.B.MODE = 0x1;		     //Single ended =1

}
/***************************************************/
/* set_MCR : Edit to configure SDADC1 conversion   */
/* 			 in Differential Mode				   */
/***************************************************/
void set_MCR_SDADC1_DIFF (void)
{
    SDADC_1.MCR.B.PDR = 0xb;			 //configure OSR 4MHz /80 = 50KHz. This Gives Cut_Off frequency of 0.333 * 50KHz = 16650 Hz
	SDADC_1.MCR.B.PGAN = 0x0;			 //configure GAIN
	SDADC_1.MCR.B.EMSEL = 0x0;			 //configure EXT/INT Modulator
	SDADC_1.MCR.B.HPFEN = 0x0;			 //configure Hi Pass filter
	SDADC_1.MCR.B.GECEN = 0x0;			 //Gain calibration mode disabled
	SDADC_1.MCR.B.VCOMSEL = 0x1;		 //Doesn't matter if 0 or 1 in Diff
	SDADC_1.MCR.B.MODE = 0x0;		     // Differential =0

}



/***************************************************/
/* set_CSR : Edit to configure SDADC conversion    */
/* 												   */
/***************************************************/


void set_CSR_SDADC1 (void)
{
    
	SDADC_1.CSR.R = 0x0;			 //configure CSR 1 VrefN connected to INP and INM
	SDADC_1.CSR.B.ANCHSEL = 0x0;	 //In Differential mode PB0(AN[0]) = INP  PB1 (AN[1]) = INM. For SE PB0 is INP, INM is VREFN
}

/****************************************************/
/* OffsetCal :configure Output settling delay       */
/* 												    */
/****************************************************/


void OSDR_SDADC1_config (void)
{
      SDADC_1.OSDR.R = 0x20;			 //configure OSDR 1 
}
/****************************************************/
/* OffsetCal :Set RESET Key. Updates configuration  */
/* 												    */
/****************************************************/

void RESET_KEY_SDADC1 (void)
{
     SDADC_1.RKR.R = 0x5AF0;			 //configure RKR 1
}

/********************************************************/
/* FIFO_INIT :Configuration of FIFO size and threshold  */
/* 												        */
/********************************************************/

 void FIFO_INIT_SDADC1 (void)
{
     SDADC_1.FCR.B.FTHLD = 0x7;
	 SDADC_1.FCR.B.FE = 1;
	 
 }
/********************************************************/
/* FIFO_INIT :Configuration of FIFO size and threshold  */
/* 												        */
/********************************************************/

 void RSER_INIT_SDADC1 (void)
{
     #ifdef Int_Drain
	 SDADC_1.RSER.R = 0x00000003;			 //Configure for  IRQ  Allow FIFO Overflow IRQs
	 #else
	 SDADC_1.RSER.R = 0x00010003;			 //Configure for  DMA  Allow FIFO Full IRQs
	 #endif
 }

/********************************************************/
/* SDADC_ERR005749  Deal with ERR005749                 */
/*                   by clearing DFORF and reading FIFO */
/********************************************************/

void SDADC_ERR005749_SDADC1 (void)
{
uint8_t i = 0;
vuint32_t unload_FIFO = 0;

while ( i<FIFO_DEPTH) 
{
unload_FIFO =  SDADC_1.CDR.R;             //Unload the FIFO by reading DFORF - Chose this method rather than comparing CDR to 0, as this may give a "False Empty" if a result is 0
i++;
}
SDADC_1.SFR.B.DFORF = 1;                  //Clear the DFORF bit

}
/********************************************************/
/* SDADC_STIME	Delay for 100uS at startup              */
/* 				for normal operations	                */
/********************************************************/
void SDADC_STIME(void)
{
 PIT_0.MCR.R = 0x4;						/*Enable Clock to PIT module*/
 PIT_0.TIMER[2].LDVAL.R = 1000;			/*40MHz OSC/4 * 1000 gives 100uS*/
 PIT_0.TIMER[2].TCTRL.B.TIE = 0x1;		/* Enable the interrupt from PITTIMER */
 PIT_0.TIMER[2].TCTRL.B.TEN = 0x1;		/* Enable the timer */
 
 }
 /********************************************************/
/* average_results :Average the results of the offset   */
/* 					calibration						    */
/********************************************************/
uint32_t average_results (void)
 {
    static uint16_t  i 	 = 0x0;
    static int32_t temp	 = 0x0;
    static uint16_t data_length = 0x1ff;		//Length of array

    while ( eDMA_0.TCD[16].CSR.B.DONE ==0)
    {
    }

    SDADC_1.MCR.B.EN = 0;			 //Disable SDADC_1 before next DMA transfer starts

    /* Calculate average */
    for ( i = 0; i < data_length; i++)  
    {
	temp = temp + SDADC1_RESULTS[i];
    }
    return (temp/data_length);
}




/********************************************************/
/* SDADC_INIT :Initialize and startSDADC                */
/* 				for normal operations	                */
/********************************************************/
void SDADC_INIT(void)
{

	enable_SDADC1 ();
	#ifdef Single_Ended
	set_MCR_SDADC1_SE();
	#else
	set_MCR_SDADC1_DIFF();
	#endif
	set_CSR_SDADC1 ();
	OSDR_SDADC1_config ();
	RSER_INIT_SDADC1 ();
	FIFO_INIT_SDADC1 ();
  	
}





