/*!
 * \file    llwu.c
 * \brief   common LLWU routines
 *
 * This file defines the functions/interrupt handlers/macros used for LLWU to be used as wakeup source.
 * And some common initializations.
 *
 * \version $Revision: 1.0 $
 * \author  Philip Drake(rxaa60)
 ***/

#include "common.h"
#include "llwu.h"
#include "mcg.h"


extern int re_init_clk;
extern int clock_freq_hz;


/* function: llwu_configure

   description: Set up the LLWU for wakeup the MCU from LLS and VLLSx modes 
   from the selected pin or module.
   
   inputs:
   pin_en - unsigned integer, bit position indicates the pin is enabled.  
            More than one bit can be set to enable more than one pin at a time.  
   
   rise_fall - 0x00 = External input disabled as wakeup
               0x01 - External input enabled as rising edge detection
               0x02 - External input enabled as falling edge detection
               0x03 - External input enablge as any edge detection
   module_en - unsigned char, bit position indicates the module is enabled.  
               More than one bit can be set to enabled more than one module                   
   
   for example:  if bit 0 and 1 need to be enabled as rising edge detect call this routine with
   pin_en = 0x0003 and rise_fall = 0x02
   
   Note: to set up one set of pins for rising and another for falling, 2 calls to this 
         function are required, 1st for rising then the second for falling.
   
*/
void llwu_configure(unsigned int pin_en, unsigned char rise_fall, unsigned char module_en ) {
    uint8 temp;

//   used on rev 1.4 of P2
    temp = LLWU_PE1;
    if( pin_en & 0x0001)
    {
        temp |= LLWU_PE1_WUPE0(rise_fall);
        printf(" LLWU configured pins PTE1/UART1_RX/I2C1_SCL /SPI1_SIN is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF0_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0002)
    {
        temp |= LLWU_PE1_WUPE1(rise_fall);
        printf(" LLWU configured pins PTE2/SPI1_SCK/SDHC0_DCLK is wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF1_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0004)
    {
        temp |= LLWU_PE1_WUPE2(rise_fall);
        printf(" LLWU configured pins PTE4/SPI1_PCS0/SDHC0_D3 is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF2_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0008)
    {
        temp |= LLWU_PE1_WUPE3(rise_fall);
        printf(" LLWU configured pins PTA4/FTM0_CH1/NMI/EZP_CSis LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF3_MASK;   // write one to clear the flag
    }
    LLWU_PE1 = temp;

    temp = LLWU_PE2;
    if( pin_en & 0x0010)
    {
        temp |= LLWU_PE2_WUPE4(rise_fall);
        printf(" LLWU configured pins PTA13/FTM1_CH1 /FTM1_QD_PHB is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF4_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0020)
    {
        temp |= LLWU_PE2_WUPE5(rise_fall);
        printf(" LLWU configured pins PTB0/I2C0_SCL/FTM1_CH0/FTM1_QD_PHA is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF5_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0040)
    {
        temp |= LLWU_PE2_WUPE6(rise_fall);
        printf(" LLWU configured pins PTC1/UART1_RTS/FTM0_CH0 is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF6_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0080)
    {
        temp |= LLWU_PE2_WUPE7(rise_fall);
        printf(" LLWU configured pins PTC3/UART1_RX/FTM0_CH2 is LLWU wakeup source \n");
        LLWU_PF1 |= LLWU_PF1_WUF7_MASK;   // write one to clear the flag
    }
    LLWU_PE2 = temp;

    temp = LLWU_PE3;
    if( pin_en & 0x0100)
    {
        temp |= LLWU_PE3_WUPE8(rise_fall);
        printf(" LLWU configured pins PTC4/SPI0_PCS0/FTM0_CH3 is LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF8_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0200)
    {
        temp |= LLWU_PE3_WUPE9(rise_fall);
        printf(" LLWU configured pins PTC5/SPI0_SCK/I2S0_RXD0 is LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF9_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0400)
    {
        temp |= LLWU_PE3_WUPE10(rise_fall);
        printf(" LLWU configured pins PTC6/PDB0_EXTRG to be an LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF10_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x0800)
    {
        temp |= LLWU_PE3_WUPE11(rise_fall);
        printf(" LLWU configured pins PTC11/I2S0_RXD1 to be an LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF11_MASK;   // write one to clear the flag
    }	
    LLWU_PE3 = temp;

    temp = LLWU_PE4;
    if( pin_en & 0x1000)
    {
        temp |= LLWU_PE4_WUPE12(rise_fall);
        printf(" LLWU configured pins PTD0/SPI0_PCS0/UART2_RTS to be an LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF12_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x2000)
    {
        temp |= LLWU_PE4_WUPE13(rise_fall);
        printf(" LLWU configured pins PTD2/UART2_RX to be an LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF13_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x4000)
    {
        temp |= LLWU_PE4_WUPE14(rise_fall);
        printf(" LLWU configured pins PTD4/UART0_RTS/FTM0_CH4/EWM_IN is LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF14_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x8000)
    {
        temp |= LLWU_PE4_WUPE15(rise_fall);
        printf(" LLWU configured pins PTD6/UART0_RX/FTM0_CH6/FTM0_FLT0 is LLWU wakeup source \n");
        LLWU_PF2 |= LLWU_PF2_WUF15_MASK;   // write one to clear the flag
    }
	
    LLWU_PE4 = temp;
	
	temp = LLWU_PE5;
    if( pin_en & 0x00010000)
    {
        temp |= LLWU_PE5_WUPE16(rise_fall);
        printf(" LLWU configured pins PTE6/SPI1_PCS3/UART3_RTS/I2S0_MCLK is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF16_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00020000)
    {
        temp |= LLWU_PE5_WUPE17(rise_fall);
        printf(" LLWU configured pins PTE9/I2S0_TXD1/LPUART0_RX is wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF17_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00040000)
    {
        temp |= LLWU_PE5_WUPE18(rise_fall);
        printf(" LLWU configured pins PTE10/I2C3_SDA/LPUART0_CTS is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF18_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00080000)
    {
        temp |= LLWU_PE5_WUPE19(rise_fall);
        printf(" LLWU configured pins PTE17/SPI0_SCK/UART2_RX/FTM_CLKIN1 is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF19_MASK;   // write one to clear the flag
    }
    LLWU_PE5 = temp;

    temp = LLWU_PE6;
    if( pin_en & 0x00100000)
    {
        temp |= LLWU_PE6_WUPE20(rise_fall);
        printf(" LLWU configured pins PTE18/SPI0_SOUT/UART2_CTS is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF20_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00200000)
    {
        temp |= LLWU_PE6_WUPE21(rise_fall);
        printf(" LLWU configured pins PTE25/CAN1_RX/UART4_RX/I2C0_SDA is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF21_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00400000)
    {
        temp |= LLWU_PE6_WUPE22(rise_fall);
        printf(" LLWU configured pins PTA10/FTM2_CH0/MII0_RXD2 is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF22_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x00800000)
    {
        temp |= LLWU_PE6_WUPE23(rise_fall);
        printf(" LLWU configured pins PTA11/FTM2_CH1/MII0_RXCLK is LLWU wakeup source \n");
        LLWU_PF3 |= LLWU_PF3_WUF23_MASK;   // write one to clear the flag
    }
    LLWU_PE6 = temp;

    temp = LLWU_PE7;
    if( pin_en & 0x01000000)
    {
        temp |= LLWU_PE7_WUPE24(rise_fall);
        printf(" LLWU configured pins PTD8/I2C0_SCL/LPUART0_RX is LLWU wakeup source \n");
        LLWU_PF4 |= LLWU_PF4_WUF24_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x02000000)
    {
        temp |= LLWU_PE7_WUPE25(rise_fall);
        printf(" LLWU configured pins PTD11/SPI2_PCS0/SDHC0_CLKIN is LLWU wakeup source \n");
        LLWU_PF4 |= LLWU_PF4_WUF25_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x04000000)
    {
        //temp |= LLWU_PE7_WUPE26(rise_fall);
        printf(" LLWU_P26 is reserved on K60-2M \n");
        //LLWU_PF4 |= LLWU_PF4_WUF26_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x08000000)
    {
        //temp |= LLWU_PE7_WUPE27(rise_fall);
        printf(" LLWU_P27 is reserved on K60-2M \n");
        //LLWU_PF4 |= LLWU_PF4_WUF27_MASK;   // write one to clear the flag
    }	
    LLWU_PE7 = temp;

    temp = LLWU_PE8;
    if( pin_en & 0x10000000)
    {
        //temp |= LLWU_PE8_WUPE28(rise_fall);
        printf(" LLWU_P28 is reserved on K60-2M \n");
        //LLWU_PF4 |= LLWU_PF4_WUF28_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x20000000)
    {
        temp |= LLWU_PE8_WUPE29(rise_fall);
        printf(" LLWU configured pins USB1_VBUS to be an LLWU wakeup source \n");
        LLWU_PF4 |= LLWU_PF4_WUF29_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x40000000)
    {
        temp |= LLWU_PE8_WUPE30(rise_fall);
        printf(" LLWU configured pins USB1_DP is LLWU wakeup source \n");
        LLWU_PF4 |= LLWU_PF4_WUF30_MASK;   // write one to clear the flag
    }
    if( pin_en & 0x80000000)
    {
        temp |= LLWU_PE8_WUPE31(rise_fall);
        printf(" LLWU configured pins USB1_DM is LLWU wakeup source \n");
        LLWU_PF4 |= LLWU_PF4_WUF31_MASK;   // write one to clear the flag
    }
	
	LLWU_PE8 = temp;
	
    LLWU_ME |= module_en;  //Set up more modules to wakeup up
    
    printf("LLWU PE1   = 0x%02X,    ",   (LLWU_PE1)) ;
    printf("LLWU PE2   = 0x%02X\n",      (LLWU_PE2)) ;
    printf("LLWU PE3   = 0x%02X,    ",   (LLWU_PE3)) ;
    printf("LLWU PE4   = 0x%02X\n",      (LLWU_PE4)) ;
    printf("LLWU PE5   = 0x%02X,    ",   (LLWU_PE5)) ;
    printf("LLWU PE6   = 0x%02X\n",      (LLWU_PE6)) ;
    printf("LLWU PE7   = 0x%02X,    ",   (LLWU_PE7)) ;
    printf("LLWU PE8   = 0x%02X\n",      (LLWU_PE8)) ;
	printf("LLWU ME    = 0x%02X,    ",    (LLWU_ME)) ;
    printf("LLWU PF1   = 0x%02X\n",       (LLWU_PF1)) ;
    printf("LLWU PF2   = 0x%02X,    ",    (LLWU_PF2)) ;
    printf("LLWU PF3   = 0x%02X\n",       (LLWU_PF3)) ;
    printf("LLWU PF4   = 0x%02X,    ",    (LLWU_PF4)) ;
	printf("LLWU MF5   = 0x%02X\n",       (LLWU_MF5)) ;
	printf("LLWU FILT1 = 0x%02X,    ", (LLWU_FILT1)) ;
    printf("LLWU FILT2 = 0x%02X\n",    (LLWU_FILT2)) ;
	printf("LLWU FILT3 = 0x%02X,    ", (LLWU_FILT3)) ;
    printf("LLWU FILT4 = 0x%02X\n",    (LLWU_FILT4)) ;
      //function ends
}


void llwu_configure_filter(unsigned int wu_pin_num, unsigned char filter_en, unsigned char rise_fall ) 
{
   //wu_pin_num is the pin number to be written to FILTSEL.  wu_pin_num is not the same as pin_en. 
    uint8 temp;
	
    printf("\nEnabling Filter %x on WU Pin %x for WU sense %x \n",filter_en, wu_pin_num, rise_fall);
   
     temp = 0;
     //first clear filter values and clear flag by writing a 1
     LLWU_FILT1 = LLWU_FILT1_FILTF_MASK;
     LLWU_FILT2 = LLWU_FILT2_FILTF_MASK;
     LLWU_FILT3 = LLWU_FILT3_FILTF_MASK;
	 LLWU_FILT4 = LLWU_FILT4_FILTF_MASK;
	 
     if(filter_en == 1)
     {
    	 //clear the flag bit and set the others
         temp |= (LLWU_FILT1_FILTF_MASK) | (LLWU_FILT1_FILTE(rise_fall) | LLWU_FILT1_FILTSEL(wu_pin_num));         
         LLWU_FILT1 = temp;
         
     }else if (filter_en == 2)
     {
    	 //clear the flag bit and set the others
    	 temp |= (LLWU_FILT2_FILTF_MASK) | (LLWU_FILT2_FILTE(rise_fall) | LLWU_FILT2_FILTSEL(wu_pin_num));         
         LLWU_FILT2 = temp;
     }else if (filter_en == 3)
     {
    	 //clear the flag bit and set the others
    	 temp |= (LLWU_FILT3_FILTF_MASK) | (LLWU_FILT3_FILTE(rise_fall) | LLWU_FILT3_FILTSEL(wu_pin_num));         
         LLWU_FILT3 = temp;
     }else if (filter_en == 4)
     {
    	 //clear the flag bit and set the others
    	 temp |= (LLWU_FILT4_FILTF_MASK) | (LLWU_FILT4_FILTE(rise_fall) | LLWU_FILT4_FILTSEL(wu_pin_num));         
         LLWU_FILT4 = temp;
     }
	 else
     {
    	 printf("\nError - invalid filter number %x\n"); 
     }
}    
    
//Interrupt handler for LLWU 
/* All the printf should be removed, for if before entering LLS mode, 
 * UART clock gate disabled to save power, then recovering from LLS, code
 * first enter LLWU ISR, then printf will cause multiple hard fault generated 
 * and cause core lock up event occur and reset MCU 
 */
void llwu_isr(void){
     GPIOB_PSOR = 0x00080000;             // set Port B19 for LLS recovery test
     //PMC_REGSC |= PMC_REGSC_ACKISO_MASK;  // shouldn't need to do this for LLS recovery	

//  GPIOE_PSOR = 0x04000000;             // set Port E 26 indicate wakeup - set it in llwu_isr
//        GPIOE_PCOR = 0x04000000;             // clear Port E 26 indicating sleep
//	GPIOE_PSOR = 0x04000000;             // set Port E 26 indicate wakeup - set it in llwu_isr
//        GPIOE_PCOR = 0x04000000;             // clear Port E 26 indicating sleep
    if ((SMC_PMCTRL & SMC_PMCTRL_STOPM_MASK) == 3){
        // LLS Mode
    }
    //printf("\n[LLWU ISR]\r\n"); 
    if (LLWU_PF1 & LLWU_PF1_WUF0_MASK) {
       LLWU_PF1 |= LLWU_PF1_WUF0_MASK;   // write one to clear the flag
       //printf("\n[LLWU ISR]  ****WUF0 was set *****\r\n"); 
    }
    if (LLWU_PF1 & LLWU_PF1_WUF1_MASK) {
       //printf("\n [LLWU ISR] ****WUF1 was set *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF1_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF2_MASK) {
       //printf("\n [LLWU ISR] ****WUF2 was set *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF2_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF3_MASK){
       //printf("\n [LLWU ISR] ****WUF3 was set *****\r\n");
       LLWU_PF1 |= LLWU_PF1_WUF3_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF4_MASK) {
       //printf("\n [LLWU ISR] ****WUF4 was set *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF4_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF5_MASK) {
       //printf("\n [LLWU ISR] ****WUF5 was set *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF5_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF6_MASK) {
       //printf("\n [LLWU ISR] ****WUF6 was set *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF6_MASK;   // write one to clear the flag
    }
    if (LLWU_PF1 & LLWU_PF1_WUF7_MASK) {
       //printf("\n [LLWU ISR] ****WUF7 was set from PTC3 input  *****\r\n"); 
       LLWU_PF1 |= LLWU_PF1_WUF7_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF8_MASK) {
       //printf("\n [LLWU ISR] ****WUF8 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF8_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF9_MASK) {
       //printf("\n [LLWU ISR] ****WUF9 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF9_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF10_MASK) {
       //printf("\n [LLWU ISR] ****WUF10 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF10_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF11_MASK) {
       //printf("\n [LLWU ISR] ****WUF11 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF11_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF12_MASK) {
       //printf("\n [LLWU ISR] ****WUF12 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF12_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF13_MASK) {
       //printf("\n [LLWU ISR] ****WUF13 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF13_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF14_MASK) {
       //printf("\n [LLWU ISR] ****WUF14 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF14_MASK;   // write one to clear the flag
    }
    if (LLWU_PF2 & LLWU_PF2_WUF15_MASK) {
       //printf("\n [LLWU ISR] ****WUF15 was set *****\r\n"); 
       LLWU_PF2 |= LLWU_PF2_WUF15_MASK;   // write one to clear the flag
    }
    if (LLWU_PF3 & LLWU_PF3_WUF16_MASK) {
       LLWU_PF3 |= LLWU_PF3_WUF16_MASK;   // write one to clear the flag
       //printf("\n[LLWU ISR]  ****WUF16 was set *****\r\n"); 
    }
    if (LLWU_PF3 & LLWU_PF3_WUF17_MASK) {
       //printf("\n [LLWU ISR] ****WUF17 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF17_MASK;   // write one to clear the flag
    }
    if (LLWU_PF3 & LLWU_PF3_WUF18_MASK) {
       //printf("\n [LLWU ISR] ****WUF18 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF18_MASK;   // write one to clear the flag
    }
    if (LLWU_PF3 & LLWU_PF3_WUF19_MASK){
       //printf("\n [LLWU ISR] ****WUF19 was set *****\r\n");
       LLWU_PF3 |= LLWU_PF3_WUF19_MASK;   // write one to clear the flag
    }
    if (LLWU_PF3 & LLWU_PF3_WUF20_MASK) {
       //printf("\n [LLWU ISR] ****WUF20 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF20_MASK;   // write one to clear the flag
    }
   if (LLWU_PF3 & LLWU_PF3_WUF21_MASK) {
       //printf("\n [LLWU ISR] ****WUF5 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF21_MASK;   // write one to clear the flag
   }
   if (LLWU_PF3 & LLWU_PF3_WUF22_MASK) {
       //printf("\n [LLWU ISR] ****WUF22 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF22_MASK;   // write one to clear the flag
    }
   if (LLWU_PF3 & LLWU_PF3_WUF23_MASK) {
       //printf("\n [LLWU ISR] ****WUF23 was set *****\r\n"); 
       LLWU_PF3 |= LLWU_PF3_WUF23_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF24_MASK) {
       //printf("\n [LLWU ISR] ****WUF24 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF24_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF25_MASK) {
       //printf("\n [LLWU ISR] ****WUF25 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF25_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF26_MASK) {
       //printf("\n [LLWU ISR] ****WUF26 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF26_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF27_MASK) {
       //printf("\n [LLWU ISR] ****WUF27 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF27_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF28_MASK) {
       //printf("\n [LLWU ISR] ****WUF28 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF28_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF29_MASK) {
       //printf("\n [LLWU ISR] ****WUF29 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF29_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF30_MASK) {
       //printf("\n [LLWU ISR] ****WUF30 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF30_MASK;   // write one to clear the flag
   }
   if (LLWU_PF4 & LLWU_PF4_WUF31_MASK) {
       //printf("\n [LLWU ISR] ****WUF31 was set *****\r\n"); 
       LLWU_PF4 |= LLWU_PF4_WUF31_MASK;   // write one to clear the flag
   }
      
   if (LLWU_MF5 & LLWU_MF5_MWUF0_MASK) {
       //printf("\n [LLWU ISR] ****MWUF0 IF  LPT0  *****\r\n"); 
//       LPTMR0_CSR |=  LPTMR_CSR_TCF_MASK;   // write 1 to TCF to clear the LPT timer compare flag
//       LPTMR0_CSR = ( LPTMR_CSR_TEN_MASK | LPTMR_CSR_TIE_MASK | LPTMR_CSR_TCF_MASK  );
       LLWU_MF5 |= LLWU_MF5_MWUF0_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF1_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF1 IF  CMP0  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF1_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF2_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF2 IF  CMP1 *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF2_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF3_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF3 IF CMP2/CMP3  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF3_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF4_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF4 IF TSI  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF4_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF5_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF5 IF RTC Alarm  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF5_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF6_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF6 IF DryIce(Tamper Detect)  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF6_MASK;   // write one to clear the flag
   }
   if (LLWU_MF5 & LLWU_MF5_MWUF7_MASK) {
	     //printf("\n [LLWU ISR] ****MWUF7 IF RTC Seconds  *****\r\n"); 
       LLWU_MF5 |= LLWU_MF5_MWUF7_MASK;   // write one to clear the flag
   }
  
   if(LLWU_FILT1 & LLWU_FILT1_FILTF_MASK){
	   
	   LLWU_FILT1 |= LLWU_FILT1_FILTF_MASK;
   }
   if(LLWU_FILT2 & LLWU_FILT2_FILTF_MASK){
	   
	   LLWU_FILT2 |= LLWU_FILT2_FILTF_MASK;
   }
   if(LLWU_FILT3 & LLWU_FILT3_FILTF_MASK){
	   
	   LLWU_FILT3 |= LLWU_FILT3_FILTF_MASK;
   }
   if(LLWU_FILT4 & LLWU_FILT4_FILTF_MASK){
	   
	   LLWU_FILT4 |= LLWU_FILT4_FILTF_MASK;
   }   
}
