/*
 * File:        uart.c
 * Purpose:     Provide common UART routines for serial IO
 *
 * Notes:       
 *              
 */

#include "common.h"
#include "lpuart.h"

/********************************************************************/
/*
 * Initialize the UART for 8N1 operation, interrupts disabled, and
 * no hardware flow-control
 *
 * NOTE: Since the UARTs are pinned out in multiple locations on most
 *       Kinetis devices, this driver does not enable UART pin functions.
 *       The desired pins should be enabled before calling this init function.
 *
 * Parameters:
 *  uartch      UART channel to initialize
 *  sysclk      UART module Clock in kHz(used to calculate baud)
 *  baud        UART baud rate
 */
void lpuart_init (LPUART_MemMapPtr uartch, int sysclk, int baud)
{
    uint16 sbr, osr;
    uint32 reg_tmp;
    
	/* Enable the clock to the selected UART */    
    if(uartch == LPUART0_BASE_PTR)
		SIM_SCGC2 |= SIM_SCGC2_LPUART0_MASK;
    else
        return;
                                
    /* Make sure that the transmitter and receiver are disabled while we 
     * change settings.
     */
    LPUART_CTRL_REG(uartch) &= ~(LPUART_CTRL_TE_MASK
				| LPUART_CTRL_RE_MASK);

    /* Configure the UART for 8-bit mode, no parity */
    LPUART_CTRL_REG(uartch) = 0;	/* We need all default settings, so entire register is cleared */
    
    /* Calculate baud settings */
    /* baud_rate = baud_clock / ((OSR+1)  SBR) */
    reg_tmp = LPUART_BAUD_REG(uartch);
    reg_tmp &= ~LPUART_BAUD_SBR_MASK;
    osr = (reg_tmp & LPUART_BAUD_OSR_MASK) >> LPUART_BAUD_OSR_SHIFT;
    sbr = (uint16)((sysclk * 1000) / ((osr + 1) * baud));
    reg_tmp |= LPUART_BAUD_SBR(sbr);
        
    /* Save off the current value of the UARTx_BDH except for the SBR field */
    LPUART_BAUD_REG(uartch) = sbr & LPUART_BAUD_SBR_MASK;

    /* Enable receiver and transmitter */
	LPUART_CTRL_REG(uartch) |= (LPUART_CTRL_TE_MASK | LPUART_CTRL_RE_MASK);
}
/********************************************************************/
/*
 * Wait for a character to be received on the specified UART
 *
 * Parameters:
 *  channel      UART channel to read from
 *
 * Return Values:
 *  the received character
 */
char lpuart_getchar (LPUART_MemMapPtr channel)
{
    /* Wait until character has been received */
    while (!(LPUART_STAT_REG(channel) & LPUART_STAT_RDRF_MASK));
    
    /* Return the 8-bit data from the receiver */
    return LPUART_DATA_REG(channel);
}
/********************************************************************/
/*
 * Wait for space in the UART Tx FIFO and then send a character
 *
 * Parameters:
 *  channel      UART channel to send to
 *  ch			 character to send
 */ 
void lpuart_putchar (LPUART_MemMapPtr channel, char ch)
{
    /* Wait until space is available in the FIFO */
    while(!(LPUART_STAT_REG(channel) & LPUART_STAT_TDRE_MASK));
    
    /* Send the character */
    LPUART_DATA_REG(channel) = (uint8)ch;
 }
/********************************************************************/
/*
 * Check to see if a character has beebn received
 *
 * Parameters:
 *  channel      UART channel to check for a character
 *
 * Return values:
 *  0       No character received
 *  1       Character has been received
 */
int lpuart_getchar_present (LPUART_MemMapPtr channel)
{
    return (LPUART_STAT_REG(channel) & UART_S1_RDRF_MASK);
}
/********************************************************************/
    
/*******************************************************************************************************/
void lpuart_configure(int mcg_clock_hz)
{
    int mcg_clock_khz;
    int periph_clock_khz;

    mcg_clock_khz = mcg_clock_hz / 1000;
    periph_clock_khz = mcg_clock_khz / (((SIM_CLKDIV1 & SIM_CLKDIV1_OUTDIV2_MASK) >> SIM_CLKDIV1_OUTDIV2_SHIFT) + 1);

    lpuart_init (LPTERM_PORT, periph_clock_khz, TERMINAL_BAUD);
}
