
/*
 * File:        smc.c
 * Purpose:     Provides routines for entering low power modes.
 *
 * Notes:	Since the wakeup mechanism for low power modes
 *              will be application specific, these routines
 *              do not include code to setup interrupts to exit
 *              from the low power modes. The desired means of
 *              low power mode exit should be configured before
 *              calling any of these functions.
 *
 *              These routines do not include protection to
 *              prevent illegal state transitions in the mode
 *              controller, and all routines that write to the
 *              PMPROT register write a value to allow all
 *              possible low power modes (it is write once, so
 *              if only the currently requested mode is enabled
 *              a different mode couldn't be enabled later on).
 *              
 */

#include "common.h"
#include "smc.h"
#include "mcg.h"

extern int mcg_clk_hz;
extern int mcg_clk_khz;
extern int core_clk_khz;
extern int core_clk_mhz;
extern int periph_clk_khz;



/********************************************************************/
/* WAIT mode entry routine. Puts the processor into wait mode.
 * In this mode the core clock is disabled (no code executing), but 
 * bus clocks are enabled (peripheral modules are operational). 
 *
 * Mode transitions:
 * RUN -> WAIT
 * VLPR -> VLPW
 *
 * This function can be used to enter normal wait mode or VLPW
 * mode. If you are executing in normal run mode when calling this
 * function, then you will enter normal wait mode. If you are in VLPR
 * mode when calling this function, then you will enter VLPW mode instead.
 *
 * NOTE: Some modules include a programmable option to disable them in 
 * wait mode. If those modules are programmed to disable in wait mode, 
 * they will not be able to generate interrupts to wake up the core.
 *
 * WAIT mode is exited using any enabled interrupt or RESET, so no
 * exit_wait routine is needed. 
 *
 * When in VLPW mode, if interrupt occurs, moves processor to VLPR mode; if 
 * RESET occurs, move processor to run mode. There is no longer LPWUI bit 
 * in PMCTRL register.
 * 
 * 
 * Parameters:
 * none
 */
void enter_wait(void)
{
    wait();
}
/********************************************************************/
/* STOP mode entry routine. Puts the processor into normal stop mode.
 * In this mode core, bus and peripheral clocks are disabled.
 *
 * Mode transitions:
 * RUN -> STOP
 *
 * This function can be used to enter normal stop mode. 
 * If you are executing in normal run mode when calling this
 * function and AVLP = 0, then you will enter normal stop mode. 
 * If AVLP = 1 with previous write to PMPROT
 * then you will enter VLPS mode instead.
 *
 * STOP mode is exited using any enabled interrupt or RESET, so no
 * exit_stop routine is needed.
 *
 * Parameters:
 * Partial Stop Option:  
 *  0x00 = STOP - Normal Stop Mode
 *  0x40 = PSTOP1 - Partial Stop with both system and bus clocks disabled
 *  0x80 = PSTOP2 - Partial Stop with system clock disabled and bus clock enabled
 *  0xC0 = Reserved
 */
void enter_stop(uint8 partial_stop_opt)
{
    /* The PMPROT register may have already been written by init code
       If so then this next write is not done since  
       PMPROT is write once after RESET  
       this write-once bit allows the MCU to enter the
       normal STOP mode.
       If AVLP is already a 1, VLPS mode is entered instead of normal STOP*/
    SMC_PMPROT = 0;  

    /* Set the STOPM field to 0b000 for normal STOP mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK;
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0); 
    SMC_STOPCTRL &= ~SMC_STOPCTRL_PSTOPO_MASK;
    SMC_STOPCTRL |= partial_stop_opt;
    /*wait for write to complete to SMC before stopping core */  
    partial_stop_opt = SMC_PMCTRL;
    
	stop();
}
/********************************************************************/
/* VLPR mode entry routine. Puts the processor into very low power
 * run mode. In this mode all clocks are enabled, but the core, bus,
 * and peripheral clocks are limited to 4MHz or less. The flash 
 * clock is limited to 1MHz or less. 
 *
 * Mode transitions:
 * RUN -> VLPR
 *
 * exit_vlpr() function can be used to switch from VLPR back to RUN. 
 * 
 * When in VLPW mode, if interrupt occurs, moves processor to VLPR mode; if 
 * RESET occurs, move processor to run mode. There is no longer LPWUI bit 
 * in PMCTRL register.
 *
 * Parameters: no
 * 
 * Return value : PMSTAT value or error code
 *                PMSTAT = 
 *                         000_0001 Current power mode is RUN
 *                         000_0100 Current power mode is VLPR
 *                ERROR Code =  0x14 - already in VLPR mode
 *                           =  0x24 - REGONS never clear indicating stop regulation
 */
int enter_vlpr(void)
{
    int i; 
	uint32 return_value = 0;	//default return value
	
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 4){
         return_value = 0x14;
    }
    
	/* The PMPROT register may have already been written by init code
       If so then this next write is not done.  
       PMPROT is write once after RESET  
       this write-once bit allows the MCU to enter the
       very low power modes: VLPR, VLPW, and VLPS   */
    SMC_PMPROT = SMC_PMPROT_AVLP_MASK;  
    
    /* Set the (for MC1)LPLLSM or (for MC2)STOPM field 
       to 0b010 for VLPS mode -
       and RUNM bits to 0b010 for VLPR mode  */
    SMC_PMCTRL &= ~SMC_PMCTRL_RUNM_MASK; 
    SMC_PMCTRL  |= SMC_PMCTRL_RUNM(0x2);       
   
    /* Wait for VLPS regulator mode to be confirmed */
    for (i = 0 ; i < 10000 ; i++)
    {     /* check that the value of REGONS bit is not 0
             once it is a zero we can stop checking */
        if ((PMC_REGSC & PMC_REGSC_REGONS_MASK) ==0x04){
        // 0 Regulator is in stop regulation or in transition to/from it
        // 1 MCU is in Run regulation mode
        }  
        else  
		    break;
    }  
    if ((PMC_REGSC & PMC_REGSC_REGONS_MASK) ==0x04)  
	    return_value = 0x24; 
	
    /* [MC2] PMSTAT only exist in Mode Controller 2       */
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) == 4)
         return_value = 0x04;
    	
    return (return_value);	
}
/********************************************************************/
/* VLPR mode exit routine. Puts the processor into normal run mode
 * from VLPR mode. You can transition from VLPR to normal run using
 * this function.
 *
 * Mode transitions:
 * VLPR -> RUN
 *
 * Parameters:
 * none
 */
void exit_vlpr(void)
{
    /* check to make sure in VLPR before exiting    */
    if  ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 4) {
      
        /* Clear RUNM */
        SMC_PMCTRL &= ~(SMC_PMCTRL_RUNM(0x3));
                      
        /* Wait for normal RUN regulation mode to be confirmed */                   
        // 0 MCU is not in run regulation mode
        // 1 MCU is in run regulation mode
        while(!(PMC_REGSC & PMC_REGSC_REGONS_MASK)){
    	     printf(" \n[exit_vlpr] Waiting on REGONS bit to set to indicate Run regulation mode ! ");
        }        
    }  //if in VLPR mode
    // else if not in VLPR ignore call
}
/********************************************************************/
/* High speed run mode entry routine. Puts the processor into high speed
 * run mode. In this mode the on-chip regulator remains in a run regulation
 * state, but with a slightly elevated voltage output. In this state, the
 * MCU is able to operate at a faster frequency compared to normal run 
 * mode.
 *
 * Mode transitions:
 * RUN -> HSRUN
 *
 * exit_hsrun() function can be used to switch from HSRUN back to RUN. 
 * 
 * Parameters: no
 * 
 * Return value : PMSTAT value or error code
 *                PMSTAT = 
 *                         0000_0001 Current power mode is RUN
 *                         1000_0000 Current power mode is HSRUN
 *                ERROR Code =  0x14 - already in HSRUN mode
 *                           =  0x24 - failed to enter HSRUN mode
 */
int enter_hsrun(void)
{
    int i;
	uint32 return_value = 0;	//default return value
	
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 0x80){
         return_value = 0x14;
    }
    
	/* The PMPROT register may have already been written by init code
       If so then this next write is not done.  
       PMPROT is write once after RESET  
       this write-once bit allows the MCU to enter the
       high speed run mode.   */
    SMC_PMPROT = SMC_PMPROT_AHSRUN_MASK;  
    
    /* Set RUNM bits to 0b011 for HSRUN mode  */
    SMC_PMCTRL &= ~SMC_PMCTRL_RUNM_MASK; 
    SMC_PMCTRL  |= SMC_PMCTRL_RUNM(0x3);       
	
	for (i = 0 ; i < 100 ; i++)
    {
	    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) == 0x80)
		    break;
	}
	
    /* [MC2] PMSTAT only exist in Mode Controller 2       */
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) == 0x80)
        return_value = 0x80;
	else
        return_value = 0x24;
	
    return (return_value);	
}
/********************************************************************/
/* HSRUN mode exit routine. Puts the processor into normal run mode
 * from HSRUN mode. You can transition from HSRUN to normal run using
 * this function.
 *
 * Mode transitions:
 * HSRUN -> RUN
 *
 * Parameters:
 * none
 */
void exit_hsrun(void)
{
    int i;
	
    /* check to make sure in HSRUN before exiting    */
    if  ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) == 0x80) {
      
        /* Clear RUNM */
        SMC_PMCTRL &= ~(SMC_PMCTRL_RUNM(0x3));
                      
        /* Wait for normal RUN mode to be confirmed */                   
        for(i = 0; i < 100; i++)
		{
		    if((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) == 0x01)
			    break;
		}
    	
		if((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK) != 0x01)
		{
		    printf("Failed to return back to normal run mode\n");  
		}
    }  //if in HSRUN mode
    // else if not in HSRUN ignore call
}
/***************************************************************/
/* VLPW mode entry routine. Puts the processor into VLPW mode 
 * from VLPR mode, directly from Run mode is not allowed. 
 *
 * Mode transitions:
 * VLPR -> VLPW
 *
 * Note, VLPW wakes up with interrupt to VLPR mode and with reset 
 * to Run mode.
 *
 * Parameters:  
 * none
 */
void enter_vlpw(void)
{
	if(enter_vlpr() == 0x04)	  
	{
	    /* Now execute the stop instruction to go into VLPS */
        stop();
	}
	else
	    printf("Currently not in VLPR mode, please enter VLPR mode first\n");
}
/***************************************************************/
/* VLPS mode entry routine. Puts the processor into VLPS mode 
 * directly from normal run mode. 
 *
 * Mode transitions:
 * RUN -> VLPS
 * VLPR -> VLPS
 *
 * Note, when VLPS is entered directly from RUN mode, 
 * exit to VLPR is disabled by hardware and the system will
 * always exit back to RUN.
 *
 *
 * Parameters:  
 * none
 */
void enter_vlps(void)
{
    volatile uint32 dummyread;
	
    /* The PMPROT register may have already been written by init code
       If so then this next write is not done since  
       PMPROT is write once after RESET 
       allows the MCU to enter the VLPR, VLPW, and VLPS modes.
       If AVLP is already writen to 0 
       Stop is entered instead of VLPS*/
    SMC_PMPROT = SMC_PMPROT_AVLP_MASK;  
	
    /* Set the STOPM field to 0b010 for VLPS mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x2); 
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_PMCTRL;
    dummyread++;

    /* Now execute the stop instruction to go into VLPS */
    stop();
}
/****************************************************************/
/* LLS2 mode entry routine. Puts the processor into LLS mode from
 * normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> LLS
 * VLPR -> LLS
 *
 * NOTE: LLS mode will exit to RUN mode if you were entered directly from 
 * RUN, else if entered from VLPR mode, it will return to VLPR mode. 
 *
 * Wakeup from LLS mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in LLS mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * none
 */
void enter_lls2(uint8 ram2po_bit)
{
    volatile uint32 dummyread;
    /* Write to PMPROT to allow LLS power modes this write-once 
       bit allows the MCU to enter the LLS low power mode*/
    SMC_PMPROT = SMC_PMPROT_ALLS_MASK;   
    /* Set the STOPM field to 0b011 for LLS2 mode  */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x3); 
    /* Set the LLSM field to 0b00 for LLS2 mode */
    if (ram2po_bit==1)       //Write RAM2PO bit if needed.
    {
        SMC_STOPCTRL |= SMC_STOPCTRL_RAM2PO_MASK;
    }
    else
    {
        SMC_STOPCTRL&= ~SMC_STOPCTRL_RAM2PO_MASK;
    }
    SMC_STOPCTRL &= ~SMC_STOPCTRL_LLSM_MASK;
    SMC_STOPCTRL |= SMC_STOPCTRL_LLSM(0);	
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_PMCTRL;
    dummyread++;

    /* Now execute the stop instruction to go into LLS2 */
    stop();
}
/****************************************************************/
/* LLS3 mode entry routine. Puts the processor into LLS mode from
 * normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> LLS
 * VLPR -> LLS
 *
 * NOTE: LLS mode will exit to RUN mode if you were entered directly from 
 * RUN, else if entered from VLPR mode, it will return to VLPR mode. 
 *
 * Wakeup from LLS mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in LLS mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * none
 */
void enter_lls3(void)
{
    volatile uint32 dummyread;
    /* Write to PMPROT to allow LLS power modes this write-once 
       bit allows the MCU to enter the LLS low power mode*/
    SMC_PMPROT = SMC_PMPROT_ALLS_MASK;   
    /* Set the STOPM field to 0b011 for LLS3 mode  */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x3); 
	/* Set the LLSM field to 0b00 for LLS3 mode */
	SMC_STOPCTRL &= ~SMC_STOPCTRL_LLSM_MASK;
	SMC_STOPCTRL |= SMC_STOPCTRL_LLSM(3);	
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_PMCTRL;
    dummyread++;

    /* Now execute the stop instruction to go into LLS3 */
    stop();
}
/***************************************************************/
/* VLLS3 mode entry routine. Puts the processor into 
 * VLLS3 mode from normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> VLLS3
 * VLPR -> VLLS3
 *
 * NOTE: VLLSx modes will always exit to RUN mode even if you were 
 * in VLPR mode before entering VLLSx.
 *
 * Wakeup from VLLSx mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in VLLSx mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * none  
 */
void enter_vlls3(void)
{
    volatile uint32 dummyread;
    /* Write to PMPROT to allow VLLS3 power modes */
    SMC_PMPROT = SMC_PMPROT_AVLLS_MASK;      
    /* Set the STOPM field to 0b100 for VLLS3 mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x4); 
    /* set VLLSM = 0b11 */
    SMC_STOPCTRL =  SMC_STOPCTRL_LLSM(3);          
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_STOPCTRL;
    dummyread++;
    
	/* Now execute the stop instruction to go into VLLS3 */
    stop();
}
/***************************************************************/
/* VLLS2 mode entry routine. Puts the processor into 
 * VLLS2 mode from normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> VLLS2
 * VLPR -> VLLS2
 *
 * NOTE: VLLSx modes will always exit to RUN mode even  
 *       if you werein VLPR mode before entering VLLSx.
 *
 * Wakeup from VLLSx mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in VLLSx mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * none  
 */
void enter_vlls2(void)
{
    volatile uint32 dummyread;
    /* Write to PMPROT to allow VLLS3 power modes */
    SMC_PMPROT = SMC_PMPROT_AVLLS_MASK;      
    /* Set the STOPM field to 0b100 for VLLS3 mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x4); 
    /* set VLLSM = 0b11 */
    SMC_STOPCTRL =  SMC_STOPCTRL_LLSM(2);          
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_STOPCTRL;
    dummyread++;
	
    /* Now execute the stop instruction to go into VLLS2 */
    stop();
}
/***************************************************************/
/* VLLS1 mode entry routine. Puts the processor into 
 * VLLS1 mode from normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> VLLS1
 * VLPR -> VLLS1
 *
 * NOTE: VLLSx modes will always exit to RUN mode even if you were 
 * in VLPR mode before entering VLLSx.
 *
 * Wakeup from VLLSx mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in VLLSx mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * none  
 */
void enter_vlls1(void)
{
    volatile uint32 dummyread;
    /* Write to PMPROT to allow VLLS3 power modes */
    SMC_PMPROT = SMC_PMPROT_AVLLS_MASK;      
    /* Set the STOPM field to 0b100 for VLLS3 mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x4); 
    /* set VLLSM = 0b11 */
    SMC_STOPCTRL =  SMC_STOPCTRL_LLSM(1);          
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_STOPCTRL;
    dummyread++;
	
    /* Now execute the stop instruction to go into VLLS1 */
    stop();
}

/********************************************************************/
/* VLLS0 mode entry routine. Puts the processor into VLLS0 mode from
 * normal run mode or VLPR. 
 *
 * Mode transitions:
 * RUN -> VLLS0
 * VLPR -> VLLS0
 *
 * NOTE: VLLSx modes will always exit to RUN mode even if you were 
 * in VLPR mode before entering VLLSx.
 *
 * Wakeup from VLLSx mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in VLLSx mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * PORPO_value - 0 POR detect circuit is enabled in VLLS0
 *               1 POR detect circuit is disabled in VLLS0
 */
 /***************************************************************/

void enter_vlls0(uint8 PORPO_value )
{
    int i;
    /* Write to PMPROT to allow all possible power modes */
    SMC_PMPROT = SMC_PMPROT_AVLLS_MASK;   
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 4){
        SMC_PMCTRL &= ~SMC_PMCTRL_RUNM_MASK;   // go back to RUN mode temporarily
        for (i=0;i<0xff;i++)
        {
            if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 1)
    	        break;
        }
    }
    /* Set the STOPM field to 0b100 for VLLS0 mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x4); 
    /* set VLLSM = 0b00 */
    SMC_STOPCTRL &= ~SMC_STOPCTRL_LLSM_MASK;
    SMC_STOPCTRL &= ~SMC_STOPCTRL_PORPO_MASK;
    SMC_STOPCTRL |=  (PORPO_value <<SMC_STOPCTRL_PORPO_SHIFT) 
                 | SMC_STOPCTRL_LLSM(0);          
    /*wait for write to complete to SMC before stopping core */  
    PORPO_value = SMC_STOPCTRL;
    /* Now execute the stop instruction to go into VLLS0 */
    
    stop();
}
/***************************************************************/
/* VLLS0 mode entry routine. Puts the processor into VLLS0 mode from
 * normal run mode or VLPR with the POR circuit disabled
 *
 * Mode transitions:
 * RUN -> VLLS0
 * VLPR -> VLLS0
 *
 * NOTE: VLLSx modes will always exit to RUN mode even if you were 
 * in VLPR mode before entering VLLSx.
 *
 * Wakeup from VLLSx mode is controlled by the LLWU module. Most
 * modules cannot issue a wakeup interrupt in VLLSx mode, so make
 * sure to setup the desired wakeup sources in the LLWU before 
 * calling this function.
 *
 * Parameters:
 * PORPO = 1-  POR detect circuit is disabled in VLLS0
 */
/***************************************************************/
void enter_vlls0_nopor(void)
{
    volatile uint32 dummyread;
    int i;
    /* Write to PMPROT to allow all possible power modes */
    SMC_PMPROT = SMC_PMPROT_AVLLS_MASK;   
    if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 4){
        SMC_PMCTRL &= ~SMC_PMCTRL_RUNM_MASK;   // go back to RUN mode temporarily
        for (i=0;i<0xff;i++)
        {
            if ((SMC_PMSTAT & SMC_PMSTAT_PMSTAT_MASK)== 1)
    	        break;
        }
    }
    /* Set the STOPM field to 0b100 for VLLS0 mode */
    SMC_PMCTRL &= ~SMC_PMCTRL_STOPM_MASK; 
    SMC_PMCTRL |=  SMC_PMCTRL_STOPM(0x4); 
    /* set VLLSM = 00 * and PORPO = 1 */
    SMC_STOPCTRL &= ~SMC_STOPCTRL_LLSM_MASK;
    SMC_STOPCTRL =  SMC_STOPCTRL_LLSM(0) | SMC_STOPCTRL_PORPO_MASK;  
    /*wait for write to complete to SMC before stopping core */  
    dummyread = SMC_STOPCTRL;
    dummyread++;
    
    /* Now execute the stop instruction to go into VLLS0 */
    stop();
}

/********************************************************************/
/********************************************************************/

int switch_to_hsrun_freq(void)
{
  int mcg_clock_hz;
  unsigned char mcg_mode;
  
  mcg_mode = what_mcg_mode();

  SIM_CLKDIV1 = (0 | SIM_CLKDIV1_OUTDIV1(CORE_CLK_DIV_HS) |	/* Core/system	 */
		     SIM_CLKDIV1_OUTDIV2(BUS_CLK_DIV_HS) |		/* Busclk		*/
		     SIM_CLKDIV1_OUTDIV3(FLEXBUS_CLK_DIV_HS) |		/* FlexBus		*/
		     SIM_CLKDIV1_OUTDIV4(FLASH_CLK_DIV_HS));		/* Flash		*/  
   enter_hsrun();
   
   mcg_clock_hz = mcg_clk_hz;
   
   if(mcg_mode == PEE){
     pee_pbe(CLK0_FREQ_HZ);
     pbe_fbe(CLK0_FREQ_HZ);
     fbe_pbe(CRYSTAL,CLK0_FREQ_HZ,PLL_0, PLL0_PRDIV_HS,PLL0_VDIV_HS);
     mcg_clock_hz = pbe_pee(CLK0_FREQ_HZ,PLL_0);
     mcg_clk_hz  = mcg_clock_hz;
     mcg_clk_khz = mcg_clk_hz / 1000;
     core_clk_khz = mcg_clk_khz / (((SIM_CLKDIV1 & SIM_CLKDIV1_OUTDIV1_MASK) >> 28)+ 1);
     core_clk_mhz = core_clk_khz/1000;
     periph_clk_khz = mcg_clk_khz / (((SIM_CLKDIV1 & SIM_CLKDIV1_OUTDIV2_MASK) >> 24)+ 1);
   }else{
     printf("MCU is not in PEE mode, switch to HRUN mode with core clock frequency is not updated\n");
   }
   return mcg_clock_hz;
}

int switch_to_run_freq(void)
{
  int mcg_clock_hz;
  unsigned char mcg_mode;
  
  mcg_mode = what_mcg_mode();
  
  mcg_clock_hz = mcg_clk_hz;
  
  if(mcg_mode == PEE){
   
   pee_pbe(CLK0_FREQ_HZ);
   pbe_fbe(CLK0_FREQ_HZ);
   fbe_pbe(CRYSTAL,CLK0_FREQ_HZ,PLL_0,PLL0_PRDIV,PLL0_VDIV);
   mcg_clock_hz = pbe_pee(CLK0_FREQ_HZ,PLL_0);
  }else{
   printf("MCU is not in PEE mode, switch to NRUN mode with core clock frequency is not updated\n");
  }
	
   SIM_CLKDIV1 = (0 | SIM_CLKDIV1_OUTDIV1(CORE_CLK_DIV) |	/* Core/system	 */
                      SIM_CLKDIV1_OUTDIV2(BUS_CLK_DIV) |		/* Busclk		*/
		      SIM_CLKDIV1_OUTDIV3(FLEXBUS_CLK_DIV) |		/* FlexBus		*/
		      SIM_CLKDIV1_OUTDIV4(FLASH_CLK_DIV));		/* Flash		*/    

   mcg_clk_hz  = mcg_clock_hz;
   mcg_clk_khz = mcg_clk_hz / 1000;
   core_clk_khz = mcg_clk_khz / (((SIM_CLKDIV1 & SIM_CLKDIV1_OUTDIV1_MASK) >> 28)+ 1);
   core_clk_mhz = core_clk_khz/1000;
   periph_clk_khz = mcg_clk_khz / (((SIM_CLKDIV1 & SIM_CLKDIV1_OUTDIV2_MASK) >> 24)+ 1);
     
   exit_hsrun();
   return mcg_clock_hz;
}