/******************************************************************************
* 
* Copyright (c) 2012 Freescale Semiconductor;
* All Rights Reserved                       
*
******************************************************************************* 
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
***************************************************************************//*!
*
* @file      PCLIB_ControllerPIAsm.c
*
* @author    B37570, B36378
* 
* @version   1.0.00.0
* 
* @date      Jan-26-2011
* 			Updated September 2013
* 
* @brief     PI controller algorithm implemented in assembler
*
*******************************************************************************
*
* PI controller algorithm implemented in assembler.
*
******************************************************************************/
#ifndef __PCLIB_CONTROL_PI_PARAMS_H__
#define __PCLIB_CONTROL_PI_PARAMS_H__

/******************************************************************************
* Includes
******************************************************************************/
#include "56800E_types.h"

/******************************************************************************
* Constants
******************************************************************************/

/******************************************************************************
* Macros 
******************************************************************************/

#define PCLIB_ControllerPIAsm(f16Error, pudtControlSturct)		\
		PCLIB_ControllerPIFAsm(f16Error, pudtControlSturct)

/******************************************************************************
* Types
******************************************************************************/

/* Digital Proportional Integrator parameters */
typedef struct _pclib_control_pi_params_t
{ 
    Frac16 f16IntegOutputN_1;		/* Previous Integral output (n-1) */ 
    Frac16 f16IntegGain;       		/* Integral Gain Ki - Data Input */
    Frac16 f16IntegUpperLimit;   	 /* Integral Upper limit - Data Input */
    Frac16 f16IntegLowerLimit;		/* Integral Lower limit - Data Input */
   	Frac16 f16PropGain;  			/* Proportional Gain Kp - Data Input */      
  	Frac16 f16UpperLimit;  			/* Control loop Upper Limit */
   	Frac16 f16LowerLimit;			/* Control loop Lower Limit */
} PCLIB_CONTROL_PI_PARAMS_T;    

/******************************************************************************
* Global variables
******************************************************************************/

/******************************************************************************
* Global functions
******************************************************************************/

/***************************************************************************//*!
*
* @brief The function calculates the Proportional-Integral
*        (PI) compensator algorithm according to equations below.The Integral 
*		 Output is limited and limit values (f16IntegUpperLimit and 
*		 f16IntegLowerLimit) are defined by user. The control loop output is 
*        also limited and limits values (f16UpperLimit and f16LowerLimit) are
*		 defined by user. 
*
*
* @param  ptr   		PCLIB_CONTROL_PI_PARAMS_T * const pudtControlStruct
*						  - Pointer to the structure of PI controller parameters
*
*                           PCLIB_CONTROL_PI_PARAMS data structure:
*							Frac16 f16IntegOutputN_1
*                               - state variable; integral part at step n-1;
*                                 can be modified outside of the function;
*                                 input/output parameter.
*								  initial value is ZERO
*							Frac16 f16IntegGain (Ki)
*                               - Integral gain; input parameter
*                                 in the following range
*                                 0 <= f16IntegGain <= 1  
*                           Frac16 f16IntegUpperLimit
*                               - upper limit of the Integral output;
*                                 input parameter;
*                                 f16IntegUpperLimit > f16IntegLowerLimit
*                           Frac16 f16IntegLowerLimit
*                               - lower limit of the Integral output;
*                                 input parameter;
*                                 f16IntegUpperLimit > f16IntegLowerLimit
*                           Frac16 f16PropGain
*                               - Proportional gain; input parameter
*                                 in the following range
*                                 0 <= f16PropGain <=1                          
*                           Frac16 f16UpperLimit
*                               - upper limit of the controller output;
*                                 input parameter;
*                                 f16UpperLimit > f16LowerLimit
*                           Frac16 f16LowerLimit
*                               - lower limit of the controller output;
*                                 input parameter;
*                                 f16UpperLimit > f16LowerLimit 
*
* 
* @param  in    		Frac16 f16ErrorK
*                         - Input error of the controller at step n processed
*                           by P and I terms of the PI algorithm
*                           fractional value in the range -: [-1:1]
*
* @return  The function PCLIB_ControllerPIFAsm returns a fractional value
*          as a result of the PI algorithm. The value returned by the algorithm
*          is in the following range:
*          f16LowerLimit <= PIresult <= f16UpperLimit
*		
* @remarks 	    PI controller algorithm:                                           
*                                                                       
*                        -----------                 
*                 e(n)  |           |  y(n)           
*               --------|    PI     |---------        
*                       |           |                 
*                        -----------                  
* 		                                                                
*       e(n) - input error in continuous time domain                    
*       y(n) - controller output in continuous time domain              
*                                                                                                              
*       The PI controller algorithm in continuous time domain:          
*                                             
*       y(n) = Integral output + Proportional output
*		
*		 Proportional output = Kp*Error(n)
*		Integral output = Ki*Error(n) + Previous Integral Output(n-1)	                                      
*                                                                                                                                                         
*       Ki = (Ki')*Ts
*		Ts - execution interval                                                             
*
*		SATURATION INDEPENDENT!
*
****************************************************************************/
 
Frac16 PCLIB_ControllerPIFAsm(
	Frac16 f16Error, 
	PCLIB_CONTROL_PI_PARAMS_T * const pudtControlSturct);  
 
/******************************************************************************
* Inline functions
******************************************************************************/

#endif /*__PCLIB_CONTROL_PI_PARAMS_H__*/


