/*-----------------------------------------------------------------------------
  Copyright (c) 2015 by Freescale Semiconductor, All Rights reserved.

  This is unpublished proprietary source code of Freescale.
  The copyright notice above does not imply any actual or
  intended publication of such source code.

  FREESCALE CONFIDENTIAL PROPRIETARY

  ---------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  FILE NAME      : memory_update_protocol_testflash.c
  DEPARTMENT     : AISG
  -----------------------------------------------------------------------------
  REVISION HISTORY
            Date D/M/Y                     DESCRIPTION              
            ----------                 ---------------------
            27/7/2015                    Initial Version
       --------------------- 	       ---------------------
  KEYWORDS :HSM, Security
  -----------------------------------------------------------------------------
  PURPOSE :To update key storage area. 
  ---------------------------------------------------------------------------*/

/*=============================================================================
  INCLUDE FILES
  ===========================================================================*/
#include "common.h"
#include "hsm.h"
/*=============================================================================
  LOCAL CONSTANTS
  ===========================================================================*/

/*=============================================================================
  LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
  ===========================================================================*/
extern struct HSM_CMD_struct HSM_CMD;
/*=============================================================================
  LOCAL VARIABLES
  ===========================================================================*/

/*=============================================================================
  LOCAL FUNCTION PROTOTYPES
  ===========================================================================*/
extern void host_pll_fxosc_setup();
extern void host_pll_firc_setup();
/*=============================================================================
  GLOBAL VARIABLES
  ===========================================================================*/

/*=============================================================================
  LOCAL FUNCTIONS
  ===========================================================================*/
void memory_update_protocol_test(uint8_t KeyUpdateStruct[]);
/*=============================================================================
  FUNCTION: memory_update_protocol_test

  DESCRIPTION:
  This function updates the key storage area 

  ARGUMENTS PASSED:
  KeyUpdateStruct:Array containing key value,KeyID,flag value and counter value
  
  RETURN VALUE:
  None
  
  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None

  ===========================================================================*/
void memory_update_protocol_test(uint8_t KeyUpdateStruct[])
{
#define UID_ADDRESS	(0x004000A0)

#pragma alignvar (4)
    /* GET ID command specific challenge*/
    uint8_t getid_challenge[16] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
    };
#pragma alignvar (4)
    /* GET ID command specific returne uid */
    uint8_t uid[15];							
    uint8_t sreg8 = 0;
    uint8_t cmac[16];	
    uint8_t *uidPtr = (unsigned char *)UID_ADDRESS;

#define KEY_SIZE_IN_WORD 4
#define AES_BLOCK_SIZE	16

    const uint32_t KEY_UPDATE_ENC_C[KEY_SIZE_IN_WORD] =		   \
    {(uint32_t)0x01015348UL, (uint32_t)0x45008000UL,
	(uint32_t)0x00000000UL, (uint32_t)0x000000B0UL};
    const uint32_t KEY_UPDATE_MAC_C[KEY_SIZE_IN_WORD] =		   \
    {(uint32_t)0x01025348UL, (uint32_t)0x45008000UL,
	(uint32_t)0x00000000UL, (uint32_t)0x000000B0UL};

    /* MASTER ECU KEY*/
    uint32_t MASTER_ECU_KEY [KEY_SIZE_IN_WORD]= 		  \
    {(uint32_t)0x00010203UL, (uint32_t)0x04050607UL,
	(uint32_t)0x08090A0BUL, (uint32_t)0x0C0D0E0FUL};
    /* Initialization Vector */
    uint32_t IV[KEY_SIZE_IN_WORD] = 				\
	{(uint32_t)0x00000000UL, (uint32_t)0x00000000UL,
	    (uint32_t)0x00000000UL, (uint32_t)0x00000000UL};

#pragma alignvar (4)
    /* Name of the key to be updated */
    uint8_t KEY_ID_val = KeyUpdateStruct[16];   				
#pragma alignvar (4)
    uint8_t ID_AuthID = (KEY_ID_val<<4 & 0xF0) | 0x01; 				
    uint32_t count_val;       							
    uint32_t flag_val;        							
    uint32_t K1[KEY_SIZE_IN_WORD];
    uint32_t K2[KEY_SIZE_IN_WORD];
    uint32_t K3[KEY_SIZE_IN_WORD];
    uint32_t K4[KEY_SIZE_IN_WORD];
#pragma alignvar (4)
    uint8_t M1_t[AES_BLOCK_SIZE];
    uint32_t M1[KEY_SIZE_IN_WORD];
    uint32_t M2_t[KEY_SIZE_IN_WORD*2];
    uint32_t M2[KEY_SIZE_IN_WORD*2];
    uint32_t M3_t[KEY_SIZE_IN_WORD*3];
    uint32_t M3[KEY_SIZE_IN_WORD];
    uint32_t M4_o[KEY_SIZE_IN_WORD*2];
    uint32_t M5_o[KEY_SIZE_IN_WORD];
    uint32_t M4_i[KEY_SIZE_IN_WORD*2];
    uint32_t M4_t[KEY_SIZE_IN_WORD];
    uint32_t M4_te[KEY_SIZE_IN_WORD];
    uint32_t M5_i[KEY_SIZE_IN_WORD];
    uint32_t plaintext[KEY_SIZE_IN_WORD*2];
    int i=0;

#pragma alignvar (4)
    uint8_t messagedebugcmaclen[AES_BLOCK_SIZE/2] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x80};
#pragma alignvar (4)
    uint8_t messagedebugcmaclen_M4[AES_BLOCK_SIZE/2] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00};

/*get UID*/
    while(HSM.HSM2HTS.B.BUSY == 1);
    /* Get ID*/
    HSM_CMD.CMD = 0x10;
    /* challenge address */
    HSM_CMD.PARAM_1 = (uint32_t)&getid_challenge;
    /* UID Output address */
    HSM_CMD.PARAM_2 = (uint32_t)&uid;
    /* HSM to HOST status HSM2HTS[7:0] value address */
    HSM_CMD.PARAM_3 = (uint32_t)&sreg8;
    /* MAC address */
    HSM_CMD.PARAM_4 = (uint32_t)&cmac;       			 		
    HSM.HT2HSMS.R   = (uint32_t) &HSM_CMD;
    /* send command */
    HSM.HT2HSMF.B.CMD_INT = 1; 					 		
    hsm_done();

/* compare UID 120bits */
    for(i=0; i<(15); i++)
    {
	if(uid[i] != *(uidPtr + i))
	{
	    while(1);
	}
    }

/*Generate K1*/
    /* K1 = KDF(KAuthID, KEY_UPDATE_ENC_C) */
    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	plaintext[i]=MASTER_ECU_KEY[i];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	plaintext[i]=KEY_UPDATE_ENC_C[i-KEY_SIZE_IN_WORD];
    /* running KDF function to get K1 */
    hsm_kdf((uint32_t *)&plaintext, (uint32_t *) &K1, 2UL);   			

/*Generate K2*/  
    /* K2 = KDF(KAuthID,KEY_UPDATE_MAC_C) */
    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	plaintext[i]=MASTER_ECU_KEY[i];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	plaintext[i]=KEY_UPDATE_MAC_C[i-KEY_SIZE_IN_WORD];
    /* running KDF function to get K2 */
    hsm_kdf((uint32_t *)&plaintext, (uint32_t *) &K2, 2UL);   			

/* generate M1 */
    /* M1 = UID|ID|AuthID */
    for(i=0;i<15;i++)
	M1_t[i] = uid[i];
    /* key to be updated and authorizing key ID */
    M1_t[15] = ID_AuthID;   							 
    /* converting uint8_t M1_t to uint32_t M1 */	
    M1[0] = (uint32_t)M1_t[0]<<24 | (uint32_t)M1_t[1]<<16 |
	(uint32_t)M1_t[2]<<8 | (uint32_t)M1_t[3] ;
    M1[1] = (uint32_t)M1_t[4]<<24 | (uint32_t)M1_t[5]<<16 |
	(uint32_t)M1_t[6]<<8 | (uint32_t)M1_t[7] ;
    M1[2] = (uint32_t)M1_t[8]<<24 | (uint32_t)M1_t[9]<<16 |
	(uint32_t)M1_t[10]<<8 | (uint32_t)M1_t[11] ;
    M1[3] = (uint32_t)M1_t[12]<<24 | (uint32_t)M1_t[13]<<16 |
	(uint32_t)M1_t[14]<<8 | (uint32_t)M1_t[15] ;

/* generate M2 */
    /* M2 = ENCCBC,K1,IV=0(CID|FID|0...0"95|KID)
       starting address of 32 bit counter value is KeyUpdateStruct[20] */
    count_val = (uint32_t)KeyUpdateStruct[20]<<24 |
	(uint32_t)KeyUpdateStruct[21]<<16 |
	(uint32_t)KeyUpdateStruct[22]<<8 |
	(uint32_t)KeyUpdateStruct[23];

    /* 8 bit flag is stored at KeyUpdateStruct[19] */
    flag_val = (uint32_t)KeyUpdateStruct[19];
    /* 28 bit counter is at MSB and 4 MSB bits out of 6 of flags at LSB */
    M2_t[0] = ((count_val<<4) & 0xFFFFFFF0) |
	((flag_val>>2) & 0x0000000F);	 

    flag_val = (uint32_t)KeyUpdateStruct[19];
    M2_t[1] = (flag_val<<30) & 0xC0000000;   					
    for(i=2;i<KEY_SIZE_IN_WORD;i++)
	M2_t[i] = 0x00000000;

    /* loading Key value to be updated */
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	M2_t[i] = (uint32_t)KeyUpdateStruct[4*(i-KEY_SIZE_IN_WORD)+0]<<24 |
	    (uint32_t)KeyUpdateStruct[4*(i-KEY_SIZE_IN_WORD)+1]<<16 |
	    (uint32_t)KeyUpdateStruct[4*(i-KEY_SIZE_IN_WORD)+2]<<8 |
	    (uint32_t)KeyUpdateStruct[4*(i-KEY_SIZE_IN_WORD)+3];

/* Load K1 in RAM key */ 
    HSM_load_plain_key((uint32_t)&K1);    					
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();
    for (i = 0;i<8; i++)
    {
	M2[i] = 0;
    }

/*CBC ENCRYPT M2_t using K1 to get M2*/ 
    HSM_encrypt_CBC(2,
	    (uint32_t) &M2_t,
	    (uint32_t) &M2,
	    (uint32_t)&IV);   
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();   

/* generate M3 */
    /* M3 = CMACK2(M1|M2) */
    /*Load K2 in RAM key*/
    HSM_load_plain_key((uint32_t)&K2);    					
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

/*Generate (M1|M2)*/
    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	M3_t[i]=M1[i];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*3;i++)
	M3_t[i]=M2[i-KEY_SIZE_IN_WORD];

/*generate MAC over (M1|M2) to get M3*/ 
    HSM_CMD.CMD = 0x5;
    /* key ID = RAM_KEY (Plain key) = K2 */
    HSM_CMD.PARAM_1 = (uint32_t)0xE;
    /* Input Message length (in bits) address */
    HSM_CMD.PARAM_2 = (uint32_t)&messagedebugcmaclen;
    /* Input Message start address */
    HSM_CMD.PARAM_3 = (uint32_t)&M3_t;
    /* Output MAC address */
    HSM_CMD.PARAM_4 = (uint32_t)&M3;   						
    HSM.HT2HSMS.R   = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();


/* load key to the Key Storage Area */
    HSM_CMD.CMD = 0x7;                						
    HSM_CMD.PARAM_1 = (uint32_t)&M1;           
    HSM_CMD.PARAM_2 = (uint32_t)&M2;        
    HSM_CMD.PARAM_3 = (uint32_t)&M3; 
    HSM_CMD.PARAM_4 = (uint32_t)&M4_o;
    HSM_CMD.PARAM_5 = (uint32_t)&M5_o;
    HSM.HT2HSMS.R   = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

/*Get K3*/
    /* K3 = KDF(KEYID,KEY_UPDATE_ENC_C) */
    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	plaintext[i]=(uint32_t)KeyUpdateStruct[4*i+0]<<24 |
	    (uint32_t)KeyUpdateStruct[4*i+1]<<16 |
	    (uint32_t)KeyUpdateStruct[4*i+2]<<8 |
	    (uint32_t)KeyUpdateStruct[4*i+3];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	plaintext[i]=KEY_UPDATE_ENC_C[i-KEY_SIZE_IN_WORD];
    hsm_kdf((uint32_t *)&plaintext, (uint32_t *) &K3, 2UL);

/*Get K4*/
    /* K4 = KDF (KEYID,KEY_UPDATE_MAC_C) */
    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	plaintext[i]=(uint32_t)KeyUpdateStruct[4*i+0]<<24 |
	    (uint32_t)KeyUpdateStruct[4*i+1]<<16 |
	    (uint32_t)KeyUpdateStruct[4*i+2]<<8 |
	    (uint32_t)KeyUpdateStruct[4*i+3];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	plaintext[i]=KEY_UPDATE_MAC_C[i-KEY_SIZE_IN_WORD];
    hsm_kdf((uint32_t *)&plaintext, (uint32_t *) &K4, 2UL);

/* Generate M4 */
    /* M4 = UID|ID|AuthID|M4*
       M4* - the encrypted counter value; prior to encryption the counter
       value (28 bits) is padded with a 1 and 99 0s.
       The key for the ECB encryption is K3
       starting address of 32 bit counter value is KeyUpdateStruct[20] */
    count_val = (uint32_t)KeyUpdateStruct[20]<<24 |
	(uint32_t)KeyUpdateStruct[21]<<16 |
	(uint32_t)KeyUpdateStruct[22]<<8 |
	(uint32_t)KeyUpdateStruct[23];;

    M4_t[0] = ((count_val<<4) & 0xFFFFFFF0) | 0x00000008;
    for(i=1;i<KEY_SIZE_IN_WORD;i++)
	M4_t[i] = 0x00000000;

/* Load K3 in RAM key */	  
    HSM_load_plain_key((uint32_t)&K3);
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

    for (i = 0;i<4; i++)
    {
	M4_te[i] = 0;
    }

/*EBC ENCRYPT */  
    HSM_encrypt_ECB(1,
	    (uint32_t)&M4_t,
	    (uint32_t)&M4_te);   
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

    for(i=0;i<KEY_SIZE_IN_WORD;i++)
	M4_i[i]=M1[i];
    for(i=KEY_SIZE_IN_WORD;i<KEY_SIZE_IN_WORD*2;i++)
	M4_i[i]=M4_te[i-KEY_SIZE_IN_WORD];

/* Generate M5 */   
    /* M5 = CMACK4(M4) */
    HSM_load_plain_key((uint32_t)&K4);
    HSM.HT2HSMS.R         = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

/* Generate MAC to get M5 */
    HSM_CMD.CMD = 0x5;
    /* key ID = RAM_KEY (Plain key) = K4 */
    HSM_CMD.PARAM_1 = (uint32_t)0xE;
    /* Input Message length (in bits) address */
    HSM_CMD.PARAM_2 = (uint32_t)&messagedebugcmaclen_M4;
    /* input Message start address */
    HSM_CMD.PARAM_3 = (uint32_t)&M4_i;
    /* output MAC address */
    HSM_CMD.PARAM_4 = (uint32_t)&M5_i; 						
    HSM.HT2HSMS.R   = (uint32_t) &HSM_CMD;
    HSM.HT2HSMF.B.CMD_INT = 1;  						
    hsm_done();

/* check if M4 is equal */
    for(i=0; i<KEY_SIZE_IN_WORD*2; i++)
    {
	if(M4_i[i] != M4_o[i])
	{
	    while(1);
	}
    }
/* check if M5 is equal */
    for(i=0; i<KEY_SIZE_IN_WORD;i++)
    {
	if(M5_i[i] != M5_o[i])
	{
	    while(1);
	}
    }

}

/*=============================================================================
  FUNCTION: main

  DESCRIPTION:
  This is the main function/entry point for the application 

  ARGUMENTS PASSED:
  None
  
  RETURN VALUE:
  None
  
  PRE-CONDITIONS:
  None

  POST-CONDITIONS:
  None

  IMPORTANT NOTES:
  None
  ===========================================================================*/

void main(void)
{
#pragma alignvar (4)
    /* key values */
    /* key values */
    /* 8 bit key_id, 16 reserved bits, 8 bit flags and 32 bit counter */
    /* reserved */
    uint8_t KeyUpdateStruct[32] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,   			
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,  			
	0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09,   			
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00    			
    };
    
#ifdef FXOSCPLLSRC
    host_pll_fxosc_setup();
#else
    host_pll_firc_setup();    
#endif

    memory_update_protocol_test(KeyUpdateStruct);
    while(1);
}

