/*****************************************************************************
 * (c) Copyright 2016, Freescale Semiconductor 
 * ALL RIGHTS RESERVED.
 ***************************************************************************//*!
 * @file     g726_enc_dec_test.c
 * @author   b01800 
 * @version  0.0.1.0
 * @date     Dec-14-2015
 * @brief    G.726 Audio Encoding /Decoding test
 * @details  KL43 device acts as I2S master, connected to SGTL5000 audio codec IC
 *           (TWR-AUDIO-SGTL)
 *           Input audio signal connected to LINE-IN / HEADSET(MIC) input
 *           Input selected by SW3
 *           Output audio signal connected to HEADPHONE output
 *           IN/OUT PCM audio data streams managed by 2ch DMA
 *           16-bit PCM audio 8kHz sample rate
 *           G.726 encoding quality can be selected as BITRATE <8,16,24,40kbit/s>
 * @target   TWR-KL43Z48M (CM0+ @48MHz) + TWR-AUDIO_SGTL
 *
 ******************************************************************************/
#include "common.h"
#include "drivers.h"
#include "derivative.h"     /* include peripheral declarations */
#include "types.h"          /* User Defined Data Types */
#include "usb_hid.h"
#include "sgtl5000.h"
#include "g72x.h"

//-------------------------- G.726 SW Codec Defs -------------------------------
#define SAMPLE_RATE 8000L
#define PTIME 10 ///< define G726 time-slot in [ms], 10ms typically 
#define SAMPLES_PER_FRAME ((SAMPLE_RATE * PTIME) / 1000) ///< 80 samples 
#define PCM_FRAME_SIZE (SAMPLES_PER_FRAME)  ///< RAW PCM audio (16bit per sample)
#define I2S_BUF_SIZE  (PCM_FRAME_SIZE)      ///< Size of I2S output buffers: equal to the PCM audio IN/OUT size

#define BITRATE  40000 ///< define G.726 Bitrate: 16, 24, 32, 40 kbit/s

#if  (BITRATE == 16000)
  #define CODE_BITS           2
  #define CODE_BIT_MASK       0x03
  #define ENC_FRAME_SIZE     (SAMPLES_PER_FRAME * CODE_BITS) / 8 ///< 20 bytes 
  #define ENCODE_FUNC        g726_16_encoder
  #define DECODE_FUNC        g726_16_decoder
#endif

#if  (BITRATE == 24000)
  #define CODE_BITS           3
  #define CODE_BIT_MASK       0x07
  #define ENC_FRAME_SIZE     (SAMPLES_PER_FRAME * CODE_BITS) / 8 ///< 30 bytes
  #define ENCODE_FUNC        g726_24_encoder
  #define DECODE_FUNC        g726_24_decoder
#endif

#if  (BITRATE == 32000)
  #define CODE_BITS           4
  #define CODE_BIT_MASK       0x0f
  #define ENC_FRAME_SIZE     (SAMPLES_PER_FRAME * CODE_BITS) / 8 ///< 40 bytes
  #define ENCODE_FUNC        g726_32_encoder
  #define DECODE_FUNC        g726_32_decoder
#endif

#if  (BITRATE == 40000)
  #define CODE_BITS           5
  #define CODE_BIT_MASK       0x1f
  #define ENC_FRAME_SIZE     (SAMPLES_PER_FRAME * CODE_BITS) / 8 ///< 50 bytes
  #define ENCODE_FUNC        g726_40_encoder
  #define DECODE_FUNC        g726_40_decoder
#endif
//------------------------------------------------------------------------------
// Auxiliary Audio Functions vars.
typedef enum {
  	ACARD_LINE_IN,
  	ACARD_MIC_IN,
} AUDIO_IN_SEL;
     
/* callback functions prototypes                                              */
static void dma0_callback (DMA_CALLBACK_TYPE type);
static void dma1_callback (DMA_CALLBACK_TYPE type);

static void porta_callback(vuint32 pin_number);

/* module function prototypes                                                 */
static void sai_playback(void);
void AudioCard_SwitchInput(AUDIO_IN_SEL input);

 void g726_codec_decode(uint8_t* input, int16_t* output);
 void g726_codec_encode(int16_t* input, uint8_t* output);

/*******************************************************************************
*//*! @addtogroup twr_sai_audio_playback_test_variables_def
* @{ 
*   Module global variables definitions are provided here.
*******************************************************************************/
int16 aud_out_buf1[I2S_BUF_SIZE];         ///< first audio buffer
int16 aud_out_buf2[I2S_BUF_SIZE];         ///< second audio buffer      

int16 aud_in_buf1[I2S_BUF_SIZE];          ///< first audio input buffer
int16 aud_in_buf2[I2S_BUF_SIZE];          ///< second audio input buffer 

static int16 *outbuf =      aud_out_buf1; ///< pointer to playack buffer
static int16 *outbuf_free = aud_out_buf2; ///< pointer to free buffer

static int16 *inbuf =       aud_in_buf1;  ///< pointer to playack buffer
static int16 *inbuf_full =  aud_in_buf2;  ///< pointer to free buffer

g726_state encoder;
g726_state decoder;

uint8_t g726_in_buf[ENC_FRAME_SIZE];
int16_t g726_out_buf[PCM_FRAME_SIZE];

/*! @} End of twr_sai_audio_playback_test_variables_def                       */

/*******************************************************************************
 * @brief   main routine
 ******************************************************************************/
int main (void)
{  
  SIM_Init (SIM_MODULE_CONFIG_ALL_PERIPH_ON);
  MCG_LITE_Init (MCG_LITE_HIRC_48MHZ); /* Set CORE clock to 48MHz, using IRC */
  SIM_SetOUTDIV1(DIV_01); /* SYSCLK = 48/2 = 24MHz = I2S MCLK frequency */
  SIM_SetOUTDIV4(DIV_02); /* BUS clock = SYSCLK 24MHz */
  
  // GPIO init for Audio input selection by SW3
  PORT_Init(PORTA, PORT_MODULE_BUTTON_IRQ_MODE, PIN_4|PIN_5, 1, porta_callback);
  GPIO_Init(GPIOA, GPIO_PIN_INPUT, PIN_4|PIN_5);
  
  // TWR LED configuration
  PORT_Init (PORTA, PORT_MODULE_ALT1_MODE, PIN_12|PIN_13, 0, NULL);
  GPIO_Init (GPIOA, GPIO_PIN_OUTPUT, PIN_12|PIN_13);
  GPIO_Clr (GPIOA, PIN_12);
  GPIO_Set (GPIOA, PIN_13);
  
  // Set LED pin (PTB19) to be toggled for g726 processing measurement 
  PORT_Init (PORTB, PORT_MODULE_ALT1_MODE, PIN_19, 0, NULL);
  GPIO_Init (GPIOB, GPIO_PIN_OUTPUT, PIN_19);
  GPIO_Clr  (GPIOB, PIN_19);
  
  /****************** SAI/I2S init **************************************/
  /* Set I2S pin mux at PTB_18, PTC_0,2,5,6 */
  PORT_Init(PORTB, PORT_MODULE_ALT4_MODE, PIN_18, 0, NULL);
  PORT_Init(PORTC, PORT_MODULE_ALT6_MODE_I2S, PIN_0|PIN_2|PIN_6, 0, NULL); 
  PORT_Init(PORTC, PORT_MODULE_ALT4_MODE, PIN_5, 0, NULL);
  // I2S bus init for Codec IC interface
  SAI_Init (SAI0, SAI_MODULE_MASTER_TX_RX_DMA_CONFIG, SAMPLE_RATE, 0, NULL);
    
  /****************** TWR-AUDIO-SGTL5000 init ***************************/
  /* Set I2C pin mux at PTE_0,1 */
  PORT_Init(PORTE, PORT_MODULE_ALT6_MODE, PIN_0|PIN_1, 0, NULL);
  I2C1_Init (I2C_MODULE_100K_MASTER_IRQ_DIS_CONFIG, 0, NULL);
  sgtl5000_Init();
  Sgtl500_Set_Volume(32,32,32); /* SGTL5000 codec volume max  */
  /*********************************************************************/
  
  /* Init g.726 state variables                                               */
  g726_init_state(&encoder);
  g726_init_state(&decoder);
  
  /* I2S input => RAM Buffer                                                  */
  DMA_Init(DMA1, DMA_DINC_SINGLE_ERQ_IRQ_CONFIG(I2S_BUF_SIZE*2, DMA_WORD, DMA_WORD),
          (uint32*)&I2S0_RDR0, (uint32*)&aud_in_buf1, 1, dma1_callback); 
  
  /* RAM Buffer => I2S output                                                 */  
  DMA_Init(DMA0, DMA_SINC_SINGLE_ERQ_IRQ_CONFIG(I2S_BUF_SIZE*2, DMA_WORD, DMA_WORD),
          (uint32*)&aud_out_buf1, (uint32*)&I2S0_TDR0, 2, dma0_callback);
    
  DMAMUX0_Init(CH1, DMAMUX_CH_DISABLE_CONFIG(DMA_SRC_I2S_RX));  
  DMAMUX0_Init(CH0, DMAMUX_CH_DISABLE_CONFIG(DMA_SRC_I2S_TX));
  
  DMA_Enable(DMA1); // Enable DMA Audio feeding
  DMA_Enable(DMA0); // Enable DMA Audio output
  
  __enable_irq();
    
  while (1)
  {
    sai_playback();
  }
}

/*******************************************************************************
*//*! @addtogroup twr_sai_audio_playback_test_functions_def
* @{ 
*   Module local function definitions are provided here.
*******************************************************************************/
/**************************************************************************//*!
 * @brief  Function fill up the DMA in/out buffers with new audio stream data   
 * @param  outbuf_free - pointer to free output buffer
 * @return none
 ******************************************************************************/

// Main CPU G726 processing function
 static void sai_playback(void)
{ 
//------------------ Encoding ------------------------
  if(inbuf_full != NULL)
  {
    GPIO_Set (GPIOB, PIN_19);  
    g726_codec_encode(inbuf_full, g726_in_buf); // encode
    inbuf_full = NULL;
    //GPIO_Clr (GPIOB, PIN_19);
  //}
//----------------------------------------------------

//----------------- Decoding -------------------------  
  if (outbuf_free != NULL) // Is there any empty buffer? (to be filled) 
  {
    g726_codec_decode (g726_in_buf, outbuf_free);   // decode
    outbuf_free = NULL; // buffer is now not free (filled with data)
    GPIO_Clr (GPIOB, PIN_19);
  }
//-----------------------------------------------------  
  }
}

/*******************************************************************************
*//*! @addtogroup sai_usecase1_functions_def
* @{ 
*   Module local function definitions are provided here.
*******************************************************************************/

/***************************************************************************//*!
 * @brief   DMA0/I2S0 IRQ Callback function definition
 * @param   outbuf, outbuf_free - pointers to audio output buffers
 * @note    Feeding output PCM audio samples data to I2S bus   
 ******************************************************************************/
 static void dma0_callback (DMA_CALLBACK_TYPE type)
{
  DMA_DSR_BCR0 |= DMA_DSR_BCR_DONE_MASK;

  if (outbuf == aud_out_buf1)     // buf1 playback completed ? 
  { 
      outbuf_free = aud_out_buf1; // buf1 to be filled 
      outbuf = aud_out_buf2;      // buf2 to be played in the next round 
  }
  
  else                     
  {
      outbuf_free = aud_out_buf2;  // exchange the roles of the buffers 
      outbuf = aud_out_buf1;
  }
  
  DMA_SAR0 = (uint32_t)outbuf;
  DMA_DSR_BCR0 |= DMA_DSR_BCR_BCR(I2S_BUF_SIZE*2);  
  DMA_DCR0 |= DMA_DCR_ERQ_MASK;   // Reneable DMA 
}

/***************************************************************************//*!
 * @brief   DMA1/I2S0 IRQ Callback function definition
 * @param   inbuf, inbuf_full - pointers to audio input buffers
 * @note    Reading input PCM audio samples from I2S bus
 ******************************************************************************/
 static void dma1_callback (DMA_CALLBACK_TYPE type)
{
  DMA_DSR_BCR1 |= DMA_DSR_BCR_DONE_MASK;
  
  if (inbuf == aud_in_buf1)     // buf1 playback completed ? 
  { 
      inbuf_full = aud_in_buf1; // buf1 to be filled 
      inbuf = aud_in_buf2;      // buf2 to be played in the next round 
  }
  
  else                     
  {
      inbuf_full = aud_in_buf2; // exchange the roles of the buffers 
      inbuf = aud_in_buf1;
  }
  
  DMA_DAR1 = (uint32_t)inbuf;
  DMA_DSR_BCR1 |= DMA_DSR_BCR_BCR(I2S_BUF_SIZE*2); 
  DMA_DCR1 |= DMA_DCR_ERQ_MASK; // Reneable DMA 
}


/***************************************************************************//*!
 * @brief   G726 Decode
 * @param   input buffer with G726 encoded data, output buffer with 16-bit PCM
 * @note    Decoding G726 compr. data stream to 16-bit PCM audio data stream
 ******************************************************************************/
 void g726_codec_decode(uint8_t* input, int16_t* output)
{
    unsigned samples, i; 
    uint8_t  *src;
    int16_t *dst;
    unsigned int in_buffer = 0;
    int in_bits = 0;
    uint8_t code;
    
    uint8_t code_bits = CODE_BITS;
    uint8_t code_bit_mask = CODE_BIT_MASK;
    
    
    /* Decode g726 */ 
    samples = SAMPLES_PER_FRAME;
    src = (uint8_t*) input;
    dst = (int16_t*) output;
    
    for (i=0; i<samples; i++) {
        
        if (in_bits < code_bits) {        
            in_buffer |= (*src++ << in_bits);
            in_bits += 8;
        }
        
        code = in_buffer & code_bit_mask;
        in_buffer >>= code_bits;
        in_bits -= code_bits;
        dst[i] = (int16_t) DECODE_FUNC(code, AUDIO_ENCODING_LINEAR, &decoder);     
    }
}

/***************************************************************************//*!
 * @brief   G726 Encode
 * @param   input buffer with 16-bit PCM data
 * @note    Encoding 16-bit PCM audio data stream to G726 compr. data stream
 ******************************************************************************/
 void g726_codec_encode(int16_t* input, uint8_t* output)
{
    unsigned samples, i; 
    int16_t *src;
    uint8_t *dst;
    unsigned int out_buffer = 0;
    int out_bits = 0;
    unsigned code;
    uint8_t code_bits = CODE_BITS;
    
    /* Encode g726 */
    samples = SAMPLES_PER_FRAME;
    src = (int16_t*) input;
    dst = (uint8_t*) output;
    
     for (i=0; i<samples; i++) {
         
        code = (uint8_t) ENCODE_FUNC(src[i], AUDIO_ENCODING_LINEAR, &encoder);
 
        out_buffer |= (code << out_bits);
 
        out_bits += code_bits;
 
        if (out_bits >= 8) {
            
            *dst++ = out_buffer & 0xff;
            out_bits -= 8;
            out_buffer >>= 8;
        }
     }
}

/***************************************************************************//*!
 * @brief   PORTA Callback function - SW2 on rissing edge
 * @param   pin_number - @ref pin_names defined in gpio.h
 * @note    Switching LINE-IN/ HEADSET(MIC) INPUT
 ******************************************************************************/
static void porta_callback(vuint32 pin_number)
{ 
  static uint32_t mic_input = 0; 

  if (pin_number&PIN_5)
  {
      if(!mic_input){
        AudioCard_SwitchInput(ACARD_MIC_IN);   
        mic_input = 1;
      }
      
      else
      {
        AudioCard_SwitchInput(ACARD_LINE_IN); 
        mic_input = 0; 
      }
    
    GPIO_Tgl(GPIOA,PIN_13);
    GPIO_Tgl(GPIOA,PIN_12);
  }
  
}

/***************************************************************************//*!
 * @brief   Audio Card Switch Input function
 * @param   ACARD_LINE_IN, ACARD_MIC_IN
 * @note    Switching LINE-IN/ HEADSET(MIC) INPUT
 ******************************************************************************/
void AudioCard_SwitchInput(AUDIO_IN_SEL input)
{
    if(input == ACARD_LINE_IN)
    {
    //For Input from Line IN  
      writeSgtl500Reg( SGTL5000_CHIP_ANA_CTRL ,0x0004);    // 4: LINEIN
      writeSgtl500Reg(SGTL5000_CHIP_ANA_ADC_CTRL, 0x00AA); // Input ADC volume
    }
  else
    {  
    // For Input from MIC
      writeSgtl500Reg(SGTL5000_CHIP_ANA_ADC_CTRL, 0x0000);   // Input ADC volume     
      writeSgtl500Reg( SGTL5000_CHIP_ANA_CTRL ,0x0000);      //0: MIC
    }
}
/*! @} End of g726_enc_dec_test_functions_def                                 */


/*******************************************************************************
 * End of main module                                                          *
 *******************************************************************************/

