/*
 * Copyright (c) 2015-2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include <math.h>

#include "lcd_hx8357_drv.h"
#include "draw_house.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define LCD_X_POINTS    LCD_H_POINTS
#define LCD_Y_POINTS    LCD_V_POINTS

/*******************************************************************************
 * Functions
 ******************************************************************************/
void Graph_DrawPoint(int32_t x, int32_t y, uint16_t color)
{
    AreaPoints_t area;

    if((x>0) && (x<=LCD_X_POINTS-1) && (y>0) && (y<=LCD_Y_POINTS-1))
    {
        area.x1 = x;
        area.y1 = y;
        area.x2 = x+1 > LCD_X_POINTS-1 ? LCD_X_POINTS-1 : x+1;
        area.y2 = y+1 > LCD_Y_POINTS-1 ? LCD_Y_POINTS-1 : y+1;

        LCD_HX8357_FillColorPolling(&area, color);
    }
}

void Graph_DrawLine(const Point_t * point1, const Point_t * point2,
                    const uint16_t color)
{
    int32_t Dif_x, Dif_y;
    int32_t Px = point1->x;
    int32_t Py = point1->y;
    int32_t Dist_x = point2->x - point1->x;
    int32_t Dist_y = point2->y - point1->y;
    int32_t Ax = 0, Ay = 0, steps;

    if(Dist_x > 0)
    {
        Dif_x = 1;
    }
    else if(Dist_x == 0)
    {
        Dif_x = 0;
    }
    else
    {
        Dif_x =-1;
        Dist_x =- Dist_x;
    }
    if(Dist_y > 0)
    {
        Dif_y = 1;
    }
    else if(Dist_y == 0)
    {
        Dif_y = 0;
    }
    else
    {
        Dif_y = -1;
        Dist_y = -Dist_y;
    }

    steps = Dist_x > Dist_y ? Dist_x : Dist_y;

    for(uint32_t i=0; i<=steps+1; i++)
    {
        Graph_DrawPoint(Px, Py, color);

        Ax += Dist_x;
        Ay += Dist_y;

        if(Ax > steps)
        {
            Ax -= steps;
            Px += Dif_x;
        }
        if(Ay > steps)
        {
            Ay -= steps;
            Py += Dif_y;
        }
    }
}

void Graph_DrawStandardRectangle(const Point_t * points, const uint16_t color)
{
    Point_t  p1, p2;

    p1.x = points->x;
    p1.y = points->y;
    p2.x = (points+1)->x;
    p2.y = points->y;
    Graph_DrawLine(&p1, &p2, color);

    p1.x = points->x;
    p1.y = points->y;
    p2.x = points->x;
    p2.y = (points+1)->y;
    Graph_DrawLine(&p1, &p2, color);

    p1.x = (points+1)->x;
    p1.y = points->y;
    p2.x = (points+1)->x;
    p2.y = (points+1)->y;
    Graph_DrawLine(&p1, &p2, color);

    p1.x = points->x;
    p1.y = (points+1)->y;
    p2.x = (points+1)->x;
    p2.y = (points+1)->y;
    Graph_DrawLine(&p1, &p2, color);
}

void Graph_DrawPolygon(const Point_t * points, uint32_t n, const uint16_t color)
{
    const Point_t * p = points;
    if(n>=3)
    {
        for(uint32_t i=0; i<n-1; i++)
        {
            Graph_DrawLine(p, p+1, color);
            p++;
        }
        Graph_DrawLine(p, points, color);
    }
}

void Graph_DrawCircle(const Point_t * center, int32_t r, const uint16_t color)
{
    int32_t px, py, rx, ry;
    int32_t py_min, py_max;

    for(px = center->x-r; px <= center->x+r; px++)
    {
        ry = (int32_t)sqrt(r*r -(px-center->x)*(px-center->x));

        Graph_DrawPoint(px, center->y + ry, color);
        Graph_DrawPoint(px, center->y - ry, color);
    }

    py_min = (int32_t)(float)(center->y-0.7*r);
    py_max = (int32_t)(float)(center->y+0.7*r);
    for(py = py_min; py <= py_max; py++)
    {
        rx = (int32_t)sqrt(r*r -(py-center->y)*(py-center->y));
        Graph_DrawPoint(center->x+rx, py, color);
        Graph_DrawPoint(center->x-rx, py, color);
    }
}

void Graph_DrawFillIsoTri(const Point_t * points, const uint16_t color)
{
    if((points->x < (points+1)->x) || (points->y > (points+1)->y))
    {
        return;
    }

    int32_t Dif_x, Dif_y;
    int32_t Px = points->x;
    int32_t Py = points->y;
    int32_t Dist_x = (points+1)->x - points->x;
    int32_t Dist_y = (points+1)->y - points->y;
    int32_t Ax = 0, Ay = 0, steps;

    if(Dist_x > 0)
    {
        Dif_x = 1;
    }
    else if(Dist_x == 0)
    {
        Dif_x = 0;
    }
    else
    {
        Dif_x =-1;
        Dist_x =- Dist_x;
    }
    if(Dist_y > 0)
    {
        Dif_y = 1;
    }
    else if(Dist_y == 0)
    {
        Dif_y = 0;
    }
    else
    {
        Dif_y = -1;
        Dist_y = -Dist_y;
    }

    steps = Dist_x > Dist_y ? Dist_x : Dist_y;

    for(uint32_t i=0; i<=steps+1; i++)
    {
#if 0   /* Method 1 */
        HLineP1.x = Px;
        HLineP1.y = Py;
        HLineP2.x = points->x + points->x - Px;
        HLineP2.y = Py;
        Graph_DrawLine(&HLineP1, &HLineP2, color);
#elif 0 /* Method 2 */
        for(int32_t ix=Px; ix<=points->x + points->x - Px; ix++)
            Graph_DrawPoint(ix, Py, color);
#else   /* Method 3 */
        AreaPoints_t area = {Px, Py, points->x + points->x - Px, Py};
        LCD_HX8357_FillColorPolling(&area, color);
#endif

        Ax += Dist_x;
        Ay += Dist_y;

        if(Ax > steps)
        {
            Ax -= steps;
            Px += Dif_x;
        }
        if(Ay > steps)
        {
            Ay -= steps;
            Py += Dif_y;
        }
    }
}

const AreaPoints_t AreaSky = {0, 0, LCD_X_POINTS-1, 119};
const AreaPoints_t AreaGround = {0, 120, LCD_X_POINTS-1, LCD_X_POINTS-1};

const Point_t Roof[] = {{90, 30}, {30, 100}};
const AreaPoints_t Wall = {40, 101, 140, 180};
const AreaPoints_t Door = {60, 130, 90, 180};
const Point_t DoorLine[] = {{75, 130}, {75, 179}};
const AreaPoints_t Window = {105, 115, 130, 140};
const Point_t WindowLine1[] = {{105, 127}, {129, 127}};
const Point_t WindowLine2[] = {{117, 115}, {117, 139}};
const AreaPoints_t Chimney = {120, 40, 135, 82};
const Point_t Road1[] = {{60, 180}, {80, 210}, {120, 239}};
const Point_t Road2[] = {{90, 180}, {110, 210}, {150, 239}};
const Point_t Road3[] = {{70, 195}, {100, 195}};
const Point_t Road4[] = {{80, 210}, {110, 210}};
const Point_t Road5[] = {{100, 225}, {130, 225}};
const Point_t Tree1[] = {{240, 50}, {220, 120}};
const Point_t Tree2[] = {{240, 80}, {215, 160}};
const AreaPoints_t Tree3 = {235, 161, 245, 185};

static uint32_t DrawHouse_Step=0;

void Graph_DrawHouse_Prepare(void)
{
    LCD_HX8357_FillColorWhole(White);
    DrawHouse_Step = 0;
}

void Graph_DrawHouse_Update(void)
{
    switch(DrawHouse_Step)
    {
        case 0:
            LCD_HX8357_FillColorWhole(White);
            break;
        case 1:
            LCD_HX8357_FillColorDMA(&AreaSky, RGB888to565(0x87CEFF));
            break;
        case 2:
            LCD_HX8357_FillColorDMA(&AreaGround, RGB888to565(0xCCFA42));
            break;
        case 3:     /* Chimney */
            LCD_HX8357_FillColorPolling(&Chimney, RGB888to565(0x556B2F));
            break;

        case 4:     /* Roof */
            Graph_DrawFillIsoTri(Roof, RGB888to565(0xEE6363));
            break;

        case 5:     /* Wall */
            LCD_HX8357_FillColorDMA(&Wall, RGB888to565(0xFFFFE0));
            break;

        case 6:     /* Window */
            LCD_HX8357_FillColorPolling(&Window, RGB888to565(0xBDB76B));
            Graph_DrawLine(WindowLine1, WindowLine1+1, RGB888to565(0x8B795E));
            Graph_DrawLine(WindowLine2, WindowLine2+1, RGB888to565(0x8B795E));
            break;

        case 7:     /* Door */
            LCD_HX8357_FillColorPolling(&Door, RGB888to565(0xBDB76B));
            Graph_DrawLine(DoorLine, DoorLine+1, RGB888to565(0x8B795E));
            break;

        case 8:     /* Road */
            Graph_DrawLine(Road1, Road1+1, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road1+1, Road1+2, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road2, Road2+1, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road2+1, Road2+2, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road3, Road3+1, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road4, Road4+1, RGB888to565(0x8B8B7A));
            Graph_DrawLine(Road5, Road5+1, RGB888to565(0x8B8B7A));
            break;

        case 9:     /* Tree1 */
            Graph_DrawFillIsoTri(Tree1, RGB888to565(0x008B00));
            break;
        case 10:    /* Tree2 */
            Graph_DrawFillIsoTri(Tree2, RGB888to565(0x008B00));
            break;
        case 11:    /* Tree3 */
            LCD_HX8357_FillColorPolling(&Tree3, RGB888to565(0x8B6914));
            break;
        default:
            break;
    }

    if(DrawHouse_Step < 11U)
    {
        DrawHouse_Step++;
    }
    else
    {
        DrawHouse_Step = 0U;
    }
}

/* EOF */
