/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_common.h"
#include "fsl_smc.h"
#include "clock_config.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define CORE_CLOCK_MODE_VLPR 8000000U
#define CORE_CLOCK_MODE_RUN 48000000U
#define CORE_CLOCK_MODE_HSRUN 96000000U

/*******************************************************************************
 * Variables
 ******************************************************************************/
/* System clock frequency. */
extern uint32_t SystemCoreClock;

/*
 * SYSOSC clock setting.
 * SYSOSC clock         : 8MHz
 * SOSCDIV1_CLK output  : Disable
 * SOSCDIV2_CLK output  : Disable
 * SOSCDIV3_CLK output  : 8MHz
 */
const scg_sosc_config_t g_scgSysOscConfig = {.freq = BOARD_XTAL0_CLK_HZ,

                                             .enableMode = kSCG_SysOscEnable | kSCG_SysOscEnableErClk,
                                             .monitorMode = kSCG_SysOscMonitorDisable,
                                             .div1 = kSCG_AsyncClkDisable,
                                             .div2 = kSCG_AsyncClkDisable,
                                             .div3 = kSCG_AsyncClkDivBy1,
                                             .capLoad = 0U,
                                             .workMode = kSCG_SysOscModeOscLowPower};

/*
 * SIRC clock setting.
 * SIRC clock           : 8MHz
 * SIRCDIV1_CLK output  : Disable
 * SIRCDIV2_CLK output  : Disable
 * SIRCDIV3_CLK output  : 4MHz
 */
const scg_sirc_config_t g_scgSircConfig = {.enableMode = kSCG_SircEnable | kSCG_SircEnableInLowPower,
                                           .div1 = kSCG_AsyncClkDisable,
                                           .div2 = kSCG_AsyncClkDisable,
                                           .div3 = kSCG_AsyncClkDivBy2,
                                           .range = kSCG_SircRangeHigh};

/*
 * FIRC clock setting.
 * FIRC clock           : 48MHz
 * FIRCDIV1_CLK output  : 48MHz
 * FIRCDIV2_CLK output  : Disable
 * FIRCDIV3_CLK output  : 48MHz
 */
const scg_firc_config_t g_scgFircConfig = {.enableMode = kSCG_FircEnable,
                                           .div3 = kSCG_AsyncClkDivBy1,
                                           .div2 = kSCG_AsyncClkDisable,
                                           .div1 = kSCG_AsyncClkDivBy1,
                                           .range = kSCG_FircRange48M,
                                           .trimConfig = NULL};

/*
 * SYSPLL clock setting in HSRUN mode.
 * SYSPLL clock       : 96MHz
 * SYSPLLDIV1 output  : 96MHz
 * SYSPLLDIV2 output  : Disable
 * SYSPLLDIV3 output  : 48MHz
 */
const scg_spll_config_t g_scgSysPllConfig = {.enableMode = kSCG_SysPllEnable,
                                             .monitorMode = kSCG_SysPllMonitorDisable,
                                             .div1 = kSCG_AsyncClkDivBy1,
                                             .div2 = kSCG_AsyncClkDisable,
                                             .div3 = kSCG_AsyncClkDivBy2,
                                             .src = kSCG_SysPllSrcFirc,
                                             .prediv = 5U,
                                             .mult = 8U};

/*
 * System clock source and divider in RUN mode.
 * Clock source : FIRC
 * Core clock   : 48MHz
 * Slow clock   : 24MHz
 */
const scg_sys_clk_config_t g_sysClkConfigRun = {.divSlow = kSCG_SysClkDivBy2,
                                                .reserved1 = 0U,
                                                .reserved2 = 0U,
                                                .reserved3 = 0U,
                                                .divCore = kSCG_SysClkDivBy1,
                                                .reserved4 = 0U,
                                                .src = kSCG_SysClkSrcFirc,
                                                .reserved5 = 0U};

/*
 * System clock source and divider in HSRUN mode.
 * Clock source : SYSPLL
 * Core clock   : 96MHz
 * Slow clock   : 24MHz
 */
const scg_sys_clk_config_t g_sysClkConfigHsrun = {.divSlow = kSCG_SysClkDivBy4,
                                                  .reserved1 = 0U,
                                                  .reserved2 = 0U,
                                                  .reserved3 = 0U,
                                                  .divCore = kSCG_SysClkDivBy1,
                                                  .reserved4 = 0U,
                                                  .src = kSCG_SysClkSrcSysPll,
                                                  .reserved5 = 0U};

/*
 * System clock source and divider in VLPR mode.
 * Clock source : SIRC
 * Core clock   : 8MHz
 * Slow clock   : 1MHz
 */
const scg_sys_clk_config_t g_sysClkConfigVlpr = {.divSlow = kSCG_SysClkDivBy8,
                                                 .reserved1 = 0U,
                                                 .reserved2 = 0U,
                                                 .reserved3 = 0U,
                                                 .divCore = kSCG_SysClkDivBy1,
                                                 .reserved4 = 0U,
                                                 .src = kSCG_SysClkSrcSirc,
                                                 .reserved5 = 0U};

/*******************************************************************************
 * Code
 ******************************************************************************/
/*
 * Clock in VLPR mode:
 * SYSOSC  : Disable
 * SIRC    : Enable
 * FIRC    : Disable
 * SYSPLL  : Disable
 */
void BOARD_BootClockVLPR(void)
{
    scg_sys_clk_config_t curConfig;

    /*
     * Should first change to use FIRC as system clock source and re-configure
     * the SIRC, after that, change to use SIRC.
     */
    CLOCK_InitFirc(&g_scgFircConfig);

    CLOCK_SetRunModeSysClkConfig(&g_sysClkConfigRun);

    /* Wait for clock source switch finished. */
    do
    {
        CLOCK_GetCurSysClkConfig(&curConfig);
    } while (curConfig.src != g_sysClkConfigRun.src);

    CLOCK_InitSirc(&g_scgSircConfig);

    CLOCK_SetVlprModeSysClkConfig(&g_sysClkConfigVlpr);

    SystemCoreClock = CORE_CLOCK_MODE_VLPR;

    SMC_SetPowerModeProtection(SMC, kSMC_AllowPowerModeAll);
    SMC_SetPowerModeVlpr(SMC);
    while (SMC_GetPowerModeState(SMC) != kSMC_PowerStateVlpr)
    {
    }
}

/*
 * Clock in RUN mode:
 * SYSOSC  : Enable
 * SIRC    : Enable
 * FIRC    : Enable
 * SYSPLL  : Disable
 */
void BOARD_BootClockRUN(void)
{
    scg_sys_clk_config_t curConfig;
#if 0
    CLOCK_InitSysOsc(&g_scgSysOscConfig);
    CLOCK_SetXtal0Freq(BOARD_XTAL0_CLK_HZ);
#endif
    CLOCK_InitFirc(&g_scgFircConfig);

    CLOCK_SetRunModeSysClkConfig(&g_sysClkConfigRun);

    /* Wait for clock source switch finished. */
    do
    {
        CLOCK_GetCurSysClkConfig(&curConfig);
    } while (curConfig.src != g_sysClkConfigRun.src);

    CLOCK_InitSirc(&g_scgSircConfig);

    SystemCoreClock = CORE_CLOCK_MODE_RUN;
}

/*
 * Clock in HSRUN mode:
 * SYSOSC  : Enable
 * SIRC    : Enable
 * FIRC    : Enable
 * SYSPLL  : Enable
 */
void BOARD_BootClockHSRUN(void)
{
    scg_sys_clk_config_t curConfig;

    SMC_SetPowerModeProtection(SMC, kSMC_AllowPowerModeAll);
    SMC_SetPowerModeHsrun(SMC);
    while (SMC_GetPowerModeState(SMC) != kSMC_PowerStateHsrun)
    {
    }

    CLOCK_InitSysOsc(&g_scgSysOscConfig);
    CLOCK_SetXtal0Freq(BOARD_XTAL0_CLK_HZ);

    CLOCK_InitFirc(&g_scgFircConfig);

    CLOCK_InitSysPll(&g_scgSysPllConfig);

    CLOCK_SetHsrunModeSysClkConfig(&g_sysClkConfigHsrun);

    /* Wait for clock source switch finished. */
    do
    {
        CLOCK_GetCurSysClkConfig(&curConfig);
    } while (curConfig.src != g_sysClkConfigHsrun.src);

    CLOCK_InitSirc(&g_scgSircConfig);

    SystemCoreClock = CORE_CLOCK_MODE_HSRUN;
}
