/*
 * Copyright (c) 2015-2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "flexio_8080_drv.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
static FLEXIO_8080_Type FlexIO_8080_Inst;
FLEXIO_8080_Type * pFlexIO_8080_Inst = &FlexIO_8080_Inst;

static flexio_8080_pins_handler_t flexio_8080_pins_handler;
flexio_8080_pins_handler_t * pflexio_8080_pins_handler = &flexio_8080_pins_handler;

static flexio_8080_SglBeat_reg_config_t SglBeatWR_Reg_Config;
static flexio_8080_SglBeat_reg_config_t SglBeatRD_Reg_Config;
static flexio_8080_MulBeatWR_reg_config_t MulBeatWR_Reg_Config;
static flexio_8080_MulBeatRD_reg_config_t MulBeatRD_Reg_Config;

bool WR_DMATransferDone;
bool RD_DMATransferDone;

static uint32_t MulBeatCountRemain;
static uint8_t * MulBeatDataRemain;

static void (*pDMAIntCallback)(void);
static uint32_t MulBeatRd_minorLoopCount;
static uint32_t MulBeatRd_majorLoopCount;
static uint32_t MulBeatRd_destAddr;

bool DMAFillColorMode = false;
/*******************************************************************************
 * Functions
 ******************************************************************************/
/*!
 * @breief Initialize FlexIO module for emulating 8080 bus.
 */
void FLEXIO_8080_Init(void)
{
    /* Configure pin MUX */
    flexio_8080_pins_handler.Config_Data_Pin();
    flexio_8080_pins_handler.Config_CS_GPIO();
    flexio_8080_pins_handler.Config_RS_GPIO();
    flexio_8080_pins_handler.Config_WR_GPIO();
    flexio_8080_pins_handler.Config_RD_GPIO();

    FLEXIO_8080_SglBeatWR_GetRegConfig(&SglBeatWR_Reg_Config);
    FLEXIO_8080_SglBeatRD_GetRegConfig(&SglBeatRD_Reg_Config);
    FLEXIO_8080_MulBeatWR_GetRegConfig(&MulBeatWR_Reg_Config);
    FLEXIO_8080_MulBeatRD_GetRegConfig(&MulBeatRD_Reg_Config);

    /* Configure FlexIO module registers */
    FLEXIO0->CTRL = FLEXIO_CTRL_DBGE_MASK | FLEXIO_CTRL_FASTACC_MASK
                                          & ~FLEXIO_CTRL_FLEXEN_MASK;
}

/*!
 * @breief Get default FlexIO 8080 type configurations.
 * You can change the constant values in this function based on your preference.
 */
void FLEXIO_8080_Type_GetDefaultConfig(void)
{
    FlexIO_8080_Inst.BusWidth = 8U;
    FlexIO_8080_Inst.TmrDiv = 4U;

    FlexIO_8080_Inst.MulBeatWR_ShfQty = 8U;
    FlexIO_8080_Inst.MulBeatRD_ShfQty = 8U;

    FlexIO_8080_Inst.SglBeatWR_ShfIdx = 0U;         /* must be 0 or 4 */
    FlexIO_8080_Inst.SglBeatRD_ShfIdx = 7U;         /* must be 3 or 7 */
    FlexIO_8080_Inst.MulBeatWR_ShfIdx[0] = 0U;      /* must be 0 or 4 */
    FlexIO_8080_Inst.MulBeatWR_ShfIdx[1] = 7U;      /* equals WR_ShfIdx[0] + (ShfQty - 1) */
    FlexIO_8080_Inst.MulBeatRD_ShfIdx[0] = 0U;      /* equals RD_ShfIdx[1] - (ShfQty - 1) */
    FlexIO_8080_Inst.MulBeatRD_ShfIdx[1] = 7U;      /* must be 3 or 7 */
    FlexIO_8080_Inst.TmrIdx = 0U;

    FlexIO_8080_Inst.MulBeatWR_EDMA_Ch = 0U;
    FlexIO_8080_Inst.MulBeatRD_EDMA_Ch = 1U;
}

/*!
 * @breief Get register configurations for single-beat writing.
 */
static void FLEXIO_8080_SglBeatWR_GetRegConfig(flexio_8080_SglBeat_reg_config_t * reg_config)
{
    reg_config->ShiftConfig =
        FLEXIO_SHIFTCFG_INSRC(1U)                                               /* Shifter input */
      | FLEXIO_SHIFTCFG_SSTOP(0U)                                               /* Shifter stop bit disabled */
      | FLEXIO_SHIFTCFG_SSTART(0U)                                              /* Shifter start bit disabled and loading data on enabled */
      | FLEXIO_SHIFTCFG_PWIDTH(FlexIO_8080_Inst.BusWidth-1U);                   /* Bus width */

    reg_config->ShiftCtl =
        FLEXIO_SHIFTCTL_TIMSEL(FlexIO_8080_Inst.TmrIdx)                         /* Shifter's assigned timer index */
      | FLEXIO_SHIFTCTL_TIMPOL(0U)                                              /* Shift on posedge of shift clock */
      | FLEXIO_SHIFTCTL_PINCFG(3U)                                              /* Shifter's pin configured as output */
      | FLEXIO_SHIFTCTL_PINSEL(FlexIO_8080_Inst.Data0PinIdx)                    /* Shifter's pin start index */
      | FLEXIO_SHIFTCTL_PINPOL(0U)                                              /* Shifter's pin active high */
      | FLEXIO_SHIFTCTL_SMOD(2U);                                               /* Shifter mode as transmit */

    reg_config->TimCmp =
        (1U * 2U -1 << 8)                                                       /* TIMCMP[15:8] = number of beats x 2 – 1 */
      | (FlexIO_8080_Inst.TmrDiv/2U - 1U);                                      /* TIMCMP[7:0] = baud rate divider / 2 – 1 */

    reg_config->TimCfg =
        FLEXIO_TIMCFG_TIMOUT(0U)                                                /* Timer output logic one when enabled and not affected by reset */
      | FLEXIO_TIMCFG_TIMDEC(0U)                                                /* Timer decrement on FlexIO clock, shift clock equals timer output */
      | FLEXIO_TIMCFG_TIMRST(0U)                                                /* Timer never reset */
      | FLEXIO_TIMCFG_TIMDIS(2U)                                                /* Timer disabled on timer compare */
      | FLEXIO_TIMCFG_TIMENA(2U)                                                /* Timer enabled on trigger high */
      | FLEXIO_TIMCFG_TSTOP(0U)                                                 /* Timer stop bit disabled */
      | FLEXIO_TIMCFG_TSTART(0U);                                               /* Timer start bit disabled */

    reg_config->TimCtl =
        FLEXIO_TIMCTL_TRGSEL((((FlexIO_8080_Inst.SglBeatWR_ShfIdx) << 2) | 1U)) /* Timer trigger selected as shifter's status flag */
      | FLEXIO_TIMCTL_TRGPOL(1U)                                                /* Timer trigger polarity as active low */
      | FLEXIO_TIMCTL_TRGSRC(1U)                                                /* Timer trigger source as internal */
      | FLEXIO_TIMCTL_PINCFG(3U)                                                /* Timer' pin configured as output */
      | FLEXIO_TIMCTL_PINSEL(FlexIO_8080_Inst.WRPinIdx)                         /* Timer' pin index: WR pin */
      | FLEXIO_TIMCTL_PINPOL(1U)                                                /* Timer' pin active low */
      | FLEXIO_TIMCTL_TIMOD(1U);                                                /* Timer mode as dual 8-bit counters baud/bit */
}

/*!
 * @breief Get register configurations for single-beat reading.
 */
static void FLEXIO_8080_SglBeatRD_GetRegConfig(flexio_8080_SglBeat_reg_config_t * reg_config)
{
    reg_config->ShiftConfig =
        FLEXIO_SHIFTCFG_INSRC(0U)                                               /* Shifter input from pin */
      | FLEXIO_SHIFTCFG_SSTOP(0U)                                               /* Shifter stop bit disabled */
      | FLEXIO_SHIFTCFG_SSTART(0U)                                              /* Shifter start bit disabled and loading data on enabled */
      | FLEXIO_SHIFTCFG_PWIDTH(FlexIO_8080_Inst.BusWidth-1U);                   /* Bus width */

    reg_config->ShiftCtl =
        FLEXIO_SHIFTCTL_TIMSEL(FlexIO_8080_Inst.TmrIdx)                         /* Shifter's assigned timer index */
      | FLEXIO_SHIFTCTL_TIMPOL(1U)                                              /* Shift on negedge of shift clock */
      | FLEXIO_SHIFTCTL_PINCFG(0U)                                              /* Shifter's pin configured as output disabled */
      | FLEXIO_SHIFTCTL_PINSEL(FlexIO_8080_Inst.Data0PinIdx)                    /* Shifter's pin start index */
      | FLEXIO_SHIFTCTL_PINPOL(0U)                                              /* Shifter's pin active high */
      | FLEXIO_SHIFTCTL_SMOD(1U);                                               /* Shifter mode as receive */

    reg_config->TimCmp =
        (1U * 2U -1 << 8)                                                       /* TIMCMP[15:8] = number of beats x 2 – 1 */
      | (FlexIO_8080_Inst.TmrDiv/2U - 1U);                                      /* TIMCMP[7:0] = baud rate divider / 2 – 1 */

    reg_config->TimCfg =
        FLEXIO_TIMCFG_TIMOUT(0U)                                                /* Timer output logic one when enabled and not affected by reset */
      | FLEXIO_TIMCFG_TIMDEC(0U)                                                /* Timer decrement on FlexIO clock, shift clock equals timer output */
      | FLEXIO_TIMCFG_TIMRST(0U)                                                /* Timer never reset */
      | FLEXIO_TIMCFG_TIMDIS(2U)                                                /* Timer disabled on timer compare */
      | FLEXIO_TIMCFG_TIMENA(2U)                                                /* Timer enabled on trigger high */
      | FLEXIO_TIMCFG_TSTOP(2U)                                                 /* Timer stop bit enabled on timer disabled */
      | FLEXIO_TIMCFG_TSTART(0U);                                               /* Timer start bit disabled */

    reg_config->TimCtl =
        FLEXIO_TIMCTL_TRGSEL((((FlexIO_8080_Inst.SglBeatRD_ShfIdx) << 2) | 1U)) /* Timer trigger selected as shifter's status flag */
      | FLEXIO_TIMCTL_TRGPOL(1U)                                                /* Timer trigger polarity as active low */
      | FLEXIO_TIMCTL_TRGSRC(1U)                                                /* Timer trigger source as internal */
      | FLEXIO_TIMCTL_PINCFG(3U)                                                /* Timer' pin configured as output */
      | FLEXIO_TIMCTL_PINSEL(FlexIO_8080_Inst.RDPinIdx)                         /* Timer' pin index: RD pin*/
      | FLEXIO_TIMCTL_PINPOL(1U)                                                /* Timer' pin active low */
      | FLEXIO_TIMCTL_TIMOD(1U);                                                /* Timer mode as dual 8-bit counters baud/bit */
}

/*!
 * @breief Get register configurations for multi-beat writing.
 */
static void FLEXIO_8080_MulBeatWR_GetRegConfig(flexio_8080_MulBeatWR_reg_config_t * reg_config)
{
    uint8_t MulBeatWR_BeatQty = FlexIO_8080_Inst.MulBeatWR_ShfQty * 32U / FlexIO_8080_Inst.BusWidth;

    reg_config->ShiftConfig =
        FLEXIO_SHIFTCFG_INSRC(1U)                                               /* Shifter input from next shifter's output */
      | FLEXIO_SHIFTCFG_SSTOP(0U)                                               /* Shifter stop bit disabled */
      | FLEXIO_SHIFTCFG_SSTART(0U)                                              /* Shifter start bit disabled and loading data on enabled */
      | FLEXIO_SHIFTCFG_PWIDTH(FlexIO_8080_Inst.BusWidth-1U);                   /* Bus width */

    reg_config->ShiftCtl =
        FLEXIO_SHIFTCTL_TIMSEL(FlexIO_8080_Inst.TmrIdx)                         /* Shifter's assigned timer index */
      | FLEXIO_SHIFTCTL_TIMPOL(0U)                                              /* Shift on posedge of shift clock */
      | FLEXIO_SHIFTCTL_PINCFG(3U)                                              /* Shifter's pin configured as output */
      | FLEXIO_SHIFTCTL_PINSEL(FlexIO_8080_Inst.Data0PinIdx)                    /* Shifter's pin start index */
      | FLEXIO_SHIFTCTL_PINPOL(0U)                                              /* Shifter's pin active high */
      | FLEXIO_SHIFTCTL_SMOD(2U);                                               /* Shifter mode as transmit */

    reg_config->BufShiftCtl =
        FLEXIO_SHIFTCTL_TIMSEL(FlexIO_8080_Inst.TmrIdx)                         /* Shifter's assigned timer index */
      | FLEXIO_SHIFTCTL_TIMPOL(0U)                                              /* Shift on posedge of shift clock */
      | FLEXIO_SHIFTCTL_PINCFG(0U)                                              /* Shifter's pin configured as output disabled */
      | FLEXIO_SHIFTCTL_PINSEL(FlexIO_8080_Inst.Data0PinIdx)                    /* Shifter's pin start index */
      | FLEXIO_SHIFTCTL_PINPOL(0U)                                              /* Shifter's pin active high */
      | FLEXIO_SHIFTCTL_SMOD(2U);                                               /* Shifter mode transmit */

    reg_config->TimCmp =
        ((MulBeatWR_BeatQty * 2U - 1) << 8)                                     /* TIMCMP[15:8] = number of beats x 2 – 1 */
      | (FlexIO_8080_Inst.TmrDiv/2U - 1U);                                      /* TIMCMP[7:0] = baud rate divider / 2 – 1 */

    reg_config->TimCfg =
        FLEXIO_TIMCFG_TIMOUT(0U)                                                /* Timer output logic one when enabled and not affected by reset */
      | FLEXIO_TIMCFG_TIMDEC(0U)                                                /* Timer decrement on FlexIO clock, shift clock equals timer output */
      | FLEXIO_TIMCFG_TIMRST(0U)                                                /* Timer never reset */
      | FLEXIO_TIMCFG_TIMDIS(2U)                                                /* Timer disabled on timer compare */
      | FLEXIO_TIMCFG_TIMENA(2U)                                                /* Timer enabled on trigger high */
      | FLEXIO_TIMCFG_TSTOP(0U)                                                 /* Timer stop bit disabled */
      | FLEXIO_TIMCFG_TSTART(0U);                                               /* Timer start bit disabled */

    reg_config->TimCtl =
        FLEXIO_TIMCTL_TRGSEL((((FlexIO_8080_Inst.MulBeatWR_ShfIdx[1]) << 2) | 1U))/* Timer trigger selected as shifter's status flag */
      | FLEXIO_TIMCTL_TRGPOL(1U)                                                /* Timer trigger polarity as active low */
      | FLEXIO_TIMCTL_TRGSRC(1U)                                                /* Timer trigger source as internal */
      | FLEXIO_TIMCTL_PINCFG(3U)                                                /* Timer' pin configured as output */
      | FLEXIO_TIMCTL_PINSEL(FlexIO_8080_Inst.WRPinIdx)                         /* Timer' pin index: WR pin */
      | FLEXIO_TIMCTL_PINPOL(1U)                                                /* Timer' pin active low */
      | FLEXIO_TIMCTL_TIMOD(1U);                                                /* Timer mode as dual 8-bit counters baud/bit */
}

/*!
 * @breief Get register configurations for multi-beat reading.
 */
static void FLEXIO_8080_MulBeatRD_GetRegConfig(flexio_8080_MulBeatRD_reg_config_t * reg_config)
{
    uint8_t MulBeatRD_BeatQty = FlexIO_8080_Inst.MulBeatWR_ShfQty * 32U / FlexIO_8080_Inst.BusWidth;

    reg_config-> BufShiftConfig =
        FLEXIO_SHIFTCFG_INSRC(1U)                                               /* Shifter input from next shifter's output  */
      | FLEXIO_SHIFTCFG_SSTOP(0U)                                               /* Shifter stop bit disabled */
      | FLEXIO_SHIFTCFG_SSTART(0U)                                              /* Shifter start bit disabled and loading data on enabled */
      | FLEXIO_SHIFTCFG_PWIDTH(FlexIO_8080_Inst.BusWidth-1U);                   /* Bus width */

    reg_config->ShiftConfig =
        FLEXIO_SHIFTCFG_INSRC(0U)                                               /* Shifter input from pin */
      | FLEXIO_SHIFTCFG_SSTOP(0U)                                               /* Shifter stop bit disabled */
      | FLEXIO_SHIFTCFG_SSTART(0U)                                              /* Shifter start bit disabled and loading data on enabled */
      | FLEXIO_SHIFTCFG_PWIDTH(FlexIO_8080_Inst.BusWidth-1U);                   /* Bus width */

    reg_config->ShiftCtl =
        FLEXIO_SHIFTCTL_TIMSEL(FlexIO_8080_Inst.TmrIdx)                         /* Shifter's assigned timer index */
      | FLEXIO_SHIFTCTL_TIMPOL(1U)                                              /* Shift on negedge of shift clock */
      | FLEXIO_SHIFTCTL_PINCFG(0U)                                              /* Shifter's pin configured as output disabled */
      | FLEXIO_SHIFTCTL_PINSEL(FlexIO_8080_Inst.Data0PinIdx)                    /* Shifter's pin start index */
      | FLEXIO_SHIFTCTL_PINPOL(0U)                                              /* Shifter's pin active high */
      | FLEXIO_SHIFTCTL_SMOD(1U);                                               /* Shifter mode as receive */

    reg_config->TimCmp =
        ((MulBeatRD_BeatQty * 2U - 1) << 8)                                     /* TIMCMP[15:8] = number of beats x 2 – 1 */
      | (FlexIO_8080_Inst.TmrDiv/2U - 1U);                                      /* TIMCMP[7:0] = baud rate divider / 2 – 1  */

    reg_config->TimCfg =
        FLEXIO_TIMCFG_TIMOUT(0U)                                                /* Timer output logic one when enabled and not affected by reset */
      | FLEXIO_TIMCFG_TIMDEC(0U)                                                /* Timer decrement on FlexIO clock, shift clock equals timer output */
      | FLEXIO_TIMCFG_TIMRST(0U)                                                /* Timer never reset  */
      | FLEXIO_TIMCFG_TIMDIS(2U)                                                /* Timer disabled on timer compare */
      | FLEXIO_TIMCFG_TIMENA(2U)                                                /* Timer enabled on trigger high */
      | FLEXIO_TIMCFG_TSTOP(2U)                                                 /* Timer stop bit enabled on timer compare */
      | FLEXIO_TIMCFG_TSTART(0U);                                               /* Timer start bit disabled */

    reg_config->TimCtl =
        FLEXIO_TIMCTL_TRGSEL((((FlexIO_8080_Inst.MulBeatRD_ShfIdx[0]) << 2) | 1U))/* Timer trigger selected as shifter's status flag */
      | FLEXIO_TIMCTL_TRGPOL(1U)                                                /* Timer trigger polarity as active low */
      | FLEXIO_TIMCTL_TRGSRC(1U)                                                /* Timer trigger source as internal */
      | FLEXIO_TIMCTL_PINCFG(3U)                                                /* Timer' pin configured as output */
      | FLEXIO_TIMCTL_PINSEL(FlexIO_8080_Inst.RDPinIdx)                         /* Timer' pin index: RD pin*/
      | FLEXIO_TIMCTL_PINPOL(1U)                                                /* Timer' pin active low */
      | FLEXIO_TIMCTL_TIMOD(1U);                                                /* Timer mode as dual 8-bit counters baud/bit */
}

/*!
 * @brief Configure FlexIO registers for 8080 single-beat writing.
 */
static void FLEXIO_8080_ConfigSglBeatWR(flexio_8080_SglBeat_reg_config_t const * reg_config)
{
    /* Disable and reset FlexIO */
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    FLEXIO0->CTRL |= FLEXIO_CTRL_SWRST_MASK;
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_SWRST_MASK;

    /* Configure the shifters */
    FLEXIO0->SHIFTCFG[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = reg_config->ShiftConfig;
    FLEXIO0->SHIFTCTL[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = reg_config->ShiftCtl;

    /* Configure the timer for shift clock */
    FLEXIO0->TIMCMP[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCmp;
    FLEXIO0->TIMCFG[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCfg;
    FLEXIO0->TIMCTL[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCtl;

    /* Enable FlexIO */
    FLEXIO0->CTRL |= FLEXIO_CTRL_FLEXEN_MASK;
}

/*!
 * @brief Configure FlexIO registers for 8080 single-beat reading.
 */
static void FLEXIO_8080_ConfigSglBeatRD(flexio_8080_SglBeat_reg_config_t const * reg_config)
{
    /* Disable and reset FlexIO */
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    FLEXIO0->CTRL |= FLEXIO_CTRL_SWRST_MASK;
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_SWRST_MASK;

    /* Configure the shifters */
    FLEXIO0->SHIFTCFG[FlexIO_8080_Inst.SglBeatRD_ShfIdx] = reg_config->ShiftConfig;
    FLEXIO0->SHIFTCTL[FlexIO_8080_Inst.SglBeatRD_ShfIdx] = reg_config->ShiftCtl;

    /* Configure the timer for shift clock */
    FLEXIO0->TIMCMP[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCmp;
    FLEXIO0->TIMCFG[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCfg;
    FLEXIO0->TIMCTL[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCtl;

    /* Enable FLexIO */
    FLEXIO0->CTRL |= FLEXIO_CTRL_FLEXEN_MASK;
}

/*!
 * @brief Configure FlexIO registers for 8080 multi-beat writing.
 */
static void FLEXIO_8080_ConfigMulBeatWR(flexio_8080_MulBeatWR_reg_config_t const * reg_config)
{
    uint32_t i;

    /* Disable and reset FlexIO */
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    FLEXIO0->CTRL |= FLEXIO_CTRL_SWRST_MASK;
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_SWRST_MASK;

    /* Configure the shifters */
    for(i=FlexIO_8080_Inst.MulBeatWR_ShfIdx[0]; i<=FlexIO_8080_Inst.MulBeatWR_ShfIdx[1]; i++)
    {
        FLEXIO0->SHIFTCFG[i] = reg_config->ShiftConfig;
    }

    FLEXIO0->SHIFTCTL[FlexIO_8080_Inst.MulBeatWR_ShfIdx[0]] = reg_config->ShiftCtl;

    for(i=FlexIO_8080_Inst.MulBeatWR_ShfIdx[0]+1U; i<=FlexIO_8080_Inst.MulBeatWR_ShfIdx[1]; i++)
    {
        FLEXIO0->SHIFTCTL[i] = reg_config->BufShiftCtl;
    }

    /* Configure the timer for shift clock */
    FLEXIO0->TIMCMP[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCmp;
    FLEXIO0->TIMCFG[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCfg;
    FLEXIO0->TIMCTL[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCtl;

    /* Enable FLexIO */
    FLEXIO0->CTRL |= FLEXIO_CTRL_FLEXEN_MASK;
}

/*!
 * @brief Configure FlexIO registers for 8080 multi-beat reading.
 */
static void FLEXIO_8080_ConfigMulBeatRD(flexio_8080_MulBeatRD_reg_config_t const * reg_config)
{
    uint32_t i;

    /* Disable and reset FlexIO */
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    FLEXIO0->CTRL |= FLEXIO_CTRL_SWRST_MASK;
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_SWRST_MASK;

    /* Configure the shifters */
    for(i=FlexIO_8080_Inst.MulBeatRD_ShfIdx[0]; i<=FlexIO_8080_Inst.MulBeatRD_ShfIdx[1]-1U; i++)
    {
        FLEXIO0->SHIFTCFG[i] = reg_config->BufShiftConfig;
    }

    FLEXIO0->SHIFTCFG[FlexIO_8080_Inst.MulBeatRD_ShfIdx[1]] = reg_config->ShiftConfig;

    for(i=FlexIO_8080_Inst.MulBeatRD_ShfIdx[0]; i<=FlexIO_8080_Inst.MulBeatRD_ShfIdx[1]; i++)
    {
        FLEXIO0->SHIFTCTL[i] = reg_config->ShiftCtl;
    }

    /* Configure the timer for shift clock */
    FLEXIO0->TIMCMP[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCmp;
    FLEXIO0->TIMCFG[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCfg;
    FLEXIO0->TIMCTL[FlexIO_8080_Inst.TmrIdx] = reg_config->TimCtl;

    /* Enable FLexIO */
    FLEXIO0->CTRL |= FLEXIO_CTRL_FLEXEN_MASK;
}

/*!
 * @brief Get DMA channel TCD configuration for multi-beat writing.
 * Then you may change some of the configurations based specific requirement.
 * After that, use tcd_config to initialize a DMA channel.
 */
static void FLEXIO_8080_MulBeatWR_DMATCD_GetConfig(edma_tcd_config_t * tcd_config,
                                                edma_user_config_t * user_config)
{
    uint32_t TransferSize;
    uint32_t MulBeatWR_DMAMinLoop = FlexIO_8080_Inst.MulBeatWR_ShfQty * 4U;      /* in bytes */

    switch(MulBeatWR_DMAMinLoop)
    {
        case 1U:
            TransferSize = 0U;
            break;
        case 2U:
            TransferSize = 1U;
            break;
        case 4U:
            TransferSize = 2U;
            break;
        case 16U:
            TransferSize = 4U;
            break;
        case 32U:
            TransferSize = 5U;
            break;
        default:                /* error */
            break;
    }

    tcd_config->srcAddr = (uint32_t)(user_config->Buffer);                      /* srcAddr */
    tcd_config->destAddr = (uint32_t)&FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.MulBeatWR_ShfIdx[0]];/* destAddr */
    tcd_config->srcOffset = MulBeatWR_DMAMinLoop;                               /* srcOffset */
    tcd_config->destOffset = 0U;                                                /* destOffset */
    tcd_config->srcLastAddrAdjust = 0;                                          /* srcLastAddrAdjust */
    tcd_config->destLastAddrAdjust = 0U;                                        /* destLastAddrAdjust */
    if(DMAFillColorMode)
    {
        tcd_config->srcModulo = 2U;                                             /* srcModulo */
    }
    else
    {
        tcd_config->srcModulo = 0U;                                             /* srcModulo */
    }
    tcd_config->destModulo = 0U;                                                /* destModulo */
    tcd_config->srcTransferSize = TransferSize;                                 /* srcTransferSize */
    tcd_config->destTransferSize = TransferSize;                                /* destTransferSize */
    tcd_config->minorLoopCount = MulBeatWR_DMAMinLoop;                          /* minorLoopCount */
    tcd_config->majorLoopCount = user_config->TotalSize/MulBeatWR_DMAMinLoop;   /* majorLoopCount */
}

/*!
 * @brief Get DMA channel TCD configuration for multi-beat reading.
 * Then you may change some of the configurations based specific requirement.
 * After that, use tcd_config to initialize a DMA channel.
 */
static void FLEXIO_8080_MulBeatRD_DMATCD_GetConfig(edma_tcd_config_t * tcd_config,
                                                edma_user_config_t * user_config)
{
    uint32_t TransferSize;
    uint32_t MulBeatRD_DMAMinLoop = FlexIO_8080_Inst.MulBeatRD_ShfQty * 4U;      /* in bytes */

    switch(MulBeatRD_DMAMinLoop)
    {
        case 1U:
            TransferSize = 0U;
            break;
        case 2U:
            TransferSize = 1U;
            break;
        case 4U:
            TransferSize = 2U;
            break;
        case 16U:
            TransferSize = 4U;
            break;
        case 32U:
            TransferSize = 5U;
            break;
        default:                /* error */
            break;
    }

    tcd_config->srcAddr = (uint32_t)&FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.MulBeatRD_ShfIdx[0]];/* srcAddr */
    tcd_config->destAddr = (uint32_t)(user_config->Buffer);                     /* destAddr */
    tcd_config->srcOffset = 0U;                                                 /* srcOffset */
    tcd_config->destOffset = MulBeatRD_DMAMinLoop;                              /* destOffset */
    tcd_config->srcLastAddrAdjust = 0U;                                         /* srcLastAddrAdjust */
    tcd_config->destLastAddrAdjust = 0;                                         /* destLastAddrAdjust */
    tcd_config->srcModulo = 0U;                                                 /* srcModulo */
    tcd_config->destModulo = 0U;                                                /* destModulo */
    tcd_config->srcTransferSize = TransferSize;                                 /* srcTransferSize */
    tcd_config->destTransferSize = TransferSize;                                /* destTransferSize */
    tcd_config->minorLoopCount = MulBeatRD_DMAMinLoop;                          /* minorLoopCount */
    tcd_config->majorLoopCount = user_config->TotalSize/MulBeatRD_DMAMinLoop - 1U;/* majorLoopCount */
}

/*!
 * @brief Configure DMA channel TCD (Transfer Control Descriptor) for FlexIO
 * 8080 multi-beat operations.
 */
static void FLEXIO_8080_MulBeat_ConfigDMATCD(uint32_t const channel,
                                edma_tcd_config_t const * tcd_config)
{
    /* Configure TCD */
    DMA0->TCD[channel].SADDR = tcd_config->srcAddr;
    DMA0->TCD[channel].SOFF = tcd_config->srcOffset;
    DMA0->TCD[channel].SLAST = tcd_config->srcLastAddrAdjust;

    DMA0->TCD[channel].DADDR = tcd_config->destAddr;
    DMA0->TCD[channel].DOFF = tcd_config->destOffset;
    DMA0->TCD[channel].DLAST_SGA = tcd_config->destLastAddrAdjust;

    if((!(DMA0->CR & DMA_CR_EMLM_MASK)))
    {
        DMA0->TCD[channel].NBYTES_MLNO = tcd_config->minorLoopCount;
    }
    else
    {
        if((!(DMA0->TCD[channel].NBYTES_MLOFFNO) & DMA_NBYTES_MLOFFNO_SMLOE_MASK) && \
           (!(DMA0->TCD[channel].NBYTES_MLOFFNO) & DMA_NBYTES_MLOFFNO_DMLOE_MASK))
        {
           DMA0->TCD[channel].NBYTES_MLOFFNO = tcd_config->minorLoopCount;
        }
        else
        {
           DMA0->TCD[channel].NBYTES_MLOFFYES = tcd_config->minorLoopCount;
        }
    }

    if(DMA0->TCD[channel].BITER_ELINKYES & DMA_BITER_ELINKYES_ELINK_MASK)
    {
        DMA0->TCD[channel].BITER_ELINKYES = DMA0->TCD[channel].BITER_ELINKYES
          &(~DMA_BITER_ELINKYES_BITER_MASK) | tcd_config->majorLoopCount;
        DMA0->TCD[channel].CITER_ELINKYES = DMA0->TCD[channel].CITER_ELINKYES
          &(~DMA_CITER_ELINKYES_CITER_MASK) | tcd_config->majorLoopCount;
    }
    else
    {
        DMA0->TCD[channel].BITER_ELINKNO = DMA0->TCD[channel].BITER_ELINKNO
          &(~DMA_BITER_ELINKNO_BITER_MASK) | tcd_config->majorLoopCount;
        DMA0->TCD[channel].CITER_ELINKNO = DMA0->TCD[channel].CITER_ELINKNO
          &(~DMA_CITER_ELINKNO_CITER_MASK) | tcd_config->majorLoopCount;
    }

    DMA0->TCD[channel].ATTR = DMA_ATTR_SMOD(tcd_config->srcModulo)
                            | DMA_ATTR_SSIZE(tcd_config->srcTransferSize)
                            | DMA_ATTR_DMOD(tcd_config->destModulo)
                            | DMA_ATTR_DSIZE(tcd_config->destTransferSize);
}

/*!
 * @brief Initialize DMA channel for multi-beat writing.
 */
static void FLEXIO_8080_MulBeatWR_DMA_Init(edma_tcd_config_t const *tcd_config)
{
    uint8_t MulBeatWR_DMAMUX_Src;

#if defined(KL28Z7_SERIES)
    MulBeatWR_DMAMUX_Src = FlexIO_8080_Inst.MulBeatWR_ShfIdx[1] + 1U;
#else
    #error "No valid MCU part is defined."
#endif

    FLEXIO_8080_MulBeat_ConfigDMATCD(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch, tcd_config);

    DMA0->CDNE |= DMA_CDNE_CDNE(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);
    DMA0->CINT |= DMA_CINT_CINT(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);

    /* Configure the DMAMUX for edma channel */
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch] &= ~DMAMUX_CHCFG_ENBL_MASK;
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch] = DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch]
        & (~DMAMUX_CHCFG_SOURCE_MASK) | DMAMUX_CHCFG_SOURCE(MulBeatWR_DMAMUX_Src);
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch] |= DMAMUX_CHCFG_ENBL_MASK;

    DMA0->TCD[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch].CSR |= DMA_CSR_DREQ_MASK;

    DMA0->TCD[FlexIO_8080_Inst.MulBeatWR_EDMA_Ch].CSR |= DMA_CSR_INTMAJOR_MASK;
    NVIC_EnableIRQ(MULBEAT_WR_DMA_CHN_IRQ);
}

/*!
 * @brief Initialize DMA channel for multi-beat reading.
*/
static void FLEXIO_8080_MulBeatRD_DMA_Init(edma_tcd_config_t const *tcd_config)
{
    uint8_t MulBeatRD_DMAMUX_Src;

#if defined(KL28Z7_SERIES)
    MulBeatRD_DMAMUX_Src = FlexIO_8080_Inst.MulBeatRD_ShfIdx[0] + 1U;
#else
    #error "No valid MCU part is defined."
#endif

    MulBeatRd_destAddr = tcd_config->destAddr;
    MulBeatRd_minorLoopCount = tcd_config->minorLoopCount;
    MulBeatRd_majorLoopCount = tcd_config->majorLoopCount;

    FLEXIO_8080_MulBeat_ConfigDMATCD(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch, tcd_config);

    DMA0->CDNE |= DMA_CDNE_CDNE(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);
    DMA0->CINT |= DMA_CINT_CINT(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);

    /* Configure the DMAMUX for edma channel */
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch] &= ~DMAMUX_CHCFG_ENBL_MASK;
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch] = DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch]
        & (~DMAMUX_CHCFG_SOURCE_MASK) | DMAMUX_CHCFG_SOURCE(MulBeatRD_DMAMUX_Src);
    DMAMUX0->CHCFG[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch] |= DMAMUX_CHCFG_ENBL_MASK;

    DMA0->TCD[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch].CSR |= DMA_CSR_DREQ_MASK;

    DMA0->TCD[FlexIO_8080_Inst.MulBeatRD_EDMA_Ch].CSR |= DMA_CSR_INTMAJOR_MASK;
    NVIC_EnableIRQ(MULBEAT_RD_DMA_CHN_IRQ);
}

/*!
* @brief Enable DMA request for multi-beat writing.
*/
static void FLEXIO_8080_MulBeatWR_DMA_Enable(bool enable)
{
    if(enable)
    {
        DMA0->SERQ |= DMA_SERQ_SERQ(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);
    }
    else
    {
        DMA0->CERQ |= DMA_CERQ_CERQ(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);
    }
}

/*!
* @brief Enable DMA request for multi-beat reading.
*/
static void FLEXIO_8080_MulBeatRD_DMA_Enable(bool enable)
{
    if(enable)
    {
        if(MulBeatRd_majorLoopCount)
        {
            DMA0->SERQ |= DMA_SERQ_SERQ(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);
        }
        else
        {
            if(pDMAIntCallback != NULL)
            {
                pDMAIntCallback();
                pDMAIntCallback = NULL;
            }
        }
    }
    else
    {
        DMA0->CERQ |= DMA_CERQ_CERQ(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);
    }
}

/*!
* @brief DMA channel 0 interrupt handler for multi-beat writing.
*/
void MULBEAT_WR_DMA_IRQ_HANDLER(void)
{
    if(false != (DMA0->INT & (1U << FlexIO_8080_Inst.MulBeatWR_EDMA_Ch)))
    {
        DMA0->CERQ |= DMA_CERQ_CERQ(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);
        DMA0->CDNE |= DMA_CDNE_CDNE(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);
        DMA0->CINT |= DMA_CINT_CINT(FlexIO_8080_Inst.MulBeatWR_EDMA_Ch);

        if(pDMAIntCallback != NULL)
        {
            pDMAIntCallback();
            pDMAIntCallback = NULL;
        }
    }
}

/*!
* @brief DMA channel 1 interrupt handler for multi-beat reading.
*/
void MULBEAT_RD_DMA_IRQ_HANDLER(void)
{
    if(false != (DMA0->INT & (1U << FlexIO_8080_Inst.MulBeatRD_EDMA_Ch)))
    {
        DMA0->CERQ |= DMA_CERQ_CERQ(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);
        DMA0->CDNE |= DMA_CDNE_CDNE(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);
        DMA0->CINT |= DMA_CINT_CINT(FlexIO_8080_Inst.MulBeatRD_EDMA_Ch);

        if(pDMAIntCallback != NULL)
        {
            pDMAIntCallback();
            pDMAIntCallback = NULL;
        }
    }
}

/*!
 * @brief Write command with single parameter in single-beat writing mode using polling method.
 */
void FLEXIO_8080_SglBeatWR_1Prm(uint32_t const cmdIdx, uint8_t const value)
{
    /* Configure RD pin as GPIO function, WR pin as FlexIO function */
    flexio_8080_pins_handler.Config_RD_GPIO();
    flexio_8080_pins_handler.Config_WR_FlexIO();

    /* Configure FlexIO with 1-beat write configuration */
    FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

    /* Assert CS, RS pins */
    flexio_8080_pins_handler.Set_CS_Pin(false);
    flexio_8080_pins_handler.Set_RS_Pin(false);

    /* Write command index */
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

    /*Wait for transfer to be completed */
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert RS pin */
    flexio_8080_pins_handler.Set_RS_Pin(true);

    /* Use polling method for data transfer */
    while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
    {
    }
    FLEXIO0->TIMSTAT |= (1U << FlexIO_8080_Inst.TmrIdx);
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = value;

    /*Wait for transfer to be completed */
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);
}

/*!
 * @brief Write command with multi parameters in single-beat writing mode using polling method.
 */
void FLEXIO_8080_SglBeatWR_nPrm(uint32_t const cmdIdx, uint8_t const * buffer,
                                                       uint32_t const length)
{
    /* Configure RD pin as GPIO function, WR pin as FlexIO function */
    flexio_8080_pins_handler.Config_RD_GPIO();
    flexio_8080_pins_handler.Config_WR_FlexIO();

    /* Configure FlexIO with 1-beat write configuration */
    FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

    /* Assert CS, RS pins */
    flexio_8080_pins_handler.Set_CS_Pin(false);
    flexio_8080_pins_handler.Set_RS_Pin(false);

    /* Write command index */
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

    /*Wait for transfer to be completed */
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert RS pin */
    flexio_8080_pins_handler.Set_RS_Pin(true);

    if(length)
    {
        /* Use polling method for data transfer */
        for(uint32_t i=0; i<length-1U; i++)
        {
            while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
            {
            }
            FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = *buffer++;
        }

        /* Write the last byte */
        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
        {
        }
        FLEXIO0->TIMSTAT |= (1U << FlexIO_8080_Inst.TmrIdx);
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = *buffer++;

        /*Wait for transfer to be completed */
        while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
        {
        }
    }

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);
}

/*!
 * @brief Write command with multi repeated parameters in single-beat writing mode using polling method.
 * It is used to fill LCD panel with solid color.
 */
void FLEXIO_8080_SglBeatWR_nSamePrm(uint32_t const cmdIdx, uint16_t const value,
                                                           uint32_t const length)
{
    /* Configure RD pin as GPIO function, WR pin as FlexIO function */
    flexio_8080_pins_handler.Config_RD_GPIO();
    flexio_8080_pins_handler.Config_WR_FlexIO();

    /* Configure FlexIO with 1-beat write configuration */
    FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

    /* Assert CS, RS pins */
    flexio_8080_pins_handler.Set_CS_Pin(false);
    flexio_8080_pins_handler.Set_RS_Pin(false);

    /* Write command index */
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

    /*Wait for transfer to be completed */
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert RS pin */
    flexio_8080_pins_handler.Set_RS_Pin(true);

    if(length)
    {
        /* Use polling method for data transfers */
        for(uint32_t i=0; i<length-1U; i++)
        {
            while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
            {
            }
            FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = value >> 8U;

            while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
            {
            }
            FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = value & 0xFF;
        }

        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
        {
        }
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = value >> 8U;

        /* Write the last byte */
        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
        {
        }
        FLEXIO0->TIMSTAT |= (1U << FlexIO_8080_Inst.TmrIdx);
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = value & 0xFF;

        /*Wait for transfer to be completed */
        while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
        {
        }
    }

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);
}

/*!
 * @brief Read command with single parameter in single-beat reading mode using polling method.
 */
uint8_t FLEXIO_8080_SglBeatRD_1Prm(uint32_t const cmdIdx)
{
    uint8_t buffer;

    /* Configure WR pin FlexIO function, RD pin as GPIO function */
    flexio_8080_pins_handler.Config_WR_FlexIO();
    flexio_8080_pins_handler.Config_RD_GPIO();

    /* Configure FlexIO with 1-beat write configuration */
    FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

    /* Assert CS, RS pins */
    flexio_8080_pins_handler.Set_CS_Pin(false);
    flexio_8080_pins_handler.Set_RS_Pin(false);

    /* Write command index */
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

    /*Wait for transfer to be completed*/
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert RS pin */
    flexio_8080_pins_handler.Set_RS_Pin(true);

    /* Configure WR pin as GPIO function, RD pin FlexIO function */
    flexio_8080_pins_handler.Config_WR_GPIO();
    flexio_8080_pins_handler.Config_RD_FlexIO();

    /* Configure FlexIO with 1-beat read configuration */
    FLEXIO_8080_ConfigSglBeatRD(&SglBeatRD_Reg_Config);

    /* Use polling method for data transfer */
    while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatRD_ShfIdx)))
    {
    }
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    buffer = FLEXIO0->SHIFTBUFBYS[FlexIO_8080_Inst.SglBeatRD_ShfIdx];           /* SHIFTBUFBYS for 8-bit reading */
    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);

    return buffer;
}

/*!
 * @brief Read command with multi parameters in single-beat reading mode using polling method.
 */
void FLEXIO_8080_SglBeatRD_nPrm(uint32_t const cmdIdx, uint8_t * buffer,
                                                uint32_t const length)
{
    /* Configure WR pin FlexIO function, RD pin as GPIO function */
    flexio_8080_pins_handler.Config_WR_FlexIO();
    flexio_8080_pins_handler.Config_RD_GPIO();

    /* Configure FlexIO with 1-beat write configuration */
    FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

    /* Assert CS, RS pins */
    flexio_8080_pins_handler.Set_CS_Pin(false);
    flexio_8080_pins_handler.Set_RS_Pin(false);

    /* Write command index */
    FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

    /*Wait for transfer to be completed*/
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }

    /* De-assert RS pin */
    flexio_8080_pins_handler.Set_RS_Pin(true);

    /* Configure WR pin as GPIO function, RD pin FlexIO function */
    flexio_8080_pins_handler.Config_WR_GPIO();
    flexio_8080_pins_handler.Config_RD_FlexIO();

    /* Configure FlexIO with 1-beat read configuration */
    FLEXIO_8080_ConfigSglBeatRD(&SglBeatRD_Reg_Config);

    /* Use polling method for data transfer */
    for(uint32_t i=0; i<length-1U; i++)
    {
        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatRD_ShfIdx)))
        {
        }
        *buffer++ = FLEXIO0->SHIFTBUFBYS[FlexIO_8080_Inst.SglBeatRD_ShfIdx];    /* SHIFTBUFBYS for 8-bit reading */
    }

    /* Read the last byte */
    while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatRD_ShfIdx)))
    {
    }
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
    *buffer = FLEXIO0->SHIFTBUFBYS[FlexIO_8080_Inst.SglBeatRD_ShfIdx];          /* SHIFTBUFBYS for 8-bit reading */

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);
}

/*!
 * @brief Write command with multi parameters in multi-beat writing mode using DMA method.
 * The total transfer size is configured in the DMA configuration function.
 */
void FLEXIO_8080_MulBeatWR_nPrm(uint32_t const cmdIdx, uint8_t const * buffer,
                                                        uint32_t const length)
{
    uint32_t BeatsPerMinLoop = FlexIO_8080_Inst.MulBeatWR_ShfQty * 4U;
    edma_tcd_config_t edma_config;
    edma_user_config_t edma_user_config;

    if(length < BeatsPerMinLoop)
    {
        FLEXIO_8080_SglBeatWR_nPrm(cmdIdx, buffer, length);
    }
    else
    {
        MulBeatCountRemain = length % BeatsPerMinLoop;
        MulBeatDataRemain = (uint8_t *)buffer + (length - MulBeatCountRemain);

        edma_user_config.Buffer = (void *)buffer;
        edma_user_config.TotalSize = length - MulBeatCountRemain;               /* in bytes */

        FLEXIO_8080_MulBeatWR_DMATCD_GetConfig(&edma_config, &edma_user_config);
        FLEXIO_8080_MulBeatWR_DMA_Init(&edma_config);

        /* Configure RD pin as GPIO function, WR pin as FlexIO function */
        flexio_8080_pins_handler.Config_RD_GPIO();
        flexio_8080_pins_handler.Config_WR_FlexIO();

        /* Configure FlexIO with 1-beat write configuration */
        FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

        /* Assert CS, RS pins */
        flexio_8080_pins_handler.Set_CS_Pin(false);
        flexio_8080_pins_handler.Set_RS_Pin(false);

        /* Write command index */
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

        /*Wait for transfer to be completed*/
        while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
        {
        }

        /* De-assert RS pin */
        flexio_8080_pins_handler.Set_RS_Pin(true);

        /* Configure FlexIO with multi-beat write configuration */
        FLEXIO_8080_ConfigMulBeatWR(&MulBeatWR_Reg_Config);
        FLEXIO0->SHIFTSDEN |= 1U << FlexIO_8080_Inst.MulBeatWR_ShfIdx[1];

        /* Start data transfer by using DMA */
        WR_DMATransferDone = false;
        pDMAIntCallback = FLEXIO_8080_MulBeatWR_Callback;
        FLEXIO_8080_MulBeatWR_DMA_Enable(true);
    }
}

/*!
 * @brief callback function for FLEXIO_8080_MulBeatWR_nPrm()
 * It is called in the DMA ISR, used to complete the transfer.
 */
static void FLEXIO_8080_MulBeatWR_Callback(void)
{
    while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.MulBeatWR_ShfIdx[1])))
    {
    }

    /* Wait the last multi-beat transfer to be completed. Clear the timer flag
    before the completing of the last beat. The last beat may has been completed
    at this point, then code would be dead in the while() below. So mask the
    while() statement and use the software delay .*/
    FLEXIO0->TIMSTAT |= (1U << FlexIO_8080_Inst.TmrIdx);

#if 0
    /* Wait timer flag to be set to ensure the completing of the last beat. */
    while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
    {
    }
#else
    /* Wait some time to ensure the completing of the last beat. */
    for(uint32_t i=0; i<50U; i++)
    {
    }
#endif

    if(MulBeatCountRemain)
    {
        /* Configure FlexIO with 1-beat write configuration */
        FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

        /* Use polling method for data transfer */
        for(uint32_t i=0; i<MulBeatCountRemain-1U; i++)
        {
            while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
            {
            }
            FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = *MulBeatDataRemain++;
        }

        /* Write the last byte */
        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatWR_ShfIdx)))
        {
        }
        FLEXIO0->TIMSTAT |= (1U << FlexIO_8080_Inst.TmrIdx);
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = *MulBeatDataRemain++;

        /*Wait for transfer to be completed */
        while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
        {
        }
    }

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);

    WR_DMATransferDone = true;
}

/*!
 * @brief Read command with multi parameters in multi-beat reading mode using DMA method.
 * The total transfer size is configured in the DMA configuration function.
 */
void FLEXIO_8080_MulBeatRD_nPrm(uint32_t const cmdIdx, uint8_t * buffer,
                                                  uint32_t const length)
{
    uint32_t BeatsPerMinLoop = FlexIO_8080_Inst.MulBeatWR_ShfQty * 4U;
    edma_tcd_config_t edma_config;
    edma_user_config_t edma_user_config;

    if(length < BeatsPerMinLoop)
    {
        FLEXIO_8080_SglBeatRD_nPrm(cmdIdx, buffer, length);
    }
    else
    {
        MulBeatCountRemain = length % BeatsPerMinLoop;
        MulBeatDataRemain = buffer + (length - MulBeatCountRemain);

        /* Multi-beat read several parameters. */
        edma_user_config.Buffer = (void *)buffer;
        edma_user_config.TotalSize = length - MulBeatCountRemain;               /* in bytes */

        FLEXIO_8080_MulBeatRD_DMATCD_GetConfig(&edma_config, &edma_user_config);
        FLEXIO_8080_MulBeatRD_DMA_Init(&edma_config);

        /* Configure RD pin as GPIO function, WR pin as FlexIO function */
        flexio_8080_pins_handler.Config_RD_GPIO();
        flexio_8080_pins_handler.Config_WR_FlexIO();

        /* Configure FlexIO with 1-beat write configuration */
        FLEXIO_8080_ConfigSglBeatWR(&SglBeatWR_Reg_Config);

        /* Assert CS, RS pins */
        flexio_8080_pins_handler.Set_CS_Pin(false);
        flexio_8080_pins_handler.Set_RS_Pin(false);

        /* Write command index */
        FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.SglBeatWR_ShfIdx] = cmdIdx;

        /*Wait for transfer to be completed */
        while(0 == (FLEXIO0->TIMSTAT & (1U << FlexIO_8080_Inst.TmrIdx)))
        {
        }

        /* De-assert RS pin */
        flexio_8080_pins_handler.Set_RS_Pin(true);

        /* Configure WR pin as GPIO function, RD pin FlexIO function */
        flexio_8080_pins_handler.Config_WR_GPIO();
        flexio_8080_pins_handler.Config_RD_FlexIO();

        /* Configure FlexIO with multi-beat write configuration */
        FLEXIO_8080_ConfigMulBeatRD(&MulBeatRD_Reg_Config);
        FLEXIO0->SHIFTSDEN |= 1U << FlexIO_8080_Inst.MulBeatRD_ShfIdx[0];

        /* Start data transfer by using DMA */
        RD_DMATransferDone = false;
        pDMAIntCallback = FLEXIO_8080_MulBeatRD_Callback;
        FLEXIO_8080_MulBeatRD_DMA_Enable(true);
    }
}

/*!
 * @brief callback function for FLEXIO_8080_MulBeatRD_nPrm()
 * It is called in the DMA ISR, used to complete the transfer.
 */
static void FLEXIO_8080_MulBeatRD_Callback(void)
{
    while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.MulBeatRD_ShfIdx[0])))
    {
    }
    FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;

    /* Read the last bytes in polling */
    uint32_t EndAddr = MulBeatRd_destAddr + (MulBeatRd_majorLoopCount+1U) * MulBeatRd_minorLoopCount - 4U;
    uint32_t Loop = MulBeatRd_minorLoopCount/4U;
    for(uint32_t i=0; i < Loop; i++)
    {
        *((uint32_t *)( EndAddr-i*4U)) =  FLEXIO0->SHIFTBUF[FlexIO_8080_Inst.MulBeatRD_ShfIdx[1]-i];
    }

    if(MulBeatCountRemain)
    {
        /* Configure FlexIO with 1-beat read configuration */
        FLEXIO_8080_ConfigSglBeatRD(&SglBeatRD_Reg_Config);

        /* Use polling method for data transfer */
        for(uint32_t i=0; i<MulBeatCountRemain-1U; i++)
        {
            while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatRD_ShfIdx)))
            {
            }
            *MulBeatDataRemain++ = FLEXIO0->SHIFTBUFBYS[FlexIO_8080_Inst.SglBeatRD_ShfIdx]; /* SHIFTBUFBYS for 8-bit reading */
        }

        /* Read the last byte */
        while(0 == (FLEXIO0->SHIFTSTAT & (1U << FlexIO_8080_Inst.SglBeatRD_ShfIdx)))
        {
        }
        FLEXIO0->CTRL &= ~FLEXIO_CTRL_FLEXEN_MASK;
        *MulBeatDataRemain = FLEXIO0->SHIFTBUFBYS[FlexIO_8080_Inst.SglBeatRD_ShfIdx];   /* SHIFTBUFBYS for 8-bit reading */
    }

    /* De-assert CS pin */
    flexio_8080_pins_handler.Set_CS_Pin(true);

    RD_DMATransferDone = true;
}

/* EOF */
