/*
 * Copyright (c) 2015-2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "fsl_common.h"
#include "board.h"
#include "pin_mux.h"

#include "flexio_8080_drv.h"
#include "lcd_hx8357_drv.h"
#include "logo_echo.h"
#include "zoo_rotate.h"
#include "draw_house.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
extern bool Button_Pressed;
static uint32_t DispMode = 0;

extern FLEXIO_8080_Type *pFlexIO_8080_Inst;
extern flexio_8080_pins_handler_t * pflexio_8080_pins_handler;

/*******************************************************************************
 * Functions
 ******************************************************************************/
/*!
 * @brief test API driver functions.
 * Please use a logic analyzer to watch the bus signal.
 */

#define BUFFER_SIZE                 256U
#define SGL_BEAT_TRANSFER_SIZE      10U
#define MUL_BEAT_TRANSFER_SIZE      150U

#pragma data_alignment = 256
uint8_t RAMBuffer[BUFFER_SIZE];

void APIDriverTest(void)
{
    /* Single-beat write several parameters. */
    for(uint32_t i=0; i<SGL_BEAT_TRANSFER_SIZE; i++)
    {
        RAMBuffer[i] = i;
    }
    FLEXIO_8080_SglBeatWR_nPrm(0x11, RAMBuffer, SGL_BEAT_TRANSFER_SIZE);

    /* Single-beat read several parameters. */
    FLEXIO_8080_SglBeatRD_nPrm(0x33, RAMBuffer, SGL_BEAT_TRANSFER_SIZE);

    /* Multi-beat write several parameters. */
    for(uint32_t i=0; i<BUFFER_SIZE; i++)
    {
        RAMBuffer[i] = i;
    }
    FLEXIO_8080_MulBeatWR_nPrm(0x55, RAMBuffer, MUL_BEAT_TRANSFER_SIZE);
    while(WR_DMATransferDone == false)
    {
    }

    /* Multi-beat read several parameters. */
    FLEXIO_8080_MulBeatRD_nPrm(0xAA, RAMBuffer, MUL_BEAT_TRANSFER_SIZE);
    while(RD_DMATransferDone == false)
    {
    }

    while(1)
    {
    }
}

/*!
 * @brief test performance of refreshing LCD.
 * Please use a oscilloscope to watch the LED pin's output.
 */
void PerformanceTest(void)
{
    uint16_t Colors[3] = {Red, Green, Blue};

    for(uint32_t i=0; i<3U; i++)
    {
        LED_GREEN_ON();
        asm("NOP");
        asm("NOP");
        asm("NOP");
        LED_GREEN_OFF();

        LCD_HX8357_FillColorWhole(Colors[i]);
    }
}

/*!
 * @brief Initialize FlexIO for this demo.
 */
void Demo_FLEXIO_8080_Init(void)
{
    /* Enable DMA and DMAMUX clock */
    PCC_DMA0 |= PCC_CLKCFG_CGC_MASK;
    PCC_DMAMUX0 |= PCC_CLKCFG_CGC_MASK;

    /* Setup FlexIO clock source and Enable Clock */
    PCC_FLEXIO0 &= ~PCC_CLKCFG_CGC_MASK;
    PCC_FLEXIO0 = PCC_FLEXIO0 & (~PCC_CLKCFG_PCS_MASK) | PCC_CLKCFG_PCS(3UL);   /* Use FIRC DIV3 48MHz as clock source */
    PCC_FLEXIO0 |= PCC_CLKCFG_CGC_MASK;

    /* Setup FlexIO 8080 pins handler functions */
    pflexio_8080_pins_handler->Config_Data_Pin = FLEXIO_8080_Config_Data_Pin;
    pflexio_8080_pins_handler->Config_WR_FlexIO = FLEXIO_8080_Config_WR_FlexIO;
    pflexio_8080_pins_handler->Config_WR_GPIO = FLEXIO_8080_Config_WR_GPIO;
    pflexio_8080_pins_handler->Config_RD_FlexIO = FLEXIO_8080_Config_RD_FlexIO;
    pflexio_8080_pins_handler->Config_RD_GPIO = FLEXIO_8080_Config_RD_GPIO;
    pflexio_8080_pins_handler->Config_CS_GPIO = FLEXIO_8080_Config_CS_GPIO;
    pflexio_8080_pins_handler->Set_CS_Pin = FLEXIO_8080_Set_CS_Pin;
    pflexio_8080_pins_handler->Config_RS_GPIO = FLEXIO_8080_Config_RS_GPIO;
    pflexio_8080_pins_handler->Set_RS_Pin = FLEXIO_8080_Set_RS_Pin;

    FLEXIO_8080_Type_GetDefaultConfig();

    pFlexIO_8080_Inst->Data0PinIdx = FLEXIO_DATA0_FXIOD_INDEX;
    pFlexIO_8080_Inst->WRPinIdx = FLEXIO_WR_PIN_FXIOD_INDEX;
    pFlexIO_8080_Inst->RDPinIdx = FLEXIO_RD_PIN_FXIOD_INDEX;

    FLEXIO_8080_Init();
}

/*!
 * @brief The main function
 */
int main (void)
{
    /* Init system clock. */
    BOARD_BootClockRUN();

    /* Enable PORT clock */
    PCC_PORTA = PCC_CLKCFG_CGC_MASK;
    PCC_PORTB = PCC_CLKCFG_CGC_MASK;
    PCC_PORTC = PCC_CLKCFG_CGC_MASK;
    PCC_PORTD = PCC_CLKCFG_CGC_MASK;
    PCC_PORTE = PCC_CLKCFG_CGC_MASK;

    /* Init button and LED pins. */
    Init_SW_LED();

    /* Init FlexIO for this demo. */
    Demo_FLEXIO_8080_Init();

    /* Mask the API driver test function for demo normal work. */
#if 0
    APIDriverTest();
#endif

    /* Init HX8357. */
    LCD_HX8357_Init();

    DispMode = 0U;
    LCD_HX8357_FillColorWhole(White);

    while(1)
    {
        if(Button_Pressed)
        {
            Button_Pressed = false;

            if(DispMode<3U)
            {
                DispMode++;
            }
            else
            {
                DispMode = 0U;
            }

            switch(DispMode)
            {
                case 0U:
                    LogoEcho_Prepare();
                    break;
                case 2U:
                    Graph_DrawHouse_Prepare();
                    break;
                default:
                    break;
            }
        }

        switch(DispMode)
        {
            case 0U:
                LogoEcho_Update();
                break;
            case 1U:
                DispZoo();
                for(volatile uint32_t t=0U; t<5000000U; t++)
                {
                }
                break;
            case 2U:
                Graph_DrawHouse_Update();
                for(volatile uint32_t t=0U; t<2000000U; t++)
                {
                }
                break;
            case 3U:
                PerformanceTest();
                break;
            default:
                break;
        }
    }
}

/* EOF */
