/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GDFLIB_FilterFIR.c
*
* @author    Andrzej Lara
* 
* @version   1.0.9.0
* 
* @date      Apr-26-2010
* 
* @brief     Source file for the FIR filter function.
*
********************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
*******************************************************************************/
/*!
@if GDFLIB_GROUP
	@addtogroup GDFLIB_GROUP
@else
	@defgroup GDFLIB_GROUP	GDFLIB	
@endif
*/ 

#include "SWLIBS_Typedefs.h"
#include "GDFLIB_FilterFIR.h"

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************
| external declarations
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs and structures (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| global variable definitions (scope: module-exported)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| global variable definitions (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototypes (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementations (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementations (scope: module-exported)
|-----------------------------------------------------------------------------*/

/***************************************************************************//*!
@brief      The function performs initialization for the #GDFLIB_FilterFIR
            function.

@param[in]  pParam   Pointer to the parameters structure.
@param[out] pState   Pointer to the state structure.
@param[in,out] ps32InBuf Pointer to a buffer for storing filter input signal
                     values, must point to R/W memory region and must
                     be a filter order + 1 long.
  
@return     void.
        
@details    The function performs initialization procedure for for the
            #GDFLIB_FilterFIR function.  In particular the function performs
            the following operations:

            -# Resets the input buffer index to zero.
            -# Initializes the input buffer pointer to the pointer
               provided as argument.
            -# Resets the input buffer.

            After initialization made by the function, the parameters
            and state structure should be provided as arguments to
            calls of the #GDFLIB_FilterFIR function.

@note       The input buffer pointer (\c State->ps32InBuf) must point to
            Read/Write memory region, which must be at least the number of the
            filter taps long. The number of taps in filter is equal the filter
            order + 1.  There are no restriction as to location of the
            parameters structure as long as it is readable.

@warning    No check is performed for R/W capability and the length of 
            the input buffer (\c pState->ps32InBuf).

@par Reentrancy:
            The function is reentrant only if the calling code is provided
            with a distinct instance of the structure pointed to by
            \c pState.

@par Code Example:
\code
#include "gdflib.h"

#define FIR_NUMTAPS 16
#define FIR_NUMTAPS_MAX 64
#define FIR_ORDER (FIR_NUMTAPS - 1)

GDFLIB_FILTERFIR_PARAM_T Param;
GDFLIB_FILTERFIR_STATE_T State;

tFrac32 as32InBuf[FIR_NUMTAPS_MAX];
tFrac32 as32CoefBuf[FIR_NUMTAPS_MAX];

#define OUT_LEN 16

void main(void)
{
    int ii;
    tFrac32 as32OutBuf[OUT_LEN];

    // Define a simple low-pass filter
    // The filter coefficients were calculated by the following
    // Matlab function (coefficients are contained in Hd.Numerator):
    //
    //function Hd = fir_example
    //FIR_EXAMPLE Returns a discrete-time filter object.
    //N    = 15;
    //F6dB = 0.5;
    // 
    //h = fdesign.lowpass('n,fc', N, F6dB);
    //
    //Hd = design(h, 'window');
    //return;
    ii = 0;
    as32CoefBuf[ii++] = 0xFFB10C14;
    as32CoefBuf[ii++] = 0xFF779D25;
    as32CoefBuf[ii++] = 0x01387DD7;
    as32CoefBuf[ii++] = 0x028E6845;
    as32CoefBuf[ii++] = 0xFB245142;
    as32CoefBuf[ii++] = 0xF7183CC7;
    as32CoefBuf[ii++] = 0x11950A3C;
    as32CoefBuf[ii++] = 0x393ED867;
    as32CoefBuf[ii++] = 0x393ED867;
    as32CoefBuf[ii++] = 0x11950A3C;
    as32CoefBuf[ii++] = 0xF7183CC7;
    as32CoefBuf[ii++] = 0xFB245142;
    as32CoefBuf[ii++] = 0x028E6845;
    as32CoefBuf[ii++] = 0x01387DD7;
    as32CoefBuf[ii++] = 0xFF779D25;
    as32CoefBuf[ii++] = 0xFFB10C14;

    Param.u32Order = 15;
    Param.ps32CoefBuf = &as32CoefBuf[0];
    
    // Initialize FIR filter
    GDFLIB_FilterFIRInit(&Param, &State, &as32InBuf[0]);

    // Compute step response of the filter
    for(ii=0; ii < OUT_LEN; ii++)
    {
        as32OutBuf[ii] = GDFLIB_FilterFIR(FRAC32(1.0), &Param, &State);
    }

    // as32Out contains step response of the filter

    return;
}
\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterFIRInit
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterFIRInit function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>24/30 </td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>26/34</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>25/33</td>
            </tr>
            </table>

*******************************************************************************/
void GDFLIB_FilterFIRInitANSIC(const GDFLIB_FILTERFIR_PARAM_T *const pParam,
                               GDFLIB_FILTERFIR_STATE_T *const pState,
                               tFrac32 *ps32InBuf)
{
    register tU32 u32I;       /* General looping variable */
    
    pState->u32Idx = 0;
    pState->ps32InBuf = ps32InBuf;

    /* Initialize the input buffer to zero */
    for(u32I=0; u32I<=pParam->u32Order; u32I++)
    {
        *ps32InBuf = 0;
        ps32InBuf++;
    }

    return;
}

/***************************************************************************//*!
@brief       The function performs a single iteration of a FIR filter.

@param[in]     s32In   Input value.
@param[in]     pParam  Pointer to the parameter structure.
@param[in,out] pState  Pointer to the filter state structure.
  
@return      A value of a filtered signal after processing by a FIR filter.
        
@details     The function performs operation of a FIR filter on the
             sample-by-sample basis.  At each new input to the FIR filter, the
             function should be called, which will return a new filtered value.

             The FIR filter is defined by the following formula:
             \f[
                y[n]=h_0 x[n] + h_1 x[n-1] + \cdots + h_N x[n-N]
             \f]

            where: \f$x[n]\f$ is the input signal, \f$y[n]\f$ is the output
            signal, \f$h_i\f$ are the filter coefficients, and \f$N\f$ is the
            filter order.  It should be noticed that the number of taps of
            the filter is \f$N + 1\f$ in this case.

            The multiply and accumulate operations are performed with 32
            accumulation guard bits present, which means that no saturation is
            performed during computations.  However, if the final value cannot
            fit in the return data type, saturation may occur. It should be
            noticed, that, although rather theoretically, no saturation
            is performed on the accumulation guard bits and an overflow
            over accumulation guard bits may occur.

            The function assumes that the filter order is at least one, which
            is equivalent to two taps.  The filter cannot contain also more
            than ffffffff(hexadecimal) taps, which is equivalent of the order
            of fffffffe (hexadecimal).

            The input values are recorded by the function in the provided state
            structure in a circular buffer, pointed to by the state structure
            member \c pState->ps32InBuf.  The buffer index is stored in \c
            pState->u32Idx, which points to the buffer element where a new
            input signal sample will be stored.
            
            The filter coefficients are stored in the parameter structure, in
            the structure member \c pParam->ps32CoefBuf.

            The first  call to the function must be preceded by an
            initialization, which can be made through the #GDFLIB_FilterFIRInit
            function.  The #GDFLIB_FilterFIRInit and then the #GDFLIB_FilterFIR
            functions should be called with the same parameters.

@par Reentrancy:
            The function is reentrant only if the calling code is provided
            with a distinct instance of the structure pointed to by
            \c pState.

@note       From the performance point of view the function is designed to
            work with filters with a larger number of taps (equal order + 1).
            As the rule of thumb, if the number of taps is lower than 5, a
            different algorithm should be considered.

@warning
             
@par Code Example:
\code
#include "gdflib.h"

#define FIR_NUMTAPS 16
#define FIR_NUMTAPS_MAX 64
#define FIR_ORDER (FIR_NUMTAPS - 1)

GDFLIB_FILTERFIR_PARAM_T Param;
GDFLIB_FILTERFIR_STATE_T State;

tFrac32 as32InBuf[FIR_NUMTAPS_MAX];
tFrac32 as32CoefBuf[FIR_NUMTAPS_MAX];

#define OUT_LEN 16

void main(void)
{
    int ii;
    tFrac32 as32OutBuf[OUT_LEN];

    // Define a simple low-pass filter
    // The filter coefficients were calculated by the following
    // Matlab function (coefficients are contained in Hd.Numerator):
    //
    //function Hd = fir_example
    //FIR_EXAMPLE Returns a discrete-time filter object.
    //N    = 15;
    //F6dB = 0.5;
    // 
    //h = fdesign.lowpass('n,fc', N, F6dB);
    //
    //Hd = design(h, 'window');
    //return;
    ii = 0;
    as32CoefBuf[ii++] = 0xFFB10C14;
    as32CoefBuf[ii++] = 0xFF779D25;
    as32CoefBuf[ii++] = 0x01387DD7;
    as32CoefBuf[ii++] = 0x028E6845;
    as32CoefBuf[ii++] = 0xFB245142;
    as32CoefBuf[ii++] = 0xF7183CC7;
    as32CoefBuf[ii++] = 0x11950A3C;
    as32CoefBuf[ii++] = 0x393ED867;
    as32CoefBuf[ii++] = 0x393ED867;
    as32CoefBuf[ii++] = 0x11950A3C;
    as32CoefBuf[ii++] = 0xF7183CC7;
    as32CoefBuf[ii++] = 0xFB245142;
    as32CoefBuf[ii++] = 0x028E6845;
    as32CoefBuf[ii++] = 0x01387DD7;
    as32CoefBuf[ii++] = 0xFF779D25;
    as32CoefBuf[ii++] = 0xFFB10C14;

    Param.u32Order = 15;
    Param.ps32CoefBuf = &as32CoefBuf[0];
    
    // Initialize FIR filter
    GDFLIB_FilterFIRInit(&Param, &State, &as32InBuf[0]);

    // Compute step response of the filter
    for(ii=0; ii < OUT_LEN; ii++)
    {
        as32OutBuf[ii] = GDFLIB_FilterFIR(FRAC32(1.0), &Param, &State);
    }

    // as32Out contains step response of the filter

    return;
}

\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterFIR
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterFIR function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>822/690</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td></td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>108/106</td>
            </tr>
            </table>

            Count of maximal clock cycles depends on selected order of the
            filter. Minimal clock cycles, noted in table
            \ref tab1_GDFLIB_FilterFIR is valid for first order FIR filter.

@internal   Because of a bug in Matlab lcc compiler, the saturation at the end
            of the function is made in an odd way.  It seems that lcc compiler
            has a bug, which does not allow to perform correctly comparison
            of signed 64-bit integer, when one of the number is 0x7fffffff.
            Except of looking strange, the required changes to code do not
            have impact on performance, so it can work also for embedded
            targets.
*******************************************************************************/
tFrac32 GDFLIB_FilterFIRANSIC(tFrac32 s32In,
                              const GDFLIB_FILTERFIR_PARAM_T *const pParam,
                              GDFLIB_FILTERFIR_STATE_T *const pState)
{
    tU32 u32Order;          /* Filter order, auxiliary variable */
    tU32 u32XIdx;           /* Input buffer index, auxiliary variable */

    tFrac32 *ps32X;         /* Pointer to number in input buffer */
    const tFrac32 *ps32H;   /* Pointer to number in coefficient buffer */

    tFrac16 s16Hk;          /* Coefficient read from memory */
    tFrac16 s16Xk;          /* Input value read from memory */

    tU32 u32K;              /* General looping variable */

    tS64 s64Y;              /* 64-bit accumulator */
    tFrac32  s32Y;          /* 32-bit final value */

    ps32H = pParam->ps32CoefBuf;
    u32Order = pParam->u32Order;
    ps32X = pState->ps32InBuf;
    u32XIdx = pState->u32Idx;

    /* Save the newest sample into the state buffer. */
    *(ps32X + u32XIdx) = s32In;

    /* Reset the accumulator and coefficient index */
    s64Y = 0;

    /* Divide the whole loop into two smaller loops in order
     * to avoid modulo computation for each iteration.
     * 
     * Note that the sample index is going downward.
     *
     * For efficiency reasons (no need to shift left every time)
     * the accumulation is done in the integer format. */
    ps32X = ps32X + u32XIdx;
    for(u32K=u32XIdx; u32K > 0; u32K--)
    {
        s16Hk = (tS16) ((*(ps32H++))>>16);
        s16Xk = (tS16) ((*(ps32X--))>>16);
        s64Y = (s64Y + (tS64)((tS32) s16Hk)*((tS32) s16Xk) );
    }

    /* u32K == 0 */
    s16Hk = (tS16) ((*(ps32H++))>>16);
    s16Xk = (tS16) ((*(ps32X))>>16);
    s64Y = (s64Y + (tS64)((tS32) s16Hk)*((tS32) s16Xk) );

    /* Wrap-up */
    ps32X += u32Order;
    for(u32K=u32Order; u32K > u32XIdx; u32K--)
    {
        s16Hk = (tS16) ((*(ps32H++))>>16);
        s16Xk = (tS16) ((*(ps32X--))>>16);
        s64Y = (s64Y + (tS64)((tS32) s16Hk)*((tS32) s16Xk) );
    }

    u32XIdx++;

    /* Wrap-up if necessary */
    u32XIdx = (u32XIdx > u32Order) ? 0 : u32XIdx;
    pState->u32Idx = u32XIdx;

    /* Perform conversion from 64-bit accumulator to 32-bit value,
     * saturate if necessary. */
    //s64Y = s64Y + s64Y;
    //324Y = (tS32) s64Y;
    s32Y = (tS32) (s64Y + s64Y);

    //if(s64Y<((tS64) 0xffffffff80000000))
    // -1073741824 = 0xffffffffc0000000))
    if( s64Y<((tS64) (-1073741824)) )
    {
        s32Y = INT32_MIN;;
    }

    //if(s64Y>((tS64) 0x000000007fffffff))
    // 1073741823 = 0x000000003fffffff
    if( s64Y>((tS64) (1073741823)) )
    {
        s32Y = INT32_MAX;
    }

    return s32Y;
}
#ifdef __cplusplus
}
#endif

/* End of file */
