/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GDFLIB_FilterIIR1.c
*
* @author   Roman Filka
*
* @version  1.0.13.0
*
* @date     Apr-26-2010
*
* @brief    Direct form II first order IIR filter.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GDFLIB_GROUP
    @addtogroup GDFLIB_GROUP
@else
    @defgroup GDFLIB_GROUP   GDFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GDFLIB_FilterIIR1.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
@ingroup    GDFLIB_GROUP
@brief      This function clears internal filter buffers used in function
            #GDFLIB_FilterIIR1.

@param[in,out]  *pParam     Pointer to filter structure with filter buffer and
                            filter parameters

@return     void

@details    This function clears internal buffers of first order IIR filter. It
            shall be called after filter parameter initialization and whenever
            the filter initialization is required.

@note       This function shall not be called together with #GDFLIB_FilterIIR1
            unless periodical clearing of filter buffers is required.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gdflib.h"

tFrac32 s32Input;
tFrac32 s32Output;

GDFLIB_FILTER_IIR1_T trMyIIR1 = GDFLIB_FILTER_IIR1_DEFAULT;

void main(void)
{
    // input value = 0.25
    s32Input  = FRAC32(0.25);

    // filter coefficients (LPF 100Hz, Ts=100e-6)
    trMyIIR1.trFiltCoeff.s32B0           = FRAC32(0.030468747091254/8);
    trMyIIR1.trFiltCoeff.s32B1           = FRAC32(0.030468747091254/8);
    trMyIIR1.trFiltCoeff.s32A1           = FRAC32(-0.939062505817492/8);
    GDFLIB_FilterIIR1Init(&trMyIIR1);
}
\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterIIR1Init
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterIIR1Init function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>4/8</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>11/11</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>9/10</td>
            </tr>
            </table>
******************************************************************************/
void GDFLIB_FilterIIR1InitANSIC(GDFLIB_FILTER_IIR1_T *pParam)
{
    pParam->s32FiltBufferX[1]   = 0;
    pParam->s32FiltBufferY[1]   = 0;
}

void GDFLIB_Filter16IIR1InitANSIC(GDFLIB_FILTER_IIR1_16_T *pParam)
{
    pParam->s16FiltBufferX[1]   = 0;
    pParam->s32FiltBufferY[1]   = 0;
}

/**************************************************************************//*!
@ingroup    GDFLIB_GROUP

@brief      This function implements Direct Form I first order IIR filter.

@param[in,out]  *pParam     Pointer to filter structure with filter buffer and
                            filter parameters
@param[in]      s32In       Value of input signal to be filtered in step (k).
                            The value is 32 bit number in 1.31 fractional format

@return     The function returns 32-bit value in fractional format 1.31,
            representing the filtered value of the input signal in step (k).

@details    The #GDFLIB_FilterIIR1ANSIC function, denoting ANSI-C compatible
            implementation, can be called via function alias #GDFLIB_FilterIIR1.

            \par
            This function calculates the first order infinite impulse (IIR)
            filter. The IIR filters are also called recursive filters because
            both, the input and the previously calculated output
            values, are used for calculation of the filter equation in each step.
            This form of feedback enables transfer of the energy from the output
            to the input, which theoretically leads to an infinitely long
            impulse response (IIR).

            A general form of the IIR filter expressed as a transfer function
            in the Z-domain is described as follows:
            \anchor eq1_GDFLIB_FilterIIR1
            \f[
                H(z)=\frac{Y(z)}{X(z)}=\frac{b_0+b_1z^{-1}+b_2z^{-2}+ \cdot +b_{N}z^{-N}}{1+a_1z^{-1}+a_2z^{-2}+ \cdot +a_{N}z^{-N}}
            \f]

            where \f$ N \f$ denotes the filter order. The first order IIR filter
            in the Z-domain is therefore given from eq. \ref eq1_GDFLIB_FilterIIR1
            as:
            \anchor eq2_GDFLIB_FilterIIR1
            \f[
                H(z)=\frac{Y(z)}{X(z)}=\frac{b_0+b_1z^{-1}}{1+a_1z^{-1}}
            \f]

            In order to implement the first order IIR filter on a microcontroller,
            the discrete time domain representation of the filter,
            described by eq. \ref eq2_GDFLIB_FilterIIR1, must be transformed into
            a time difference equation as follows:
            \anchor eq3_GDFLIB_FilterIIR1
            \f[
                y(k)= b_0 x(k) + b_1 x(k-1) - a_1 y(k-1)
            \f]

            Equation \ref eq3_GDFLIB_FilterIIR1 represents a Direct Form I
            implementation of first order IIR filter. It is well known, that
            Direct Form I (DF-I) and Direct Form II (DF-II) implementations of
            IIR filter are generally sensitive to parameter quantization, if
            a finite precision arithmetic is considered. This however can be
            neglected when the filter transfer function is broken up to low order
            sections, i.e. first or second order.
            The main difference between DF-I and DF-II implementations of IIR
            filter is in number of delay buffers and in number of guard bits
            required to handle potential overflow.
            The DF-II implementation requires less delay buffers than DF-I,
            hence less data memory is utilized. On the other hand,
            since the poles come first in the DF-II realization,
            the signal entering the state delay-line typically requires a larger
            dynamic range than the output signal \f$y(k)\f$. Therefore overflow
            can occur at the delay-line input of DF-II implementation,
            unlike in the DF-I implementation.

            \par
            Because there are two delay buffers necessary for both DF-I and DF-II
            implementation of the first order IIR filter, DF-I implementation
            was chosen to be used in #GDFLIB_FilterIIR1 function.

            \anchor fig1_GDFLIB_FilterIIR1
            \image latex GDFLIB_FilterIIR1_Figure1.eps "Direct Form 1 first order IIR filter" width=10cm
            \image html GDFLIB_FilterIIR1_Figure1.jpg "Direct Form 1 first order IIR filter"

            \par
            The coefficients of the filter depicted in Fig. \ref fig1_GDFLIB_FilterIIR1
            can be designed to meet requirements for the first order Low (LPF)
            or High Pass filter (HPF). Filter coefficients can be calculated
            using various tools, for example Matlab \e butter function. In order
            to avoid overflow during the calculation of the #GDFLIB_FilterIIR1
            function, filter coefficients must be divided by eight.
            The coefficient quantization error due to
            finite precision arithmetic can be neglected in case of a first order
            filter. Therefore calculation of coefficients can be done using
            Matlab as follows:

\code
freq_cut    = 100;
T_sampling  = 100e-6;

[b,a]=butter(1,[freq_cut*T_sampling*2],'low');
sys=tf(b,a,T_sampling);
bode(sys)

s32B0 = b(1)/8;
s32B1 = b(2)/8;
s32A1 = a(2)/8;
disp('Coefficients for GDFLIB_FilterIIR1 function:');
disp(['s32B0 = FRAC32(' num2str(s32B0) ')']);
disp(['s32B1 = FRAC32(' num2str(s32B1) ')']);
disp(['s32A1 = FRAC32(' num2str(s32A1) ')']);
\endcode

@note       Filter delay line includes two delay buffers which should be reset
            after filter initialization. This can be
            done by assigning filter instance a #GDFLIB_FILTER_IIR1_DEFAULT
            macro during instance declaration or by calling #GDFLIB_FilterIIR1Init
            function.

@warning    Because of fixed point implementation and to avoid overflow
            during the calculation of the #GDFLIB_FilterIIR1 function,
            filter coefficients must be divided by eight. Function output
            is internally multiplied by eight to correct the coefficient scaling.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gdflib.h"

tFrac32 s32Input;
tFrac32 s32Output;

GDFLIB_FILTER_IIR1_T trMyIIR1 = GDFLIB_FILTER_IIR1_DEFAULT;

void main(void)
{
    // input value = 0.25
    s32Input  = FRAC32(0.25);

    // filter coefficients (LPF 100Hz, Ts=100e-6)
    trMyIIR1.trFiltCoeff.s32B0           = FRAC32(0.030468747091254/8);
    trMyIIR1.trFiltCoeff.s32B1           = FRAC32(0.030468747091254/8);
    trMyIIR1.trFiltCoeff.s32A1           = FRAC32(-0.939062505817492/8);
    GDFLIB_FilterIIR1Init(&trMyIIR1);

    // output should be 0x00F99998
    s32Output = GDFLIB_FilterIIR1(s32Input,&trMyIIR1);
}
\endcode

@par Performance:
            \anchor tab1_GDFLIB_FilterIIR1
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GDFLIB_FilterIIR1 function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>200/118</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>108/79</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>85/61</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GDFLIB_FilterIIR1ANSIC(tFrac32 s32In, GDFLIB_FILTER_IIR1_T *pParam)
{

#if defined (USE_FRAC32_ARITHMETIC) && defined(USE_ASM)
    /*
     * Implemented equation:
     * y(k) = b0*x(k) + b1*x(k-1) - a1*y(k-1)
     *
     * r3 <- x(k)
     * 0x0(r4) <- b0>>3     parameter is scaled down by eight to format Q4.28
     * 0x4(r4) <- b1>>3     parameter is scaled down by eight to format Q4.28
     * 0x8(r4) <- a1>>3     parameter is scaled down by eight to format Q4.28
     * 0xC(r4) <- x(k-1)
     * 0x10(r4) <- y(k-1)
     */

    /*--------------
    (b1>>3)*x(k-1)
    --------------*/
    asm("e_lwz      r5,0x4(r4)");   // r5 = b1>>3
    asm("e_lwz      r6,0xC(r4)");   // r6 = x(k-1)
    asm("se_stw     r3,0xC(r4)");   // store x(k) into x(k-1) in filter structure

    asm("mullw      r0,r5,r6");     // r0 = r5 * r6; multiply low word
    asm("mulhw      r5,r5,r6");     // r5 = (r5*r6)>>32; multiply high word
    asm("e_slwi     r6,r5,1");      // r6 = r5<<1; convert high word to frac format
    asm("e_rlwimi   r6,r0,1,31,31");// r6 = r6 & r0(MSB)

    /*--------------
    (b0>>3)*x(k)
    --------------*/
    asm("e_lwz      r5,0(r4)");     // r5 = b0>>3
    asm("mullw      r0,r3,r5");     // r0 = r3 * r5; multiply low word
    asm("mulhw      r3,r3,r5");     // r3 = (r3*r5)>>32; multiply high word
    asm("e_slwi     r3,r3,1");      // r3 = r3<<1; convert high word to frac format
    asm("e_rlwimi   r3,r0,1,31,31");// r3 = r3 & r0(MSB)

    /*--------------
    (b0>>3)*x(k) + (b1>>3)*x(k-1)
    --------------*/
    asm("se_add     r3,r6");        // r3 = (b0>>3)*x(k) + (b1>>3)*x(k-1)

    /*--------------
    (a1>>3)*y(k-1)
    --------------*/
    asm("e_lwz      r5,0x8(r4)");   // r5 = a1>>3
    asm("e_lwz      r6,0x10(r4)");  // r6 = y(k-1)

    asm("mullw      r0,r5,r6");     // r0 = r5 * r6; multiply low word
    asm("mulhw      r5,r5,r6");     // r5 = (r5*r6)>>32; multiply high word
    asm("e_slwi     r5,r5,1");      // r5 = r5<<1; convert high word to frac format
    asm("e_rlwimi   r5,r0,1,31,31");// r5 = r5 & r0(MSB)

    /*--------------
    Acc = (b0>>3)*x(k) + (b1>>3)*x(k-1) - (a1>>3)*y(k-1)
    --------------*/
    asm("se_sub     r3,r5");        // r3 = (b0>>3)*x(k) + (b1>>3)*x(k-1) - (a1>>3)*y(k-1)

    /*--------------
     y(k) = Acc * 8;
     Correction due to scaled filter coefficients.
     Result saturates if overflow caused by arithmetic left shift occurs.
    --------------*/
    asm("e_slwi     r5,r3,3");      // temp = y(k)<<3;  r5 = r3<<3;
                                    // transform result from Q4.28 back to
                                    // Q1.31 format

    //y(k) = (Acc > Xmax)? 0x7fff ffff : y(k)
    asm("se_bmaski  r6,31");        // r6 = 0x7FFFFFFF
    asm("se_bmaski  r7,28");        // r7 = 0x7FFFFFFF>>3 = Xmax
    asm("se_cmp     r3,r7");        //
    asm("iselgt     r3,r6,r5");     // r3 = (r3 > r7) ? r6 : r5

    //y(k) = (temp <= Xmin)? 0x8000 0000:y(k)
    asm("e_lis      r6,0x8000");    // r6 = 0x80000000
    asm("e_lis      r7,0xf000");    // r7 = 0x80000000>>3 = Xmin
    asm("se_cmp     r3,r7");        //
    asm("isellt     r3,r6,r3");     // r3 = (r3 < r7) ? r6 : r3
    asm("iseleq     r3,r6,r3");     // r3 = (r3 == r7)? r6 : r3

    asm("se_stw     r3,0x10(r4)");  // y(k-1) <- r3
#endif

#if defined (USE_FRAC32_ARITHMETIC) && !defined(USE_ASM)
//USE_FRAC32_ARITHMETIC
    register tFrac32 s32M1;
    register tFrac32 s32M2;
    register tFrac32 s32M3;
    register tFrac32 s32Acc;
    register tFrac32 s32Out;

    /*
     * Implemented equation:
     * y(k) = b0*x(k) + b1*x(k-1) - a1*y(k-1)
     *
     * Calculation steps:
     * M1   = b0*x(k)
     * M2   = b1*x(k-1)
     * M3   = a1*y(k-1)
     * Acc  = M1+M2
     * Acc  = Acc-M3
     * y(k) = Acc*2
     */

    // M1   = b0*x(k), number format Q1.31
    s32M1   = F32Mul(pParam->trFiltCoeff.s32B0,s32In);

    // M2   = b1*x(k-1), number format Q1.31
    s32M2   = F32Mul(pParam->trFiltCoeff.s32B1,pParam->s32FiltBufferX[1]);

    // M3   = a1*y(k-1), number format Q1.31
    s32M3   = F32Mul(pParam->trFiltCoeff.s32A1,pParam->s32FiltBufferY[1]);

    // Acc  = M1+M2, number format Q1.31
    s32Acc  = F32Add(s32M2,F32Neg(s32M3));

    // Acc  = Acc-M3, number format Q1.31
    s32Acc  = F32Add(s32Acc,s32M1);

    /* y(k) = Acc * 8; correction due to scaled filter coefficients
     * Result saturates if overflow caused by arithmetic left shift occurs.
     */
    s32Out  = F32ShlSat(s32Acc,3);

    // Storing filter states in the buffer
    pParam->s32FiltBufferX[1]   = s32In;
    pParam->s32FiltBufferY[1]   = s32Out;

    // Returning de-scaled value of internal accumulator
    return(s32Out);
#endif

#if !defined (USE_FRAC32_ARITHMETIC) && defined(USE_ASM)
// ASM 16bit arithmetic
    /*
     * Implemented equation:
     * y(k) = b0*x(k) + b1*x(k-1) - a1*y(k-1)
     *
     * r3 <- x(k)
     * 0x0(r4) <- b0>>3     parameter is scaled down by eight to format Q4.28
     * 0x4(r4) <- b1>>3     parameter is scaled down by eight to format Q4.28
     * 0x8(r4) <- a1>>3     parameter is scaled down by eight to format Q4.28
     * 0xC(r4) <- x(k-1)
     * 0x10(r4) <- y(k-1)
     */

    /*--------------
    (b1>>3)*x(k-1)
    --------------*/
    asm("e_lwz      r5,0x4(r4)");   // r5 = b1>>3
    asm("e_lwz      r6,0xC(r4)");   // r6 = x(k-1)
    asm("se_stw     r3,0xC(r4)");   // store x(k) into x(k-1) in filter structure

    asm("mulhw      r5,r5,r6");     // r5 = (r5*r6)>>32; multiply high word
    asm("e_slwi     r6,r5,1");      // r6 = r5<<1; convert high word to frac format

    /*--------------
    (b0>>3)*x(k)
    --------------*/
    asm("e_lwz      r5,0(r4)");     // r5 = b0>>3
    asm("mulhw      r3,r3,r5");     // r3 = (r3*r5)>>32; multiply high word
    asm("e_slwi     r3,r3,1");      // r3 = r3<<1; convert high word to frac format

    /*--------------
    (b0>>3)*x(k) + (b1>>3)*x(k-1)
    --------------*/
    asm("se_add     r3,r6");        // r3 = (b0>>3)*x(k) + (b1>>3)*x(k-1)

    /*--------------
    (a1>>3)*y(k-1)
    --------------*/
    asm("e_lwz      r5,0x8(r4)");   // r5 = a1>>3
    asm("e_lwz      r6,0x10(r4)");  // r6 = y(k-1)

    asm("mulhw      r5,r5,r6");     // r5 = (r5*r6)>>32; multiply high word
    asm("e_slwi     r5,r5,1");      // r5 = r5<<1; convert high word to frac format

    /*--------------
    Acc = (b0>>3)*x(k) + (b1>>3)*x(k-1) - (a1>>3)*y(k-1)
    --------------*/
    asm("se_sub     r3,r5");        // r3 = (b0>>3)*x(k) + (b1>>3)*x(k-1) - (a1>>3)*y(k-1)

    /*--------------
     y(k) = Acc * 8;
     Correction due to scaled filter coefficients.
     Result saturates if overflow caused by arithmetic left shift occurs.
    --------------*/
    asm("e_slwi     r5,r3,3");      // temp = y(k)<<3;  r5 = r3<<3;
                                    // transform result from Q4.28 back to
                                    // Q1.31 format

    //y(k) = (Acc > Xmax)? 0x7fff ffff : y(k)
    asm("se_bmaski  r6,31");        // r6 = 0x7FFFFFFF
    asm("se_bmaski  r7,28");        // r7 = 0x7FFFFFFF>>3 = Xmax
    asm("se_cmp     r3,r7");        //
    asm("iselgt     r3,r6,r5");     // r3 = (r3 > r7) ? r6 : r5

    //y(k) = (temp <= Xmin)? 0x8000 0000:y(k)
    asm("e_lis      r6,0x8000");    // r6 = 0x80000000
    asm("e_lis      r7,0xf000");    // r7 = 0x80000000>>3 = Xmin
    asm("se_cmp     r3,r7");        //
    asm("isellt     r3,r6,r3");     // r3 = (r3 < r7) ? r6 : r3
    asm("iseleq     r3,r6,r3");     // r3 = (r3 == r7)? r6 : r3

    asm("se_stw     r3,0x10(r4)");  // y(k-1) <- r3
#endif

#if !defined (USE_FRAC32_ARITHMETIC) && !defined(USE_ASM)
//USE_FRAC16_ARITHMETIC
    register tFrac32 s32M1;
    register tFrac32 s32A1;
    register tFrac32 s32Acc;
    register tFrac32 s32Out;

    /*
     * Implemented equation:
     * y(k) = b0*x(k) + b1*x(k-1) - a1*y(k-1)
     *
     * Calculation steps:
     * M1   = b0*x(k)
     * A1   = -a1*y(k-1) + b0*x(k)
     * Acc  = -a1*y(k-1) + b0*x(k) + b1*x(k-1)
     * y(k) = Acc*2
     */

    //M1    = -a1*y(k-1)
    s32M1   = F32Mul(F32Neg(pParam->trFiltCoeff.s32A1),pParam->s32FiltBufferY[1]);

    //A1    = -a1*y(k-1) + b0*x(k)
    s32A1   = F32MacF16F16(s32M1,
                              (tFrac16)((pParam->trFiltCoeff.s32B0)>>16),
                              (tFrac16)(s32In>>16));

    //Acc   = -a1*y(k-1) + b0*x(k) + b1*x(k-1)
    s32Acc  = F32MacF16F16(s32A1,
                              (tFrac16)((pParam->trFiltCoeff.s32B1)>>16),
                              (tFrac16)((pParam->s32FiltBufferX[1])>>16));

    /* y(k) = Acc * 8; correction due to scaled filter coefficients
     * Result saturates if overflow caused by arithmetic left shift occurs.
     */
    s32Out  = F32ShlSat(s32Acc,3);

    // Storing filter states in the buffer
    pParam->s32FiltBufferX[1]   = s32In;
    pParam->s32FiltBufferY[1]   = s32Out;

    // Returning de-scaled value of internal accumulator
    return(s32Out);
#endif
}

/**************************************************************************//*!
@ingroup    GDFLIB_GROUP
@brief      This function clears internal filter buffers used in function
            #GDFLIB_FilterIIR1.

@param[in,out]  *pParam

@return     void

@details    

@note       

@par Reentrancy:
            The function is reentrant.
******************************************************************************/
tFrac16 GDFLIB_Filter16IIR1ANSIC(tFrac16 s16In, GDFLIB_FILTER_IIR1_16_T *pParam)
{
    register tFrac32 s32M1;
    register tFrac32 s32M2;
    register tFrac32 s32M3;
    register tFrac32 s32Acc;
    register tFrac32 s32Out;

    /*
     * Implemented equation:
     * y(k) = b0*x(k) + b1*x(k-1) - a1*y(k-1)
     *
     * Calculation steps:
     * M1   = b0*x(k)
     * M2   = b1*x(k-1)
     * M3   = a1*y(k-1)
     * Acc  = M1+M2
     * Acc  = Acc-M3
     * y(k) = Acc*2
     */

    // M1   = b0*x(k), number format Q1.31
    s32M1   = (F32MulF16F16(pParam->trFiltCoeff.s16B0,s16In))>>3;

    // M2   = b1*x(k-1), number format Q1.31
    s32M2   = (F32MulF16F16(pParam->trFiltCoeff.s16B1,pParam->s16FiltBufferX[1]))>>3;

    // M3   = a1*y(k-1), number format Q1.31
    s32M3   = (F32Mul(((tFrac32)(pParam->trFiltCoeff.s16A1))<<16,pParam->s32FiltBufferY[1]))>>3;

    // Acc  = M1+M2, number format Q1.31
    s32Acc  = F32Add(s32M2,F32Neg(s32M3));

    // Acc  = Acc-M3, number format Q1.31
    s32Acc  = F32Add(s32Acc,s32M1);

    /* y(k) = Acc * 8; correction due to scaled filter coefficients
     * Result saturates if overflow caused by arithmetic left shift occurs.
     */
    s32Out  = F32ShlSat(s32Acc,3);

    // Storing filter states in the buffer
    pParam->s16FiltBufferX[1]   = s16In;
    pParam->s32FiltBufferY[1]   = s32Out;

    // Returning de-scaled value of internal accumulator
    return((tFrac16)(s32Out>>16));
}

#ifdef __cplusplus
}
#endif

/* End of file */
