/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GFLIB_Asin.c
*
* @author    B04459 
* 
* @version    1.0.14.0
* 
* @date      Apr-26-2010
* 
* @brief     Source file for Arcsine function
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
	@addtogroup GFLIB_GROUP
@else
	@defgroup GFLIB_GROUP	GFLIB	
@endif
*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Defines.h"
#include "SWLIBS_Inlines.h"

#include "GFLIB_Asin.h"
#include "GFLIB_Sqrt.h"

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
| external declarations
|-----------------------------------------------------------------------*/

/************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
    
/************************************************************************
| typedefs and structures (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-exported)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-local)
|-----------------------------------------------------------------------*/

GFLIB_ASIN_TAYLOR_T gflibAsinCoef = 
{
     91918582,  66340080,   9729967,    682829947,      12751,      // Coef. <0;1/2)
     -52453538, -36708911,  -15136243,  -964576326,     1073652175  // Coef.  <1/2;1)
};
/************************************************************************
| function prototypes (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-exported)
|-----------------------------------------------------------------------*/
/**************************************************************************//*!
@brief     Arcsine function - Piece-wise polynomial approximation.

@param[in]  *pParam     Pointer to array of Taylor coefficients. Using the
                        function alias #GFLIB_Asin, default coefficients are
                        used.
@param[in]  s32In       Input argument is a 32bit number that contains the
                        value between \f$\left[-1,1\right)\f$.

@return     The function returns \f$\frac{\arcsin(s32In)}{\pi} \f$ as fixed point
            32-bit number, normalized between \f$\left[-1,1\right)\f$.

@details    The #GFLIB_AsinANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Asin.

            \par
            The #GFLIB_Asin function provides a computational method for
            calculation of a standard inverse trigonometric \e arcsine function
            \f$\arcsin(x)\f$, using the piece-wise polynomial approximation.
            Function \f$\arcsin(x)\f$ takes an ratio of the length of the opposite
            side to the length of the hypotenuse and returns the angle. 
            
            \anchor fig1_GFLIB_Asin
            \image latex GFLIB_Asin_Figure1.eps "Course of the functionGFLIB_Asin" width=10cm
            \image html GFLIB_Asin_Figure1.jpg "Course of the functionGFLIB_Asin"

            The computational algorithm uses the symmetry of the \f$ arcsin(x) \f$ function
            around the point \f$ \left(0, \pi/2 \right) \f$, which allows to compute the function values
            just in the interval \f$ \left[0, 1 \right) \f$ and to compute the function values in the
            interval \f$ \left[-1, 0 \right) \f$ by the simple formula:
            
            \anchor eq1_GFLIB_Asin
            \f[
                y_{\left[-1, 0 \right)} = - y_{\left[ 1, 0 \right)},
            \f]
            
            where: 
            - \f$ y_{\left[-1, 0 \right)} \f$ is the \f$arcsin(x)\f$ function value in the interval
              \f$ \left[-1, 0 \right) \f$
            - \f$ y_{\left[1, 0 \right)} \f$ is the \f$arcsin(x)\f$ function value in the interval
              \f$ \left[1, 0 \right) \f$
             
            Additionally, because the \f$arcsin(x)\f$ function is difficult for
            polynomial approximation for \f$ x \f$ approaching 1 (or -1 by symmetry)
            due to its derivatives approaching infinity, special transformation 
            is used to transform the range of \f$ x \f$ from \f$ \left[0.5, 1 \right)\f$ to
            \f$ \left(0, 0.5 \right] \f$:
            
            \anchor eq2_GFLIB_Acos
            \f[
                \arcsin(\sqrt{1-x}) = \frac{\pi}{2} - \arcsin(\sqrt{x})
            \f]
             
            In this way the computation of the \f$ arcsin(x) \f$ function in the range 
            \f$ \left[0.5, 1 \right) \f$ can be replaced by the computation in the range 
            \f$ \left(0, 0.5 \right] \f$, in which approximation is easier.
            
            For the interval \f$ \left(0, 0.5 \right]\f$ the algorithm uses polynomial approximation
            as follows. 
            
            \anchor eq2_GFLIB_Asin
            \f[
                \mbox{s32Dump} = a_1 \cdot s32In^4 + a_2 \cdot s32In^3 + a_3 \cdot s32In^2 + a_4 \cdot s32In + a_5
            \f]

            \anchor eq3_GFLIB_Asin
            \f[
                \arcsin(s32In) =
                    \left\{\begin{array}{lclcl}
                       \mbox{-s32Dump}  & \mbox{if} & -1 \leq \mbox{s32In} < 0 \\
                       \\
                       \mbox{s32Dump} & \mbox{if} & 0 \leq \mbox{s32In} < 1
                    \end{array}\right.
            \f]

            Division of \f$\left[0,1\right)\f$ interval into two
            sub-intervals with polynomial coefficients calculated for each
            sub-interval, are noted in table \ref tab1_GFLIB_Asin.


            \anchor tab1_GFLIB_Asin
            <table border="1" CELLPADDING="5" align = "center">
            <caption>Integer Polynomial coefficients for each interval:</caption>
            <tr>
              <th>Interval</th>
              <th>\f$a_1\f$</th>
              <th>\f$a_2\f$</th>
              <th>\f$a_3\f$</th>
              <th>\f$a_4\f$</th>
              <th>\f$a_5\f$</th>
            </tr>
            <tr align="center">
              <td style="font-weight:bold"> \f$ \left< 0 , \frac{1}{2} \right) \f$ </td>
              <td>91918582</td> <td>66340080</td> <td>9729967</td> <td>682829947</td><td>12751</td>
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{1}{2}  , 1 \right) \f$</td>
              <td>-52453538</td><td>-36708911</td><td>-15136243</td><td>-964576326</td><td>1073652175</td>
            </tr>
            </table>

            Polynomial coefficients were obtained using Matlab fitting function,
            where polynomial of 5th order was used for fitting of each respective
            sub-interval. The Matlab was used as follows:

            \par
\code
clear all
clc

number_of_range = 2;
i = 1;
range = 0;

    Range = 1 / number_of_range;
    x(i,:) = (((i-1)*Range):(1/(2^15)):((i)*Range))';
    y(i,:) = asin(x(i,:))/pi;
    p(i,:) = polyfit((x(i,:)),(y(i,:)),4);
    
    i=i+1;

    Range = 1 / number_of_range;
    x(i,:) = (((i-1)*Range):(1/(2^15)):((i)*Range))';
    y(i,:) = asin(x(i,:))/pi;
    x1(i,:) = ((x(i,:) - ((i-1)*Range)));
    x1(i,:) = 0.5 - x1(i,:);
    x2(i,:) = sqrt(x1(i,:));  
    p(i,:) = polyfit((x2(i,:)),(y(i,:)),4);
    i=i+1;

f(2,:) = polyval(p(2,:),x2(2,:));
f(1,:) = polyval(p(1,:),x(1,:));
error_1 = abs(f(2,:) - y(2,:));
max(error_1 * (2^15))
error_2 = abs(f(1,:) - y(1,:));
max(error_2 * (2^15))
plot(x(2,:),y(2,:),'-',x(2,:),f(2,:),'-',x(1,:),y(1,:),'-',x(1,:),f(1,:),'-');
coef = round(p * (2^31))
\endcode

            \anchor fig2_GFLIB_Asin
            \image latex GFLIB_Asin_Figure2.eps "asin(x) vs. GFLIB_Asin(s32In)" width=14cm
            \image html GFLIB_Asin_Figure2.jpg "asin(x) vs. GFLIB_Asin(s32In)"

            \par
            Fig.\ref fig2_GFLIB_Asin depicts a floating
            point \e arcsine function generated from Matlab and approximated value of
            \e arcsine function obtained from #GFLIB_Asin, plus their difference.
            The course of calculation accuracy as a function of input ratio can be
            observed from this figure.
            The achieved accuracy with considering the 5th order piece-wise polynomial
            approximation and described fixed point scaling, is less 1.3LSB on upper
            16-bit of the 32bit result.

@note       Output angle is normalized into range \f$\left[-0.5,0.5\right)\f$.
            The polynomial coefficients are stored in locally defined structure,
            call of which is masked by function alias #GFLIB_Asin.
            The polynomial coefficients can be calculated by the user and in
            such case full #GFLIB_AsinANSIC function call with pointer to
            the newly defined coefficients shall be used instead of the function
            alias.


@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32Input;
tFrac32 s32Angle;

void main(void)
{
    // input s32Input = 1
    s32Input  = FRAC32(1);

    // output should be 0x3FFE A1CF = 0.49995 => pi/2
    s32Angle = GFLIB_Asin(s32Input);
}
\endcode

@par Performance:
            \anchor tab2_GFLIB_Asin
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Asin function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>168/148</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>40/40</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>171/168</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>64/64</td>
            </tr>
            </table>

****************************************************************************/
tFrac32 GFLIB_AsinANSIC(tFrac32 s32In, const GFLIB_ASIN_TAYLOR_T *const pParam) 
{
    tFrac32 s32Sign,s32Dump;
    tS32    nSector = 0;
    
    s32Sign = s32In;
    s32In = F32AbsSat(s32In);
    if (s32In >= INT32_MAX/2)
    {
        s32In = F32NegSat(s32In);
        s32In = (s32In ^ INT32_MIN);
        s32In = GFLIB_Sqrt(s32In);
        nSector = 1;
    }
    
    s32In = s32In >> 15;
        
    //Polynom
    // (a1 * x) + a2
    s32Dump = ((pParam->GFLIB_ASIN_SECTOR[nSector].s32a[0] >> 16) * (s32In)) + pParam->GFLIB_ASIN_SECTOR[nSector].s32a[1];
    // ((a1 * x) + a2) * x + a3
    s32Dump = ((s32Dump >> 14) * (s32In>>2)) + pParam->GFLIB_ASIN_SECTOR[nSector].s32a[2];
    // (((a1 * x) + a2) * x + a3) * x + a4
    s32Dump = ((s32Dump >> 14) *(s32In>>2)) + pParam->GFLIB_ASIN_SECTOR[nSector].s32a[3];
    // ((((a1 * x) + a2) * x + a3) * x + a4) * x + a5
    s32Dump = ((s32Dump >> 14) *(s32In>>2)) + pParam->GFLIB_ASIN_SECTOR[nSector].s32a[4];
    
    // Sign is added to the result value 
    s32Dump = (s32Sign > 0)?s32Dump:-s32Dump;
    return (s32Dump);
}

#ifdef __cplusplus
}
#endif

/* End of file */
