/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_ControllerPIp.c
*
* @author   Roman Filka
*
* @version  1.0.12.0
*
* @date     Apr-26-2010
*
* @brief    Source file containing routines for calculation of a standard
*           parallel form Proportional-Integral controller without implemented
*           integral anti-windup.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_ControllerPIp.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
\nosubgrouping
@brief      The function calculates the parallel form of the
            Proportional-Integral (PI) regulator.

@param[in]      s32InErr    Input error signal to the controller is a 32bit
                            number normalized between \f$\left[-1,1\right)\f$.

@param[in,out]  *pParam     Pointer to controller parameters structure.

@return     The function returns 32-bit value in format 1.31, representing
            the signal to be applied to the controlled system so that the input
            error is forced to zero.

@details    The #GFLIB_ControllerPIpANSIC function, denoting ANSI-C compatible
            implementation, can be called via function alias
            #GFLIB_ControllerPIp.

            \par
            A PI controller attempts to correct the error between a measured
            process variable and a desired set-point by calculating and then
            outputting a corrective action that can adjust the process
            accordingly. The #GFLIB_ControllerPIpANSIC function calculates the
            Proportional-Integral (PI) algorithm according to the equations
            below. The PI algorithm is implemented in the parallel
            (non-interacting) form allowing user to define the P and I
            parameters independently without interaction.

            An anti-windup strategy is not implemented in this function.
            Nevertheless, the accumulator overflow is prevented by correct
            saturation of the controller output at maximal values: \f$\left[-1,1\right)\f$
            in fractional interpretation or \f$\left[-2^{31},2^{31}-1\right)\f$ in integer
            interpretation.

            The PI algorithm in the continuous time domain can be described as:
            \anchor eq1_GFLIB_ControllerPIp
            \f[
                u(t)=e(t) \cdot K_P + K_I \int^t_0 e(t) \, dt
            \f]

            where
            - \f$e(t)\f$ - input error in the continuous time domain
            - \f$u(t)\f$ - controller output in the continuous time domain
            - \f$K_P\f$ - proportional gain
            - \f$K_I\f$ - integral gain

            Eq. \ref eq1_GFLIB_ControllerPIp can be described using
            the Laplace transformation as follows:
            \anchor eq2_GFLIB_ControllerPIp
            \f[
                H(s)=\frac{U(s)}{E(s)} = K_P + K_I\frac{1}{s}
            \f]

            The proportional part of equation \ref eq2_GFLIB_ControllerPIp is
            transformed into discrete time domain simply as:
            \anchor eq3_GFLIB_ControllerPIp
            \f[
                u_P(k) = K_P \cdot e(k)
            \f]

            Transforming the integral part of equation \ref eq2_GFLIB_ControllerPIp
            into a discrete time domain using Billinear method, also known as
            trapezoidal approximation, leads to a following equation:
            \anchor eq4_GFLIB_ControllerPIp
            \f[
                u_I(k) = u_I(k-1) + e(k) \cdot \frac{K_I T_s}{2} + e(k-1) \cdot \frac{K_I T_s}{2}
            \f]

            where \f$ T_s \f$ [sec] is sampling time.

            In order to implement the discrete equation of the controller on the
            fixed point arithmetic platform, maximal values (scales) of input
            and output signals have to be known a priori. This is essential for
            correct casting of the physical signal values into fixed point
            values:
            - \f$ E^{MAX} \f$ - maximal value of controller input error signal
            - \f$ U^{MAX} \f$ - maximal value of controller output signal

            The fractional representation of both input and output
            signals, normalized between \f$\left[-1,1\right)\f$, is obtained
            as follows:
            \anchor eq5_GFLIB_ControllerPIp
            \f[
                e_{f}(k) = \frac{e(k)}{E^{MAX}}
            \f]

            \anchor eq6_GFLIB_ControllerPIp
            \f[
                u_{f}(k) = \frac{u(k)}{U^{MAX}}
            \f]

            Applying such scaling (normalization) on the proportional term of eq.
            \ref eq3_GFLIB_ControllerPIp results in:
            \anchor eq7_GFLIB_ControllerPIp
            \f[
                u_{Pf}(k) = e_{f}(k) \cdot K_{P\_sc}
            \f]
            where
            \f$  K_{P\_sc} = K_P \cdot \frac{E^{MAX}}{U^{MAX}} \f$
            is proportional gain parameter considering input/output scaling.

            Analogically, scaling the integral term of eq. \ref eq4_GFLIB_ControllerPIp
            results in:
            \anchor eq8_GFLIB_ControllerPIp
            \f[
                u_{If}(k) = u_{If}(k-1) + K_{I\_sc} \cdot e_{f}(k) + K_{I\_sc} \cdot e_{f}(k-1)
            \f]

            where

            \f$ K_{I\_sc} = \frac{K_I T_s}{2} \cdot \frac{E^{MAX}}{U^{MAX}} \f$
            is integral gain parameter considering input/output scaling.

            Sum of scaled proportional and integral terms gives complete
            equation of the controller. The problem is however, that
            either of the gain parameters  \f$ K_{P\_sc}, K_{I\_sc} \f$ can
            be out of the \f$\left[-1,1\right)\f$ range, hence can not be
            directly interpreted as a fractional value. To overcome that, it is
            necessary to scale these gain parameters using the shift values as
            follows:
            \anchor eq9_GFLIB_ControllerPIp
            \f[
                 s32PropGain = K_{P\_sc} \cdot 2^{s16PropGainShift}
            \f]

            and

            \anchor eq10_GFLIB_ControllerPIp
            \f[
                 s32IntegGain = K_{I\_sc} \cdot 2^{s16IntegGainShift}
            \f]

            where
            - \f$s16PropGain\f$ - is the scaled value of proportional gain \f$\left[-1,1\right)\f$
            - \f$s16PropGainShift\f$ - is the scaling shift for proportional gain \f$\left[-31,31\right]\f$
            - \f$s16IntegGain\f$ - is the scaled value of integral gain \f$\left[-1,1\right)\f$
            - \f$s16IntegGainShift\f$ - is the scaling shift for integral gain \f$\left[-31,31\right]\f$

@note       All controller parameters and states can be reset during declaration
            using #GFLIB_CONTROLLER_PI_P_DEFAULT macro.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32InErr;
tFrac32 s32Output;

GFLIB_CONTROLLER_PI_P_T trMyPI = GFLIB_CONTROLLER_PI_P_DEFAULT;

void main(void)
{
    // input error = 0.25
    s32InErr  = FRAC32(0.25);

    // controller parameters
    trMyPI.s32PropGain        = FRAC32(0.01);
    trMyPI.s32IntegGain       = FRAC32(0.02);
    trMyPI.s16PropGainShift    = 1;
    trMyPI.s16IntegGainShift   = 1;
    trMyPI.s32IntegPartK_1    = 0;

    // output should be 0x01EB851E
    s32Output = GFLIB_ControllerPIp(s32InErr,&trMyPI);
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_ControllerPIp
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_ControllerPIp function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>472/310</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>155/126</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>132/109</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GFLIB_ControllerPIpANSIC(tFrac32 s32InErr,
                                 GFLIB_CONTROLLER_PI_P_T *pParam)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32    s32Proportional;
    register tFrac32    s32IntegralX;
    register tFrac32    s32IntegralX1;
    register tFrac32    s32IntegralX2;
    register tFrac32    s32IntegralXshifted;
    register tFrac32    s32Out;

    // Calculation of proportional term
    s32Proportional     = F32MulSat(pParam->s32PropGain,s32InErr);

    /* Limitation of proportional term applied during calculation, to prevent
     * overflow caused by scaling shift. */
    s32Proportional     = F32ShlSat(s32Proportional,pParam->s16PropGainShift);

    // Calculation of integral term
    s32IntegralX1       = F32MulSat(pParam->s32IntegGain,s32InErr);
    s32IntegralX2       = F32MulSat(pParam->s32IntegGain,pParam->s32InK_1);
    s32IntegralX        = F32AddSat(s32IntegralX1,s32IntegralX2);
    s32IntegralXshifted = F32ShlSat(s32IntegralX,pParam->s16IntegGainShift);
    pParam->s32IntegPartK_1 = F32AddSat(s32IntegralXshifted,pParam->s32IntegPartK_1);

    /* Calculation of the controller output by adding the proportional and
     * integral terms, saturation is handled correctly here. */
    s32Out              = F32AddSat(s32Proportional, pParam->s32IntegPartK_1);

    // Updating controller input state variables
    pParam->s32InK_1    = s32InErr;

    return (s32Out);
#else
    register tFrac32    s32Proportional;
    register tFrac32    s32IntegralX;
    register tFrac32    s32IntegralX1;
    register tFrac32    s32IntegralXshifted;
    register tFrac32    s32Out;

    // Calculation of proportional term
    s32Proportional     = F32MulF16F16((tFrac16)((pParam->s32PropGain)>>16),((tFrac16)(s32InErr>>16)));

    /* Limitation of proportional term applied during calculation, to prevent
     * overflow caused by scaling shift. */
    s32Proportional     = F32ShlSat(s32Proportional,pParam->s16PropGainShift);

    // Calculation of integral term
    s32IntegralX1       = F32MulF16F16((tFrac16)(pParam->s32IntegGain>>16),(tFrac16)(s32InErr>>16));
    s32IntegralX        = F32MacSatF16F16(s32IntegralX1,((tFrac16)(pParam->s32IntegGain)>>16),(tFrac16)((pParam->s32InK_1)>>16));
    s32IntegralXshifted = F32ShlSat(s32IntegralX,pParam->s16IntegGainShift);
    pParam->s32IntegPartK_1 = F32AddSat(s32IntegralXshifted,pParam->s32IntegPartK_1);

    /* Calculation of the controller output by adding the proportional and
     * integral terms, saturation is handled correctly here. */
    s32Out              = F32AddSat(s32Proportional, pParam->s32IntegPartK_1);

    // Updating controller input state variables
    pParam->s32InK_1    = s32InErr;

    return (s32Out);
#endif
}
#ifdef __cplusplus
}
#endif

/* End of file */
