/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GFLIB_ControllerPIr.c
*
* @author   Roman Filka
*
* @version  1.0.14.0
*
* @date     Apr-26-2010
*
* @brief    Source file containing routines for calculation of a  standard
*           recurrent form of Proportional-Integral controller without
*           implemented integral anti-windup.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_ControllerPIr.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
\nosubgrouping
@brief      Recurrent form Proportional-Integral controller without integrator
            anti\--windup functionality.

@param[in]      s32InErr    Input error signal to the controller is a 32bit
                            number normalized between \f$\left[-1,1\right)\f$.

@param[in,out]  *pParam     Pointer to controller parameters structure.

@return     The function returns 32-bit value in fractional format 1.31, representing
            the signal to be applied to the controlled system so that the input
            error is forced to zero.

@details    The #GFLIB_ControllerPIrANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_ControllerPIr.

            \par
            The function #GFLIB_ControllerPIr calculates a standard
            recurrent form of Proportional\--Integral controller, without integral
            anti\--windup.

            The continuous time domain representation of the PI controller is
            defined as:
            \anchor eq1_GFLIB_ControllerPIr
            \f[
                u(t)=e(t) \cdot K_P + K_I \int^t_0 e(t) \, dt
            \f]

            The transfer function for this kind of PI controller, in continuous
            time domain, is described using the Laplace transformation as
            follows:
            \anchor eq2_GFLIB_ControllerPIr
            \f[
                H(s)=\frac{U(s)}{E(s)}=\frac{K_P+sK_I}{s}
            \f]

            Transforming equation \ref eq2_GFLIB_ControllerPIr into a discrete
            time domain leads to a following equation:
            \anchor eq3_GFLIB_ControllerPIr
            \f[
                u(k) = u(k-1) + e(k)\cdot CC1 + e(k-1)\cdot CC2
            \f]

            where
            \f$Kp\f$ is proportional gain,
            \f$Ki\f$ is integral gain,
            \f$Ts\f$ is sampling period,
            \f$u(k)\f$ is controller output,
            \f$e(k)\f$ is controller input error signal,
            \f$CC1\f$ and \f$CC2\f$ are controller coefficients calculated
            depending on used discretization method as shown in table \ref tab1_GFLIB_ControllerPIr.

            \anchor tab1_GFLIB_ControllerPIr
            <table border="1" CELLPADDING="5" align = "center">
            <caption>Calculation of coefficients CC1 and CC2 of using various discretization methods.</caption>
            <tr>
              <th></th> <th>Trapezoidal</th> <th>Bakward Rect.</th> <th>Forward Rect.</th>
            </tr>
            <tr>
              <th>\f$CC1=\f$</th> <th>\f$K_p+K_i\frac{T_s}{2}\f$</th> <th>\f$K_p+K_iT_s\f$</th> <th>\f$K_p\f$</th>
            </tr>
            <tr>
              <th>\f$CC2=\f$</th> <th>\f$-K_p+K_i\frac{T_s}{2}\f$</th> <th>\f$-K_p\f$</th> <th>\f$-K_p+K_iT_s\f$</th>
            </tr>
            </table>

            In order to implement the discrete equation of the controller
            \ref eq3_GFLIB_ControllerPIr on the fixed point arithmetic platform,
            maximal values (scales) of input
            and output signals
            - \f$ E^{MAX} \f$ - maximal value of controller input error signal
            - \f$ U^{MAX} \f$ - maximal value of controller output signal

            have to be known a priori. This is essential for
            correct casting of the physical signal values into fixed point
            values \f$\left[-1,1\right)\f$.

            Then the fractional representation \f$\left[-1,1\right)\f$ of both
            input and output signals is obtained as follows:
            \anchor eq4_GFLIB_ControllerPIr
            \f[
                e_{f}(k) = \frac{e(k)}{E^{MAX}}
            \f]

            \anchor eq5_GFLIB_ControllerPIr
            \f[
                u_{f}(k) = \frac{u(k)}{U^{MAX}}
            \f]

            The resulting controller discrete time domain equation in fixed
            point fractional representation is therefore given as:
            \anchor eq6_GFLIB_ControllerPIr
            \f[
                u_f(k)\cdot U^{MAX} =
                        u_f(k-1)\cdot U^{MAX} +
                        e_f(k)\cdot E^{MAX}\cdot CC1 +
                        e_f(k-1)\cdot E^{MAX}\cdot CC2
            \f]

            which can be rearranged into a following form:
            \anchor eq7_GFLIB_ControllerPIr
            \f[
                u_f(k) =
                        u_f(k-1) +
                        e_f(k) \cdot CC1_f +
                        e_f(k-1)\cdot CC2_f
            \f]
            where

            \anchor eq8_GFLIB_ControllerPIr
            \f[
                CC1_f = CC1 \cdot \frac{E^{MAX}}{U^{MAX}}   \qquad,\qquad
                CC2_f = CC2 \cdot \frac{E^{MAX}}{U^{MAX}}
            \f]

            are the controller coefficients adapted according to the input and
            output scale values. In order to implement both coefficients
            as fractional numbers, both \f$ CC1_f \f$ and \f$ CC2_f \f$ must
            reside in a fractional range \f$\left[-1,1\right)\f$.
            However, depending on values
            \f$ CC1, CC2, E^{MAX}, U^{MAX} \f$, calculation of
            \f$ CC1_f \f$ and \f$ CC2_f\f$ may result in values outside this
            fractional range. Therefore a scaling of \f$ CC1_f, CC2_f\f$ is
            introduced as follows:
            \anchor eq9_GFLIB_ControllerPIr
            \f[
                s32CC1sc    = CC1_f \cdot 2^{-u16NShift}
            \f]

            \anchor eq10_GFLIB_ControllerPIr
            \f[
                s32CC2sc    = CC2_f \cdot 2^{-u16NShift}
            \f]

            Introduced scaling shift \f$u16NShift\f$ is chosen such that
            both coefficients \f$s32CC1sc\f$,\f$s32CC2sc\f$ reside in range
            \f$\left[-1,1\right)\f$.
            To simplify the implementation, this scaling shift is chosen to
            be a power of 2, so the final scaling is a simple shift operation.
            Moreover the scaling shift can not be a negative number, so
            the operation of scaling is always to scale numbers with absolute
            value larger than 1 down to fit in range \f$\left[-1,1\right)\f$.
            \anchor eq11_GFLIB_ControllerPIr
            \f[
                u16NShift = \max \left(
                \mbox{ceil} \left( \frac{\log(\mbox{abs}(CC1_f))}{\log(2)} \right),
                \mbox{ceil} \left( \frac{\log(\mbox{abs}(CC2_f))}{\log(2)} \right)
                \right)
            \f]

            The final, scaled, fractional equation of reccurent PI controller
            on 32-bit fixed point platform is therefore implemented as follows:
            \anchor eq12_GFLIB_ControllerPIr
            \f[
                u_f(k) \cdot (2^{-u16NShift}) =
                    u_f(k-1) \cdot (2^{-u16NShift}) +
                    e_f(k) \cdot s32CC1sc +
                    e_f(k-1) \cdot s32CC2sc
            \f]

            where:
            - \f$u_f(k)\f$ - fractional representation \f$\left[-1,1\right)\f$ of controller output
            - \f$e_f(k)\f$ - fractional representation \f$\left[-1,1\right)\f$ of controller input
            (error)
            - \f$s32CC1sc\f$ - fractional representation \f$\left[-1,1\right)\f$ of 1st controller
            coefficient
            - \f$s32CC2sc\f$ - fractional representation \f$\left[-1,1\right)\f$ of 2nd controller
            coefficient
            - \f$u16NShift\f$ - in range \f$\left[0,31\right]\f$ - is chosen such that both
            coefficients \f$s32CC1sc\f$ and \f$s32CC2sc\f$ are in the range \f$\left[-1,1\right)\f$

@note       All controller parameters and states can be reset during declaration
            using #GFLIB_CONTROLLER_PI_R_DEFAULT macro.


@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32InErr;
tFrac32 s32Output;

GFLIB_CONTROLLER_PI_R_T trMyPI = GFLIB_CONTROLLER_PI_R_DEFAULT;

void main(void)
{
    // input error = 0.25
    s32InErr  = FRAC32(0.25);

    // controller parameters
    trMyPI.s32CC1sc           = FRAC32(0.01);
    trMyPI.s32CC2sc           = FRAC32(0.02);
    trMyPI.u16NShift          = 1;

    // output should be 0x00A3D70A
    s32Output = GFLIB_ControllerPIr(s32InErr,&trMyPI);
}
\endcode

@par Performance:
            \anchor tab2_GFLIB_ControllerPIr
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_ControllerPIr function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>260/202</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>105/89</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>90/77</td>
            </tr>
            </table>


******************************************************************************/
tFrac32 GFLIB_ControllerPIrANSIC(tFrac32 s32InErr,
                                 GFLIB_CONTROLLER_PI_R_T *pParam)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32 s32M1;
    register tFrac32 s32M2;
    register tFrac32 s32A1;

    /*
     * Implemented equation:
     * u(k) = u(k-1) + e(k)*CC1 + e(k-1)*CC2
     *
     * Calculation steps:
     * M1   = e(k)*CC1sc
     * M2   = e(k-1)*CC2sc
     * A1   = M1+M2
     * Acc  = u(k-1)+A1
     */

    // M1   = e(k)*CC1, number format Q1.31
    s32M1   = F32MulSat(s32InErr, pParam->s32CC1sc);

    // M2   = e(k-1)*CC2, number format Q1.31
    s32M2   = F32MulSat(pParam->s32InErrK1, pParam->s32CC2sc);

    // A1 = M1+M2
    s32A1   = F32AddSat(s32M1,s32M2);

    // A2 = u(k-1)+A1
    pParam->s32Acc  = F32AddSat(pParam->s32Acc,s32A1);

    // Storing input error state into the controller structure
    pParam->s32InErrK1  = s32InErr;

    // Returning de-scaled value of internal accumulator
    return(F32ShlSat(pParam->s32Acc,(tS32)pParam->u16NShift));
#else
    register tFrac32 s32A1;
    register tFrac32 s32A2;

    /*
     * Implemented equation:
     * u(k) = u(k-1) + e(k)*CC1 + e(k-1)*CC2
     *
     * Calculation steps:
     * M1   = e(k)*CC1sc
     * M2   = e(k-1)*CC2sc
     * A1   = M1+M2
     * Acc  = u(k-1)+A1
     */

    // A1   = u(k-1) + e(k)*CC1sc
    s32A1   = F32MacSatF16F16(pParam->s32Acc,
                             (tFrac16)(s32InErr>>16),
                             (tFrac16)((pParam->s32CC1sc)>>16));

    // A2   = A1 + e(k-1)*CC2sc
    s32A2   = F32MacSatF16F16(s32A1,
                             (tFrac16)((pParam->s32InErrK1)>>16),
                             (tFrac16)((pParam->s32CC2sc)>>16));

    // A2 = u(k-1)+A1
    pParam->s32Acc  = s32A2;

    // Storing input error state into the controller structure
    pParam->s32InErrK1  = s32InErr;

    // Returning de-scaled value of internal accumulator
    return(F32ShlSat(pParam->s32Acc,pParam->u16NShift));
#endif
}

#ifdef __cplusplus
}
#endif

/* End of file */
