/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
 *
 * @file    GFLIB_Cos.c
 *
 * @author  B04459
 *
 * @version 1.0.16.0
 *
 * @date    Apr-26-2010
 *
 * @brief   Source file containing routines for calculation of the cosine
 *          function.
 *
 *******************************************************************************
 *
 * Function implemented as ANSIC ISO/IEC 9899:1990, C90.
 *
 ******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
 * Includes
 ******************************************************************************/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_Cos.h"

/******************************************************************************
 * External declarations
 ******************************************************************************/

/******************************************************************************
 * Defines and macros            (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Typedefs and structures       (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Global variable definitions   (scope: module-exported)
 ******************************************************************************/

/******************************************************************************
 * Global variable definitions   (scope: module-local)
 ******************************************************************************/
const GFLIB_COSTLR_T gflibCosCoef = {188656, -5026995, 85569306, \
                                     -693598668, 1686629713};

/******************************************************************************
 * Function prototypes           (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function implementations      (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function implementations      (scope: module-exported)
 ******************************************************************************/

/**************************************************************************//*!
@brief      Cosine function - Taylor polynomial approximation.

@param[in]  *pParam     Pointer to array of Taylor coefficients. Using the
                        function alias #GFLIB_Cos, default coefficients are
                        used.
@param[in]  s32In       Input argument is a 32bit number that contains the
                        angle in radians between \f$\left[-\pi,\pi\right)\f$
                        normalized between \f$\left[-1,1\right)\f$.

@return     The function returns the cos of the input argument as fixed point
            32-bit number, normalized between \f$\left[-1,1 \right)\f$.

@details    The #GFLIB_CosANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Cos.

            \par
            The #GFLIB_Cos function provides a computational method for
            calculation of a standard trigonometric \e cosine function
            \f$\cos(x)\f$, using the 9th order Taylor polynomial approximation
            of the \e sine function. Following two equations describe chosen
            approach of calculating \e cosine function:
            \anchor eq1_GFLIB_Cos
            \f[
                \cos(s32In)=\sin(\frac{\pi}{2} + s32In)=\sin(x)
            \f]

            \anchor eq2_GFLIB_Cos
            \f[
                \sin(x)=x-\frac{x^{3}}{3!}+\frac{x^{5}}{5!}-\frac{x^{7}}{7!}+
                \frac{x^{9}}{9!} + \cdots=\sum_{n=0}^{\infty} \frac{(-1)^n x^{2n+1}}{(2n+1)!}
            \f]

            \par
            The 9th order polynomial approximation is chosen as sufficient in
            order to achieve best ratio between calculation accuracy and
            speed of calculation. Accuracy criterion is to have the output
            error within 3LSB on upper 16-bit of the 32bit result.
            Because the #GFLIB_Sin function is implemented with considering fixed
            point fractional arithmetic, all variables are normalized to fit
            into \f$\left[-1,1\right)\f$ range. Therefore in order to cast the
            fractional value of the input angle s32In \f$\left[-1,1 \right)\f$
            into correct range \f$\left[-\pi,\pi \right)\f$, the input s32In
            must be multiplied by \f$\pi\f$. So the fixed point fractional
            implementation of #GFLIB_Sin function, using 9th order Taylor
            approximation, is given as follows:
            \anchor eq2_GFLIB_Cos
            \f[
                \sin(\pi\cdot s32In)=(\pi\cdot s32In)-
                \frac{(\pi\cdot s32In)^{3}}{3!}+
                \frac{(\pi\cdot s32In)^{5}}{5!}-
                \frac{(\pi\cdot s32In)^{7}}{7!}+
                \frac{(\pi\cdot s32In)^{9}}{9!}
            \f]

            The 9th order polynomial approximation of the sine function
            has a very good accuracy in the range \f$\left[-\frac{\pi}{2},\frac
            {\pi}{2}\right)\f$ of the argument, but in wider ranges the
            calculation error quickly grows up. To minimize the error without
            having to use a high order polynomial,
            the symmetry of the sine function \f$\sin(x) = \sin(\pi - x)\f$ is
            utilized. Therefore the input argument is transferred to be always
            in the range \f$ \left[-\frac{\pi}{2},\frac{\pi}{2}\right)\f$ and
            the Taylor polynomial is calculated only in the range of the argument
            \f$\left[-\frac{\pi}{2},\frac{\pi}{2}\right)\f$.

            To make calculations more precise (because in calculations there is
            used value of input angle rounded to a 16-bit fractional number),
            the given argument value s32In (that is to be transferred into
            the range \f$\left[-0.5,0.5\right)\f$ due to the \e sine function
            symmetry) is shifted by 1 bit to the left (multiplied by 2).
            Then the value of \f$s32In^{2}\f$, used in the calculations,
            is in the range \f$\left[-1,1\right)\f$ instead of \f$\left[-0.25,0.25\right]\f$.
            Shifting the input value by 1 bit to the left will increase the
            accuracy of the calculated \f$\sin(\pi\cdot s32In)\f$ function.
            Implementing such scale on approximation function described by
            eq.\ref eq2_GFLIB_Cos, results in following:
            \anchor eq3_GFLIB_Cos
            \f[
            \begin{array}{l}
                \sin \left(s32In \cdot 2\cdot \frac{\pi}{2}\right) = \ldots\\
                \ldots = \left(
                \frac{\left(s32In \cdot 2 \cdot \frac{\pi}{2}\right)}{2}-
                \frac{\left(s32In \cdot 2 \cdot \frac{\pi}{2}\right)^{3}}{3!\cdot 2}+
                \frac{\left(s32In \cdot 2 \cdot \frac{\pi}{2}\right)^{5}}{5!\cdot 2}-
                \frac{\left(s32In \cdot 2 \cdot \frac{\pi}{2}\right)^{7}}{7!\cdot 2}+
                \frac{\left(s32In \cdot 2 \cdot \frac{\pi}{2}\right)^{9}}{9!\cdot 2}
                \right)\cdot 2
            \end{array}
            \f]

            Equation \ref eq3_GFLIB_Cos can be further rewritten into a following
            form:
            \anchor eq4_GFLIB_Cos
            \f[
                \begin{array}{rcl}
                \sin\left(s32In \cdot \pi \right) & = & (s32In \cdot 2)(a_1 + \\
                                      & + & (s32In \cdot 2)^2(a_2 + \\
                                      & + & (s32In \cdot 2)^2(a_3 + \\
                                      & + & (s32In \cdot 2)^2(a_4 + \\
                                      & + & (s32In \cdot 2)^2(a_5)))))) \cdot 2
                \end{array}
            \f]

            where \f$a_1\ldots a_5\f$ are coefficients of the approximation polynomial,
            which are calculated as follows (represented as 32-bit signed
            fractional numbers):
            \anchor eq4_GFLIB_Cos
            \f[
                \begin{array}{lclcccl}
                a_1 = & \frac{\left(\frac{\pi}{2}\right)}{2} & = & 0.785398163397448
                            & \Rightarrow &
                            \frac{\left(\frac{\pi}{2}\right)}{2} \cdot 2^{31} & =
                            0x6487ED51\\
                a_3 = & - \frac{\left(\frac{\pi}{2}\right)^{3}}{3!\cdot 2} & = & -0.322982048753123
                            & \Rightarrow & -
                            \frac{\left(\frac{\pi}{2}\right)^{3}}{3!\cdot 2} \cdot 2^{31} & =
                            0xD6A88634\\
                a_5 = & \frac{\left(\frac{\pi}{2}\right)^{5}}{5!\cdot 2} & = & 0.0398463131230835
                            & \Rightarrow &
                            \frac{\left(\frac{\pi}{2}\right)^{5}}{5!\cdot 2} \cdot 2^{31} & =
                            0x0519AF1A\\
                a_7 = & - \frac{\left(\frac{\pi}{2}\right)^{7}}{7!\cdot 2} & = & -0.00234087706765934
                            & \Rightarrow & -
                            \frac{\left(\frac{\pi}{2}\right)^{7}}{7!\cdot 2} \cdot 2^{31} & =
                            0xFFB34B4D\\
                a_9 = & \frac{\left(\frac{\pi}{2}\right)^{9}}{9!\cdot 2} & = & 8.02205923936799e^{-005}
                            & \Rightarrow &
                            \frac{\left(\frac{\pi}{2}\right)^{9}}{9!\cdot 2} \cdot 2^{31} & =
                            0x0002A0F0
                \end{array}
            \f]

            Therefore the resulting equation has the following form:
            \anchor eq5_GFLIB_Cos
            \f[
                \begin{array}{rcl}
                \sin\left(s32In \cdot \pi \right) & = & (s32In \cdot 2)(0x6487ED51 + \\
                                      & + & (s32In \cdot 2)^2(0xD6A88634 + \\
                                      & + & (s32In \cdot 2)^2(0x0519AF1A + \\
                                      & + & (s32In \cdot 2)^2(0xFFB34B4D + \\
                                      & + & (s32In \cdot 2)^2(0x0002A0F0)))))) \cdot 2
                \end{array}
            \f]

            \par
            Fig.\ref fig1_GFLIB_Cos depicts a floating
            point \e cosine function generated from Matlab and approximated value of
            \e cosine function obtained from #GFLIB_Cos, plus their difference.
            The course of calculation accuracy as a function of input angle can be
            observed from this figure.
            The achieved accuracy with considering the 9th order Taylor
            approximation and described fixed point scaling, is less 3LSB on upper
            16-bit of the 32bit result.

            \anchor fig1_GFLIB_Cos
            \image latex GFLIB_Cos_Figure1.eps "cos(x) vs. GFLIB_Cos(s32In)" width=14cm
            \image html GFLIB_Cos_Figure1.jpg "cos(x) vs. GFLIB_Cos(s32In)"

@note       Input angle (s32In) is normalized into range \f$\left[-1,1\right)\f$.
            The polynomial
            coefficients are stored in locally defined structure with five
            members, call of which is masked by function alias #GFLIB_Cos.
            The polynomial coefficients can be calculated by the user and in
            such case full #GFLIB_CosANSIC function call with pointer to
            the newly defined coefficients shall be used instead of the function
            alias.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32Angle;
tFrac32 s32Output;

void main(void)
{
    // input angle = 0.25 => pi/4
    s32Angle  = FRAC32(0.25);

    // output should be 0x5A824000
    s32Output = GFLIB_Cos(s32Angle);
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_Cos
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Cos function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>96/94</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>20/20</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>59/64</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>51/55</td>
            </tr>
            </table>

 ******************************************************************************/
tFrac32 GFLIB_CosANSIC(tFrac32 s32In, const GFLIB_COSTLR_T * const pParam)
{
    tS32 s32X;
    tS32 s32Y;
    tS32 s32Z;

    //cos(pi * s32In) = sin((0.5 + s32In)*pi)
    s32In = s32In + 0x40000000;

    //multiplication  s32In * 2 = s32Y for better accuracy is sin
    //calculation into half range
    s32Y = s32In << 1;

    //ABS//
    s32Y = F32Abs(s32Y);

    //Sine is calculation into the half and negative range
    s32X = -s32Y;

    s32X = s32X >> 16;

    //preparing for fraction multi 16 x 16 = 32
    //32 frac value is transfer into lower 16bits
    //fraction multi 16 x 16 = 32
    //x^2 = s32Y preparing for next uses
    //x = s32X
    s32Y = (s32X * s32X) >> 16;

    //------------------------------------------------------------------------//
    //x(a1 + x^2*(a3 + x^2(a5 + x^2(a7 + a9 * x^2))))
    //------------------------------------------------------------------------//
    //a7 + a9 * x^2
    s32Z = (((pParam->s32A[0] >> 14) * s32Y)) + pParam->s32A[1];

    // a5 + x^2(a7 + a9 * x^2)
    s32Z = (((s32Z >> 14) * s32Y)) + pParam->s32A[2];

    // a3 + x^2(a5 + x^2(a7 + a9 * x^2))
    s32Z = (((s32Z >> 14) * s32Y)) + pParam->s32A[3];

    // a1 + x^2*(a3 + x^2(a5 + x^2(a7 + a9 * x^2)))
    s32Z = (((s32Z >> 14) * s32Y)) + pParam->s32A[4];

    // x(a1 + x^2*(a3 + x^2(a5 + x^2(a7 + a9 * x^2))))
    s32Z = ((s32Z >> 14) * s32X);

    //sign is add into the result
    return ((s32In > 0) ? -s32Z : s32Z);
}

#ifdef __cplusplus
}
#endif

/* End of file */
