/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GFLIB_Limit.c
*
* @author    R29302
* 
* @version   1.0.11.0
* 
* @date      Apr-26-2010
* 
* @brief     Source file for two side limit function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB   
@endif
*/
#include "SWLIBS_Typedefs.h"
#include "GFLIB_Limit.h"

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
| external declarations
|-----------------------------------------------------------------------*/

/************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| typedefs and structures (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-exported)
|-----------------------------------------------------------------------*/

/************************************************************************
| global variable definitions (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function prototypes (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| function implementations (scope: module-exported)
|-----------------------------------------------------------------------*/

/**************************************************************************//*!
@brief       The function limits the input value to the defined upper and
             lower limits.

@param[in]   s32In   Input value.
@param[in]   pParam  Pointer to the limits structure.
  
@return      The input value or the upper or lower limit if the input
             value is beyond these limits.
        
@details     The function tests whether the input value is within the
             upper and lower limits. If so the input value will be returned.
             If the input value is above the upper limit, the upper limit
             will be returned.  Similarily if the input value is below the
             lower limit, the lower limit will be returned.

             The upper and lower limits can be found in the limits
             structure, supplied to the function as a pointer \c pParam.
             
@note        The function assumes that the upper limit is greater than
             the lower limit. Otherwise the function returns an undefined
             value.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 x;
tFrac32 y;
GFLIB_LIMIT_T LimitParam;

void main(void)
{
    // 0.5 = 0x40000000
    LimitParam.s32LowerLimit = FRAC32(-0.5);

    // -0.5 = 0xc0000000
    LimitParam.s32UpperLimit = FRAC32(0.5);

    // x = 0.75 = 0x60000000
    x  = FRAC32(0.75);
    // y should be 0x40000000
    y = GFLIB_Limit(x, &LimitParam);

    // x = -0.75 = 0xa0000000
    x  = FRAC32(-0.75);
    // y should be 0xc0000000
    y = GFLIB_Limit(x, &LimitParam);

    // x = 0.25 = 0x20000000
    x  = FRAC32(0.25);
    // y should be 0x20000000
    y = GFLIB_Limit(x, &LimitParam);

    return;
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_Limit
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Limit function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>26/18</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>19/16</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>18/14</td>
            </tr>
            </table>

****************************************************************************/
tFrac32 GFLIB_LimitANSIC(tFrac32 s32In,
                         const GFLIB_LIMIT_T *const pParam)
{           
    s32In = (s32In > pParam->s32UpperLimit) ? pParam->s32UpperLimit : s32In; 
    s32In = (s32In < pParam->s32LowerLimit) ? pParam->s32LowerLimit : s32In; 

    return s32In;
}

#ifdef __cplusplus
}
#endif

/* End of file */
