/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
 *
 * @file     GFLIB_Tan.c
 *
 * @author   B04459
 *
 * @version  1.0.12.0
 *
 * @date     Apr-26-2010
 *
 * @brief    Source file containing routines for calculation of the tangent
 *           function.
 *
 *******************************************************************************
 *
 * Function implemented as ANSIC ISO/IEC 9899:1990, C90.
 *
 ******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB
@endif
*/

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
 * Includes
 ******************************************************************************/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GFLIB_Tan.h"

/******************************************************************************
 * External declarations
 ******************************************************************************/

/******************************************************************************
 * Defines and macros            (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Typedefs and structures       (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Global variable definitions   (scope: module-exported)
 ******************************************************************************/

/******************************************************************************
 * Global variable definitions   (scope: module-local)
 ******************************************************************************/
const GFLIB_TANTLR_T gflibTanCoef =
{
          86016,  256000, 105668608,  105498624,  // Coef. <0;pi/32)
          92160,  786432, 107732992,  318550016,  // Coef. <pi/32;pi*2/32)
         106496, 1380352, 112027648,  537917440,  // Coef. <pi*2/32;pi*3/32)
         133120, 2091008, 118910976,  768380928,  // Coef. <pi*3/32;pi*4/32)
         174080, 3000320, 128995328, 1015683072,  // Coef. <pi*4/32;pi*5/32)
         239616, 4225024, 143284224, 1287151616,  // Coef. <pi*5/32;pi*6/32)
         348160, 5963776, 163397632, 1592680448,  // Coef. <pi*6/32;pi*7/32)
         536576, 8568832, 192008192, 1946363904  // Coef. <pi*7/32;pi*8/32)
};

/******************************************************************************
 * Function prototypes           (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function implementations      (scope: module-local)
 ******************************************************************************/

/******************************************************************************
 * Function implementations      (scope: module-exported)
 ******************************************************************************/

/**************************************************************************//*!
@brief     Tangent function - Piece-wise polynomial approximation.

@param[in]  *pParam     Pointer to array of Taylor coefficients. Using the
                        function alias #GFLIB_Tan, default coefficients are
                        used.
@param[in]  s32In       Input argument is a 32bit number that contains the
                        angle in radians between \f$\left[-\pi,\pi\right)\f$
                        normalized between \f$\left[-1,1\right)\f$.

@return     The function returns \f$\tan(\pi\cdot s32In) \f$ as fixed point
            32-bit number, normalized between \f$\left[-1,1\right)\f$.

@details    The #GFLIB_TanANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GFLIB_Tan.

            \par
            The #GFLIB_Tan function provides a computational method for
            calculation of a standard trigonometric \e tangent function
            \f$\tan(x)\f$, using the piece-wise polynomial approximation.
            Function \f$\tan(x)\f$ takes an angle and returns the ratio of two
            sides of a right triangle. The ratio is the length of the side
            opposite the angle divided by the length of the side adjacent to
            the angle. Therefore the tangent function is defined by:
            \anchor eq1_GFLIB_Tan
            \f[
                \tan(x) \equiv \frac{\sin(x)}{\cos(x)}
            \f]

            Because both \f$\sin(x)\f$ and \f$\cos(x)\f$ are defined on interval
            \f$\left[-\pi,\pi \right)\f$, function \f$\tan(x)\f$ is equal zero
            when \f$\sin(x)=0\f$ and is equal infinity when \f$\cos(x)=0\f$.
            Therefore \e tangent function has asymptotes at \f$ n\cdot\frac{\pi}{2}\f$
            for \f$ n=\pm 1,\pm 3,\pm 5 \ldots\f$. The graph of \f$\tan(x)\f$
            is shown on Fig. \ref fig1_GFLIB_Tan.

            \anchor fig1_GFLIB_Tan
            \image latex TanFigure1.eps "Course of the functionGFLIB_Tan" width=14cm
            \image html TanFigure1.jpg "Course of the functionGFLIB_Tan"

            #GFLIB_Tan function is implemented with considering fixed
            point fractional arithmetic, hence all tangent values falling
            beyond \f$\left[-1,1\right)\f$, are truncated to -1 and 1 respectively.
            This truncation is applied for angles in range
            \f$\left[-\frac{3\pi}{4},-\frac{\pi}{4}\right)\f$
            and
            \f$\left[\frac{\pi}{4},\frac{3\pi}{4}\right)\f$.
            As can be furhter seen from Fig. \ref fig1_GFLIB_Tan, tangent
            values are identical for angles in range:
            -#   \f$\left[-\frac{\pi}{4},0\right)\f$ and
                 \f$\left[\frac{3\pi}{4},\pi\right)\f$
            -#   \f$\left[-\pi,-\frac{3\pi}{4}\right)\f$ and
                 \f$\left[0,\frac{\pi}{4}\right)\f$

            Moreover it can be observed from Fig. \ref fig1_GFLIB_Tan,
            that course of the \f$\tan(x)\f$ function
            output for angles in interval 1. is identical, but with opposite sign,
            to output for angles in interval 2. Therefore the approximation of
            \e tangent function for whole defined range of input angles can be
            simplified to approximation for angles in range \f$\left[0,\frac{\pi}{4}\right)\f$,
            and then depending on the input angle the result will be negated.
            In order to increase accuracy of approximation without need for a
            high order polynomial, the interval \f$\left[0,\frac{\pi}{4}\right)\f$
            is further divided into eight equally spaced sub intervals, and
            polynomial approximation is done for each interval respectively.
            Such division results in eight sets of
            polynomial coefficients. Moreover it allows using polynomial only of 4th order
            to achieve accuracy of less than 0.5LSB (on upper 16 bits of 32 bit
            results) across full range of input angles.

            \par
            #GFLIB_Tan function uses fixed point fractional arithmetic so to
            cast the fractional value of the input angle s32In
            \f$\left[-1,1 \right)\f$
            into correct range \f$\left[-\pi,\pi \right)\f$, the fixed point input angle
            s32In must be multiplied by \f$\pi\f$. Then the fixed point fractional
            implementation of the approximation polynomial, used for calculation
            of each sub sector, is defiend as follows:
            \anchor eq2_GFLIB_Tan
            \f[
                \mbox{s32Dump} = a_1 \cdot s32In^3 + a_2 \cdot s32In^2 + a_3 \cdot s32In + a_4
            \f]

            \anchor eq3_GFLIB_Tan
            \f[
                \tan(\pi \cdot s32In) =
                    \left\{\begin{array}{lclcl}
                       \mbox{s32Dump}  & \mbox{if} & -1 \leq \mbox{s32In} < -0.5 & \mbox{or} & 0 \leq \mbox{s32In} < 0.5\\
                       \\
                       \mbox{-s32Dump} & \mbox{if} & -0.5 \leq \mbox{s32In} < 0  & \mbox{or} & 0.5 \leq \mbox{s32In} < 1
                    \end{array}\right.
            \f]

            Division of \f$\left[0,\frac{\pi}{4}\right)\f$ interval into eight
            sub-intervals with polynomial coefficients calculated for each
            sub-interval, are noted in table \ref tab1_GFLIB_Tan. Polynomial
            coefficients were obtained using Matlab fitting function,
            where polynomial of 4th order was used for fitting of each respective
            sub-interval.

            \anchor tab1_GFLIB_Tan
            <table border="1" CELLPADDING="5" align = "center">
            <caption>Integer Polynomial coefficients for each interval:</caption>
            <tr>
              <th>Interval</th>
              <th>\f$a_1\f$</th>
              <th>\f$a_2\f$</th>
              <th>\f$a_3\f$</th>
              <th>\f$a_4\f$</th>
            </tr>
            <tr align="center">
              <td style="font-weight:bold"> \f$ \left< 0 , \frac{\pi}{32} \right) \f$ </td>
              <td>86016</td> <td>256000</td> <td>105668608</td> <td>105498624</td>
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{\pi}{32}  , \frac{2\pi}{32} \right) \f$</td>
              <td>92160</td><td>786432</td><td>107732992</td><td>318550016</td>
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{2\pi}{32}  , \frac{3\pi}{32} \right) \f$</td>
              <td>106496<td>1380352<td>112027648<td>537917440
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{3\pi}{32}  , \frac{4\pi}{32} \right) \f$</td>
              <td>133120<td>2091008<td>118910976<td>768380928
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{4\pi}{32}  , \frac{5\pi}{32} \right) \f$</td>
              <td>174080<td>3000320<td>128995328<td>1015683072
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{5\pi}{32}  , \frac{6\pi}{32} \right) \f$</td>
              <td>239616<td>4225024<td>143284224<td>1287151616
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{6\pi}{32}  , \frac{7\pi}{32} \right) \f$</td>
              <td>348160<td>5963776<td>163397632<td>1592680448
            </tr>
            <tr align="center">
              <td style="font-weight:bold">\f$ \left< \frac{7\pi}{32}  , \frac{8\pi}{32} \right) \f$</td>
              <td>536576<td>8568832<td>192008192<td>1946363904
            </tr>
            </table>

            \anchor fig2_GFLIB_Tan
            \image latex GFLIB_Tan_Figure2.eps "tan(x) vs. GFLIB_Tan(s32In)" width=14cm
            \image html GFLIB_Tan_Figure2.jpg "tan(x) vs. GFLIB_Tan(s32In)"

            \par
            Fig.\ref fig2_GFLIB_Tan depicts a floating
            point \e tangent function generated from Matlab and approximated value of
            \e tangent function obtained from #GFLIB_Tan, plus their difference.
            The course of calculation accuracy as a function of input angle can be
            observed from this figure.
            The achieved accuracy with considering the 4th order piece-wise polynomial
            approximation and described fixed point scaling, is less 0.5LSB on upper
            16-bit of the 32bit result.

@note       Input angle (s32In) is normalized into range \f$\left[-1,1\right)\f$.
            The polynomial coefficients are stored in locally defined structure,
            call of which is masked by function alias #GFLIB_Tan.
            The polynomial coefficients can be calculated by the user and in
            such case full #GFLIB_TanANSIC function call with pointer to
            the newly defined coefficients shall be used instead of the function
            alias.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

tFrac32 s32Angle;
tFrac32 s32Output;

void main(void)
{
    // input angle = 0.25 => pi/4
    s32Angle  = FRAC32(0.25);

    // output should be 0x7FFFFFFF = 1
    s32Output = GFLIB_Tan(s32Angle);
}
\endcode

@par Performance:
            \anchor tab2_GFLIB_Tan
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_Tan function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>114/116</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>128/128</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>62/64</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>55/57</td>
            </tr>
            </table>

******************************************************************************/
tFrac32 GFLIB_TanANSIC(tFrac32 s32In, const GFLIB_TANTLR_T * const pParam)
{
    tFrac32 s32Dump;
    tFrac32 s32Sign;
    tFrac32 s32Stamp;
    tS32    nSector;

    // The Tan function is same on the ranges (-pi;-pi/2) and (0;pi/2)
    // And function is same on the ranges (-pi/2;0) and (pi/2;pi) too.
    // So range (-pi;-pi/2) is scaled onto range (0;pi/2) and range (pi/2;pi)
    // is scaled onto range (-pi/2;0)
    s32In = (s32In << 1)>>1;

    // Sign is saves in the s32Sign.
    // Function -(ABS(s32In))
    s32Sign = (s32In >>31);
    s32In = (s32In ^ s32Sign)-s32Sign;
    s32Stamp = s32In;

    // Simple round of the input value for better accuracy, round value 0x800
    s32In = (s32In + 0x800);

    // Polynom sector choose
    nSector =  s32In >> 26;

    // Input value is scaled on the range FRAC <0,0.5)
    s32In = s32In - (nSector << 26);
    s32In = s32In << 5;

    // Input value is scaled on the range FRAC <-0.5,0.5)
    s32In = s32In - INT32_MAX/2;

    // Input value is scaled on the range FRAC <-1,1) and value prepar for
    // multiple 16 x 16 = 32
    s32In = s32In >> 15;

    //Polynom
    s32Dump = ((pParam->GFLIB_TAN_SECTOR[nSector].s32A[0] >> 12) * (s32In>>3))
    + pParam->GFLIB_TAN_SECTOR[nSector].s32A[1];
    // (a1 * x) + a2

    s32Dump = ((s32Dump >> 13) * (s32In>>2))
    + pParam->GFLIB_TAN_SECTOR[nSector].s32A[2];
    // ((a1 * x) + a2) * x + a3

    s32Dump = ((s32Dump >> 14) *(s32In>>1))
    + pParam->GFLIB_TAN_SECTOR[nSector].s32A[3];
    // (((a1 * x) + a2) * x + a3) * x + a4

    // select undefine range where is fuction |tan(x)| higher then 1 and result
    // is repaired default value +1
    s32Dump = (s32Stamp >= (INT32_MAX/4)-(0x800))?INT32_MAX:s32Dump;

    // Sign is added to the result value
    s32Dump = (s32Sign >= 0)?s32Dump:-s32Dump;

    return (s32Dump);
}

#ifdef __cplusplus
}
#endif

/* End of file */
