/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GFLIB_VectorLimit.c
*
* @author    r29302
* 
* @version   1.0.16.0
* 
* @date      Apr-26-2010
* 
* @brief     Source file for the Vector-Limit function.
*
********************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
*******************************************************************************/
/*!
@if GFLIB_GROUP
    @addtogroup GFLIB_GROUP
@else
    @defgroup GFLIB_GROUP   GFLIB   
@endif
*/ 
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Defines.h"
#include "SWLIBS_Inlines.h"

#include "GFLIB_VectorLimit.h"
#include "GFLIB_Sqrt.h"

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************
| external declarations
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*!
  \def F32SQRT2BY2
  Local define for the GFLIB_VectorLimit function holding 32-bit sqrt(2)/2.
*/

#define F32SQRT2BY2 0x5A82799A

/*!
  \def F32MULBY2
  Local define for the GFLIB_VectorLimit performing 32-bit fractional
  multiplication with division by 2.
*/
#define F32MULBY2(x, y) ((tS32) ((((tS64) (x))*((tS64) (y)))>>32))

/*******************************************************************************
| typedefs and structures (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| global variable definitions (scope: module-exported)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| global variable definitions (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototypes (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementations (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementations (scope: module-exported)
|-----------------------------------------------------------------------------*/
/***************************************************************************//*!
@brief        Limit magnitude of the input vector.  Function alias
              #GFLIB_VectorLimit.

@param[out]     pOut    Pointer to structure of the limited output vector.
@param[in]      pIn     Pointer to structure of the input vector.
@param[in]      pParam  Pointer to parameters structure. 

@return         The function will return true (\c TRUE) if the input
                vector is being limited or false (\c FALSE) otherwise.
        
@details      The #GFLIB_VectorLimit function limits the magnitude of the
              input vector keeping its direction unchanged.  Limitation is
              performed as follows:
            
              \f[
               y_{out} =
               \left\{
               \begin{array}{rl}
                   \frac{\displaystyle y_{in}}{\displaystyle \sqrt{x_{in}^2 + y_{in}^2}} \cdot L & \mbox{if }  \sqrt{x_{in}^2 + y_{in}^2} > L \\
                   y_{in} & \mbox{if } \sqrt{x_{in}^2 + y_{in}^2} \le L
                                 \end{array}
                                 \right.
              \f]                     

              \f[
               x_{out} =
               \left\{
               \begin{array}{rl}
                   \frac{\displaystyle x_{in}}{\displaystyle \sqrt{x_{in}^2 + y_{in}^2}} \cdot L & \mbox{if }  \sqrt{x_{in}^2 + y_{in}^2} > L \\
                   x_{in} & \mbox{if } \sqrt{x_{in}^2 + y_{in}^2} \le L
                                 \end{array}
                                 \right.
              \f]                     

              Where:
              - \f$x_{in}, y_{in}\f$ and \f$x_{out}, y_{out}\f$ are the
              co-ordinates of the input and output vector, respectively
              - \f$L\f$ is the maximum magnitude of the vector

              The input vector co-ordinates are defined by the structure
              pointed to by the \c pIn parameter, and the output vector
              co-ordinates be found in the structure pointed by the \c pOut
              parameter.  The maximum vector magnitude is defined in the
              parameters structure pointed to by the \c pParam function
              parameter.

              Graphical interpretation of the function can be seen in the
              figure below.

              \anchor fig1_GFLIB_VectorLimit
              \image latex GFLIB_VectorLimit_Figure1.eps "Graphical interpretation of the GFLIB_VectorLimit function." width=10cm
              \image html GFLIB_VectorLimit_Figure1.gif "Graphical interpretation of the GFLIB_VectorLimit function." width=10cm

              If an actual limitation occurs, the function will return
              the logical true (\c TRUE), otherwise the logical false
              will be returned (\c FALSE). 

              For computational reasons, the output vector will be computed
              as zero, if the input vector magnitude is lower than
              \f$2^{-15}\f$ regardless of the set maximum magnitude of the
              input vector. The function returns the logical true
              (\c TRUE) in this case.

              Also, the 16 least significant bits of the maximum vector
              magnitude in the parameters structure, the \c pParam->s32Lim, are
              ignored.  It means that the defined magnitude must be equal to
              or greater than \f$2^{-15}\f$, otherwise the result is undefined.

@note         The function calls the square root routine (\c #GFLIB_Sqrt).
                
@warning      The maximum vector magnitude in the parameters structure,
              the \c pParam->s32Lim, must be positive and equal to or greater
              than \f$2^{-15}\f$, otherwise the result is undefined.  The
              function does not check for the valid range of \c
              pParam->s32Lim.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gflib.h"

GFLIB_VECTORLIMIT_T VectorLimitParam;
SWLIBS_2Syst In = { 0, 0};
SWLIBS_2Syst Out = { 0, 0};
tBool bLim;

void main(void)
{
    VectorLimitParam.s32Lim = 0x20000000; // 2^-2
    In.s32Arg1 = 0x20000000; // 2^-2
    In.s32Arg2 = 0x20000000; // 2^-2
    bLim = GFLIB_VectorLimit(&Out, &In, &VectorLimitParam);
    // Calculations:
    // Len = sqrt(2)*2^-2
    // Len > 2^-2, limitation required
    // xout = 2^-2/Len * Lim = sqrt(2)/2 * 2^-2
    // yout = 2^-2/Len * Lim = sqrt(2)/2 * 2^-2
    // sqrt(2)/2*2^-2 = 0x16A09E66
    // The output should be:
    // Out.s32Arg1 = 0x16a08000
    // Out.s32Arg2 = 0x16a08000
    // bLim = TRUE

    return;
}
\endcode

@par Performance:
            \anchor tab1_GFLIB_VectorLimit
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GFLIB_VectorLimit function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>590/366</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>67/64</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>65/60</td>
            </tr>
            </table>
*******************************************************************************/ 
tBool GFLIB_VectorLimitANSIC(SWLIBS_2Syst *const pOut,
                             const SWLIBS_2Syst *const pIn, 
                             const GFLIB_VECTORLIMIT_T *const pParam)
{
    tS32    s32A;
    tS32    s32B;
    tU32    u32InSquare;
    tU32    u32InLen;
    tS32    s32InLenNorm;
    tU32    u32LimitSquare;
    tS32    s32Sin;
    tS32    s32Cos;

    s32A = pIn->s32Arg1;
    s32B = pIn->s32Arg2;

    u32InSquare = (tU32) F32MULBY2(s32A, s32A);
    u32InSquare += (tU32) F32MULBY2(s32B, s32B);
    u32LimitSquare = (tU32) F32MULBY2(pParam->s32Lim, pParam->s32Lim);

    if( u32InSquare == 0x00000000 )
    {
        pOut->s32Arg1 = 0;
        pOut->s32Arg2 = 0;

        return TRUE;
    }

    if( u32InSquare > u32LimitSquare )
    {
        /* NOTE:
         *   u32InSquare is the magnitude of the input vector
         *   divided by 2.
         */
        u32InLen = (tU32)GFLIB_Sqrt((tFrac32)u32InSquare);

        /* Normalize for the best accuracy */
        s32InLenNorm = (tS32)F32Norm((tFrac32)u32InLen);
        u32InLen <<= s32InLenNorm;

        /* For the x direction */
        /* NOTE:
         *   Multiplication by sqrt(2)/2 have two purposes:
         *   (1) adjusts for the square root of the vector
         *   magnitude divided by 2 and (2) protects from
         *   division overflow.
         */
        s32A = F32MULBY2(F32SQRT2BY2, s32A);
        s32Cos = (s32A<<s32InLenNorm)/((tS32) (u32InLen>>16));
        s32Cos = s32Cos<<15;
        s32Cos = F32Mul(s32Cos, pParam->s32Lim);
        s32Cos = F32AddSat(s32Cos, s32Cos);
        pOut->s32Arg1 = s32Cos;

        /* For the y direction */
        /* NOTE:
         *   Multiplication by sqrt(2)/2 have two purposes:
         *   (1) adjusts for the square root of the vector
         *   magnitude divided by 2 and (2) protects from
         *   division overflow.
         */
        s32B = F32MULBY2(F32SQRT2BY2, s32B);
        s32Sin = (s32B<<s32InLenNorm)/((tS32) (u32InLen>>16));
        s32Sin = s32Sin<<15;
        s32Sin = F32Mul(s32Sin, pParam->s32Lim);
        s32Sin = F32AddSat(s32Sin, s32Sin);
        pOut->s32Arg2 = s32Sin;
        
        /* Limitation has occurred, return logical 1 */
        return TRUE;
    }
    else
    {
        pOut->s32Arg1 = s32A;
        pOut->s32Arg2 = s32B;

        /* Limitation has not occurred, return logical 0 */
        return FALSE;
    }

}

#ifdef __cplusplus
}
#endif

/* End of file */
