/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GMCLIB_DecouplingPMSM.c
*
* @author   Roman Filka
*
* @version  1.0.12.0
*
* @date     Apr-26-2010
*
* @brief    ANSI-C source file for #GMCLIB_DecouplingPMSM function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90. 
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
    @addtogroup GMCLIB_GROUP
@else
    @defgroup GMCLIB_GROUP   GMCLIB
@endif
*/ 

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_DecouplingPMSM.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/
/**************************************************************************//*!
@brief      This function calculates the cross-coupling voltages to eliminate 
            dq axis coupling causing non-linearity of the field oriented control.

@param[out] *pUdqDec    Pointer to the structure containing direct
                    \f$\left(u_{df_{dec}}\right)\f$
                    and quadrature \f$\left(u_{qf_{dec}}\right)\f$ components of the
                    decoupled stator voltage vector to be applied on motor
                    terminals

@param[in]  *pUdq   Pointer to the structure containing direct \f$\left(u_{df}\right)\f$
                    and quadrature \f$\left(u_{qf}\right)\f$ components of the
                    stator voltage vector generated by the current controllers

@param[in]  *pIdq   Pointer to the structure containing direct \f$\left(i_{df}\right)\f$
                    and quadrature \f$\left(i_{qf}\right)\f$ components of the
                    stator current vector measured on motor terminals

@param[in]  s32AngularVel   Rotor angular velocity in rad/sec, referred to as
                    \f$\left(\omega_{ef}\right)\f$ in detailed section of
                    the documentation

@param[in]  *pParam Pointer to the structure containing \f$k_{df}\f$ and \f$k_{qf}\f$
                    coefficients (see detailed section of the documentation)
                    and scale parameters \f$\left(k_{d\_shift}\right)\f$ and
                    \f$\left(k_{q\_shift}\right)\f$

@return     void

@details    The #GMCLIB_DecouplingPMSMANSIC function, denoting ANSI-C compatible
            source code implementation, can be called via function alias
            #GMCLIB_DecouplingPMSM.

            \par
            Quadrature phase model of PMSM motor, in synchronous reference frame,
            is very popular for field oriented control structures, because both
            controllable quantities, current and voltage, are DC values. This
            allows employing only simple controllers to force the machine
            currents into the defined states.

            \par
            The voltage equations of this model can be obtained by transforming
            the motor three phase voltage equations into quadrature phase
            rotational frame, which is aligned and rotates synchronously with rotor.
            Such transformation, after some mathematical corrections, yields the
            following set of equations, describing quadrature phase model of
            PMSM motor, in synchronous reference frame:
            \anchor eq1_GMCLIB_DecouplingPMSM
            \f[
                \left[ \begin{array}{c} u_d \\ u_q \end{array} \right] =
                \underbrace{
                R_s \left[ \begin{array}{c} i_d \\ i_q \end{array} \right] +
                \left[ \begin{array}{cc} L_d & 0\\ 0 & L_q \end{array} \right]
                \frac{\mathrm{d}}{\mathrm{d} t} \left[ \begin{array}{c} i_d \\ i_q \end{array}
                \right]
                }_{linear}+
                \underbrace{
                \omega_e \left[ \begin{array}{c} -Lq \\ Ld \end{array} \right]
                \left[ \begin{array}{c} i_q \\ i_d \end{array} \right]
                }_{cross-coupling}+
                \underbrace{
                \omega_e \psi_{pm} \left[ \begin{array}{c} 0 \\ 1 \end{array} \right]
                }_{back EMF}\\
            \f]

            It can bee seen that \ref eq1_GMCLIB_DecouplingPMSM represents
            a non-linear cross dependent system. The linear voltage components
            cover the model of phase winding, which is simplified to resistance
            in series with inductance (R-L circuit). The cross-coupling components
            represent the mutual coupling between the two phases of the quadrature
            phase model and backEMF component (visible only in q-axis voltage)
            represents the generated back EMF voltage caused by rotor rotation.

            \par
            In order to achieve dynamic torque, speed and position control,
            non-linear and backEMF components from \ref eq1_GMCLIB_DecouplingPMSM
            must be compensated. This will result in fully decoupled flux and
            torque control of the machine and simplifies PMSM motor model into two
            independent R-L circuit models as follows:
            \anchor eq2_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_d &=& R_si_d + L_d\frac{\mathrm{d} i_d}{\mathrm{d} t}\\
                u_q &=& R_si_q + L_q\frac{\mathrm{d} i_q}{\mathrm{d} t}
            \end{array}
            \f]
            Such simplification of the PMSM model also greatly simplifies
            design of both d-q current controllers.

            Therefore it is advantageous to compensate the cross-coupling terms
            in \ref eq1_GMCLIB_DecouplingPMSM,
            using a feed-forward voltages \f$\overline{u}_{dq_{comp}}\f$, given
            from \ref eq1_GMCLIB_DecouplingPMSM as follows:
            \anchor eq3_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_{d_{comp}}& = & - \omega_e \cdot L_q \cdot i_q\\
                u_{q_{comp}}& = & \omega_e \cdot L_d \cdot i_d
            \end{array}
            \f]
            The feed-forward voltages \f$\overline{u}_{dq_{comp}}\f$ are added
            to the voltages generated by the current controllers
            \f$\overline{u}_{dq}\f$, which cover the R-L model.
            Resulting voltages represents direct \f$u_{d_{dec}}\f$ and
            quadrature \f$u_{q_{decq}}\f$ components of the decoupled voltage vector
            that is to be applied on motor terminals (using a pulse width
            modulator). Back EMF voltage component is considered to be already
            compensated by external function.

            \par
            The function #GMCLIB_DecouplingPMSM calculates the cross-coupling
            voltages \f$\overline{u}_{dq_{comp}}\f$ and adds these to the input
            \f$\overline{u}_{dq}\f$ voltage vector. Because back EMF voltage
            component is considered compensated, this component is equal zero.
            Therefore calculations performed by #GMCLIB_DecouplingPMSM are derived
            from these two equations:

            \anchor eq4_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_{d_{dec}}& = & u_d + u_{d_{comp}}\\
                u_{q_{dec}}& = & u_q + u_{q_{comp}}\\
            \end{array}
            \f]
            where \f$\overline{u}_{dq}\f$ is a voltage vector calculated by the
            controllers (with already compensated back EMF component),
            \f$\overline{u}_{dq_{comp}}\f$ is the feed-forward
            compensating voltage vector described in \ref eq3_GMCLIB_DecouplingPMSM
            and \f$\overline{u}_{dq_{dec}}\f$ is the resulting decoupled voltage
            vector to be applied on motor terminals. Substituting
            \ref eq3_GMCLIB_DecouplingPMSM into \ref eq4_GMCLIB_DecouplingPMSM
            and normalizing \ref eq4_GMCLIB_DecouplingPMSM results into
            following set of equations:

            \anchor eq5_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_{df_{dec}} \cdot U_{max}& = & u_{df} \cdot U_{max} -
                    \omega_{ef} \cdot \Omega_{max} \cdot L_q \cdot i_{qf} \cdot I_{max}\\
                u_{qf_{dec}} \cdot U_{max}& = & u_{qf} \cdot U_{max} +
                    \omega_{ef} \cdot \Omega_{max} \cdot L_d \cdot i_{df} \cdot I_{max}
            \end{array}
            \f]
            where subscript \f$f\f$ denotes fractional representation of
            respective quantity, and \f$U_{max}\f$, \f$I_{max}\f$, \f$\Omega_{max}\f$
            are the maximal values (scale values) for voltage, current and
            angular velocity, respectively.

            Real quantities are converted to fractional range \f$\left[-1,1\right)\f$
            using following equations:
            \anchor eq6_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{lcllcl}
                u_{df_{dec}} &=& \frac{u_{d_{dec}}}{U_{max}}& \qquad u_{qf_{dec}}&=& \frac{u_{q_{dec}}}{U_{max}}\\
                u_{df} &=& \frac{u_d}{U_{max}}& \qquad u_{qf} &=& \frac{u_q}{U_{max}}\\
                i_{df} &=& \frac{i_d}{I_{max}}& \qquad i_{qf} &=& \frac{i_q}{I_{max}}\\
                \omega_{ef} &=& \frac{\omega_e}{\Omega_{max}} & & &
            \end{array}
            \f]

            Further rearranging \ref eq5_GMCLIB_DecouplingPMSM results in:
            \anchor eq11_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_{df_{dec}}& = & u_{df} - \omega_{ef} \cdot i_{qf}
                              \frac{L_q \cdot \Omega_{max} \cdot I_{max}}{U_{max}}
                              = u_{df} - \omega_{ef} \cdot i_{qf} \cdot k_d\\
                u_{qf_{dec}}& = & u_{qf} + \omega_{ef} \cdot i_{df}
                              \frac{L_d \cdot \Omega_{max} \cdot I_{max}}{U_{max}}
                              = u_{qf} + \omega_{ef} \cdot i_{df} \cdot k_q
            \end{array}
            \f]
            where \f$k_d\f$ and \f$k_q\f$ are coefficients calculated
            as:

            \anchor eq12_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                k_d &=& L_q \cdot \Omega_{max} \cdot \frac{I_{max}}{U_{max}}\\
                k_q &=& L_d \cdot \Omega_{max} \cdot \frac{I_{max}}{U_{max}}
            \end{array}
            \f]
            Because function #GMCLIB_DecouplingPMSM is implemented using
            the fractional arithmetics, both \f$k_d\f$ and \f$k_q\f$
            coefficients also have to be scaled
            to fit into the fractional range \f$\left[-1,1\right)\f$. For that
            purpose additional two scaling coefficients are defined as:

            \anchor eq13_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                k_{d\_shift} &=& ceil\left(\frac{\log(k_d)}{\log(2)}\right)\\
                k_{q\_shift} &=& ceil\left(\frac{\log(k_q)}{\log(2)}\right)
            \end{array}
            \f]
            Using scaling coefficients \ref eq13_GMCLIB_DecouplingPMSM, the
            fractional representation of coefficients \f$k_d\f$ and \f$k_q\f$
            from \ref eq12_GMCLIB_DecouplingPMSM is derived as follows:

            \anchor eq14_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                 k_{df} &=& k_d \cdot 2^{-k_{d\_shift}}\\
                 k_{qf} &=& k_q \cdot 2^{-k_{q\_shift}}
            \end{array}
            \f]
            Substituting \ref eq12_GMCLIB_DecouplingPMSM - \ref eq14_GMCLIB_DecouplingPMSM
            into \ref eq11_GMCLIB_DecouplingPMSM results in final form of
            the equation set, actually implemented in #GMCLIB_DecouplingPMSM
            function:

            \anchor eq15_GMCLIB_DecouplingPMSM
            \f[
            \begin{array}{ccl}
                u_{df_{dec}}& = & u_{df} - \omega_{ef} \cdot i_{qf} \cdot k_{df} \cdot 2^{k_{d\_shift}}\\
                u_{qf_{dec}}& = & u_{qf} + \omega_{ef} \cdot i_{df} \cdot k_{qf} \cdot 2^{k_{q\_shift}}
            \end{array}
            \f]
            Scaling of both equations into fractional range is done using
            multiplication by \f$2^{k_{d\_shift}}\f$, \f$2^{k_{q\_shift}}\f$,
            respectively. Therefore it is implemented as a simple left shift
            with overflow protection.

@note       All parameters can be reset during
            declaration using #GMCLIB_DECOUPLINGPMSM_DEFAULT macro.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"
#define L_D     (50.0e-3)   // Ld inductance = 50mH
#define L_Q     (100.0e-3)  // Lq inductance = 100mH
#define U_MAX   (50.0)      // scale for voltage = 50V
#define I_MAX   (10.0)      // scale for current = 10A
#define W_MAX   (2000.0)    // scale for angular velocity = 2000rad/sec

// Example of calculation of function scale coefficients (remove backslashes
// and copy paste into Matlab)
// L_D          = 50e-3;
// L_Q          = 100e-3;
// U_MAX        = 50;
// I_MAX        = 10;
// W_MAX        = 2000;
// k_d          = (L_Q*W_MAX*I_MAX/U_MAX)
// k_d_shift    = ceil(log(k_d)/log(2))     = 6
// k_df         = k_d * 2^(-k_d_shift)      = 0.625
// k_q          = (L_D*W_MAX*I_MAX/U_MAX)
// k_q_shift    = ceil(log(k_q)/log(2))     = 5
// k_qf         = k_q * 2^(-k_q_shift)      = 0.625

GMCLIB_DECOUPLINGPMSM_T trMyDec = GMCLIB_DECOUPLINGPMSM_DEFAULT;
SWLIBS_2Syst    trUsDQ;
SWLIBS_2Syst    trIsDQ;
SWLIBS_2Syst    trUsDecDQ;
tFrac32         s32We;

void main(void)
{
    // scaling coefficients of given decoupling algorithm
    trMyDec.s32Kd       = FRAC32(0.625);
    trMyDec.s16KdShift  = 6;
    trMyDec.s32Kq       = FRAC32(0.625);
    trMyDec.s16KqShift  = 5;
    trUsDQ.s32Arg1 = FRAC32(5.0/U_MAX);  // d quantity of input voltage vector 5[V]
    trUsDQ.s32Arg2 = FRAC32(10.0/U_MAX); // q quantity of input voltage vector 10[V]
    trIsDQ.s32Arg1 = FRAC32(6.0/I_MAX);  // d quantity of measured current vector 6[A]
    trIsDQ.s32Arg2 = FRAC32(4.0/I_MAX);  // q quantity of measured current vector 4[A]
    s32We          = FRAC32(100.0/W_MAX) // rotor angular velocity

    // Output should be
    // trUsDecDQ.s32Arg1 = 0xA666668C ~= -35[V]
    // trUsDecDQ.s32Arg1 = 0x66666659 ~=  40[V]
    GMCLIB_DecouplingPMSM(&trUsDecDQ,&trUsDq,&trIsDq,s32We,&trMyDec);
}
\endcode

@par Performance:
            \anchor tab1_GMCLIB_DecouplingPMSM
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_DecouplingPMSM function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>480/300</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>190/150</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>159/126</td>
            </tr>
            </table>

******************************************************************************/
void GMCLIB_DecouplingPMSMANSIC(SWLIBS_2Syst *pUdqDec,
                                const SWLIBS_2Syst *const pUdq,
                                const SWLIBS_2Syst *const pIdq,
                                tFrac32      s32AngularVel,
                                const GMCLIB_DECOUPLINGPMSM_T *const pParam)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32 s32Md;
    register tFrac32 s32Mq;

    // m_d = w_e * i_q * k_d
    s32Md    = F32ShlSat(F32Mul(F32MulSat(s32AngularVel,pIdq->s32Arg2),pParam->s32Kd),pParam->s16KdShift);

    // m_q = w_e * i_d * k_q
    s32Mq    = F32ShlSat(F32Mul(F32MulSat(s32AngularVel,pIdq->s32Arg1),pParam->s32Kq),pParam->s16KqShift);

    // u_d = u_ind - m_d
    pUdqDec->s32Arg1     = F32SubSat(pUdq->s32Arg1,s32Md);

    // u_q = u_inq + m_q
    pUdqDec->s32Arg2     = F32AddSat(pUdq->s32Arg2,s32Mq);
#else
    register tFrac32 s32Md;
    register tFrac32 s32Mq;

    // m_d = w_e * i_q * k_d
    s32Md    = F32ShlSat(F32MulSatF16F16(F16MulSat((tFrac16)(s32AngularVel>>16),(tFrac16)(pIdq->s32Arg2>>16)),(tFrac16)(pParam->s32Kd>>16)),pParam->s16KdShift);

    // m_q = w_e * i_d * k_q
    s32Mq    = F32ShlSat(F32MulSatF16F16(F16MulSat((tFrac16)(s32AngularVel>>16),(tFrac16)(pIdq->s32Arg1>>16)),(tFrac16)(pParam->s32Kq>>16)),pParam->s16KqShift);

    // u_d = u_ind - m_d
    pUdqDec->s32Arg1     = F32SubSat(pUdq->s32Arg1,s32Md);

    // u_q = u_inq + m_q
    pUdqDec->s32Arg2     = F32AddSat(pUdq->s32Arg2,s32Mq);

#endif
}

#ifdef __cplusplus
}
#endif

/* End of file */
