/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file     GMCLIB_ElimDcBusRip.c
*
* @author   B04459, R29302
*
* @version  1.0.18.0
*
* @date     Apr-26-2010
*
* @brief    Source file for the ElimDcBusRip function.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
    @addtogroup GMCLIB_GROUP
@else
    @defgroup GMCLIB_GROUP  GMCLIB
@endif
*/
#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_ElimDcBusRip.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-exported)
------------------------------------------------------------------------------*/
/**************************************************************************//*!
@brief       Elimination of the DC bus voltage ripple.

@param[out]  pOut   Pointer to structure with direct \f$\left(\alpha\right)\f$
                    and quadrature \f$\left(\beta\right)\f$ components of the
                    required stator voltage vector re-calculated such as to
                    compensate for voltage ripples on DC bus.

@param[in]   pIn    Pointer to structure with direct \f$\left(\alpha\right)\f$
                    and quadrature \f$\left(\beta\right)\f$ components of the
                    required stator voltage vector before compensation of
                    voltage ripples on DC bus.

@param[in]   pParam Pointer to the parameters structure.

@return      void

@details     The #GMCLIB_ElimDcBusRipANSIC function, denoting ANSI-C compatible
             implementation, can be called via function alias
             #GMCLIB_ElimDcBusRip.

             The #GMCLIB_ElimDcBusRip function provides a computational method
             for recalculation of direct \f$\left(\alpha\right)\f$ and
             quadrature \f$\left(\beta\right)\f$ components of the required
             stator voltage vector, such as to compensate voltage ripples on
             the DC bus of the power stage.

             Considering a cascaded type structure of the control system in a
             standard motor control application, the required voltage vector,
             to be applied on motor terminals, is generated by a set of
             controllers (usually P, PI or PID) only with knowledge of maximal
             value of DC bus voltage. Amplitude and phase of the required
             voltage vector are then used by the pulse width modulator (PWM)
             for generation of appropriate duty\--cycles for power inverter
             switches. Obviously the amplitude of the generated phase voltage
             (averaged across one switching period) does not only depend on the
             actual on/off times of the given phase switches and maximal value
             of the DC bus voltage.  The actual amplitude of phase voltage is
             directly affected also by actual value of the available DC bus
             voltage. Therefore any variations in amplitude of actual DC bus
             voltage must be accounted for by modifying amplitude of required
             voltage so that the output phase voltage remains unaffected.

             For better understanding let's consider following two simple examples:

             \par
             Example 1:
             - amplitude of required phase voltage \f$U_{req}=50\f$[V]
             - maximal amplitude of DC bus voltage \f$U_{DC\_BUS\_MAX}=100\f$[V]
             - actual amplitude of DC bus voltage \f$U_{DC\_BUS\_ACTUAL}=100\f$[V]
             - voltage to be applied to PWM modulator to generate \f$U_{req}=50\f$[V]
             on the inverter phase output:
             \anchor eq1_GMCLIB_ElimDcBusRip
             \f[
                  U_{req\_new} = \frac{U_{req}\cdot U_{DC\_BUS\_MAX}}{U_{DC\_BUS\_ACTUAL}} = 50V
             \f]

             \par
             Example 2:
             - amplitude of required phase voltage \f$U_{req}=50\f$[V]
             - maximal amplitude of DC bus voltage \f$U_{DC\_BUS\_MAX}=100\f$[V]
             - actual amplitude of DC bus voltage \f$U_{DC\_BUS\_ACTUAL}=90\f$[V]
             - voltage to be applied to PWM modulator to generate \f$U_{req}=50\f$[V]
             on the inverter phase output:
             \anchor eq2_GMCLIB_ElimDcBusRip
             \f[
                  U_{req\_new} = \frac{U_{req}\cdot U_{DC\_BUS\_MAX}}{U_{DC\_BUS\_ACTUAL}} = 55.5V
             \f]

             \par
             The imperfections of the DC bus voltage are compensated by the
             modification of amplitudes of the direct-\f$\alpha\f$ and the
             quadrature-\f$\beta\f$ components of the stator reference voltage
             vector.  The following formulas are used:

              - for the \f$\alpha\f$-component:

             \anchor eq3_GMCLIB_ElimDcBusRip
             \f[
             u_\alpha^* = \left\{\begin{array}{cc}
                        \frac{\displaystyle s32ModIndex\cdot u_\alpha}{\displaystyle s32ArgDcBusMsr/2}  & \mbox{if } abs\left(s32ModIndex\cdot u_\alpha\right) < \frac{s32ArgDcBusMsr}{2}\\
                        sign\left(u_\alpha \right)  & \mbox{otherwise}
                      \end{array}\right.
             \f]

              - for the \f$\beta\f$-component:

             \anchor eq4_GMCLIB_ElimDcBusRip
             \f[
             u_\beta^* = \left\{\begin{array}{cc}
                        \frac{\displaystyle s32ModIndex\cdot u_\beta}{\displaystyle s32ArgDcBusMsr/2}  & \mbox{if } abs\left(s32ModIndex\cdot u_\beta\right) < \frac{s32ArgDcBusMsr}{2}\\
                        sign\left(u_\beta \right)  & \mbox{otherwise}
                      \end{array}\right.
             \f]

              where: \f$s32ModIndex\f$ is the inverse modulation index,
              \f$s32ArgDcBusMsr\f$ is the measured DC bus voltage, the \f$u_\alpha\f$ and
              \f$u_\beta\f$ are the input voltages, and the \f$u_\alpha^*\f$
              and \f$u_\beta^*\f$ are the output duty-cycle ratios.

              The \f$s32ModIndex\f$ and \f$s32ArgDcBusMsr\f$ are supplied to the
              function within the parameters structure through its members.  The
              \f$u_\alpha\f$, \f$u_\beta\f$ correspond to respectively the \c
              s32Arg1 and \c s32Arg2 members of the input structure, and the
              \f$u_\alpha^*\f$ and \f$u_\beta^*\f$ to respectively the \c
              s32Arg1 and \c s32Arg2 members of the  output structure.

              It should be noted that although the modulation index (see the
              parameters structure, the \c s32ModIndex member) is assumed
              to be equal to or greater than zero, the possible values
              are restricted to values resulting from the use of Space Vector
              Modulation techniques.

              \par
              In order to correctly handle the discontinuity at
              \f$s32ArgDcBusMsr\f$ approaching \f$0\f$ and for efficiency
              reasons, the function will assign \f$0\f$ to the output duty cycle
              ratios, if the \f$s32ArgDcBusMsr\f$ is below the threshold of
              \f$2^{-15}\f$. In other words the 16 least significant bits of of
              the \c s32DcBusMsr are ignored.  Also the computed output of the
              \f$u_\alpha*\f$ and \f$u_\beta^*\f$ components may have the 16
              least significant bits inaccurate.

@note         Both the inverse modulation index \c pIn\-->s32ModIndex and the
              measured DC bus voltage \c pIn\-->s32DcBusMsr must be equal to or
              greater then 0, otherwise the results are undefined.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"

#define U_MAX   (36.0)      // Voltage scale

SWLIBS_2Syst tr32InVoltage;
SWLIBS_2Syst tr32OutVoltage;

GMCLIB_ELIMDCBUSRIP_T trMyElimDCB = GMCLIB_ELIMDCBUSRIP_DEFAULT;

void main(void)
{
    // Input voltage vector 15V @ angle 30deg
    // alpha component of input voltage vector = 12.99[V]
    // beta component of input voltage vector  = 7.5[V]
    tr32InVoltage.s32Arg1       = FRAC32(12.99/U_MAX);
    tr32InVoltage.s32Arg2       = FRAC32(7.5/U_MAX);

    // inverse modulation coefficient for standard space vector modulation
    trMyElimDCB.s32ModIndex     = FRAC32(0.866025403784439);

    // value of "measured" DC bus voltage 17V
    // When used in final application this randomly chosen value shall be
    // replaced by actual value of measured voltage on DC bus terminals of
    // the power inverter
    trMyElimDCB.s32ArgDcBusMsr  = FRAC32(17.0/U_MAX);

    // output should be
    // alpha component of the output voltage vector:
    //    (12.99/36)*0.8660/(17.0/36/2) = 1.3235 -> 1.0 -> 0x7fffffff
    // beta component of the output voltage vector :
    //    (7.5/36)*0.8660/(17.0/36/2) = 0.7641 -> 0x61cf5770
    // due to 16-bit accuracy the result will be 0x61cf8000

    GMCLIB_ElimDcBusRip(&tr32OutVoltage,&tr32InVoltage,&trMyElimDCB);

    return;
}
\endcode

@par Performance:
            \anchor tab1_GMCLIB_ElimDcBusRip
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_ElimDcBusRip function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>378/340</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>251/243</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>163/156</td>
            </tr>
            </table>

******************************************************************************/
void GMCLIB_ElimDcBusRipANSIC(SWLIBS_2Syst *pOut,
                              const SWLIBS_2Syst *const pIn,
                              const GMCLIB_ELIMDCBUSRIP_T *const pParam)
{
	register tFrac32 s32Alpha;
	register tFrac32 s32Beta;
	register tFrac32 s32AlphaAbs;
	register tFrac32 s32BetaAbs;
	register tFrac32 s32DcBusBy2;
    register tFrac16 s16DcBus;
    register tS32 s32DcBusBy2Norm;
    register tFrac32 aux;

    s32DcBusBy2 = pParam->s32ArgDcBusMsr>>1;
    s16DcBus = (tS16) (s32DcBusBy2>>15);

    /* The condition causes, that for any value below the fractional
     * 2^-15 threshold, the output alpha, beta will be zero.  In this
     * way the continuity at Udc approaching 0 is handled.
     */
    if((s16DcBus) == 0)
    {
        pOut->s32Arg1 = 0;
        pOut->s32Arg2 = 0;
        return;
    }

    /* NOTE: s32ModIndex can be 0 or positive, which means that there is no
     * danger of saturation (only if both multiplication arguments are -1).
     * Therefore F32Mul is used instead of F32MulSat.
     */
    s32Alpha = F32Mul(pParam->s32ModIndex, pIn->s32Arg1);
    s32Beta = F32Mul(pParam->s32ModIndex, pIn->s32Arg2);
    s32AlphaAbs = F32Abs(s32Alpha);
    s32BetaAbs = F32Abs(s32Beta);

    /* Normalization to minimize error resulting from dropped off bits
     * while computing s16DcBus.
     */
    s32DcBusBy2Norm = F32Norm(s32DcBusBy2);
    s16DcBus = (tS16) (s32DcBusBy2>>(16 - s32DcBusBy2Norm));

    /* NOTE: for both alpha and beta, overflow may occur at division, if
     * s32Alpha or s32Beta are very close to s32DcBusBy2. For example if
     * s32Alpha = 0x40000000 (ModIndex=0.5)*0x7ffffffc (alpha), s32DcBusBy2 =
     * 0x3fffffff then s32Alpha/s16DcBus = 0x00008000. Therefore it is
     * necessary to saturate the division result.
     */

    if (s32AlphaAbs < s32DcBusBy2)
    {
        s32Alpha = s32Alpha<<s32DcBusBy2Norm;
        aux = (s32Alpha/s16DcBus)<<14;
        pOut->s32Arg1 = F32AddSat(aux, aux);
    }
    else
    {
        pOut->s32Arg1 = (pIn->s32Arg1 < 0) ? INT32_MIN:INT32_MAX;
    }

    if (s32BetaAbs < s32DcBusBy2)
    {
        s32Beta = s32Beta<<s32DcBusBy2Norm;
        aux = (s32Beta/s16DcBus)<<14;
        pOut->s32Arg2 = F32AddSat(aux, aux);
    }
    else
    {
        pOut->s32Arg2 = (pIn->s32Arg2 < 0) ? INT32_MIN:INT32_MAX;
    }

    return;
}

#ifdef __cplusplus
}
#endif

/* End of file */
