/******************************************************************************
*
* (c) Copyright 2009, Freescale & STMicroelectronics
*
***************************************************************************//*!
*
* @file      GMCLIB_ParkInv.c
*
* @author    Roman Filka
*
* @version   1.0.14.0
*
* @date      Apr-26-2010
*
* @brief     Source file for Inverse Park Transformation algorithm.
*
*******************************************************************************
*
* Function implemented as ANSIC ISO/IEC 9899:1990, C90.
*
******************************************************************************/
/*!
@if GMCLIB_GROUP
    @addtogroup GMCLIB_GROUP
@else
    @defgroup GMCLIB_GROUP  GMCLIB
@endif
*/
#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
| Includes
------------------------------------------------------------------------------*/
#include "SWLIBS_Typedefs.h"
#include "SWLIBS_Inlines.h"
#include "SWLIBS_Defines.h"

#include "GMCLIB_ParkInv.h"

/******************************************************************************
| external declarations
------------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| typedefs and structures (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-exported)
------------------------------------------------------------------------------*/

/******************************************************************************
| global variable definitions (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function prototypes (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-local)
------------------------------------------------------------------------------*/

/******************************************************************************
| function implementations (scope: module-exported)
------------------------------------------------------------------------------*/
/**************************************************************************//*!
@brief          Inverse Park Trasformation algorithm implementation.

@param[out]     *pOut       Pointer to structure containing data of two-phase
                            stationary orthogonal system
                            \f$ \left( \alpha-\beta \right) \f$.
@param[in]      *pInAngle   Pointer to structure where the values of sine and
                            cosine of the rotor position are stored.
@param[in]      *pIn        Pointer to structure containing data of two-phase
                            rotational orthogonal system
                            \f$ \left( d-q \right) \f$.


@return         void

@details        The #GMCLIB_ParkInvANSIC function, denoting ANSI-C compatible
                implementation, can be called via function alias
                #GMCLIB_ParkInv.

                The #GMCLIB_ParkInv function calculates the Inverse Park
                Transformation, which transforms quantities (flux, voltage,
                current) from the two-phase \f$\left(d-q\right)\f$ rotational orthogonal
                coordinate system to the two-phase \f$\left(\alpha - \beta\right)\f$
                stationary orthogonal coordinate system, according to these
                equations:
                \anchor eq1_GMCLIB_ParkInv
                \f[
                \alpha = \cos(\theta_e) \cdot d - \sin(\theta_e) \cdot q
                \f]

                \anchor eq2_GMCLIB_ParkInv
                \f[
                \beta = \sin(\theta_e) \cdot d + \cos(\theta_e) \cdot q
                \f]

@note           The inputs and the outputs are normalized to fit in range
                \f$\left[-1,1\right)\f$.

@par Reentrancy:
            The function is reentrant.

@par Code Example:
\code
#include "gmclib.h"

SWLIBS_2Syst tr32Angle;
SWLIBS_2Syst tr32Dq;
SWLIBS_2Syst tr32AlBe;

void main(void)
{
    // input angle sin(60) = 0.866025403
    // input angle cos(60) = 0.5
    tr32Angle.s32Arg1  = FRAC32(0.866025403);
    tr32Angle.s32Arg2  = FRAC32(0.5);

    // input d  = 0.123
    // input q  = 0.654
    tr32Dq.s32Arg1  = FRAC32(0.123);
    tr32Dq.s32Arg2  = FRAC32(0.654);

    // output should be
    // tr32AlBe.s32Arg1 ~ alpha = 0xBF601273
    // tr32AlBe.s32Arg2 ~ beta  = 0x377D9EE4
    GMCLIB_ParkInv(&tr32AlBe,&tr32Angle,&tr32Dq);
}
\endcode

@par Performance:
            \anchor tab1_GMCLIB_ParkInv
            <table border="1" CELLPADDING="5" align = "center">
            <caption>#GMCLIB_ParkInv function performance</caption>
            <tr>
              <th>Code size [bytes] GHS/CW</th> <td>270/178</td>
            </tr>
            <tr>
              <th>Data size [bytes] GHS/CW</th> <td>0/0</td>
            </tr>
            <tr>
              <th>Execution clock cycles max [clk] GHS/CW</th> <td>142/98</td>
            </tr>
            <tr>
              <th>Execution clock cycles min [clk] GHS/CW</th> <td>115/75</td>
            </tr>
            </table>

****************************************************************************/
void GMCLIB_ParkInvANSIC (SWLIBS_2Syst *pOut,
                          const SWLIBS_2Syst *const pInAngle,
                          const SWLIBS_2Syst *const pIn)
{
#ifdef USE_FRAC32_ARITHMETIC
    register tFrac32    s32D;
    register tFrac32    s32Q;
    register tFrac32    s32Sin;
    register tFrac32    s32Cos;
    register tFrac32    s32Temp;
    register tFrac32    s32Sat;

    s32D    = (tFrac32)((pIn->s32Arg1)>>1);
    s32Q    = (tFrac32)((pIn->s32Arg2)>>1);
    s32Sin  = (tFrac32)((pInAngle->s32Arg1)>>1);
    s32Cos  = (tFrac32)((pInAngle->s32Arg2)>>1);

    pOut->s32Arg1   = F32Sub(F32Mul(s32Cos,s32D),
                             F32Mul(s32Sin,s32Q));

    pOut->s32Arg2   = F32Add(F32Mul(s32Cos,s32Q),
                             F32Mul(s32Sin,s32D));

    s32Temp  = pOut->s32Arg1<<2;
    s32Sat   = (pOut->s32Arg1<0) ? INT32_MIN : INT32_MAX;
    pOut->s32Arg1  = ((s32Temp^pOut->s32Arg1)<0)? s32Sat : s32Temp;

    s32Temp  = pOut->s32Arg2<<2;
    s32Sat   = (pOut->s32Arg2<0) ? INT32_MIN : INT32_MAX;
    pOut->s32Arg2  = ((s32Temp^pOut->s32Arg2)<0)? s32Sat : s32Temp;

#else
    register tFrac16    s16D;
    register tFrac16    s16Q;
    register tFrac16    s16Sin;
    register tFrac16    s16Cos;
    register tFrac32    s32Temp;
    register tFrac32    s32Sat;

    s16D    = (tFrac16)((pIn->s32Arg1)>>16);
    s16Q    = (tFrac16)((pIn->s32Arg2)>>16);
    s16Sin  = (tFrac16)((pInAngle->s32Arg1)>>16);
    s16Cos  = (tFrac16)((pInAngle->s32Arg2)>>16);

    pOut->s32Arg1   = ((tFrac32)(F16SubSat(F16MulSat(s16Cos,s16D),
                                           F16MulSat(s16Sin,s16Q)))<<16);

    pOut->s32Arg2   = ((tFrac32)(F16AddSat(F16MulSat(s16Cos,s16Q),
                                           F16MulSat(s16Sin,s16D)))<<16);
#endif
}
#ifdef __cplusplus
}
#endif

/* End of file */
