/*
 * main implementation: use this 'C' sample to create your own application
 *
 */

#include "derivative.h" /* include peripheral declarations */
/******************************************************************************
* User includes
******************************************************************************/
#include "etpu_set\etpu\pwmm\etpu_pwmm.h"  /* eTPU PWMM function API */
#include "etpu_set\etpu_gct.h"   /* eTPU configuration */
#include "MCLIB\gmclib.h" /* General Motor Control Library */
#include "MCLIB\gdflib.h" /* General Motor Control Library */
#include "FreeMASTER\MPC57xx\freemaster.h" /* FreeMASTER Communication driver */

/******************************************************************************
* Defines
******************************************************************************/
#define PWMM_ENABLE       1
#define PWMM_DISABLE      2
#define PWMM_NO_OPERATION 0
#define DEBUG_PIN_0 145
#define ETPU_PWMM_IRQ_SOURCE	749
#define ETPU_PWMM_IRQ_PRIORITY	1
/******************************************************************************
* Global function prototypes
******************************************************************************/
extern void xcptn_xmpl(void);
extern void MC_MODE_Init(void);
extern void etpu_pwmm_missing_update_irq(void);

/******************************************************************************
* Global data
******************************************************************************/
uint32_t IntcIsrVectorTable[];
uint32_t etpu_engine_load;
uint8_t  pwmm_enable_disable = PWMM_NO_OPERATION;

/* Motor Voltage Vector - global variables to be updated from a FreeMASTER */
 int24_t mvv_angle     = 0;
 int24_t mvv_angle_inc = SFRACT24(0.01*1/180);
 int24_t mvv_amplitude = SFRACT24(0.8);

/******************************************************************************
* Local function Prototypes
******************************************************************************/
uint32_t get_etpu_load_a(void);
void LINFlex_init(void);
void SIUL2_Configure(void);
void my_interrupt_init(uint32_t source, uint8_t priority);
void etpu_pwmm_update(void);

/******************************************************************************
* FreeMASTER TSA tables
******************************************************************************/
/*
 * With TSA enabled, the user describes the global and static variables using
 * so-called TSA tables. There can be any number of tables defined in
 * the project files. Each table does have the identifier which should be
 * unique across the project.
 *
 * Note that you can declare variables as Read-Only or Read-Write.
 * The FreeMASTER driver denies any write access to the Read-Only variables
 * when TSA_SAFETY is enabled.
 */
FMSTR_TSA_TABLE_BEGIN(fmstr_tsa_table_etpu_load)
    FMSTR_TSA_RO_VAR(etpu_engine_load, FMSTR_TSA_UINT32)
FMSTR_TSA_TABLE_END()

FMSTR_TSA_TABLE_BEGIN(fmstr_tsa_table_pwm_mvv)
    FMSTR_TSA_RO_VAR(mvv_angle,     FMSTR_TSA_SINT32)
    FMSTR_TSA_RW_VAR(mvv_angle_inc, FMSTR_TSA_SINT32)
    FMSTR_TSA_RW_VAR(mvv_amplitude, FMSTR_TSA_SINT32)
FMSTR_TSA_TABLE_END()

FMSTR_TSA_TABLE_BEGIN(fmstr_tsa_table_etpu_pwmm)
    FMSTR_TSA_RW_VAR(pwmm_enable_disable, FMSTR_TSA_UINT8)
    FMSTR_TSA_RW_VAR(pwmm_config, FMSTR_TSA_USERTYPE(struct pwmm_config_t))
    FMSTR_TSA_RW_VAR(pwmm_inputs, FMSTR_TSA_USERTYPE(struct pwmm_inputs_t))
    FMSTR_TSA_RO_VAR(pwmm_states, FMSTR_TSA_USERTYPE(struct pwmm_states_t))

    FMSTR_TSA_STRUCT(struct pwmm_config_t)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, update_option, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, modulation, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, mode, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, period, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, dead_time, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, min_pulse_width, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_config_t, update_time, FMSTR_TSA_UINT32)

    FMSTR_TSA_STRUCT(struct pwmm_inputs_t)
    FMSTR_TSA_MEMBER(struct pwmm_inputs_t, input_a, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_inputs_t, input_b, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_inputs_t, input_c, FMSTR_TSA_UINT32)

    FMSTR_TSA_STRUCT(struct pwmm_states_t)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, sector, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, state, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, update, FMSTR_TSA_UINT8)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, duty_a, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, duty_b, FMSTR_TSA_UINT32)
    FMSTR_TSA_MEMBER(struct pwmm_states_t, duty_c, FMSTR_TSA_UINT32)
FMSTR_TSA_TABLE_END()

/*
 * This list describes all TSA tables which should be exported to the
 * FreeMASTER application.
 */
FMSTR_TSA_TABLE_LIST_BEGIN()
    FMSTR_TSA_TABLE(fmstr_tsa_table_etpu_load)
    FMSTR_TSA_TABLE(fmstr_tsa_table_pwm_mvv)
    FMSTR_TSA_TABLE(fmstr_tsa_table_etpu_pwmm)
FMSTR_TSA_TABLE_LIST_END()

/******************************************************************************
* Interrupt handlers
******************************************************************************/
void etpu_pwmm_missing_update_irq(void)
{
	    static uint32_t missing_update_cnt = 0;

	    missing_update_cnt++;
	    /* Clear the channel interrupt flag */
	    fs_etpu_clear_chan_interrupt_flag(ETPU_PWMM_MASTER_CH);
}

int main(void)
{
	  /* MCU initialization */
	  MC_MODE_Init();
	  /* Configure and Enable Interrupts */
	  xcptn_xmpl ();
	  /* Configure custom interrupt */
	  my_interrupt_init(ETPU_PWMM_IRQ_SOURCE,ETPU_PWMM_IRQ_PRIORITY);
	  /* Initialize LINFlex, UART */
	  LINFlex_init();
	  /* SIUL2 pin configuration */
	  SIUL2_Configure();
	  /* FreeMASTER initialization */
	  FMSTR_Init();
	  /* Initialize and run the eTPU */
	  my_system_etpu_init();
	  my_system_etpu_start();
	  get_etpu_load_a();

	  pwmm_enable_disable = PWMM_ENABLE;

	  /* Loop forever */
	  for(;;) {

		  	 /* FreeMASTER processing on background */
		  	 FMSTR_Poll();
			 /* PWMM Enable/Disable */
			 if(pwmm_enable_disable == PWMM_ENABLE)
			 {
				 fs_etpu_pwmm_enable(&pwmm_instance);
				 pwmm_enable_disable = PWMM_NO_OPERATION;
			 }
			 else if(pwmm_enable_disable == PWMM_DISABLE)
			 {
				 fs_etpu_pwmm_disable(&pwmm_instance);
				 pwmm_enable_disable = PWMM_NO_OPERATION;
			 }
			 /* PWMM update */
			 fs_etpu_pwmm_get_states(&pwmm_instance, &pwmm_states);
			 if(pwmm_states.update == FS_ETPU_PWMM_UPDATE_APPLIED)
			 {
				 etpu_pwmm_update();
			 }
	  }
}

/***************************************************************************
*
* @brief   function to perform eTPU PWMM update.
*
* @return  N/A
*
****************************************************************************/

void etpu_pwmm_update(void)
{
		SWLIBS_2Syst f32_sin_cos;
	    SWLIBS_2Syst f32_alpha_beta;
	    SWLIBS_2Syst f32_d_q;

	    /* Debug pin ON */
	    SIUL2.GPDO[DEBUG_PIN_0].B.PDO = 1;

	    /* PWM inputs update */
	    mvv_angle += mvv_angle_inc;

	    switch(pwmm_config.modulation)
	  	    {
	  	       case FS_ETPU_PWMM_MODULATION_SINE_TABLE:
	  	    	pwmm_inputs.input_a = mvv_angle;
	  	    	pwmm_inputs.input_b = mvv_amplitude;
	  	    	break;

	  	       case FS_ETPU_PWMM_MODULATION_SVMSTD:
	  	       case FS_ETPU_PWMM_MODULATION_SVMU0N:
	  	       case FS_ETPU_PWMM_MODULATION_SVMU7N:
	  	       case FS_ETPU_PWMM_MODULATION_PWMICT:
	  	     	/* Sin and Cos of current motor angle */
	  	     	f32_sin_cos.s32Arg1 = GFLIB_Sin(mvv_angle<<8);
	  	     	f32_sin_cos.s32Arg2 = GFLIB_Cos(mvv_angle<<8);

	  	     	/* DQ coordinates of current Motor Voltage Vector */
	  	     	f32_d_q.s32Arg1 = 0;
	  	     	f32_d_q.s32Arg2 = -mvv_amplitude;

	  	     	/* Inverse Park transformation to get motor Alpha-Beta coordinates */
	  	     	GMCLIB_ParkInv(&f32_alpha_beta, &f32_sin_cos, &f32_d_q);

	  	       pwmm_inputs.input_a = f32_alpha_beta.s32Arg1;
	  	       pwmm_inputs.input_b = f32_alpha_beta.s32Arg2;

	  	       break;
	  	     }

	  	  /* PWMM update */
	  	  fs_etpu_pwmm_config(&pwmm_instance, &pwmm_config, &pwmm_inputs);

	  	  /* FreeMASTER Recorder */
	  	  FMSTR_Recorder();

	  	  /* eTPU load */
	  	  etpu_engine_load = get_etpu_load_a();

	  	  /* debug pin OFF */
	  	  SIUL2.GPDO[DEBUG_PIN_0].B.PDO = 0;

}

/***************************************************************************
*
* @brief    Init LINFlex #2 to 115kbd @ 80MHz, for FreeMASTER.
*
* @return  N/A
*
****************************************************************************/
void LINFlex_init(void)
{
	    /* UART LINFlexM #2 Configuration */
	    LINFlexD_M1.LINCR1.B.SLEEP = 0x0; // exit sleep mode
	    LINFlexD_M1.LINCR1.B.INIT = 0x1;  // Put LINFlex hardware in init mode
	    LINFlexD_M1.UARTCR.B.UART = 0x1;  // Enable UART mode
	    LINFlexD_M1.UARTCR.B.WL0 = 0x1; // 8 bit Word Length in UART mode
	    LINFlexD_M1.UARTCR.B.WL1 = 0x0; // 8 bit Word Length in UART mode
	    LINFlexD_M1.UARTCR.B.PCE = 0x0; // no Parity
	    LINFlexD_M1.UARTCR.B.TxEn = 0x1;  // Enable transmission of data now
	    LINFlexD_M1.UARTCR.B.TDFL_TFC = 0x0;  // Transmitter Data Field Length/TX FIFO Counter = 1 byte
	    /* Baud rate settings to 115200 */
	    LINFlexD_M1.LINIBRR.B.IBR = 0x36; //0x2B; // LDIV Mantissa baud rate divider component (Integer Baud Rate Register - IBR)
	    LINFlexD_M1.LINFBRR.B.FBR = 0x6;    // Fraction baud rate divider component (Fractional Baud Rate Register - FBR)
	    LINFlexD_M1.LINCR1.B.INIT = 0x0;    // set LINFlex hardware to normal mode
}

/***************************************************************************//*!
*
* @brief   SIUL2 configuration for eTPU input/output
*
* @return  N/A
*
******************************************************************************/
void SIUL2_Configure(void)
{
    /* SIUL2 configuration for LINFlex M1    Port Pin Number  MSCR_SSS  Function  Description */
    SIUL2.MSCR0_255[12].R = 0x32810009;   // PA[12] 12  0000_1001 LINM1TX   LINFlexD_M1 Transmit Data Output
    SIUL2.MSCR0_255[11].R = 0x00080000;   // PA[11] 11  0000_0000 GPIO	 General Purpose
    SIUL2.MSCR512_995[245].R = 0x00000001;// PA[11] 757 0000_0001 LINM1RX   LINFlexD_M1 Receive Data Input

    /* Setup eTPU A outputs */
    SIUL2.MSCR0_255[101].R = 0x02380001; // eTPU A channel 11 = PG[5] - PWM_PHASEA_BASE
    SIUL2.MSCR0_255[52].R = 0x02380001; // eTPU A channel 12 = PD[4] - PWM_PHASEA_COMPL
    SIUL2.MSCR0_255[107].R = 0x02380001; // eTPU A channel 17 = PG[11] - PWM_PHASEB_BASE
    SIUL2.MSCR0_255[6].R = 0x02380001; // eTPU A channel 18 = PA[6] - PWM_PHASEB_COMPL
    SIUL2.MSCR0_255[105].R = 0x02380001; // eTPU A channel 15 = PG[9] - PWM_PHASEC_BASE
    SIUL2.MSCR0_255[106].R = 0x02380001; // eTPU A channel 16 = PG[10] - PWM_PHASEC_COMPL

    /* Debug pin */
    SIUL2.MSCR0_255[145].R = 0x02380000;   // PJ[1]   145 0000_0000 GPIO
}

/***************************************************************************//*!
*
* @brief   Set the interrupt source and priority.
*
* @return  N/A
*
******************************************************************************/

void my_interrupt_init(uint32_t source, uint8_t priority)
{
	INTC_0.PSR[source].R = 0x4000 | (priority & 0xFF) ;
}

/***************************************************************************//*!
*
* @brief   Evaluate eTPU engine A load.
*
* @warning - This function is applicable on eTPU2 only.
*          - The first call of this function does not return a correct result.
*          - This function presumes the TCR1 clock is configured for system
*            clock div 1.
*
* @return  eTPU engine A load percentage (0 - 10000).
*
******************************************************************************/

uint32_t get_etpu_load_a(void)
{
  static uint24_t tcr1;
         uint24_t tcr1_last;
         uint24_t idle_cnt;
         uint24_t time_cnt;

  tcr1_last = tcr1;
  tcr1 = eTPU->TB1R_A.R;
  idle_cnt = fs_etpu_get_idle_cnt_a();
  fs_etpu_clear_idle_cnt_a();

  time_cnt = 0xFFFFFF & (tcr1 - tcr1_last);
  //time_cnt = time_cnt/2;
  return(10000*(time_cnt - idle_cnt)/time_cnt);
}
/*******************************************************************************
 *
 * Copyright:
 *  NXP Semiconductors. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of NXP
 *  Semiconductors. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, NXP
 *  Semiconductors DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL NXP Semiconductors BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  NXP Semiconductors assumes no responsibility for the
 *  maintenance and support of this software
 ******************************************************************************/
