/*
 * File:    mcg.c
 *
 * Notes:
 * Assumes the MCG mode is in the default FEI mode out of reset
 */

#include "common.h"
#include "mcg.h"

/******************************************************************************
 * macro definitions                                                          *
 ******************************************************************************/
#define WAIT_FOR_FLAG(reg,name1,name2)                                        \
{                                                                             \
  while (((MCG_S&MCG_S_##name2##_MASK)>>MCG_S_##name2##_SHIFT)!=              \
        ((reg&reg##_##name1##_MASK)>>reg##_##name1##_SHIFT));                 \
}

typedef struct
{
    uint32_t ref_range_min;
    uint32_t ref_range_max;
    uint32_t fll_factor;
    uint32_t dco_range_min;
    uint32_t dco_range_max;
} MCG_GetFllFreq_param_t;

MCG_GetFllFreq_param_t MCG_GetFllFreq_param_table[] =
{
    /* ref_range_min ref_range_max fll_factor dco_range_min dco_range_max */
    {  31250,        39063,        640,       20000000,     25000000},
    {  32768,        32768,        732,       24000000,     24000000},
    {  31250,        39063,        1280,      40000000,     45000000},
    {  32768,        32768,        1460,      48000000,     48000000},
    {  31250,        39063,        1920,      60000000,     75000000},
    {  32768,        32768,        2190,      72000000,     72000000},
    {  31250,        39063,        2560,      80000000,     100000000},
    {  32768,        32768,        2929,      96000000,     96000000}
};

/********************************************************************/

/********************************************************************/


#if defined(__ICCARM__)
  #pragma diag_suppress=Pa082
#endif

/********************************************************************/
static uint32_t MCG_CalcRangeVal(uint32_t crystal_val)
{
    uint32_t range_val;

    if (crystal_val <= 40000)
        range_val = 0;
    else if (crystal_val <= 8000000)
        range_val = 1;
    else
        range_val = 2;
    
    return range_val;
}

static uint32_t MCG_GetFrdivVal(void)
{
    uint32_t frdiv_val = (MCG_C1 & MCG_C1_FRDIV_MASK) >> MCG_C1_FRDIV_SHIFT;

    if (((MCG_C2 & MCG_C2_RANGE0_MASK) >> MCG_C2_RANGE0_SHIFT) > 0)
        return (1 << frdiv_val);
    else
        return (32 << frdiv_val);
}

static uint32_t MCG_CalcFrdivVal(uint32_t crystal_val, uint32_t *frdiv_reg_val,
                                   uint32_t *frdiv_val)
{
    /* determine FRDIV based on reference clock frequency
       since the external frequency has already been checked only the maximum
       frequency for each FRDIV value needs to be compared here.
    */
    if (frdiv_reg_val)
    {
        if (crystal_val <= 1250000)
            *frdiv_reg_val = 0;
        else if (crystal_val <= 2500000)
            *frdiv_reg_val = 1;
        else if (crystal_val <= 5000000)
            *frdiv_reg_val = 2;
        else if (crystal_val <= 10000000)
            *frdiv_reg_val = 3;
        else if (crystal_val <= 20000000)
            *frdiv_reg_val = 4;
        else
            *frdiv_reg_val = 5;
    }

    if (frdiv_val)
    {
        if (((MCG_C2 & MCG_C2_RANGE0_MASK) >> MCG_C2_RANGE0_SHIFT) > 0)
            *frdiv_val = (32 << *frdiv_reg_val);
        else
            *frdiv_val = (1 << *frdiv_reg_val);
    }
    
    return 0;
}

static int32_t MCG_CheckIRCRange(uint32_t irc_freq, uint32_t irc_type)
{
    int32_t rtn_val = 0;

    /* Check that the irc frequency matches the selected IRC
       Select the desired IRC */
    switch (irc_type)
    {
    case IRC_SLOW:
        if (!CHECK_SLOW_IRC_RANGE(irc_freq))
            rtn_val = -1;
        break;
    case IRC_FAST:
        if (!CHECK_FAST_IRC_RANGE(irc_freq))
            rtn_val = -1;
        break;
    default:
        rtn_val = -1;
    }
    
    return rtn_val;
}

int32_t MCG_PEE_To_PBE(int32_t crystal_val)
{
    if (MCG_MODE_PEE != MCG_GetCurMode())
        return -1;

    /*
       As we are running from the PLL by default the PLL and external clock settings are valid
       To move to PBE from PEE simply requires the switching of the CLKS mux to select the ext clock
       As CLKS is already 0 the CLKS value can simply be OR'ed into the register
    */
    /* switch CLKS mux to select external reference clock as MCG_OUT */
    MCG_C1 |= MCG_C1_CLKS(2);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in PBE mode */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_PBE_To_PEE(int32_t pll_src)
{
    uint32_t ref_freq = 0, extal_freq = 0;

    if (MCG_MODE_PBE != MCG_GetCurMode())
        return -1;

    /*
    * As the PLL settings have already been checked when PBE mode was
    * enterred they are not checked here
    */

    /* Check the PLL state before transitioning to PEE mode */

    switch (pll_src)
    {
    case ERC_SLOW:
        ref_freq = 32768;
        break;
    case ERC_FRDIV:
        if (MCG_C7 & MCG_C7_OSCSEL_MASK)
            extal_freq = ERC_SLOW_FREQ;
        else
            extal_freq = ERC_FAST_FREQ;
        ref_freq = extal_freq / MCG_GetFrdivVal();
        break;
    default:
        return -2;
    }
    
    /* Check LOCK bit is set before transitioning MCG to PLL output
      (already checked in fbe_pbe but good practice
       to re-check before switch to use PLL)
    */
    WAIT_FOR_FLAG(MCG_C5, PLLCLKEN0, LOCK0);
    /* clear CLKS to switch CLKS mux to select PLL as MCG_OUT */
    MCG_C1 &= ~MCG_C1_CLKS_MASK;
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in PEE */
    return (ref_freq * 375);
}


int32_t MCG_PBE_To_FBE(int32_t crystal_val)
{
    if (MCG_MODE_PBE != MCG_GetCurMode())
        return -1;

    /*
      As we are running from the ext clock, by default the external clock settings are valid
      To move to FBE from PBE simply requires the switching of the PLLS mux to disable the PLL
    */
    /* clear PLLS to disable PLL, still clocked from ext ref clk */
    MCG_C6 &= ~MCG_C6_PLLS_MASK;
    WAIT_FOR_FLAG(MCG_C6, PLLS, PLLST);

    /* Now in FBE mode */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


/********************************************************************/
/* Functon name : fbe_pbe
 *
 * Mode transition: FBE to PBE mode
 *
 * This function transitions the MCG from FBE mode to PBE mode.
 * This function presently only supports OSC0 and PLL0. Support for OSC1 and PLL1 will be added soon
 * The function requires the desired OSC and PLL be passed in to it for compatibility with the
 * future support of OSC/PLL selection
 *
 * Parameters: osc_select  - Selects which OSC is to be used in PBE mode, 0 or 1
 *             crystal_val - external clock frequency in Hz
 *             pll_select  - 0 to select PLL0, non-zero to select PLL1.
 *             prdiv_val   - value to divide the external clock source by to create the desired
 *                           PLL reference clock frequency
 *             vdiv_val    - value to multiply the PLL reference clock frequency by
 *
 * Return value : MCGCLKOUT frequency (Hz) or error code
 */
int32_t MCG_FBE_To_PBE(int32_t crystal_val)
{
    /* Check MCG is in FBE mode */
    if (MCG_MODE_FBE != MCG_GetCurMode())
        return -1;

    /*
      Configure MCG_C5
      If the PLL is to run in STOP mode then the PLLSTEN bit needs to be OR'ed in here or in user code.
    */
    MCG_C5 |= (MCG_C5_PLLCLKEN0_MASK);

    /*
      Configure MCG_C6
      The PLLS bit is set to enable the PLL, MCGOUT still sourced from ext ref clk
      The clock monitor is not enabled here as it has likely been enabled previously and so the value of CME
      is not altered here.
      The loss of lock interrupt can be enabled by seperately OR'ing in the LOLIE bit in MCG_C6
    */
    MCG_C6 |= MCG_C6_PLLS_MASK;

    WAIT_FOR_FLAG(MCG_C6, PLLS, PLLST);
    WAIT_FOR_FLAG(MCG_C5, PLLCLKEN0, LOCK0);

    /* now in PBE */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_PBE_To_BLPE(int32_t crystal_val)
{
    if (MCG_MODE_PBE != MCG_GetCurMode())
        return -1;

    // To enter BLPE mode the LP bit must be set, disabling the PLL
    MCG_C2 |= MCG_C2_LP_MASK;

    // Now in BLPE mode
    return crystal_val;
}

/*
 Since PBE mode can be entered via FBE -> BLPE modes, it cannot be assumed that the PLL has been
 previously configured correctly. That is why this general purpose driver has the PLL settings as
 passed parameters.
*/
int32_t MCG_BLPE_To_PBE(int32_t crystal_val)
{
    // Check MCG is in BLPE mode
    if (MCG_MODE_BLPE != MCG_GetCurMode())
        return -1;

    /*
     Configure MCG_C5
     If the PLL is to run in STOP mode then the PLLSTEN bit needs to be OR'ed in here or in user code.
    */
    MCG_C5 |= (MCG_C5_PLLCLKEN0_MASK);

    /*
     Configure MCG_C6
     The PLLS bit is set to enable the PLL, MCGOUT still sourced from ext ref clk
     The clock monitor is not enabled here as it has likely been enabled previously and so the value of CME
     is not altered here.
     The loss of lock interrupt can be enabled by seperately OR'ing in the LOLIE bit in MCG_C6
    */
    MCG_C6 |= MCG_C6_PLLS_MASK;

    /* Now that PLL is configured, LP is cleared to enable the PLL */
    MCG_C2 &= ~MCG_C2_LP_MASK;

    WAIT_FOR_FLAG(MCG_C6, PLLS, PLLST);
    WAIT_FOR_FLAG(MCG_C5, PLLCLKEN0, LOCK0);

    /* now in PBE */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_BLPE_To_FBE(int32_t crystal_val)
{
    if (MCG_MODE_BLPE != MCG_GetCurMode())
        return -1;

    /* To move from BLPE to FBE the PLLS mux be set to select 
    the FLL output and the LP bit must be cleared */
    /* clear PLLS to select the FLL */
    MCG_C6 &= ~MCG_C6_PLLS_MASK;
    /* clear LP bit */
    MCG_C2 &= ~MCG_C2_LP_MASK;

    /* wait for PLLST status bit to clear */
    WAIT_FOR_FLAG(MCG_C6, PLLS, PLLST);

    /* now in FBE mode */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_FBE_To_BLPE(int32_t crystal_val)
{
    if (MCG_MODE_FBE != MCG_GetCurMode())
        return -1;

    /* To move from FBE to BLPE the LP bit must be set */
    MCG_C2 |= MCG_C2_LP_MASK;

    /* now in FBE mode */
    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_FBE_To_FEI(int32_t slow_irc_freq)
{
    uint8_t temp_reg;
    int32_t mcg_out;

    if (MCG_MODE_FBE != MCG_GetCurMode())
        return -1;

    /* Check IRC frequency is within spec. */
    if (MCG_CheckIRCRange(slow_irc_freq, IRC_SLOW))
        return -2;

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(slow_irc_freq);
    if (mcg_out < 0)
        return mcg_out;

    /* Need to make sure the clockmonitor is disabled before moving to an "internal" clock mode */
    MCG_C6 &= ~MCG_C6_CME0_MASK; //This assumes OSC0 is used as the external clock source

    /* Move to FEI by setting CLKS to 0 and enabling the slow IRC as the FLL reference clock */
    temp_reg = MCG_C1;
    /* clear CLKS to select FLL output */
    temp_reg &= ~MCG_C1_CLKS_MASK;
    /* select internal reference clock */
    temp_reg |= MCG_C1_IREFS_MASK;
    MCG_C1 = temp_reg;

    /* wait for Reference clock Status bit to clear */
    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);

    /* Wait for clock status bits to show clock source is ext ref clk */
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in FEI mode */
    return mcg_out;
}


/********************************************************************/
/* Functon name : fei_fbe
 *
 * Mode transition: FEI to FBE mode
 *
 * This function transitions the MCG from FEI mode to FBE mode. This is
 * achieved by setting the MCG_C2[LP] bit. There is no status bit to
 * check so 0 is always returned if the function was called with the MCG
 * in FBI mode. The MCGCLKOUT frequency does not change
 *
 * Parameters: crystal_val - external clock frequency in Hz
 *             hgo_val     - selects whether low power or high gain mode is selected
 *                           for the crystal oscillator. This has no meaning if an
 *                           external clock is used.
 *             erefs_val   - selects external clock (=0) or crystal osc (=1)
 *
 * Return value : MCGCLKOUT frequency (Hz) or error code
 */
int32_t MCG_FEI_To_FBE(int32_t crystal_val, uint8_t hgo_val, uint8_t erefs_val)
{
    uint8_t temp_reg;
    uint32_t range_val = 0, frdiv_val = 0;

    if (MCG_MODE_FEI != MCG_GetCurMode())
        return -1;

    /* check external frequency is less than the maximum frequency */
    if  (crystal_val > 32000000)
        return -2;

    /* check crystal frequency is within spec. if crystal osc is being used */
    if (erefs_val)
    {
        if (!(CHECK_SLOW_ERC_RANGE(crystal_val) || CHECK_FAST_ERC_RANGE(crystal_val)))
            return -3;
    }

    /* make sure HGO will never be greater than 1. Could return an error instead if desired. */
    hgo_val = (hgo_val > 0) ? 1 : 0;

    /* configure the MCG_C2 register
       the RANGE value is determined by the external frequency. Since the RANGE
       parameter affects the FRDIV divide value
       it still needs to be set correctly even if the oscillator is not being used
    */
    temp_reg = MCG_C2;
    temp_reg &= ~(MCG_C2_RANGE0_MASK | MCG_C2_HGO0_MASK | MCG_C2_EREFS0_MASK);
    range_val = MCG_CalcRangeVal(crystal_val);
    temp_reg |= (MCG_C2_RANGE0(range_val) | (hgo_val << MCG_C2_HGO0_SHIFT) | (erefs_val << MCG_C2_EREFS0_SHIFT));
    MCG_C2 = temp_reg;

    /*
     determine FRDIV based on reference clock frequency
     since the external frequency has already been checked only the maximum
     frequency for each FRDIV value needs to be compared here.
    */
    MCG_CalcFrdivVal(crystal_val, &frdiv_val, NULL);

    /*
     Select external oscilator and Reference Divider and clear IREFS to start ext osc
     If IRCLK is required it must be enabled outside of this driver, existing state will be maintained
     CLKS=2, FRDIV=frdiv_val, IREFS=0, IRCLKEN=0, IREFSTEN=0
    */
    temp_reg = MCG_C1;
    temp_reg &= ~(MCG_C1_CLKS_MASK | MCG_C1_FRDIV_MASK | MCG_C1_IREFS_MASK);
    temp_reg = MCG_C1_CLKS(2) | MCG_C1_FRDIV(frdiv_val);
    MCG_C1 = temp_reg;

    /* if the external oscillator is used need to wait for OSCINIT to set */
    if (erefs_val)
    {
        WAIT_FOR_FLAG (MCG_C2, EREFS0, OSCINIT0);
    }

    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in FBE
       It is recommended that the clock monitor is enabled when using an
       external clock as the clock source/reference.
       It is enabled here but can be removed if this is not required. */
    MCG_C6 |= MCG_C6_CME0_MASK;

    /* MCGOUT frequency equals external clock frequency */
    return crystal_val;
}


int32_t MCG_FBE_To_FEE(int32_t crystal_val)
{
    int16_t fll_ref_freq;
    int32_t mcg_out;

    if (MCG_MODE_FBE != MCG_GetCurMode())
        return -1;

    /* The FLL ref clk divide value depends on FRDIV and the RANGE value */
    fll_ref_freq = crystal_val / MCG_GetFrdivVal();

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(fll_ref_freq);
    if (mcg_out < 0)
        return mcg_out;

    /* Clear CLKS field to switch CLKS mux to select FLL output */
    MCG_C1 &= ~MCG_C1_CLKS_MASK;

    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);
  
    /* Now in FEE mode */
    return mcg_out;
}

int32_t MCG_FEE_To_FBE(int32_t crystal_val)
{
    if (MCG_MODE_FEE != MCG_GetCurMode())
        return -1;

    /* 
     Set CLKS field to 2 to switch CLKS mux to select ext ref clock
     MCG is current in FEE mode so CLKS field = 0 so can just OR in new value
    */
    /* set CLKS to select ext ref clock */
    MCG_C1 |= MCG_C1_CLKS(2);

    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in FBE mode */
    return crystal_val;
}

int32_t MCG_FBE_To_FBI(int32_t irc_freq, int32_t irc_type)
{
    uint8_t temp_reg;

    if (MCG_MODE_FBE != MCG_GetCurMode())
        return -1;

    /* Check that the irc frequency matches the selected IRC */
    if (MCG_CheckIRCRange(irc_freq, irc_type))
        return -2;

    /* Select the required IRC */
    if (IRC_FAST == irc_type)
        MCG_C2 |= MCG_C2_IRCS_MASK;
    else
        MCG_C2 &= ~MCG_C2_IRCS_MASK;

    /* Make sure the clock monitor is disabled before switching modes otherwise it will trigger */
    MCG_C6 &= ~MCG_C6_CME0_MASK;

    /* Select the IRC as the CLKS mux selection */
    temp_reg = MCG_C1;
    temp_reg &= ~MCG_C1_CLKS_MASK;
    /* select IRC as MCGOUT and enable IREFS */
    temp_reg |= MCG_C1_CLKS(1) | MCG_C1_IREFS_MASK;
    MCG_C1 = temp_reg;

    WAIT_FOR_FLAG(MCG_C2, IRCS, IRCST);
    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);
  
    /* Now in FBI mode */
    return irc_freq;
}


int32_t MCG_FBI_To_FBE(int32_t crystal_val, uint8_t hgo_val, uint8_t erefs_val)
{
    uint8_t temp_reg;
    uint32_t range_val = 0, frdiv_val = 0;

    if (MCG_MODE_FBI != MCG_GetCurMode())
        return -1;

    /* check external frequency is less than the maximum frequency */
    if  (crystal_val > 60000000)
        return -2;

    /* check crystal frequency is within spec. if crystal osc is being used */
    if (erefs_val)
    {
        if (!(CHECK_SLOW_ERC_RANGE(crystal_val) || CHECK_FAST_ERC_RANGE(crystal_val)))
            return -3;
    }

    /* make sure HGO will never be greater than 1. Could return an error instead if desired. */
    hgo_val = (hgo_val > 0) ? 1 : 0;

    /*
     configure the MCG_C2 register
     the RANGE value is determined by the external frequency.
     Since the RANGE parameter affects the F#RDIV divide value
     it still needs to be set correctly even if the oscillator is not being used
    */
    temp_reg = MCG_C2;
    temp_reg &= ~(MCG_C2_RANGE0_MASK | MCG_C2_HGO0_MASK | MCG_C2_EREFS0_MASK);
    range_val = MCG_CalcRangeVal(crystal_val);
    temp_reg |= (MCG_C2_RANGE0(range_val) | (hgo_val << MCG_C2_HGO0_SHIFT) | (erefs_val << MCG_C2_EREFS0_SHIFT));
    MCG_C2 = temp_reg;

    /* determine FRDIV based on reference clock frequency
       since the external frequency has already been checked only the maximum
       frequency for each FRDIV value needs to be compared here. */
    MCG_CalcFrdivVal(crystal_val, &frdiv_val, NULL);

    /* Select external oscilator and Reference Divider and clear IREFS to start ext osc
       If IRCLK is required it must be enabled outside of this driver, existing state will be maintained
       CLKS=2, FRDIV=frdiv_val, IREFS=0, IRCLKEN=0, IREFSTEN=0 */
    temp_reg = MCG_C1;
    temp_reg &= ~(MCG_C1_CLKS_MASK | MCG_C1_FRDIV_MASK | MCG_C1_IREFS_MASK);
    temp_reg = MCG_C1_CLKS(2) | MCG_C1_FRDIV(frdiv_val);
    MCG_C1 = temp_reg;

    /* if the external oscillator is used need to wait for OSCINIT to set */
    if (erefs_val)
        WAIT_FOR_FLAG (MCG_C2, EREFS0, OSCINIT0);

    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /*
     Now in FBE
     It is recommended that the clock monitor is enabled when using an external clock as the clock source/reference.
     It is enabled here but can be removed if this is not required.
    */
    MCG_C6 |= MCG_C6_CME0_MASK;

    return crystal_val;
}


/********************************************************************/
/* Functon name : fbi_blpi
 *
 * Mode transition: FBI to BLPI mode
 *
 * This function transitions the MCG from FBI mode to BLPI mode. This is
 * achieved by setting the MCG_C2[LP] bit. There is no status bit to
 * check so 0 is always returned if the function was called with the MCG
 * in FBI mode.
 *
 * Parameters: irc_freq - internal reference clock frequency
 *             ircs_select - 0 if slow irc, 1 if fast irc
 *
 * Return value : MCGOUT frequency or error code 0x13
 */
int32_t MCG_FBI_To_BLPI(int32_t irc_freq, int32_t irc_type)
{
    if (MCG_MODE_FBI != MCG_GetCurMode())
        return -1;

    /* Set LP bit to disable the FLL and enter BLPI */
    MCG_C2 |= MCG_C2_LP_MASK;

    /* Now in BLPI */
    return irc_freq;
}

/********************************************************************/
/* Functon name : blpi_fbi
 *
 * Mode transition: BLPI to FBI mode
 *
 * This function transitions the MCG from BLPI mode to FBI mode. This is
 * achieved by clearing the MCG_C2[LP] bit. There is no status bit to
 * check so 0 is always returned if the function was called with the MCG
 * in BLPI mode.
 *
 * Parameters: irc_freq - internal reference clock frequency
 *             ircs_select - 0 if slow irc, 1 if fast irc
 *
 * Return value : MCGOUT frequency or error code 0x15
 */
int32_t MCG_BLPI_To_FBI(int32_t irc_freq, int32_t irc_type)
{
    if (MCG_MODE_BLPI != MCG_GetCurMode())
        return -1;

    /* Clear LP bit to enable the FLL and enter FBI mode */
    MCG_C2 &= ~MCG_C2_LP_MASK;

    /* Now in FBI mode */
    return irc_freq;
}


int32_t MCG_FBI_To_FEE(int32_t crystal_val, uint8_t hgo_val, uint8_t erefs_val)
{
    uint8_t temp_reg;
    int32_t mcg_out, fll_ref_freq;
    uint32_t range_val = 0, fll_div = 0, frdiv_val = 0;

    if (MCG_MODE_FBI != MCG_GetCurMode())
        return -1;

    /* check external frequency is less than the maximum frequency */
    if  (crystal_val > 60000000)
        return -2;

    /* check crystal frequency is within spec. if crystal osc is being used */
    if (erefs_val)
    {
        if (!(CHECK_SLOW_ERC_RANGE(crystal_val) || CHECK_FAST_ERC_RANGE(crystal_val)))
            return -3;
    }

    /* make sure HGO will never be greater than 1. Could return an error instead if desired. */
    hgo_val = (hgo_val > 0) ? 1 : 0;

    /* configure the MCG_C2 register
       the RANGE value is determined by the external frequency. Since the RANGE parameter affects the FRDIV divide value
       it still needs to be set correctly even if the oscillator is not being used
    */
    temp_reg = MCG_C2;
    /* clear fields before writing new values */
    temp_reg &= ~(MCG_C2_RANGE0_MASK | MCG_C2_HGO0_MASK | MCG_C2_EREFS0_MASK);
    range_val = MCG_CalcRangeVal(crystal_val);
    temp_reg |= (MCG_C2_RANGE0(range_val) | (hgo_val << MCG_C2_HGO0_SHIFT) | (erefs_val << MCG_C2_EREFS0_SHIFT));
    MCG_C2 = temp_reg;

    /* determine FRDIV based on reference clock frequency
       since the external frequency has already been checked only the maximum
       frequency for each FRDIV value needs to be compared here.
    */
    MCG_CalcFrdivVal(crystal_val, &frdiv_val, &fll_div);
    /* The FLL ref clk divide value depends on FRDIV and the RANGE value */
    fll_ref_freq = crystal_val / fll_div;

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(fll_ref_freq);
    if (mcg_out < 0)
        return mcg_out;

    /* 
     Select external oscilator and Reference Divider and clear IREFS to start ext osc
     If IRCLK is required it must be enabled outside of this driver,
     existing state will be maintained
     CLKS=0, FRDIV=frdiv_val, IREFS=0, IRCLKEN=?, IREFSTEN=?
    */
    temp_reg = MCG_C1;
    /* Clear CLKS, FRDIV and IREFS fields */
    temp_reg &= ~(MCG_C1_CLKS_MASK | MCG_C1_FRDIV_MASK | MCG_C1_IREFS_MASK);
    /* Set the required CLKS and FRDIV values */
    temp_reg = MCG_C1_CLKS(0) | MCG_C1_FRDIV(frdiv_val);
    MCG_C1 = temp_reg;

    /* if the external oscillator is used need to wait for OSCINIT to set */
    if (erefs_val)
        WAIT_FOR_FLAG (MCG_C2, EREFS0, OSCINIT0);

    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /*
     Now in FEE
     It is recommended that the clock monitor is enabled when using an external clock as the clock source/reference.
     It is enabled here but can be removed if this is not required.
     The clock monitor MUST be disabled when returning to a non-external clock mode (FEI, FBI and BLPI)
    */
    //MCG_C6 |= MCG_C6_CME0_MASK;

    /* MCGOUT frequency equals FLL frequency */
    return mcg_out;
}


int32_t MCG_FEE_To_FBI(int32_t irc_freq, int32_t irc_type)
{
    if (MCG_MODE_FEE != MCG_GetCurMode())
        return -1;

    if (MCG_CheckIRCRange(irc_freq, irc_type))
        return -2;

    if (IRC_FAST == irc_type)
    {
        /* select fast IRC by setting IRCS */
        MCG_C2 |= MCG_C2_IRCS_MASK;
    }
    else
    {
        /* select slow IRC by clearing IRCS */
        MCG_C2 &= ~MCG_C2_IRCS_MASK;
    }

    /* Make sure the clock monitor is disabled before switching modes otherwise it will trigger */
    MCG_C6 &= ~MCG_C6_CME0_MASK;

    /* Select the IRC as the CLKS mux selection */
    /* set IREFS and select IRC as MCGOUT */
    MCG_C1 |= MCG_C1_CLKS(1) | MCG_C1_IREFS_MASK;

    /* wait until internal reference switches to requested irc. */
    WAIT_FOR_FLAG(MCG_C2, IRCS, IRCST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);
    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);

    /* Now in FBI mode */
    return irc_freq;
}


int32_t MCG_FBI_To_FEI(int32_t slow_irc_freq)
{
    uint8_t temp_reg;
    int32_t mcg_out;

    if (MCG_MODE_FBI != MCG_GetCurMode())
        return -1;

    /* Check IRC frequency is within spec. */
    if (MCG_CheckIRCRange(slow_irc_freq, IRC_SLOW))
        return -2;

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(slow_irc_freq);
    if (mcg_out < 0)
        return mcg_out;

    /* Change the CLKS mux to select the FLL output as MCGOUT */
    temp_reg = MCG_C1;
    /* clear CLKS field */
    temp_reg &= ~MCG_C1_CLKS_MASK;
    /* select FLL as MCGOUT */
    temp_reg |= MCG_C1_CLKS(0);
    /* make sure IRC is FLL reference */
    temp_reg |= MCG_C1_IREFS_MASK;
    /* update MCG_C1 */
    MCG_C1 = temp_reg;

    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in FEI mode */
    return mcg_out;
}


int32_t MCG_FEI_To_FBI(int32_t irc_freq, int32_t irc_type)
{
    uint8_t temp_reg;;

    /* Check MCG is in FEI mode */
    if (MCG_MODE_FEI != MCG_GetCurMode())
        return -1;

    /* Check that the irc frequency matches the selected IRC
       Select the desired IRC */
    if (MCG_CheckIRCRange(irc_freq, irc_type))
        return -2;

    if (IRC_FAST == irc_type)
    {
        /* select fast IRC by setting IRCS */
        MCG_C2 |= MCG_C2_IRCS_MASK;
    }
    else
    {
        /* select slow IRC by clearing IRCS */
        MCG_C2 &= ~MCG_C2_IRCS_MASK;
    }

    /* Change the CLKS mux to select the IRC as the MCGOUT */
    temp_reg = MCG_C1;
    /* clear CLKS */
    temp_reg &= ~MCG_C1_CLKS_MASK;
    /* select IRC as the MCG clock sourse */
    temp_reg |= MCG_C1_CLKS(1);
    MCG_C1 = temp_reg;

    /* wait until internal reference switches to requested irc. */
    WAIT_FOR_FLAG(MCG_C2, IRCS, IRCST);
    WAIT_FOR_FLAG(MCG_C1, CLKS, CLKST);

    /* Now in FBI mode */
    return irc_freq;
}


/********************************************************************/
/* Functon name : fei_fee
 *
 * Mode transition: FEI to FEE mode
 *
 * This function transitions the MCG from FEI mode to FEE mode. This is
 * achieved by setting the MCG_C2[LP] bit. There is no status bit to
 * check so 0 is always returned if the function was called with the MCG
 * in FBI mode. The MCGCLKOUT frequency does not change
 *
 * Parameters: crystal_val - external clock frequency in Hz
 *             hgo_val     - selects whether low power or high gain mode is selected
 *                           for the crystal oscillator. This has no meaning if an
 *                           external clock is used.
 *             erefs_val   - selects external clock (=0) or crystal osc (=1)
 *
 * Return value : MCGCLKOUT frequency (Hz) or error code
 */
int32_t MCG_FEI_To_FEE(int32_t crystal_val, uint8_t hgo_val, uint8_t erefs_val)
{
    uint8_t temp_reg = 0;
    uint32_t fll_div = 0, range_val = 0, frdiv_val;
    int32_t mcg_out, fll_ref_freq;

    /* check if in FEI mode */
    if (MCG_MODE_FEI != MCG_GetCurMode())
        return -1;

    // check external frequency is less than the maximum frequency
    if  (crystal_val > 48000000)
        return -2;

    /* check crystal frequency is within spec. if crystal osc is being used */
    if (erefs_val)
    {
        if (!(CHECK_SLOW_ERC_RANGE(crystal_val) || CHECK_FAST_ERC_RANGE(crystal_val)))
            return -3;
    }

    /*
      make sure HGO will never be greater than 1. Could return an error instead if desired.
      force hgo_val to 1 if > 0
    */
    hgo_val = (hgo_val > 0) ? 1 : 0;

    /* configure the MCG_C2 register
       the RANGE value is determined by the external frequency. Since the RANGE parameter affects the FRDIV divide value
       it still needs to be set correctly even if the oscillator is not being used */
    temp_reg = MCG_C2;
    /* clear fields before writing new values */
    temp_reg &= ~(MCG_C2_RANGE0_MASK | MCG_C2_HGO0_MASK | MCG_C2_EREFS0_MASK);
    range_val = MCG_CalcRangeVal(crystal_val);
    temp_reg |= (MCG_C2_RANGE0(range_val) | (hgo_val << MCG_C2_HGO0_SHIFT) | (erefs_val << MCG_C2_EREFS0_SHIFT));
    MCG_C2 = temp_reg;

    MCG_CalcFrdivVal(crystal_val, &frdiv_val, &fll_div);
    /* The FLL ref clk divide value depends on FRDIV and the RANGE value */
    fll_ref_freq = crystal_val / fll_div;

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(fll_ref_freq); // FLL reference frequency calculated from ext ref freq and FRDIV
    if (mcg_out < 0)
        return mcg_out; // If error code returned, return the code to calling function

    /* Select external oscilator and Reference Divider and clear IREFS to start ext osc
       If IRCLK is required it must be enabled outside of this driver, existing state will be maintained
       CLKS=0, FRDIV=frdiv_val, IREFS=0, IRCLKEN=0, IREFSTEN=0 */
    temp_reg = MCG_C1;
    temp_reg &= ~(MCG_C1_CLKS_MASK | MCG_C1_FRDIV_MASK | MCG_C1_IREFS_MASK); // Clear values in these fields
    temp_reg = MCG_C1_CLKS(0) | MCG_C1_FRDIV(frdiv_val); // Set the required CLKS and FRDIV values
    MCG_C1 = temp_reg;

    /* if the external oscillator is used need to wait for OSCINIT to set */
    if (erefs_val)
        WAIT_FOR_FLAG (MCG_C2, EREFS0, OSCINIT0);
    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);

    /* Now in FBE
       It is recommended that the clock monitor is enabled when using an external clock as the clock source/reference.
       It is enabled here but can be removed if this is not required. */
    MCG_C6 |= MCG_C6_CME0_MASK;

    /* MCGOUT frequency equals FLL frequency */
    return mcg_out;
}


int32_t MCG_FEE_To_FEI(int32_t slow_irc_freq)
{
    int32_t mcg_out;

    if (MCG_MODE_FEE != MCG_GetCurMode())
        return -1;

    /* Check IRC frequency is within spec. */
    /* FEI can only use slow irc clock, 32K */
    if ((slow_irc_freq < 31250) || (slow_irc_freq > 39063))
        return -2;

    /* Check resulting FLL frequency */
    mcg_out = MCG_GetFllFreq(slow_irc_freq);
    if (mcg_out < 0)
        return mcg_out;

    /* Ensure clock monitor is disabled before switching to FEI
       otherwise a loss of clock will trigger */
    MCG_C6 &= ~MCG_C6_CME0_MASK;

    /* Change FLL reference clock from external to internal by
       setting IREFS bit */
    /* select internal reference */
    MCG_C1 |= MCG_C1_IREFS_MASK;

    /* wait for Reference clock to switch to internal reference */
    WAIT_FOR_FLAG(MCG_C1, IREFS, IREFST);
  
    /* Now in FEI mode */
    return mcg_out;
}

int32_t MCG_GetFllFreq(int32_t fll_ref)
{
    int32_t MCG_GetFllFreq_hz;
    uint8_t drst_drs = (MCG_C4 & MCG_C4_DRST_DRS_MASK) >> MCG_C4_DRST_DRS_SHIFT;
    uint8_t dmx32 = (MCG_C4 & MCG_C4_DMX32_MASK) >> MCG_C4_DMX32_SHIFT;
    uint8_t fll_index = drst_drs * 2 + dmx32;
    
    MCG_GetFllFreq_hz = fll_ref * MCG_GetFllFreq_param_table[fll_index].fll_factor;
    /* We may allow 1MHz frequency difference */
    if ((MCG_GetFllFreq_hz < (MCG_GetFllFreq_param_table[fll_index].dco_range_min - 1000000)) ||
        (MCG_GetFllFreq_hz > (MCG_GetFllFreq_param_table[fll_index].dco_range_max + 1000000)))
        return -1;
    else
        return MCG_GetFllFreq_hz;
}

uint8_t MCG_GetCurMode(void)
{
    if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x0) &&      // check CLKS mux has selcted FLL output
        (MCG_S & MCG_S_IREFST_MASK) &&                                     // check FLL ref is internal ref clk
        (!(MCG_S & MCG_S_PLLST_MASK)))                                     // check PLLS mux has selected FLL
        return MCG_MODE_FEI;                                                          // return FEI code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x3) && // check CLKS mux has selcted PLL output
            (!(MCG_S & MCG_S_IREFST_MASK)) &&                              // check FLL ref is external ref clk
            (MCG_S & MCG_S_PLLST_MASK))                                    // check PLLS mux has selected PLL
        return MCG_MODE_PEE;                                                          // return PEE code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x2) && // check CLKS mux has selcted external reference
            (!(MCG_S & MCG_S_IREFST_MASK)) &&                              // check FLL ref is external ref clk
            (MCG_S & MCG_S_PLLST_MASK) &&                                  // check PLLS mux has selected PLL
            (!(MCG_C2 & MCG_C2_LP_MASK)))                                  // check MCG_C2[LP] bit is not set
        return MCG_MODE_PBE;                                                          // return PBE code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x2) && // check CLKS mux has selcted external reference
            (!(MCG_S & MCG_S_IREFST_MASK)) &&                              // check FLL ref is external ref clk
            (!(MCG_S & MCG_S_PLLST_MASK)) &&                               // check PLLS mux has selected FLL
            (!(MCG_C2 & MCG_C2_LP_MASK)))                                  // check MCG_C2[LP] bit is not set
        return MCG_MODE_FBE;                                                          // return FBE code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x2) && // check CLKS mux has selcted external reference
            (!(MCG_S & MCG_S_IREFST_MASK)) &&                              // check FLL ref is external ref clk
            (MCG_C2 & MCG_C2_LP_MASK))                                     // check MCG_C2[LP] bit is set
        return MCG_MODE_BLPE;                                                         // return BLPE code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x1) && // check CLKS mux has selcted int32_t ref clk
            (MCG_S & MCG_S_IREFST_MASK) &&                                 // check FLL ref is internal ref clk
            (!(MCG_S & MCG_S_PLLST_MASK)) &&                               // check PLLS mux has selected FLL
            (MCG_C2 & MCG_C2_LP_MASK))                                     // check LP bit is set
        return MCG_MODE_BLPI;                                                         // return BLPI code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x1) && // check CLKS mux has selcted int32_t ref clk
            (MCG_S & MCG_S_IREFST_MASK) &&                                 // check FLL ref is internal ref clk
            (!(MCG_S & MCG_S_PLLST_MASK)) &&                               // check PLLS mux has selected FLL
            (!(MCG_C2 & MCG_C2_LP_MASK)))                                  // check LP bit is clear
        return MCG_MODE_FBI;                                                          // return FBI code
    else if ((((MCG_S & MCG_S_CLKST_MASK) >> MCG_S_CLKST_SHIFT) == 0x0) && // check CLKS mux has selcted FLL
            (!(MCG_S & MCG_S_IREFST_MASK)) &&                              // check FLL ref is external ref clk
            (!(MCG_S & MCG_S_PLLST_MASK)))                                 // check PLLS mux has selected FLL
        return MCG_MODE_FEE;                                                          // return FEE code
    else
        return MCG_MODE_UNKNOWN;                                                            // error condition
}

void MCG_ClkMonitorEnable(uint8_t en_dis)
{
    if (en_dis)
        MCG_C6 |= MCG_C6_CME0_MASK;
    else
        MCG_C6 &= ~MCG_C6_CME0_MASK;
}


