/*****************************************************************************
 * (c) Copyright 2010-2012, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************//*!
 * @file      spi.c
 * @version   1.0.3.0
 * @date      Mar-12-2013
 * @brief     SPI driver source code.
 *
******************************************************************************/

#include "drivers.h"
#include <stdio.h>
#include <string.h>

static struct spi_flash gSpiFlash;

static int32_t spi_flash_chk_status(uint32_t timeout,
			   uint8_t cmd, uint8_t poll_bit);

static void spi_flash_addr2cmd(uint32_t addr, uint8_t *cmd)
{
    cmd[1] = addr >> 16;
    cmd[2] = addr >> 8;
    cmd[3] = addr >> 0;
}

static int32_t spi_flash_rw(uint8_t *cmd, uint32_t cmd_len,
				uint8_t *data_out, uint8_t *data_in,
				uint32_t data_len)
{
    gSpiFlash.ss_ctrl(SS_CLEAR);

    gSpiFlash.spi_comm_byte(cmd, cmd_len, NULL, cmd_len);
    if (data_len != 0)
    {
        gSpiFlash.spi_comm_byte(data_out, data_len, data_in, data_len);
    }
    
    gSpiFlash.ss_ctrl(SS_SET);

    return 0;
}

int32_t spi_flash_send_cmd(uint8_t cmd, void *response, uint32_t len)
{
    return spi_flash_rw(&cmd, 1, NULL, response, len);
}

static inline int32_t spi_flash_enable_write(uint8_t is_enabled)
{
    if (is_enabled)
        return spi_flash_send_cmd(CMD_WRITE_ENABLE, NULL, 0);
    else
        return spi_flash_send_cmd(CMD_WRITE_DISABLE, NULL, 0);
}

static int32_t spi_flash_write_page(uint8_t *buf, uint32_t page_offset,
                                    uint32_t byte_offset, uint32_t len)
{
    uint8_t cmd[4] = { 0 };
    
    cmd[0] = CMD_PAGE_PROGRAM;
    cmd[1] = page_offset >> 8;
    cmd[2] = page_offset;
    cmd[3] = byte_offset;
    
    /* Each write need to enable write */
    if (spi_flash_enable_write(1))
    {
        SF_DEBUG("SF: enabling write failed\n");
        goto err_out;
    }

    if (spi_flash_rw(cmd, 4, buf, NULL, len))
    {
        SF_DEBUG("SF: write failed\n");
        goto err_out;
    }

    if (spi_flash_chk_status(SPI_FLASH_TIMEOUT, CMD_READ_STATUS, STATUS_WIP))
    {
        SF_DEBUG("SF: check status failed\n");
        goto err_out;
    }

    return 0;
err_out:
    return 1;
}

int32_t spi_flash_write(uint32_t offset, uint32_t len, const void *buf)
{
    uint32_t page_offset = 0, byte_offset = 0, page_size = 0;
    uint32_t data_chunk_len = 0, data_transferred = 0;
    int ret = 0;
    struct spi_flash *flash = &gSpiFlash;

    page_size = flash->page_size;
    page_offset = offset / page_size;
    byte_offset = offset % page_size;
    
    while (data_transferred < len)
    {
        /* First and last sector might be unaligned to page_size,
           So transfer unaligned sector first. */
        data_chunk_len = min(len - data_transferred, page_size - byte_offset);

        if (spi_flash_write_page(((uint8_t *)buf + data_transferred), page_offset,
                             byte_offset, data_chunk_len))
        {
            break;
        }

        byte_offset += data_chunk_len;
        if (byte_offset == page_size)
        {
            page_offset++;
            byte_offset = 0;
        }
        data_transferred += data_chunk_len;
    }

    if (ret)
    {
        SF_DEBUG("SF: program failed!\r\n");
    }
    else
    {
        SF_DEBUG("SF: program success!\r\n");
    }

    return ret;
}

int32_t spi_flash_read(uint32_t offset,
		uint32_t data_len, void *data)
{
    uint8_t cmd[5];

    cmd[0] = CMD_READ_ARRAY_FAST;
    spi_flash_addr2cmd(offset, cmd);
    cmd[4] = 0x00;

    return spi_flash_rw(cmd, sizeof(cmd), NULL, data, data_len);
}

static int32_t spi_flash_chk_status(uint32_t timeout,
			   uint8_t cmd, uint8_t poll_bit)
{
    uint32_t i = 0;
    uint8_t status;

    gSpiFlash.spi_comm_byte(&cmd, 1, NULL, 0);

    for (i = 0; i < timeout; ++i)
    {
        gSpiFlash.spi_comm_byte(NULL, 0, &status, 1);

        if ((status & poll_bit) == 0)
            break;
    };

    if (i == timeout)
    {
        SF_DEBUG("SF: time out!\n");
        return -1;
    }

    return 0;
}

int32_t spi_flash_erase(uint32_t offset, uint32_t len)
{
    uint32_t erase_start, erase_end, erase_size;
    int32_t ret;
    uint8_t cmd[4];
    struct spi_flash *flash = &gSpiFlash;

    erase_size = flash->sector_size;
    if ((offset % erase_size) || (len % erase_size))
    {
        SF_DEBUG("SF: Erase offset or length is not multiple of erase size\n");
        return -1;
    }

    switch (erase_size)
    {
    case ERASE_SECTOR_4K_SIZE:
        cmd[0] = CMD_ERASE_4K;
        break;
    case ERASE_SECTOR_32K_SIZE:
        cmd[0] = CMD_ERASE_32K;
        break;
    case ERASE_SECTOR_64K_SIZE:
        cmd[0] = CMD_ERASE_64K;
        break;
    default:
        SF_DEBUG("SF: Unreconized erase sector size, use 4K for default!\n");
        cmd[0] = CMD_ERASE_4K;
    }

    erase_start = offset;
    erase_end = erase_start + len;

    while (offset < erase_end)
    {
        spi_flash_addr2cmd(offset, cmd);
        offset += erase_size;

        ret = spi_flash_enable_write(1);
        if (ret)
            goto out;

        ret = spi_flash_rw(cmd, sizeof(cmd), NULL, NULL, 0);
        if (ret)
            goto out;

        ret = spi_flash_chk_status(SPI_FLASH_TIMEOUT,
                                         CMD_READ_STATUS, STATUS_WIP);
        if (ret)
            goto out;
    }

    SF_DEBUG("SF: Successfully erased!\r\n");

 out:
    return ret;
}

int spi_flash_write_status(uint8_t sts_reg)
{
    uint8_t cmd;
    int32_t ret;

    ret = spi_flash_enable_write(1);
    if (ret < 0)
    {
        SF_DEBUG("SF: enabling write failed\n");
        return ret;
    }

    cmd = CMD_WRITE_STATUS;
    ret = spi_flash_rw(&cmd, 1, &sts_reg, NULL, 1);
    if (ret)
    {
        SF_DEBUG("SF: fail to write status register\n");
        return ret;
    }

    ret = spi_flash_chk_status(SPI_FLASH_TIMEOUT,
                                 CMD_READ_STATUS, STATUS_WIP);
    if (ret < 0)
    {
        SF_DEBUG("SF: write status register timed out\n");
        return ret;
    }

    return 0;
}

int32_t spi_flash_init(SPI_FLASH_SS_CTRL ss_ctrl, SPI_FLASH_SPI_COMM_BYTE spi_comm_byte)
{
    int32_t ret;
    uint8_t idcode[IDCODE_LEN] = { 0 };
    struct spi_flash *flash = &gSpiFlash;
    uint8_t cmd = CMD_READ_ID;

    memset(flash, 0, sizeof(struct spi_flash));
    flash->ss_ctrl = ss_ctrl;
    flash->spi_comm_byte = spi_comm_byte;

    /* Send CMD_READ_ID to get flash chip ID codes */
    ret = spi_flash_rw(&cmd, 1, NULL, idcode, sizeof(idcode));
    if (ret)
    {
        return -1;
    }

    if (SPI_FLASH_MACRONIX_VENDER_ID == idcode[0]) {
        if (spi_flash_init_macronix(flash, idcode)) {
            return -1;
        }
    }
    else
    {
        SF_DEBUG("Can't find spi flash chip!\r\n");
        return -1;
    }

    SF_DEBUG("SF: Detected SPI Flash!\r\n");

    return 0;
}

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/