/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************/

#ifndef __1PH4CH_POWER_METERING_H__
#define __1PH4CH_POWER_METERING_H__

#include "drivers.h"
#include "fraclib.h"
#include "meterlib_cfg.h"

/* Configurations */
//#define POWER_METERING_DEBUG
/* Smart card debug macro */
#define POWER_METERING_SC_DEBUG                 (0)

/******************************************************************************
 * application password - must be entered via FreeMASTER prior recalibration
 * and/or saving parameters into flash
 ******************************************************************************/
/*!< default password             */
#define POWER_METERING_PASSWORD                 (11111)           

/*
 * The Calibration environment is used 220V, 5.0A.
 * User could changed it according to test condition.
 * preset calibration angle is 0 degree.
 * */
#define CAL_CURR  5.0                     /*!< Calibration point -  current */
#define CAL_VOLT  220.0                   /*!< Calibration point - voltage */

#define METER_CL  " C 5-120A"             /*!< Meter class shown on LCD     */ 
#define METER_SN  17                      /*!< Meter serial number          */

#define POWER_METERING_PHRASE_NUM               (3)

/* ------------------------ System clock configuration ---------------------- */
#define POWER_METERING_SYS_CLK_FREQ             (48000000)
#if (POWER_METERING_SYS_CLK_FREQ == 72000000)
    /* From RM, 32768 * 2197 / 3 = 23997098.6666.. ~23997099 */
    #define POWER_METERING_BUS_CLK_FREQ          (23997099)
#elif (POWER_METERING_SYS_CLK_FREQ == 48000000)
    /* From RM, 32768 * 1464 / 2 = 23986176 */
    #define POWER_METERING_BUS_CLK_FREQ          (23986176)
#elif (POWER_METERING_SYS_CLK_FREQ == 24000000)
    /* From RM, 32768 * 732 = 23986176 */
    #define POWER_METERING_BUS_CLK_FREQ          (23986176)
#else
    #error "Wrong system frequency!"
#endif

#define POWER_METERING_SYS_CLK_FEE_MODE         (1)
#define POWER_METERING_SYS_CLK_FEI_MODE         (0)
#if (POWER_METERING_SYS_CLK_FEE_MODE + POWER_METERING_SYS_CLK_FEI_MODE) > 1
    #error "Only one clock mode config can be used!"
#endif

#if (POWER_METERING_SYS_CLK_FEE_MODE + POWER_METERING_SYS_CLK_FEI_MODE) == 0
    #error "One clock mode config need to be used!"
#endif
/* -------------------------------------------------------------------------- */

/* ---------------------- AFE, ADC sampling config macros --------------------*/
/* SAR conversion time 10us / ~12MHz     */

#define POWER_METERING_AFE_PLL_CLK              (12.288e6)
#define POWER_METERING_AFE_DIV                  (AFE_DIV2)
#define POWER_METERING_SAR_CONV_TIME            (10.000e-6)
/* SAR count value is 61.44 ~61 */
#define POWER_METERING_SAR_CONT                 (int16_t)(POWER_METERING_SAR_CONV_TIME * POWER_METERING_AFE_PLL_CLK / (1 << POWER_METERING_AFE_DIV))
#define POWER_METERING_AFE_OSR                  (DEC_OSR1024)
#define POWER_METERING_SAMPLES_PER_SEC          (POWER_METERING_AFE_PLL_CLK / (1 << POWER_METERING_AFE_DIV) / (1 << (POWER_METERING_AFE_OSR + 6)))

/*
 * AFE sampling interval
 * We use 1024 OSR, according to rm, we will sample 6144000 / 1024 = 6000 samples
 * Sampling interval is 1000000 / 6000 = 166.67 us
*/
/* PDB default delay value, 22us, 22 * BUS_CLK / 1000000 = 22 * 24 = 528 */
/* The duration between two afe triggers is about 2 * 61.44 / 6.144 = 20us
 * The adc need to be triggered after afe, so this value better to be
 * set to (2 * 61.44 / 6.144) * (BUS_CLK / 1000000) = 480
 */
#define POWER_METERING_PDB_CONT                 (480)

/*
 CT polarity;
 If current ahead of voltage, ct_comp_polarity = 0;
 if voltage ahead of current, ct_comp_polarity = 1;
 */
#define POWER_METERING_CT_COMP_POLARITY         (1)

/* AFE and ADC macro */
#define POWER_METERING_VOLTAGE_ADC_CH_PH1       (AD8) /* ADCSE0 */
#define POWER_METERING_VOLTAGE_ADC_CH_PH2       (AD9) /* ADCSE1 */
#define POWER_METERING_VOLTAGE_ADC_CH_PH3       (AD10) /* ADCSE2 */
/* If set SIM_MISC to use pdb to trigger adc directly, we don't need to config xbar
 * It is said in RM 42.2.6 that:
 * DebugCounter is paused when processor is in Debug mode,
 * and the counter for the DAC trigger is also paused in Debug mode.
 * PDB and ADC counter will be paused.
 * So this config will make ADC not working in IAR debug mode.
 */
#define POWER_METERING_PDB_TRIG_ADC_DIRECTLY    (0)
/* -------------------------------------------------------------------------- */

/* ------------------------ EMC Testing Control Macro ----------------------- */
/* Macros to enable EMC testing mode */
#define POWER_METERING_ENABLE_ESD_TEST_MODE        (0)
/* -------------------------------------------------------------------------- */

/* ------------------------ CPU Loading measurment Macro ----------------------- */
#define POWER_METERING_ENABLE_CPU_LOADING_TST      (0)
/* This is the average systick counts that an idle loop takes.
   Disable interrupt and let idle run. You can get this value.
*/
#define POWER_METERING_LOADING_IDLE_LOOP_COUNT     (21)
/* -------------------------------------------------------------------------- */

/* -------------------------- Calibration macros -----------------------------*/
/* Enable iteration, when this macro is open, software will do iteration in calibration
 * The iteration will try to make all Qs to 0.
 * Iteration once macro will only do iteration once.
 * Iteration no limit macro will do iteration untill all Qs are 0.
 */
#define POWER_METERING_CALI_ENABLE_ITERATION    (1)
#if POWER_METERING_CALI_ENABLE_ITERATION
    #define POWER_METERING_CALI_ITERATION_ONCE      (0)
    #define POWER_METERING_CALI_ITERATION_NO_LIMIT  (1)

    #if (POWER_METERING_CALI_ITERATION_ONCE + POWER_METERING_CALI_ITERATION_NO_LIMIT) > 1
        #error "Only one iteration config can be used!"
    #endif

    #if (POWER_METERING_CALI_ITERATION_ONCE + POWER_METERING_CALI_ITERATION_NO_LIMIT) == 0
        #error "One iteration config need to be used!"
    #endif
#endif

/* This macro sets the waiting time for data to be stable for calibration */
#define POWER_METERING_CALI_SKIP_TIMEOUT        (25)

/* when calibrate the meter, the init angle, unit is rad 
    0.5L = 60 degree, 60 = pi / 3 = 1.047197551196598
    0.5C = -60, 
    Default is PF=1.0, angle=0.
*/
#define POWER_METERING_CALI_INIT_ANGLE          1.047197551196598 //0.5L
/* CALI_V * CALI_I * 0.5 * SQRT(3), ~952 */
#define POWER_METERING_CALI_Q_TARGET            952.0
//#define POWER_METERING_CALI_INIT_ANGLE        (-1.047197551196598) //0.5C
//#define POWER_METERING_CALI_INIT_ANGLE        (0.0)
//#define POWER_METERING_CALI_Q_TARGET          (0.0)

/* Load calibration data or default data */
#define POWER_METERING_ENABLE_LOAD_CALIB_DATA   (1)
/* -------------------------------------------------------------------------- */

/* -------------------------- PDB or QTMR for phase delay --------------------*/
/* Use PDB to trigger ADC */
/* Use QTimer to trigger ADC */
#define POWER_METERING_USE_QTMR_TRIGGER_ADC     (0)
/* Use software shift, that is afe will trigger adc directly */
#define POWER_METERING_USE_AFE_TRIGGER_ADC      (1)

#if (POWER_METERING_USE_QTMR_TRIGGER_ADC + POWER_METERING_USE_AFE_TRIGGER_ADC) > 1
#error "Only one source can be used to trigger ADC!"
#endif

#if (POWER_METERING_USE_QTMR_TRIGGER_ADC + POWER_METERING_USE_AFE_TRIGGER_ADC) == 0
#error "One source need to be used to trigger ADC!"
#endif
/* -------------------------------------------------------------------------- */

/* ---------------------------- Frequency detection --------------------------*/
/* The grid freq used for calibration */
#define POWER_METERING_CALIB_GRID_FREQ             (F_NOM)
#define POWER_METERING_GRID_FREQ_MIN               (45.0)
#define POWER_METERING_GRID_FREQ_MAX               (65.0)
#define POWER_METERING_FREQ_DET_CH1                (IN0)
#define POWER_METERING_FREQ_DET_CH2                (IN1)
#define POWER_METERING_FREQ_DET_CH3                (IN2)
#define POWER_METERING_ENABLE_FREQ_AVERAGE         (0)

/* Macros for frequency detection */
#define POWER_METERING_FREQ_CALC_TMR_DIV           (BUS_CLK_DIV16)
#if ((POWER_METERING_FREQ_CALC_TMR_DIV) == (BUS_CLK_DIV16))
    #define POWER_METERING_TMRPRCLK        (double)(POWER_METERING_BUS_CLK_FREQ / 16.0)
    #define POWER_METERING_TMR2FREQ(x)     (double)(POWER_METERING_TMRPRCLK / (double)x)
#else
    #error "Please change POWER_METERING_FREQ_CALC_TMR_DIV value according to your timer divider!"
#endif
/* -------------------------------------------------------------------------- */

/* ------------------------------- VLPR --------------------------------------*/
#define POWER_METERING_ENABLE_BATTERY_LOW_POWER         (0)
/* Wakeup time every minute */
#define POWER_METERING_VLPR_WAKEUP_SEC          (5)
/* -------------------------------------------------------------------------- */

/* --------------------------- Enable watchdog -------------------------------*/
#define POWER_METERING_ENABLE_WATCHDOG          (0)
/* -------------------------------------------------------------------------- */

/* --------------------------- Startup delay ---------------------------------*/
/* Startup delay to make test bench stable */
#define POWER_METERING_STARTUP_DELAY            (3 * POWER_METERING_SYS_CLK_FREQ)
/* -------------------------------------------------------------------------- */

#define POWER_METERING_ENABLE_OSC32K_LOAD_CAPS     (1)

/*----------------------- NVM config and runtime data ------------------------*/
#define POWER_METERING_FLASH_CONFIG_ADDR           (0x0001f000)

#define POWER_METERING_RUNTIME_DATA_VALID_FLAG      (0x5555)
/* -------------------------------------------------------------------------- */

/* ---------------------------Enable freemaster ------------------------------*/
#define POWER_METERING_ENABLE_FMSTR             (0)
/* -------------------------------------------------------------------------- */

/*----------------------------- External vref --------------------------------*/
/* Enable vref compensation */
/* Photon256 don't need verf compensation */
#define POWER_METERING_ENABLE_VREF_COMP         (0)

/* Use external vref */
#define POWER_METERING_USE_EXTERNAL_VREF        (0)
/* -------------------------------------------------------------------------- */

/*--------------------------RTC comp and Temp sensor -------------------------*/
/* Enable RTC compensation */
#define POWER_METERING_ENABLE_RTC_COMP          (1)

/* Use internal temperature sensor */
#define POWER_METERING_EXTERNAL_TEMP_AD         (AD11)
/* Buffer for temperature samples, these samples will be averaging. */
#define POWER_METERING_TEMP_SAMPLE_NUM          (32)
#define POWER_METERING_TEMP_COMP_ARRAY_SIZE     (126)
#define POWER_METERING_TEMP_MAX_TEMP            (85)
#define POWER_METERING_TEMP_MIN_TEMP            (-40)
#define POWER_METERING_TEMP_MAX_TEMP_ADC_VAL    (61991)
#define POWER_METERING_TEMP_MIN_TEMP_ADC_VAL    (8574)
/* -------------------------------------------------------------------------- */

/* Macro to RESET pin set as GPIO and driven low to improve EMC susceptibility */
#define POWER_METERING_SET_RESET_PIN_TO_GPIOOUT (0)

/* Enable and set clock source */
#define POWER_METERING_ENABLE_CLKOUT_SRC        (0)

/*----------------------------- SLCD Macros -----------------------------*/
#define POWER_METERING_SLCD_SUPPORT             (0)

/* Display macros, 1 Hz */
#define POWER_METERING_SLCD_REFRESH_FREQ_HZ     (1)
/* -------------------------------------------------------------------------- */

/*----------------------------- Pulse Out Macros -----------------------------*/
/* Const pulse number */
#define POWER_METERING_PULSE_NUM                (COUNTER_RES)

/* 80ms pulse width */
/* 80ms = 80000 us, here unit is us */
#define POWER_METERING_PULSE_WIDTH              (80000)
/* Suppose PIT interval is 50KHz for pulse out, set it to 1000000 / 50000 = 20us here */
/* PIT pulse interval is 20us. If 20us, 50% duty, that is 40us, is not enough,
   User need to change pit pulse interval */
#define POWER_METERING_PULSE_PIT_INTERVAL       (100)
#define POWER_METERING_PULSE_PIT_FREQ           (1000000 / POWER_METERING_PULSE_PIT_INTERVAL)
/* PIT pulse counter setting */
#define POWER_METERING_PULSE_PIT_LDVAL           (POWER_METERING_BUS_CLK_FREQ / POWER_METERING_PULSE_PIT_FREQ)
/* Default pulse counter for 80ms holding time */
#define POWER_METERING_PULSE_HOLDING_COUNT      (POWER_METERING_PULSE_WIDTH / POWER_METERING_PULSE_PIT_INTERVAL)

/*
 P is Wt/s, pulse_out_update_period will be called every
 POWER_METERING_PULSE_PIT_INTERVAL = 50us here.
 So, energy increases (P / ( 1000000 / POWER_METERING_PULSE_PIT_INTERVAL)) every (POWER_METERING_PULSE_PIT_INTERVAL)us here.
 POWER_METERING_PULSE_THRES is (3600 * 1000) / POWER_METERING_PULSE_NUM.
 Magnitude two sides by  1000000 / POWER_METERING_PULSE_PIT_INTERVAL = ).
 */
/*
 pulse and led output 
 default pulse number is 400.

 keep 80ms for pulse out if period > 100ms
 if period < 100ms, duty = 50%

 period = 1000 * 3600 * 1000000 / C / P , 
 C = PULSE_ACTIVE_NUM or PULSE_REACTIVE_NUM
 unit is ms.

 POWER_100MS: power value when period is 100ms.
 POWER_100MS = 1000 * 3600 * 1s / C / 100ms = 10000 * 3600 / C 
*/
#define POWER_METERING_ENERGY_PER_PERIOD(us, pulse)      (3600 * 1000 * 1000000ll) / us / pulse
#define POWER_METERING_PULSE_THRES                 POWER_METERING_ENERGY_PER_PERIOD(POWER_METERING_PULSE_PIT_INTERVAL, POWER_METERING_PULSE_NUM)

/* 100ms threshhold, 100ms = 0.1s */
/* If every 0.1s need a pulse, every pulse means (1000 * 3600 / POWER_METERING_PULSE_NUM) energy
  The P is (1000 * 3600 / POWER_METERING_PULSE_NUM / 0.1) then */
#define POWER_METERING_POWER_100MS                 POWER_METERING_ENERGY_PER_PERIOD(100000, POWER_METERING_PULSE_NUM)
/* -------------------------------------------------------------------------- */

/*------------------------------- Smart Card----------------------------------*/
#define POWER_METERING_SMART_CARD_SUPPORT               (0)
/* Smart Card Clock, 1M - 5M */
/* 3571200 / 372 = 9600 b/s, 4915200 / 512 = 9600 b/s */
#define POWER_METERING_SMART_CARD_CLOCK_FREQ            (3000000)
#if (POWER_METERING_SMART_CARD_CLOCK_FREQ < 1000000) || (POWER_METERING_SMART_CARD_CLOCK_FREQ > 5000000)
    #error "Smart Card Clock should be in the range of 1M to 5M!"
#endif
#define POWER_METERING_SMART_CARD_ISO7816_CARD
/* -------------------------------------------------------------------------- */

/*------------------------------- RS485 --------------------------------------*/
#define POWER_METERING_RS485_SUPPORT                    (0) 
#define POWER_METERING_RS485_UART_BAUD_RATE	        (9600)
#define POWER_METERING_RS485_PARSE_CMD_NUM              (9)
#define POWER_METERING_RS485_SHELL_BUFFER_SIZE          (80)
#define POWER_METERING_RS485_SHELL_MAX_ARGS             (8)
#define POWER_METERING_RS485_SHELL_PROMPT               ">>"
#define POWER_METERING_RS485_SHELL_DISP_LINE_LEN        (16)
/* -------------------------------------------------------------------------- */

/* -----------------------VBAT conversion macros -----------------------------*/
#define POWER_METERING_MAXVBAT                  ((double)4.13)
#define POWER_METERING_VBAT2ADC(x)              ((uint16_t)((x / MAXVBAT) * 65535.0))
#define POWER_METERING_ADC2VBAT(x)              ((MAXVBAT / 65535.0) * (double)x)
/* -------------------------------------------------------------------------- */

/* ----------------------- IrDA macros -----------------------------*/
#define POWER_METERING_IRDA_SUPPORT                     (0)
#define POWER_METERING_IRDA_BAUDRATE                    (4800)
#define POWER_METERING_IRDA_MOD_PWM_FREQ                (38e3)
/* -------------------------------------------------------------------------- */

/* ----------------------- SPI Flash -----------------------------*/
#define POWER_METERING_SF_SUPPORT                       (0)
/* BaudRateDivisor = (SPPR + 1)  2^(SPR + 1) */
/* BaudRate = BUS_CLOCK / BaudRateDivisor */
#define POWER_METERING_SF_SPI_SPPR                      (1)
#define POWER_METERING_SF_SPI_SPR                       (0)
/* -------------------------------------------------------------------------- */

/* 0xffff=read default configuration data           */
/* 0xfff5=perform calibration pre-processing        */
/* 0xffa5=calculate calibration data                */
/* 0xa5a5=calibration completed and stored          */
typedef enum
{
    CALI_FLAG_DEFAULT = 0xfffff,
    CALI_FLAG_INIT = 0xffff5,
    CALI_FLAG_DATA_READY = 0xfffa5,
    CALI_FLAG_ITERA_INIT = 0xff5a5,
    CALI_FLAG_ITERA = 0xfa5a5,
    CALI_FLAG_FIN = 0x5a5a5,
} tMETERING_Calib_Flag;

typedef enum
{
    SAMPLE_FLAG_3DEGREE = 0,    /* delay angle < 3 degree */
    SAMPLE_FLAG_6DEGREE,        /* delay angle > 3 degree and < 6 degree */
    SAMPLE_FLAG_ERROR           /* delay angle > 6 degree, error */
} tMETERING_Sample_Flag;

typedef struct
{
    /* Pre-calibration data - user/measured input                               */  
    double      urms_cal;  /* preset calibration voltage [Vrms]                */
    double      urms_msr;  /* measured voltage [Vrms]                          */
    double      irms_cal;  /* preset calibration current [Arms]                */
    double      irms_msr;  /* measured current [Arms]                          */
    double      angle_cal; /* preset calibration angle [rad] = 45 degrees      */
    double      angle_msr; /* measured angle between voltage and current [rad] */
    double      P_msr;     /* measured active power [W]                        */
    double      Q_msr;     /* measured reactive power [VAR]                    */
    frac32      u_msrmax;  /* measured maximum voltage                         */
    frac32      u_msrmin;  /* measured minimum voltage                         */
    frac32      i_msrmax;  /* measured maximum current                         */
    frac32      i_msrmin;  /* measured minimum current                         */
} tMETERING_Calib_Param_1PH;

typedef struct
{
    /* Post-calibration data - calculated phase delay, offsets and gains        */
    uint8_t     CT_flagPh; /* enum sample_flag ;         */
    int16_t     delay;     /* AFE ch3 vs ch1 delay in modulator clocks */
    int32_t     angle;     /* angel */
    frac32      i_offset;  /* current measurement offset */
    frac32      i_gain;    /* current measurement gain   */
    frac32      u_offset;  /* voltage measurement offset */
    frac32      u_gain;    /* voltage measurement gain   */
} tMETERING_Calib_Data_1PH;

typedef struct
{
    /* Basic power meter configuration data                                     */
    uint16_t      id;         /* Electricity meter serial number                  */
    uint16_t      tarif;      /* tarif T1=1,T2=2,T3=3 and T4=4                    */ 
    uint16_t      wh_idx;     /* number of pulses index for Wh generation         */
    uint16_t      varh_idx;   /* number of pulses index for VARh generation       */ 
    uint16_t      vrefh;      /* VREFH trimming value                             */
    uint16_t      vrefl;      /* VREFL trimming value                             */

    /* pre-calibration data - user/measured input                               */
    tMETERING_Calib_Param_1PH  calib_param1;
    tMETERING_Calib_Param_1PH  calib_param2;
    tMETERING_Calib_Param_1PH  calib_param3;

    /* post-calibration data - calculated phase delay, offsets and gains        */
    tMETERING_Calib_Data_1PH   calib_data1;
    tMETERING_Calib_Data_1PH   calib_data2;
    tMETERING_Calib_Data_1PH   calib_data3;

    /* configuration flag                                                       */
    tMETERING_Calib_Flag       calib_flag;                        
} tMETERING_CONFIG_FLASH_DATA;

typedef struct
{
    int32_t       wh_cnt;     /* active energy counter                            */
    int32_t       varh_cnt;   /* reactive energy counter                          */ 
    uint16_t      hardfault;  /* hardfault flag:TRUE-occurred, FALSE-didn't occur */                                     
    uint16_t      flag;       /* 0x5555= valid data                               */
                              /* 0x----= not valid data - initialization needed   */
} tMETERING_RUNTIME_DATA;

/* Saved the sampled data, 6000/1200 = 5 */
typedef struct {
    frac32 i24_ph[5];
    frac32 u24_ph[5];
    uint8_t index;	
} tMETERING_Sample_Data;

#endif
