/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************//*!
 * @file      appconfig.h
 * @author    
 * @version   
 * @date      
 * @brief     Application configuration and interrupt vector definitions.
 ******************************************************************************/
#ifndef __APPCONFIG_H
#define __APPCONFIG_H

/***************************************************************************//*!
 * @brief   Reserve trace buffer in RAM for Micro Trace Buffer (MTB) trace 
 *          capabilities.  
 * @param   MTB_RAM_RESERVE - this define reserves trace buffer in RAM for use 
 *          by Micro Trace Buffer (MTB). The trace buffer starts at 0x1FFFF000 
 *          and it has size of 64-bytes. You can modify size of the trace buffer 
 *          for example to 256-bytes by the following definition: 
 *          #define MTB_RAM_SIZE 0x100
 * @note    Tracing capabilities reduces performance of the application.
 ******************************************************************************/
//#define MTB_RAM_RESERVE

/***************************************************************************//*!
 * @brief   Modify default boot options. The boot options allows the MCU to 
 *          modify default operations from NVM at reset. 
 *          Supported options:
 * @param   BOOT_FLASH_SECURE - MCU Secured (MCU is unsecured by default; secure
 *                              command is blocked in IAR 6.40.2 flash loader)
 * @param   BOOT_DIV_CLK_BY_8 - Divide clock by 8 (MCU boots up with system clock 
 *                              divide by 1 by default).   
 * @param   BOOT_NMI_DISABLE  - Disable NMI (NMI is enabled by default)
 * @param   BOOT_EXT_CLK      - Device uses external slower clock for flash  
 *                              calibration after reset. Internal 25 MHz flash 
 *                              clock is disabled.
 * @param   BOOT_HWVLPR_MODE  - Device boots in VLPR mode starting next VDD power 
 *                              cycle provided VBAT power is retained (forced by HW).
 * @param   BOOT_SWVLPR_MODE  - Device boots in RUN mode however device enters 
 *                              VLPR by first software instructions of the device
 *                              startup (forced by SW).
 * @note    If neither BOOT_EXE_MODE, BOOT_VLPR_MODE nor BOOT_EXT_FLASH_CLK 
 *          constants are defined then device boots in default mode which is RUN 
 *          mode with internal 25 MHz clock enabled and used for flash calibration.
 ******************************************************************************/
#define BOOT_NMI_DISABLE  

/***************************************************************************//*!
 * @brief   Define specific peripheral configuration structures.
 * @example 
 *          #define PIT_CH_CONFIG                                              \
 *          (tPIT_CH){                                                         \
 *            SET(PIT_TCTRL_TIE_MASK)|SET(PIT_TCTRL_TEN_MASK),                 \
 *          } 
 ******************************************************************************/
/**************************************************************************/ /*!
  * @brief Comparator Frequency Detection Setting:
  * - filter enabled ~ 10us;  if filter disabled, CMP will get 6000Hz which is AFE sample rate(jitter).
  * - hysteresis 5mv,
  * - sample, windowing and trigger modes disabled,
  * - high speed mode,
  * - non-inverting polarity,
  * - unfiltered signal propagated to output (disabled),
  * - comparator enabled,
  * - CMP filter disabled when CR1[SE]=0, 
  * - IRQ and DMA disabled,
  * - DAC disabled.
  *****************************************************************************/
#define CMP_FREQ_DETECT_MODE_CONFIG_EN_DAC(dac)                               \
(tCMP){                                                                       \
/* CR0    */  SET(CMP_CR0_FILTER_CNT(4)) |SET(CMP_CR0_HYSTCTR(0)),            \
/* CR1    */  CLR(CMP_CR1_SE_MASK)       |CLR(CMP_CR1_WE_MASK)|               \
              CLR(CMP_CR1_TRIGM_MASK)    |SET(CMP_CR1_PMODE_MASK)|            \
              CLR(CMP_CR1_INV_MASK)      |CLR(CMP_CR1_COS_MASK)|              \
              CLR(CMP_CR1_OPE_MASK)      |SET(CMP_CR1_EN_MASK),               \
/* FPR    */  SET(CMP_FPR_FILT_PER(0x40)),                                    \
/* SCR    */  CLR(CMP_SCR_DMAEN_MASK)    |SET(CMP_SCR_IER_MASK)|              \
              SET(CMP_SCR_IEF_MASK),                                          \
/* DACCR  */  SET(CMP_DACCR_DACEN_MASK)  |CLR(CMP_DACCR_VRSEL_MASK)| dac      \
}

/**************************************************************************/ /*!
 * @brief   Input Capture Mode Frequency Detection Setting
 * Other characteristics:
 * - Count rising edges
 * - Rollover enabled
 * - One shot Timer mode disabled
 * - Overflow interrupt disabled
 * - Co-channel initialization disabled
 * - Compare interrupt disabled
 * - Input capture interrupt disabled
 * - Input signal polarity not inverted
 * - Continue with normal operation during debug mode (default)
 * - Reload counter on capture event
 * @param   prm_clk_src     Use one of the following @ref tmr_prm_src
 * @param   sec_clk_src     Use one of the following @ref tmr_sec_src 
 * @param   input_mode      Use one of the following @ref tmr_capt_mode
 *****************************************************************************/
#define TMR_FREQ_DETECT_MODE(prm_clk_src,sec_clk_src,input_mode)              \
(tTMR_CH){                                                                    \
/* CTRL   */    SET(TMR_CTRL_CM(COUNT_POSEDGE))|                          \
                SET(TMR_CTRL_PCS(prm_clk_src))|SET(TMR_CTRL_SCS(sec_clk_src))|\
                CLR(TMR_CTRL_DIR_MASK)|CLR(TMR_CTRL_LENGTH_MASK)|             \
                CLR(TMR_CTRL_ONCE_MASK)|CLR(TMR_CTRL_COINIT_MASK)|            \
                SET(TMR_CTRL_OUTMODE(ASSERT_ON_CNTR_ACTIVE)),                 \
/* SCTRL    */  CLR(TMR_SCTRL_TCFIE_MASK)|CLR(TMR_SCTRL_TOFIE_MASK)|          \
                CLR(TMR_SCTRL_IEFIE_MASK)|CLR(TMR_SCTRL_IPS_MASK)|            \
                SET(TMR_SCTRL_CAPTURE_MODE(input_mode))|                      \
                CLR(TMR_SCTRL_MSTR_MASK)|CLR(TMR_SCTRL_EEOF_MASK)|            \
                CLR(TMR_SCTRL_VAL_MASK)|CLR(TMR_SCTRL_TOF_MASK)|              \
                CLR(TMR_SCTRL_OPS_MASK)|CLR(TMR_SCTRL_OEN_MASK)|              \
                CLR(TMR_SCTRL_FORCE_MASK),                                    \
 /* CSCTRL  */  SET(TMR_CSCTRL_DBG_EN(0))|CLR(TMR_CSCTRL_FAULT_MASK)|         \
                CLR(TMR_CSCTRL_ALT_LOAD_MASK)|SET(TMR_CSCTRL_ROC_MASK)|       \
                CLR(TMR_CSCTRL_TCI_MASK)|CLR(TMR_CSCTRL_TCF2EN_MASK)|         \
                CLR(TMR_CSCTRL_TCF1EN_MASK)|SET(TMR_CSCTRL_CL2(0))|           \
                SET(TMR_CSCTRL_CL1(0)),                                       \
}

/**************************************************************************/ /*!
 * @brief   Input Capture Bus Clock Mode to Low Power Oscillator Clock Setting
 * Other characteristics:
 * - Count rising edges
 * - Rollover enabled
 * - One shot Timer mode disabled
 * - Overflow interrupt disabled
 * - Co-channel initialization disabled
 * - Compare interrupt disabled
 * - Input capture interrupt disabled
 * - Input signal polarity not inverted
 * - Continue with normal operation during debug mode (default)
 * - Reload counter on capture event
 * @param   prm_clk_src     Use one of the following @ref tmr_prm_src
 * @param   sec_clk_src     Use one of the following @ref tmr_sec_src 
 * @param   input_mode      Use one of the following @ref tmr_capt_mode
 *****************************************************************************/
#define TMR_BUSCLK_LPOCLK_RATIO_DETECT_MODE(prm_clk_src,sec_clk_src,          \
                                                          input_mode)         \
(tTMR_CH){                                                                    \
/* CTRL   */    SET(TMR_CTRL_CM(COUNT_RISING_EDGE))|                          \
                SET(TMR_CTRL_PCS(prm_clk_src))|SET(TMR_CTRL_SCS(sec_clk_src))|\
                CLR(TMR_CTRL_DIR_MASK)|CLR(TMR_CTRL_LENGTH_MASK)|             \
                CLR(TMR_CTRL_ONCE_MASK)|CLR(TMR_CTRL_COINIT_MASK)|            \
                SET(TMR_CTRL_OUTMODE(ASSERT_ON_CNTR_ACTIVE)),                 \
/* SCTRL    */  CLR(TMR_SCTRL_TCFIE_MASK)|CLR(TMR_SCTRL_TOFIE_MASK)|          \
                CLR(TMR_SCTRL_IEFIE_MASK)|CLR(TMR_SCTRL_IPS_MASK)|            \
                SET(TMR_SCTRL_CAPTURE_MODE(input_mode))|                      \
                CLR(TMR_SCTRL_MSTR_MASK)|CLR(TMR_SCTRL_EEOF_MASK)|            \
                CLR(TMR_SCTRL_VAL_MASK)|CLR(TMR_SCTRL_TOF_MASK)|              \
                CLR(TMR_SCTRL_OPS_MASK)|CLR(TMR_SCTRL_OEN_MASK)|              \
                CLR(TMR_SCTRL_FORCE_MASK),                                    \
 /* CSCTRL  */  SET(TMR_CSCTRL_DBG_EN(0))|CLR(TMR_CSCTRL_FAULT_MASK)|         \
                CLR(TMR_CSCTRL_ALT_LOAD_MASK)|SET(TMR_CSCTRL_ROC_MASK)|       \
                CLR(TMR_CSCTRL_TCI_MASK)|CLR(TMR_CSCTRL_TCF2EN_MASK)|         \
                CLR(TMR_CSCTRL_TCF1EN_MASK)|SET(TMR_CSCTRL_CL2(0))|           \
                SET(TMR_CSCTRL_CL1(0)),                                       \
}

/**************************************************************************/ /*!
 * @brief   Input Capture Bus Clock Mode to Low Power Oscillator Clock Setting
 * Other characteristics:
 * - Count rising edges
 * - Rollover enabled
 * - One shot Timer mode disabled
 * - Overflow interrupt disabled
 * - Co-channel initialization disabled
 * - Compare interrupt disabled
 * - Input capture interrupt disabled
 * - Input signal polarity not inverted
 * - Continue with normal operation during debug mode (default)
 * - Reload counter on capture event
 * @param   prm_clk_src     Use one of the following @ref tmr_prm_src
 * @param   sec_clk_src     Use one of the following @ref tmr_sec_src 
 * @param   input_mode      Use one of the following @ref tmr_capt_mode
 *****************************************************************************/
#define TMR_PHASE_ERROR_COMPENSATION_MODE(prm_clk_src,sec_clk_src,            \
                                                          input_mode)         \
(tTMR_CH){                                                                    \
/* CTRL   */    SET(TMR_CTRL_CM(COUNT_TRIGGERED))|                            \
                SET(TMR_CTRL_PCS(prm_clk_src))|SET(TMR_CTRL_SCS(sec_clk_src))|\
                CLR(TMR_CTRL_DIR_MASK)|SET(TMR_CTRL_LENGTH_MASK)|             \
                CLR(TMR_CTRL_ONCE_MASK)|CLR(TMR_CTRL_COINIT_MASK)|            \
                SET(TMR_CTRL_OUTMODE(SET_ON_COMP_CLR_ON_SEC_EDGE)),           \
/* SCTRL    */  CLR(TMR_SCTRL_TCFIE_MASK)|CLR(TMR_SCTRL_TOFIE_MASK)|          \
                CLR(TMR_SCTRL_IEFIE_MASK)|CLR(TMR_SCTRL_IPS_MASK)|            \
                SET(TMR_SCTRL_CAPTURE_MODE(input_mode))|                      \
                CLR(TMR_SCTRL_MSTR_MASK)|CLR(TMR_SCTRL_EEOF_MASK)|            \
                CLR(TMR_SCTRL_VAL_MASK)|CLR(TMR_SCTRL_TOF_MASK)|              \
                CLR(TMR_SCTRL_OPS_MASK)|CLR(TMR_SCTRL_OEN_MASK)|              \
                CLR(TMR_SCTRL_FORCE_MASK),                                    \
 /* CSCTRL  */  SET(TMR_CSCTRL_DBG_EN(0))|CLR(TMR_CSCTRL_FAULT_MASK)|         \
                CLR(TMR_CSCTRL_ALT_LOAD_MASK)|SET(TMR_CSCTRL_ROC_MASK)|       \
                CLR(TMR_CSCTRL_TCI_MASK)|CLR(TMR_CSCTRL_TCF2EN_MASK)|         \
                CLR(TMR_CSCTRL_TCF1EN_MASK)|SET(TMR_CSCTRL_CL2(0))|           \
                SET(TMR_CSCTRL_CL1(0)),                                       \
}

/**************************************************************************/ /*!
 * @brief Selects 16-bit hardware triggered mode with internal voltage reference.
 *****************************************************************************/
#define ADC_MODULE_DIV4_16B_HWTRG_IREF_CONFIG                                 \
(tADC){                                                                       \
/* CFG1   */  CLR(ADC_CFG1_ADLPC_MASK)|SET(ADC_CFG1_ADIV(2))|                 \
              SET(ADC_CFG1_ADLSMP_MASK)|SET(ADC_CFG1_MODE(3))|                \
              SET(ADC_CFG1_ADICLK(0)),                                        \
/* CFG2   */  CLR(ADC_CFG2_MUXSEL_MASK)|CLR(ADC_CFG2_ADACKEN_MASK)|           \
              SET(ADC_CFG2_ADHSC_MASK)|SET(ADC_CFG2_ADLSTS(0)),               \
/* CV1    */  0l,                                                             \
/* CV2    */  0l,                                                             \
/* SC2    */  SET(ADC_SC2_ADTRG_MASK)|CLR(ADC_SC2_ACFE_MASK)|                 \
              CLR(ADC_SC2_ACFGT_MASK)|CLR(ADC_SC2_ACREN_MASK)|                \
              CLR(ADC_SC2_DMAEN_MASK)|SET(ADC_SC2_REFSEL(1)),                 \
/* SC3    */  CLR(ADC_SC3_CAL_MASK)|CLR(ADC_SC3_ADCO_MASK)|                   \
              CLR(ADC_SC3_AVGE_MASK)|SET(ADC_SC3_AVGS(0)),                    \
}

/***************************************************************************//*!
 * @brief PBD init with HW trigger.
 * @details  Delay registers are updated when the counter reaches the MOD 
 * register value, HW trigger is enabled, interrupt is disabled, DMA is disabled,
 * PDB is enabled, one-shot mode is configured, delay value will be loaded from
 * buffer immediately.
 * @param trgs    Select one of the @ref pdb_triggers.
 * @param modulo  PDB modulus (0 - 65535).
 * @param idly    PDB interrupt delay (0 - 65535).
 * @showinitializer
 ******************************************************************************/
#define PDB_MODULE_HWTRG_ONESHOT_LOAD_DELAY_IMMEDIATELY_CONFIG(trgs,presc,mult,modulo,idly)                     \
(tPDB){                                                                       \
/* SC   */ SET(PDB_SC_LDMOD(0))|CLR(PDB_SC_PDBEIE_MASK)|                      \
/* .... */ CLR(PDB_SC_SWTRIG_MASK)|CLR(PDB_SC_DMAEN_MASK)|                    \
/* .... */ SET(PDB_SC_PRESCALER(presc))|SET(PDB_SC_TRGSEL(trgs))|                 \
/* .... */ SET(PDB_SC_PDBEN_MASK)|CLR(PDB_SC_PDBIE_MASK)|                     \
/* .... */ SET(PDB_SC_MULT(mult))|CLR(PDB_SC_CONT_MASK)|SET(PDB_SC_LDOK_MASK),   \
/* MOD  */ modulo,                                                            \
/* IDLY */ idly,                                                              \
}

/***************************************************************************//*!
 * @brief   Configures LPUART to operate in interrupt mode. Receive and transmit
 *          data is not inverted.
 * @details Configures LPUART to operate in interrupt mode. Receive and transmit
 *          data is not inverted.
 * @param   brate     Baud rate.
 * @param   clk       Module clock in Hz.
 * @showinitializer
 ******************************************************************************/
#define LPUART_MODULE_INTRMODE_IRDA_CONFIG(brate,clk, osr)                    \
(tLPUART){                                                                    \
/* BAUD  */ CLR(LPUART_BAUD_MAEN1_MASK)|CLR(LPUART_BAUD_MAEN2_MASK)|          \
/* ..... */ CLR(LPUART_BAUD_M10_MASK)|((15<<24)&0x1f000000)|                  \
/* ..... */ CLR(LPUART_BAUD_TDMAE_MASK)|CLR(LPUART_BAUD_RDMAE_MASK)|          \
/* ..... */ SET(LPUART_BAUD_MATCFG(0x0))|CLR(LPUART_BAUD_BOTHEDGE_MASK)|      \
/* ..... */ CLR(LPUART_BAUD_RESYNCDIS_MASK)|CLR(LPUART_BAUD_LBKDIE_MASK)|     \
/* ..... */ CLR(LPUART_BAUD_RXEDGIE_MASK)|CLR(LPUART_BAUD_SBNS_MASK)|         \
/* ..... */ (LPUART_CALC_SBR(brate,clk)&0x1FFF),                              \
/* STAT  */ CLR(LPUART_STAT_MSBF_MASK)|CLR(LPUART_STAT_RXINV_MASK)|           \
/* ..... */ CLR(LPUART_STAT_RWUID_MASK)|CLR(LPUART_STAT_BRK13_MASK)|          \
/* ..... */ CLR(LPUART_STAT_LBKDE_MASK),                                      \
/* CTRL  */ CLR(LPUART_CTRL_R8T9_MASK)|CLR(LPUART_CTRL_R9T8_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_TXDIR_MASK)|CLR(LPUART_CTRL_TXINV_MASK)|          \
/* ..... */ CLR(LPUART_CTRL_ORIE_MASK)|CLR(LPUART_CTRL_NEIE_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_FEIE_MASK)|CLR(LPUART_CTRL_PEIE_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_TIE_MASK)|CLR(LPUART_CTRL_TCIE_MASK)|             \
/* ..... */ CLR(LPUART_CTRL_RIE_MASK)|CLR(LPUART_CTRL_ILIE_MASK)|             \
/* ..... */ CLR(LPUART_CTRL_TE_MASK)|CLR(LPUART_CTRL_RE_MASK)|                \
/* ..... */ CLR(LPUART_CTRL_RWU_MASK)|CLR(LPUART_CTRL_SBK_MASK)|              \
/* ..... */ CLR(LPUART_CTRL_MA1IE_MASK)|CLR(LPUART_CTRL_MA2IE_MASK)|          \
/* ..... */ SET(LPUART_CTRL_IDLECFG(0x0))|CLR(LPUART_CTRL_LOOPS_MASK)|        \
/* ..... */ CLR(LPUART_CTRL_DOZEEN_MASK)|CLR(LPUART_CTRL_RSRC_MASK)|          \
/* ..... */ CLR(LPUART_CTRL_M_MASK)|CLR(LPUART_CTRL_WAKE_MASK)|               \
/* ..... */ CLR(LPUART_CTRL_ILT_MASK)|CLR(LPUART_CTRL_PE_MASK)|               \
/* ..... */ CLR(LPUART_CTRL_PT_MASK),                                         \
/* MATCH */ SET(LPUART_MATCH_MA2(0x0))|SET(LPUART_MATCH_MA1(0x0)),            \
/* MODDIR*/ CLR(LPUART_MODIR_IREN_MASK)|CLR(LPUART_MODIR_TNP_MASK(osr))|      \
/* ..... */ CLR(LPUART_MODIR_TXCTSSRC_MASK)|CLR(LPUART_MODIR_TXCTSC_MASK)|    \
/* ..... */ CLR(LPUART_MODIR_RXRTSE_MASK)|CLR(LPUART_MODIR_TXRTSPOL_MASK)|    \
/* ..... */ CLR(LPUART_MODIR_TXRTSE_MASK)|CLR(LPUART_MODIR_TXCTSE_MASK)       \
}

#define LPUART_MODULE_INTRMODE_TXINV_IRDA_CONFIG(brate,clk, osr)                    \
(tLPUART){                                                                    \
/* BAUD  */ CLR(LPUART_BAUD_MAEN1_MASK)|CLR(LPUART_BAUD_MAEN2_MASK)|          \
/* ..... */ CLR(LPUART_BAUD_M10_MASK)|((15<<24)&0x1f000000)|                  \
/* ..... */ CLR(LPUART_BAUD_TDMAE_MASK)|CLR(LPUART_BAUD_RDMAE_MASK)|          \
/* ..... */ SET(LPUART_BAUD_MATCFG(0x0))|CLR(LPUART_BAUD_BOTHEDGE_MASK)|      \
/* ..... */ CLR(LPUART_BAUD_RESYNCDIS_MASK)|CLR(LPUART_BAUD_LBKDIE_MASK)|     \
/* ..... */ CLR(LPUART_BAUD_RXEDGIE_MASK)|CLR(LPUART_BAUD_SBNS_MASK)|         \
/* ..... */ (LPUART_CALC_SBR(brate,clk)&0x1FFF),                              \
/* STAT  */ CLR(LPUART_STAT_MSBF_MASK)|CLR(LPUART_STAT_RXINV_MASK)|           \
/* ..... */ CLR(LPUART_STAT_RWUID_MASK)|CLR(LPUART_STAT_BRK13_MASK)|          \
/* ..... */ CLR(LPUART_STAT_LBKDE_MASK),                                      \
/* CTRL  */ CLR(LPUART_CTRL_R8T9_MASK)|CLR(LPUART_CTRL_R9T8_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_TXDIR_MASK)|SET(LPUART_CTRL_TXINV_MASK)|          \
/* ..... */ CLR(LPUART_CTRL_ORIE_MASK)|CLR(LPUART_CTRL_NEIE_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_FEIE_MASK)|CLR(LPUART_CTRL_PEIE_MASK)|            \
/* ..... */ CLR(LPUART_CTRL_TIE_MASK)|CLR(LPUART_CTRL_TCIE_MASK)|             \
/* ..... */ CLR(LPUART_CTRL_RIE_MASK)|CLR(LPUART_CTRL_ILIE_MASK)|             \
/* ..... */ CLR(LPUART_CTRL_TE_MASK)|CLR(LPUART_CTRL_RE_MASK)|                \
/* ..... */ CLR(LPUART_CTRL_RWU_MASK)|CLR(LPUART_CTRL_SBK_MASK)|              \
/* ..... */ CLR(LPUART_CTRL_MA1IE_MASK)|CLR(LPUART_CTRL_MA2IE_MASK)|          \
/* ..... */ SET(LPUART_CTRL_IDLECFG(0x0))|CLR(LPUART_CTRL_LOOPS_MASK)|        \
/* ..... */ CLR(LPUART_CTRL_DOZEEN_MASK)|CLR(LPUART_CTRL_RSRC_MASK)|          \
/* ..... */ CLR(LPUART_CTRL_M_MASK)|CLR(LPUART_CTRL_WAKE_MASK)|               \
/* ..... */ CLR(LPUART_CTRL_ILT_MASK)|CLR(LPUART_CTRL_PE_MASK)|               \
/* ..... */ CLR(LPUART_CTRL_PT_MASK),                                         \
/* MATCH */ SET(LPUART_MATCH_MA2(0x0))|SET(LPUART_MATCH_MA1(0x0)),            \
/* MODDIR*/ CLR(LPUART_MODIR_IREN_MASK)|CLR(LPUART_MODIR_TNP_MASK(osr))|      \
/* ..... */ CLR(LPUART_MODIR_TXCTSSRC_MASK)|CLR(LPUART_MODIR_TXCTSC_MASK)|    \
/* ..... */ CLR(LPUART_MODIR_RXRTSE_MASK)|CLR(LPUART_MODIR_TXRTSPOL_MASK)|    \
/* ..... */ CLR(LPUART_MODIR_TXRTSE_MASK)|CLR(LPUART_MODIR_TXCTSE_MASK)       \
}

/**************************************************************************/ /*!
  * @brief   LCD front plane pins initialization:
  * @details Enables LCD pins for LCD glas driving:
  *          - Use "CLR" for disable pin, 
  *          - Use "SET" for enable pin
  * @showinitializer
  *****************************************************************************/
#define LCD_FRONT_PLANE_PINS_METERING                                          \
(tLCD_FP){                                                                     \
/* BPENL */ CLR(LCD_00) |CLR(LCD_01) |SET(LCD_02) |SET(LCD_03)|                \
/* ..... */ SET(LCD_04) |SET(LCD_05) |SET(LCD_06) |SET(LCD_07)|                \
/* ..... */ SET(LCD_08) |SET(LCD_09) |SET(LCD_10) |SET(LCD_11)|                \
/* ..... */ SET(LCD_12) |SET(LCD_13) |SET(LCD_14) |SET(LCD_15)|                \
/* ..... */ SET(LCD_16) |SET(LCD_17) |SET(LCD_18) |SET(LCD_19)|                \
/* ..... */ SET(LCD_20) |CLR(LCD_21) |CLR(LCD_22) |SET(LCD_23)|                \
/* ..... */ SET(LCD_24) |SET(LCD_25) |SET(LCD_26) |SET(LCD_27)|                \
/* ..... */ SET(LCD_28) |SET(LCD_29) |CLR(LCD_30) |SET(LCD_31),                \
/* BPENH */ CLR(LCD_32) |CLR(LCD_33) |CLR(LCD_34) |CLR(LCD_35)|                \
/* ..... */ CLR(LCD_36) |CLR(LCD_37) |CLR(LCD_38) |CLR(LCD_39)|                \
/* ..... */ CLR(LCD_40) |CLR(LCD_41) |CLR(LCD_42) |CLR(LCD_43)|                \
/* ..... */ SET(LCD_44) |SET(LCD_45) |SET(LCD_46) |SET(LCD_47)|                \
/* ..... */ CLR(LCD_48) |CLR(LCD_49) |SET(LCD_50) |CLR(LCD_51)|                \
/* ..... */ SET(LCD_52) |SET(LCD_53) |SET(LCD_54) |SET(LCD_55)|                \
/* ..... */ SET(LCD_56) |SET(LCD_57) |SET(LCD_58) |SET(LCD_59)|                \
/* ..... */ CLR(LCD_60) |CLR(LCD_61) |CLR(LCD_62) |CLR(LCD_63)                 \
}

/**************************************************************************/ /*!
  * @brief   LCD back plane pins initialization:
  * @details Enables LCD backplane pins for LCD glass driving:
  *          - Use "CLR" for disable pin, 
  *          - Use "SET" for enable pin
  * @showinitializer
  *****************************************************************************/
#define LCD_BACK_PLANE_PINS_METERING                                           \
(tLCD_BP){                                                                     \
/* BPENL */ CLR(LCD_00) |CLR(LCD_01) |CLR(LCD_02) |CLR(LCD_03)|                \
/* ..... */ CLR(LCD_04) |CLR(LCD_05) |CLR(LCD_06) |CLR(LCD_07)|                \
/* ..... */ CLR(LCD_08) |CLR(LCD_09) |CLR(LCD_10) |CLR(LCD_11)|                \
/* ..... */ CLR(LCD_12) |CLR(LCD_13) |CLR(LCD_14) |CLR(LCD_15)|                \
/* ..... */ CLR(LCD_16) |CLR(LCD_17) |CLR(LCD_18) |CLR(LCD_19)|                \
/* ..... */ CLR(LCD_20) |CLR(LCD_21) |CLR(LCD_22) |SET(LCD_23)|                \
/* ..... */ CLR(LCD_24) |SET(LCD_25) |CLR(LCD_26) |SET(LCD_27)|                \
/* ..... */ CLR(LCD_28) |CLR(LCD_29) |CLR(LCD_30) |SET(LCD_31),                \
/* BPENH */ CLR(LCD_32) |CLR(LCD_33) |CLR(LCD_34) |CLR(LCD_35)|                \
/* ..... */ CLR(LCD_36) |CLR(LCD_37) |CLR(LCD_38) |CLR(LCD_39)|                \
/* ..... */ CLR(LCD_40) |CLR(LCD_41) |CLR(LCD_42) |CLR(LCD_43)|                \
/* ..... */ SET(LCD_44) |CLR(LCD_45) |SET(LCD_46) |CLR(LCD_47)|                \
/* ..... */ CLR(LCD_48) |CLR(LCD_49) |SET(LCD_50) |CLR(LCD_51)|                \
/* ..... */ CLR(LCD_52) |CLR(LCD_53) |CLR(LCD_54) |CLR(LCD_55)|                \
/* ..... */ CLR(LCD_56) |CLR(LCD_57) |SET(LCD_58) |CLR(LCD_59)|                \
/* ..... */ CLR(LCD_60) |CLR(LCD_61) |CLR(LCD_62) |CLR(LCD_63)                 \
}

/**************************************************************************/ /*!
  * @brief   LCD segments initialization:
  * @details Set back planes pins for LCD panel. 
  *          - Use "LCD_OFF" for no back plane assign, 
  *          - Use "LCD_BP1" for back plane 0, 
  *          - Use "LCD_BP2" for back plane 1, 
  *          - Use "LCD_BP3" for back plane 2, 
  *          - Use "LCD_BP4" for back plane 3, 
  *          - Use "LCD_BP5" for back plane 4, 
  *          - Use "LCD_BP6" for back plane 5, 
  *          - Use "LCD_BP7" for back plane 6, 
  *          - Use "LCD_BP8" for back plane 7
  * @showinitializer
  *****************************************************************************/
#define LCD_ASSIGN_BACK_PLANES_METERING                                        \
(tLCD_CP){                                                                     \
/* WF3TO0   */  LCD_WF_WF0 (LCD_OFF) |LCD_WF_WF1 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF2 (LCD_OFF) |LCD_WF_WF3 (LCD_OFF),                    \
/* WF7TO4   */  LCD_WF_WF4 (LCD_OFF) |LCD_WF_WF5 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF6 (LCD_OFF) |LCD_WF_WF7 (LCD_OFF),                    \
/* WF11TO8  */  LCD_WF_WF8 (LCD_OFF) |LCD_WF_WF9 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF10(LCD_OFF) |LCD_WF_WF11(LCD_OFF),                    \
/* WF15TO12 */  LCD_WF_WF12(LCD_OFF) |LCD_WF_WF13(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF14(LCD_OFF) |LCD_WF_WF15(LCD_OFF),                    \
/* WF19TO16 */  LCD_WF_WF16(LCD_OFF) |LCD_WF_WF17(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF18(LCD_OFF) |LCD_WF_WF19(LCD_OFF),                    \
/* WF23TO20 */  LCD_WF_WF20(LCD_OFF) |LCD_WF_WF21(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF22(LCD_OFF) |LCD_WF_WF23(LCD_BP3),                    \
/* WF27TO24 */  LCD_WF_WF24(LCD_OFF) |LCD_WF_WF25(LCD_BP5)|                    \
/* ........ */  LCD_WF_WF26(LCD_OFF) |LCD_WF_WF27(LCD_BP6),                    \
/* WF31TO28 */  LCD_WF_WF28(LCD_OFF) |LCD_WF_WF29(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF30(LCD_OFF) |LCD_WF_WF31(LCD_BP7),                    \
/* WF35TO32 */  LCD_WF_WF32(LCD_OFF) |LCD_WF_WF33(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF34(LCD_OFF) |LCD_WF_WF35(LCD_OFF),                    \
/* WF39TO36 */  LCD_WF_WF36(LCD_OFF) |LCD_WF_WF37(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF38(LCD_OFF) |LCD_WF_WF39(LCD_OFF),                    \
/* WF43TO40 */  LCD_WF_WF40(LCD_OFF) |LCD_WF_WF41(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF42(LCD_OFF) |LCD_WF_WF43(LCD_OFF),                    \
/* WF47TO44 */  LCD_WF_WF44(LCD_BP2) |LCD_WF_WF45(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF46(LCD_BP4) |LCD_WF_WF47(LCD_OFF),                    \
/* WF51TO48 */  LCD_WF_WF48(LCD_OFF) |LCD_WF_WF49(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF50(LCD_BP8) |LCD_WF_WF51(LCD_OFF),                    \
/* WF55TO52 */  LCD_WF_WF52(LCD_OFF) |LCD_WF_WF53(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF54(LCD_OFF) |LCD_WF_WF55(LCD_OFF),                    \
/* WF59TO56 */  LCD_WF_WF56(LCD_OFF) |LCD_WF_WF57(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF58(LCD_BP1) |LCD_WF_WF59(LCD_OFF),                    \
/* WF63TO60 */  LCD_WF_WF60(LCD_OFF) |LCD_WF_WF61(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF62(LCD_OFF) |LCD_WF_WF63(LCD_OFF),                    \
}

/**************************************************************************/ /*!
  * @brief   LCD segments initialization:
  * @details Set all LCD panel segments to ON state. 
  *          - Use "LCD_BP1" for back plane 0, 
  *          - Use "LCD_BP2" for back plane 1, 
  *          - Use "LCD_BP3" for back plane 2, 
  *          - Use "LCD_BP4" for back plane 3, 
  *          - Use "LCD_BP5" for back plane 4, 
  *          - Use "LCD_BP6" for back plane 5, 
  *          - Use "LCD_BP7" for back plane 6, 
  *          - Use "LCD_BP8" for back plane 7, 
  *          - Use "LCD_OFF" for unused segment, 
  *          - Use "LCD_ON1" for segment ON with 1 back plane, 
  *          - Use "LCD_ON2" for segment ON with 2 back planes, 
  *          - Use "LCD_ON3" for segment ON with 3 back planes,  
  *          - Use "LCD_ON4" for segment ON with 4 back planes, 
  *          - Use "LCD_ON5" for segment ON with 5 back planes, 
  *          - Use "LCD_ON6" for segment ON with 6 back planes,  
  *          - Use "LCD_ON7" for segment ON with 7 back planes,  
  *          - Use "LCD_ON8" for segment ON with 8 back planes
  * @showinitializer
  *****************************************************************************/
#define LCD_ALL_SEGMENTS_ON_METERING                                           \
(tLCD_CP){                                                                     \
/* WF3TO0   */  LCD_WF_WF0 (LCD_OFF) |LCD_WF_WF1 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF2 (LCD_ON8) |LCD_WF_WF3 (LCD_ON8),                    \
/* WF7TO4   */  LCD_WF_WF4 (LCD_ON8) |LCD_WF_WF5 (LCD_ON8)|                    \
/* ........ */  LCD_WF_WF6 (LCD_ON8) |LCD_WF_WF7 (LCD_ON8),                    \
/* WF11TO8  */  LCD_WF_WF8 (LCD_ON8) |LCD_WF_WF9 (LCD_ON8)|                    \
/* ........ */  LCD_WF_WF10(LCD_ON8) |LCD_WF_WF11(LCD_ON8),                    \
/* WF15TO12 */  LCD_WF_WF12(LCD_ON8) |LCD_WF_WF13(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF14(LCD_ON8) |LCD_WF_WF15(LCD_ON8),                    \
/* WF19TO16 */  LCD_WF_WF16(LCD_ON8) |LCD_WF_WF17(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF18(LCD_ON8) |LCD_WF_WF19(LCD_ON8),                    \
/* WF23TO20 */  LCD_WF_WF20(LCD_ON8) |LCD_WF_WF21(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF22(LCD_OFF) |LCD_WF_WF23(LCD_BP3),                    \
/* WF27TO24 */  LCD_WF_WF24(LCD_ON8) |LCD_WF_WF25(LCD_BP5)|                    \
/* ........ */  LCD_WF_WF26(LCD_ON8) |LCD_WF_WF27(LCD_BP6),                    \
/* WF31TO28 */  LCD_WF_WF28(LCD_ON8) |LCD_WF_WF29(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF30(LCD_OFF) |LCD_WF_WF31(LCD_BP7),                    \
/* WF35TO32 */  LCD_WF_WF32(LCD_OFF) |LCD_WF_WF33(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF34(LCD_OFF) |LCD_WF_WF35(LCD_OFF),                    \
/* WF39TO36 */  LCD_WF_WF36(LCD_OFF) |LCD_WF_WF37(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF38(LCD_OFF) |LCD_WF_WF39(LCD_OFF),                    \
/* WF43TO40 */  LCD_WF_WF40(LCD_OFF) |LCD_WF_WF41(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF42(LCD_OFF) |LCD_WF_WF43(LCD_OFF),                    \
/* WF47TO44 */  LCD_WF_WF44(LCD_BP2) |LCD_WF_WF45(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF46(LCD_BP4) |LCD_WF_WF47(LCD_ON8),                    \
/* WF51TO48 */  LCD_WF_WF48(LCD_OFF) |LCD_WF_WF49(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF50(LCD_BP8) |LCD_WF_WF51(LCD_OFF),                    \
/* WF55TO52 */  LCD_WF_WF52(LCD_ON8) |LCD_WF_WF53(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF54(LCD_ON8) |LCD_WF_WF55(LCD_ON8),                    \
/* WF59TO56 */  LCD_WF_WF56(LCD_ON8) |LCD_WF_WF57(LCD_ON8)|                    \
/* ........ */  LCD_WF_WF58(LCD_BP1) |LCD_WF_WF59(LCD_ON8),                    \
/* WF63TO60 */  LCD_WF_WF60(LCD_OFF) |LCD_WF_WF61(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF62(LCD_OFF) |LCD_WF_WF63(LCD_OFF),                    \
}

/**************************************************************************/ /*!
  * @brief   LCD segments initialization:
  * @details Set all LCD panel segments to ON state. 
  *          - Use "LCD_BP1" for back plane 0, 
  *          - Use "LCD_BP2" for back plane 1, 
  *          - Use "LCD_BP3" for back plane 2, 
  *          - Use "LCD_BP4" for back plane 3, 
  *          - Use "LCD_BP5" for back plane 4, 
  *          - Use "LCD_BP6" for back plane 5, 
  *          - Use "LCD_BP7" for back plane 6, 
  *          - Use "LCD_BP8" for back plane 7, 
  *          - Use "LCD_OFF" for unused segment
  * @showinitializer
  *****************************************************************************/
#define LCD_ALL_SEGMENTS_OFF_METERING                                          \
(tLCD_CP){                                                                     \
/* WF3TO0   */  LCD_WF_WF0 (LCD_OFF) |LCD_WF_WF1 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF2 (LCD_OFF) |LCD_WF_WF3 (LCD_OFF),                    \
/* WF7TO4   */  LCD_WF_WF4 (LCD_OFF) |LCD_WF_WF5 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF6 (LCD_OFF) |LCD_WF_WF7 (LCD_OFF),                    \
/* WF11TO8  */  LCD_WF_WF8 (LCD_OFF) |LCD_WF_WF9 (LCD_OFF)|                    \
/* ........ */  LCD_WF_WF10(LCD_OFF) |LCD_WF_WF11(LCD_OFF),                    \
/* WF15TO12 */  LCD_WF_WF12(LCD_OFF) |LCD_WF_WF13(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF14(LCD_OFF) |LCD_WF_WF15(LCD_OFF),                    \
/* WF19TO16 */  LCD_WF_WF16(LCD_OFF) |LCD_WF_WF17(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF18(LCD_OFF) |LCD_WF_WF19(LCD_OFF),                    \
/* WF23TO20 */  LCD_WF_WF20(LCD_OFF) |LCD_WF_WF21(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF22(LCD_OFF) |LCD_WF_WF23(LCD_BP3),                    \
/* WF27TO24 */  LCD_WF_WF24(LCD_OFF) |LCD_WF_WF25(LCD_BP5)|                    \
/* ........ */  LCD_WF_WF26(LCD_OFF) |LCD_WF_WF27(LCD_BP6),                    \
/* WF31TO28 */  LCD_WF_WF28(LCD_OFF) |LCD_WF_WF29(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF30(LCD_OFF) |LCD_WF_WF31(LCD_BP7),                    \
/* WF35TO32 */  LCD_WF_WF32(LCD_OFF) |LCD_WF_WF33(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF34(LCD_OFF) |LCD_WF_WF35(LCD_OFF),                    \
/* WF39TO36 */  LCD_WF_WF36(LCD_OFF) |LCD_WF_WF37(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF38(LCD_OFF) |LCD_WF_WF39(LCD_OFF),                    \
/* WF43TO40 */  LCD_WF_WF40(LCD_OFF) |LCD_WF_WF41(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF42(LCD_OFF) |LCD_WF_WF43(LCD_OFF),                    \
/* WF47TO44 */  LCD_WF_WF44(LCD_BP2) |LCD_WF_WF45(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF46(LCD_BP4) |LCD_WF_WF47(LCD_OFF),                    \
/* WF51TO48 */  LCD_WF_WF48(LCD_OFF) |LCD_WF_WF49(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF50(LCD_BP8) |LCD_WF_WF51(LCD_OFF),                    \
/* WF55TO52 */  LCD_WF_WF52(LCD_OFF) |LCD_WF_WF53(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF54(LCD_OFF) |LCD_WF_WF55(LCD_OFF),                    \
/* WF59TO56 */  LCD_WF_WF56(LCD_OFF) |LCD_WF_WF57(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF58(LCD_BP1) |LCD_WF_WF59(LCD_OFF),                    \
/* WF63TO60 */  LCD_WF_WF60(LCD_OFF) |LCD_WF_WF61(LCD_OFF)|                    \
/* ........ */  LCD_WF_WF62(LCD_OFF) |LCD_WF_WF63(LCD_OFF)                     \
}

/***************************************************************************//*!
 * @brief   Configures UART to operate in polling mode. Receive and transmit data 
 *          is not inverted.
 * @details Configures UART to operate in polling mode. Receive and transmit data 
 *          is not inverted.
 * @param   brate     Baud rate.
 * @param   bclk      Bus clock in Hz.
 * @showinitializer
 ******************************************************************************/
#define UART_MODULE_POLLMODE_ISO7816_EMV_CONFIG(brate,bclk)                       \
(tUART){                                                                      \
/* BDH */ CLR(UART_BDH_RXEDGIE_MASK)|((CALC_SBR(brate,bclk)>>8)&0x1f),        \
/* BDL */ ((CALC_SBR(brate,bclk)>>0)&0xff),                                   \
/* C1  */ SET(UART_C1_LOOPS_MASK)|SET(UART_C1_RSRC_MASK)|                     \
/* ... */ SET(UART_C1_M_MASK)|CLR(UART_C1_WAKE_MASK)|                         \
/* ... */ CLR(UART_C1_ILT_MASK)|SET(UART_C1_PE_MASK)|                         \
/* ... */ CLR(UART_C1_PT_MASK),                                               \
/* C2  */ CLR(UART_C2_TIE_MASK)|CLR(UART_C2_TCIE_MASK)|                       \
/* ... */ CLR(UART_C2_RIE_MASK)|CLR(UART_C2_ILIE_MASK)|                       \
/* ... */ SET(UART_C2_TE_MASK)|SET(UART_C2_RE_MASK)|                          \
/* ... */ CLR(UART_C2_RWU_MASK)|CLR(UART_C2_SBK_MASK),                        \
/* S2  */ CLR(UART_S2_RXEDGIF_MASK)|CLR(UART_S2_MSBF_MASK)|                   \
/* ... */ CLR(UART_S2_RXINV_MASK)|CLR(UART_S2_RWUID_MASK)|                    \
/* ... */ CLR(UART_S2_BRK13_MASK),                                            \
/* C3  */ CLR(UART_C3_T8_MASK)|CLR(UART_C3_TXDIR_MASK)|                       \
/* ... */ CLR(UART_C3_TXINV_MASK)|CLR(UART_C3_ORIE_MASK)|                     \
/* ... */ CLR(UART_C3_NEIE_MASK)|CLR(UART_C3_FEIE_MASK)|                      \
/* ... */ CLR(UART_C3_PEIE_MASK),                                             \
/* MA1 */ SET(UART_MA1_MA(0x00)),                                             \
/* MA2 */ SET(UART_MA2_MA(0x00)),                                             \
/* C4  */ CLR(UART_C4_MAEN1_MASK)|CLR(UART_C4_MAEN2_MASK)|                    \
/* ... */ CLR(UART_C4_M10_MASK)|((CALC_BRFA(brate,bclk)>>0)&0x1f),            \
/* C5  */ CLR(UART_C5_TDMAS_MASK)|CLR(UART_C5_RDMAS_MASK)                     \
}

/***************************************************************************//*!
 * @brief   Configures UART to operate in interrupt mode. Receive and transmit data 
 *          is not inverted.
 * @details Configures UART to operate in interrupt mode. Receive and transmit data 
 *          is not inverted.
 * @param   brate     Baud rate.
 * @param   bclk      Bus clock in Hz.  
 * @showinitializer
 ******************************************************************************/
#define UART_MODULE_INTRMODE_ISO7816_EMV_CONFIG(brate,bclk)                       \
(tUART){                                                                      \
/* BDH */ CLR(UART_BDH_RXEDGIE_MASK)|((CALC_SBR(brate,bclk)>>8)&0x1f),        \
/* BDL */ ((CALC_SBR(brate,bclk)>>0)&0xff),                                   \
/* C1  */ SET(UART_C1_LOOPS_MASK)|SET(UART_C1_RSRC_MASK)|                     \
/* ... */ SET(UART_C1_M_MASK)|CLR(UART_C1_WAKE_MASK)|                         \
/* ... */ CLR(UART_C1_ILT_MASK)|SET(UART_C1_PE_MASK)|                         \
/* ... */ CLR(UART_C1_PT_MASK),                                               \
/* C2  */ CLR(UART_C2_TIE_MASK)|CLR(UART_C2_TCIE_MASK)|                       \
/* ... */ SET(UART_C2_RIE_MASK)|CLR(UART_C2_ILIE_MASK)|                       \
/* ... */ SET(UART_C2_TE_MASK)|SET(UART_C2_RE_MASK)|                          \
/* ... */ CLR(UART_C2_RWU_MASK)|CLR(UART_C2_SBK_MASK),                        \
/* S2  */ CLR(UART_S2_RXEDGIF_MASK)|CLR(UART_S2_MSBF_MASK)|                   \
/* ... */ CLR(UART_S2_RXINV_MASK)|CLR(UART_S2_RWUID_MASK)|                    \
/* ... */ CLR(UART_S2_BRK13_MASK),                                            \
/* C3  */ CLR(UART_C3_T8_MASK)|CLR(UART_C3_TXDIR_MASK)|                       \
/* ... */ CLR(UART_C3_TXINV_MASK)|CLR(UART_C3_ORIE_MASK)|                     \
/* ... */ CLR(UART_C3_NEIE_MASK)|CLR(UART_C3_FEIE_MASK)|                      \
/* ... */ CLR(UART_C3_PEIE_MASK),                                             \
/* MA1 */ SET(UART_MA1_MA(0x00)),                                             \
/* MA2 */ SET(UART_MA2_MA(0x00)),                                             \
/* C4  */ CLR(UART_C4_MAEN1_MASK)|CLR(UART_C4_MAEN2_MASK)|                    \
/* ... */ CLR(UART_C4_M10_MASK)|((CALC_BRFA(brate,bclk)>>0)&0x1f),            \
/* C5  */ CLR(UART_C5_TDMAS_MASK)|CLR(UART_C5_RDMAS_MASK),                    \
}

/***************************************************************************//*!
 * @brief Configures pin(s) to Alternative 1 (GPIO) IRQ on either edge mode.
 * @details Configures port pin(s) with the following characteristics:
 * - Internal pull-up disabled
 * - Digital filter disabled
 * - Interrupt on either edge enabled
 * - Slew rate control disabled
 * - Open-drain mode disabled
 * @showinitializer
 ******************************************************************************/
#define PORT_MODULE_SMART_CARD_DETECT_ALT1_ANYEDGE_IRQ_MODE                                     \
(tPORT){                                                                      \
/* PCR  */ SET(PORT_PCR_IRQC(11))|CLR(PORT_PCR_LK_MASK)|SET(PORT_PCR_MUX(1))| \
/* ...  */ CLR(PORT_PCR_ODE_MASK)|CLR(PORT_PCR_SRE_MASK)|                     \
/* ...  */ SET(PORT_PCR_PE_MASK)|SET(PORT_PCR_PS_MASK),                       \
/* DFER */ SET(PORT_DFER_DFE_MASK), /* use mask for all pins */               \
}

#define PORT_MODULE_BATTERY_PIN_ALT1_ANYEDGE_IRQ_MODE                                     \
(tPORT){                                                                      \
/* PCR  */ SET(PORT_PCR_IRQC(11))|CLR(PORT_PCR_LK_MASK)|SET(PORT_PCR_MUX(1))| \
/* ...  */ CLR(PORT_PCR_ODE_MASK)|CLR(PORT_PCR_SRE_MASK)|                     \
/* ...  */ SET(PORT_PCR_PE_MASK)|SET(PORT_PCR_PS_MASK),                       \
/* DFER */ SET(PORT_DFER_DFE_MASK), /* use mask for all pins */               \
}

/***************************************************************************//*!
 * @brief   Output Compare Mode.
 * @details Output Compare Mode. The timer operates in Output Compare Mode with 
 * the following characteristics:
 * - Counts up
 * - Count until compare then re-initialize.
 * - One shot Timer mode disabled
 * - Overflow interrupt disabled
 * - Co-channel initialization disabled
 * - Compare interrupt enabled
 * - Input capture interrupt disabled
 * - Input signal polarity not inverted
 * - Master operation disabled
 * - External OFLAG force enabled.
 * - Polarity of OFLAG output signal not inverted.
 * - Output not forced to external pin
 * @param   prm_clk_src     Select one of the @ref tmr_prm_src.
 * @showinitializer
 ******************************************************************************/
#define TMR_CH_FIXED_FREQ_PWM_MODE_CONFIG(prm_clk_src)  \
(tTMR_CH){                                                                    \
/* CTRL     */  SET(TMR_CTRL_CM(COUNT_POSEDGE))|                                 \
/* ....     */  SET(TMR_CTRL_PCS(prm_clk_src))|SET(TMR_CTRL_SCS(SEC_CNTR0_INP))|\
/* ....     */  CLR(TMR_CTRL_DIR_MASK)|SET(TMR_CTRL_LENGTH_MASK)|             \
/* ....     */  CLR(TMR_CTRL_ONCE_MASK)|CLR(TMR_CTRL_COINIT_MASK)|            \
/* ....     */  SET(TMR_CTRL_OUTMODE(TGL_ON_COMP)),                           \
/* SCTRL    */  CLR(TMR_SCTRL_TCFIE_MASK)|CLR(TMR_SCTRL_TOFIE_MASK)|          \
/* .....    */  CLR(TMR_SCTRL_IEFIE_MASK)|CLR(TMR_SCTRL_IPS_MASK)|            \
/* .....    */  SET(TMR_SCTRL_CAPTURE_MODE(CAPTURE_DISABLED))|                \
/* .....    */  CLR(TMR_SCTRL_MSTR_MASK)|CLR(TMR_SCTRL_EEOF_MASK)|            \
/* .....    */  CLR(TMR_SCTRL_VAL_MASK)|CLR(TMR_SCTRL_TOF_MASK)|              \
/* .....    */  SET(TMR_SCTRL_FORCE_MASK) |                                                         \
/* .....    */  CLR(TMR_SCTRL_OPS_MASK)|SET(TMR_SCTRL_OEN_MASK),              \
/* CSCTRL   */  SET(TMR_CSCTRL_DBG_EN(0))|CLR(TMR_CSCTRL_FAULT_MASK)|         \
/* ......   */  CLR(TMR_CSCTRL_ALT_LOAD_MASK)|CLR(TMR_CSCTRL_ROC_MASK)|       \
/* ......   */  CLR(TMR_CSCTRL_TCI_MASK)|CLR(TMR_CSCTRL_TCF2EN_MASK)|         \
/* ......   */  CLR(TMR_CSCTRL_TCF1EN_MASK)|SET(TMR_CSCTRL_CL2(0))|           \
/* ......   */  SET(TMR_CSCTRL_CL1(0)),                                       \
}

/***************************************************************************//*!
 * @brief   SPI setting in 8bit POLLING mode.
 * @details SPI setting in 8bit POLLING mode. Baud rate is 1/120 of BUSCLK, 
 *          8-bit polling mode, module enabled after initialization, master mode, 
 *          SS is configured as GPIO (must be driven manually!), no FIFO no DMA 
 *          no MATCH functionality, full-duplex mode.
 * @showinitializer
 ******************************************************************************/
/* BaudRateDivisor = (SPPR + 1) * 2^(SPR + 1) */
#define SPI_MODULE_8B_POLL_CONFIG(sppr, spr)                            \
(tSPI){                                                                        \
/* C1   */ CLR(SPI_C1_SPIE_MASK)|SET(SPI_C1_SPE_MASK)|CLR(SPI_C1_SPTIE_MASK)|  \
/* ..   */ SET(SPI_C1_MSTR_MASK)|CLR(SPI_C1_CPOL_MASK)|CLR(SPI_C1_CPHA_MASK)|  \
/* ..   */ CLR(SPI_C1_SSOE_MASK)|CLR(SPI_C1_LSBFE_MASK),                       \
/* C2   */ CLR(SPI_C2_SPMIE_MASK)|CLR(SPI_C2_SPIMODE_MASK)|                    \
/* ..   */ CLR(SPI_C2_TXDMAE_MASK)|CLR(SPI_C2_MODFEN_MASK)|                    \
/* ..   */ CLR(SPI_C2_RXDMAE_MASK)|CLR(SPI_C2_SPISWAI_MASK)|                   \
/* ..   */ CLR(SPI_C2_SPC0_MASK),                                              \
/* C3   */ 0,   /* C3 is not used for SPI0 module due to no FIFO support */    \
/* BR   */ SET(SPI_BR_SPPR(sppr))|SET(SPI_BR_SPR(spr)),                        \
/* ML   */ 0,                                                                  \
/* MH   */ 0,                                                                  \
}

/***************************************************************************//*!
 * @brief Configures pin(s) to Alternative 1 (GPIO) open-drain mode.
 * @details Configures port pin(s) with the following characteristics:
 * - Internal pull-up enabled
 * - Digital filter disabled
 * - Interrupt disabled
 * - Slew rate control disabled
 * - Open-drain mode enabled
 * @showinitializer
 ******************************************************************************/
#define PORT_MODULE_OD_EN_PULLUP_EN_MODE(alt_val)                             \
(tPORT){                                                                      \
/* PCR  */ SET(PORT_PCR_IRQC(0))|CLR(PORT_PCR_LK_MASK)|SET(PORT_PCR_MUX(alt_val))|  \
/* ...  */ SET(PORT_PCR_ODE_MASK)|CLR(PORT_PCR_SRE_MASK)|                     \
/* ...  */ SET(PORT_PCR_PE_MASK)|SET(PORT_PCR_PS_MASK),                       \
/* DFER */ CLR(PORT_DFER_DFE_MASK), /* use mask for all pins */               \
}

/***************************************************************************//*!
 * @brief   FLL Engaged Internal (DCO Range: 20-25 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock.
 * @details FLL Engaged Internal (DCO Range: 20-25 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock.
 * @showinitializer
 ******************************************************************************/
#define FLL_MODULE_FEI_24MHZ_CONFIG                                        \
(tFLL){                                                                       \
/* C1 */ SET(MCG_C1_CLKS(0x00))|SET(MCG_C1_FRDIV(0x00))|                      \
/* .. */ SET(MCG_C1_IREFS_MASK)|CLR(MCG_C1_IRCLKEN_MASK)|                     \
/* .. */ CLR(MCG_C1_IREFSTEN_MASK),                                           \
/* C2 */ SET(MCG_C2_LOCRE0_MASK)|SET(MCG_C2_RANGE0(0x00))|CLR(MCG_C2_HGO0_MASK)|\
/* .. */ CLR(MCG_C2_EREFS0_MASK)|CLR(MCG_C2_LP_MASK)|CLR(MCG_C2_IRCS_MASK),   \
/* C4 */ SET(MCG_C4_DMX32_MASK)|SET(MCG_C4_DRST_DRS(0x00)),                   \
/* C6 */ CLR(MCG_C6_LOLIE0_MASK)|CLR(MCG_C6_PLLS_MASK)|CLR(MCG_C6_CME0_MASK)| \
/* .. */ SET(MCG_C6_CHGPMP_BIAS(0x08)),                                       \
/* SC */ CLR(MCG_SC_ATME_MASK)|CLR(MCG_SC_ATMS_MASK)|                         \
/* .. */ CLR(MCG_SC_FLTPRSRV_MASK)|SET(MCG_SC_FCRDIV(0x01)),                  \
/* C7 */ SET(MCG_C7_PLL32KREFSEL(0))|CLR(MCG_C7_OSCSEL_MASK),                 \
/* C8 */ SET(MCG_C8_LOCRE1_MASK)|CLR(MCG_C8_LOLRE_MASK)|CLR(MCG_C8_CME1_MASK)|\
/* .. */ CLR(MCG_C8_COARSE_LOLIE_MASK),                                       \
}

/***************************************************************************//*!
 * @brief   FLL Engaged Internal (DCO Range: 40-50 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock.
 * @details FLL Engaged Internal (DCO Range: 40-50 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock.
 * @showinitializer
 ******************************************************************************/
#define FLL_MODULE_FEI_48MHZ_CONFIG                                        \
(tFLL){                                                                       \
/* C1 */ SET(MCG_C1_CLKS(0x00))|SET(MCG_C1_FRDIV(0x00))|                      \
/* .. */ SET(MCG_C1_IREFS_MASK)|CLR(MCG_C1_IRCLKEN_MASK)|                     \
/* .. */ CLR(MCG_C1_IREFSTEN_MASK),                                           \
/* C2 */ SET(MCG_C2_LOCRE0_MASK)|SET(MCG_C2_RANGE0(0x00))|CLR(MCG_C2_HGO0_MASK)|\
/* .. */ CLR(MCG_C2_EREFS0_MASK)|CLR(MCG_C2_LP_MASK)|CLR(MCG_C2_IRCS_MASK),   \
/* C4 */ SET(MCG_C4_DMX32_MASK)|SET(MCG_C4_DRST_DRS(0x01)),                   \
/* C6 */ CLR(MCG_C6_LOLIE0_MASK)|CLR(MCG_C6_PLLS_MASK)|CLR(MCG_C6_CME0_MASK)| \
/* .. */ SET(MCG_C6_CHGPMP_BIAS(0x08)),                                       \
/* SC */ CLR(MCG_SC_ATME_MASK)|CLR(MCG_SC_ATMS_MASK)|                         \
/* .. */ CLR(MCG_SC_FLTPRSRV_MASK)|SET(MCG_SC_FCRDIV(0x01)),                  \
/* C7 */ SET(MCG_C7_PLL32KREFSEL(0))|CLR(MCG_C7_OSCSEL_MASK),                 \
/* C8 */ SET(MCG_C8_LOCRE1_MASK)|CLR(MCG_C8_LOLRE_MASK)|CLR(MCG_C8_CME1_MASK)|\
/* .. */ CLR(MCG_C8_COARSE_LOLIE_MASK),                                       \
}

/***************************************************************************//*!
 * @brief   FLL Engaged Internal (DCO Range: 60-75 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock.
 * @details FLL Engaged Internal (DCO Range: 60-75 MHz). FLL clocked by 
 *          32 KHz Internal Reference Clock. Use @ref SIM_SetClkDiv and @ref 
 *          SIM_SetClkMode to divide core and bus clocks to allowed ranges.
 * @showinitializer
 ******************************************************************************/
#define FLL_MODULE_FEI_72MHZ_CONFIG                                        \
(tFLL){                                                                       \
/* C1 */ SET(MCG_C1_CLKS(0x00))|SET(MCG_C1_FRDIV(0x00))|                      \
/* .. */ SET(MCG_C1_IREFS_MASK)|CLR(MCG_C1_IRCLKEN_MASK)|                     \
/* .. */ CLR(MCG_C1_IREFSTEN_MASK),                                           \
/* C2 */ SET(MCG_C2_LOCRE0_MASK)|SET(MCG_C2_RANGE0(0x00))|CLR(MCG_C2_HGO0_MASK)|\
/* .. */ CLR(MCG_C2_EREFS0_MASK)|CLR(MCG_C2_LP_MASK)|CLR(MCG_C2_IRCS_MASK),   \
/* C4 */ SET(MCG_C4_DMX32_MASK)|SET(MCG_C4_DRST_DRS(0x02)),                   \
/* C6 */ CLR(MCG_C6_LOLIE0_MASK)|CLR(MCG_C6_PLLS_MASK)|CLR(MCG_C6_CME0_MASK)| \
/* .. */ SET(MCG_C6_CHGPMP_BIAS(0x08)),                                       \
/* SC */ CLR(MCG_SC_ATME_MASK)|CLR(MCG_SC_ATMS_MASK)|                         \
/* .. */ CLR(MCG_SC_FLTPRSRV_MASK)|SET(MCG_SC_FCRDIV(0x01)),                  \
/* C7 */ SET(MCG_C7_PLL32KREFSEL(0))|CLR(MCG_C7_OSCSEL_MASK),                 \
/* C8 */ SET(MCG_C8_LOCRE1_MASK)|CLR(MCG_C8_LOLRE_MASK)|CLR(MCG_C8_CME1_MASK)|\
/* .. */ CLR(MCG_C8_COARSE_LOLIE_MASK),                                       \
}

/**************************************************************************/ /*!
  * @brief   SIM initialization:
  * @showinitializer
  *****************************************************************************/
#define SIM_MODULE_POWER_METERING_RUN_MODE_CONFIG                               \
(tSIM){                                                                         \
/* SOPT1    */ SET(SIM_SOPT1_OSC32KSEL(0)),                                     \
/* SOPT1_CFG*/ CLR(SIM_SOPT1_CFG_RAMBPEN_MASK)|CLR(SIM_SOPT1_CFG_RAMSBDIS_MASK)|\
/* .........*/ CLR(SIM_SOPT1_CFG_CMPOLPTMR0SEL_MASK)|                           \
/* .........*/ SET(SIM_SOPT1_CFG_LPTMR3SEL(0))|SET(SIM_SOPT1_CFG_LPTMR2SEL(0))| \
/* .........*/ SET(SIM_SOPT1_CFG_LPTMR1SEL(0)),                                 \
/* CTRL_REG */ SET(SIM_CTRL_REG_CLKOUTSEL(0))|                                  \
/* ........ */ SET(SIM_CTRL_REG_SAR_TRG_CLK_SEL(0))|                            \
/* ........ */ CLR(SIM_CTRL_REG_PTC2_HD_EN_MASK)|                               \
/* ........ */ CLR(SIM_CTRL_REG_PLL_VLP_EN_MASK)|SET(SIM_CTRL_REG_NMIDIS_MASK), \
/* SCGC4    */ CLR(SIM_SCGC4_SPI1_MASK)|CLR(SIM_SCGC4_SPI0_MASK)|               \
               SET(SIM_SCGC4_CMP1_MASK)|SET(SIM_SCGC4_CMP0_MASK)|               \
/* .....    */ SET(SIM_SCGC4_VREF_MASK)|CLR(SIM_SCGC4_UART3_MASK)|              \
/* .....    */ SET(SIM_SCGC4_UART2_MASK)|CLR(SIM_SCGC4_UART1_MASK)|             \
/* .....    */ CLR(SIM_SCGC4_UART0_MASK)|CLR(SIM_SCGC4_I2C1_MASK)|              \
/* .....    */ CLR(SIM_SCGC4_I2C0_MASK)|CLR(SIM_SCGC4_OSC_MASK)|                \
/* .....    */ SET(SIM_SCGC4_MCG_MASK)|CLR(SIM_SCGC4_EWM_MASK),                 \
/* SCGC5    */ SET(SIM_SCGC5_TMR3_MASK)|SET(SIM_SCGC5_TMR2_MASK)|               \
               SET(SIM_SCGC5_TMR1_MASK)|SET(SIM_SCGC5_TMR0_MASK)|               \
/* .....    */ SET(SIM_SCGC5_XBAR_MASK)|SET(SIM_SCGC5_WDOG_MASK)|               \
/* .....    */ CLR(SIM_SCGC5_IRTCREGFILE_MASK)|SET(SIM_SCGC5_IRTC_MASK)|        \
/* .....    */ CLR(SIM_SCGC5_PORTI_MASK)|SET(SIM_SCGC5_PORTH_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTG_MASK)|SET(SIM_SCGC5_PORTF_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTE_MASK)|SET(SIM_SCGC5_PORTD_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTC_MASK)|SET(SIM_SCGC5_PORTB_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTA_MASK)|SET(SIM_SCGC5_SLCD_MASK),              \
/* SCGC6    */ SET(SIM_SCGC6_SIM_LP_MASK)|                                      \
/* .....    */ SET(SIM_SCGC6_LPTMR_MASK)|CLR(SIM_SCGC6_CRC_MASK)|               \
/* .....    */ SET(SIM_SCGC6_AFE_MASK)|SET(SIM_SCGC6_PIT1_MASK)|                \
               SET(SIM_SCGC6_PIT0_MASK)|SET(SIM_SCGC6_ADC_MASK)|                \
/* .....    */ CLR(SIM_SCGC6_RNGA_MASK)|CLR(SIM_SCGC6_DMAMUX3_MASK)|            \
/* .....    */ CLR(SIM_SCGC6_DMAMUX2_MASK)|CLR(SIM_SCGC6_DMAMUX1_MASK)|         \
/* .....    */ CLR(SIM_SCGC6_DMAMUX0_MASK)|SET(SIM_SCGC6_FTFA_MASK),            \
/* SCGC7    */ CLR(SIM_SCGC7_DMA_MASK)|CLR(SIM_SCGC7_MPU_MASK),                 \
/* CLKDIV1  */ SET(SIM_CLKDIV1_SYSDIV(SYSCLK_DIV1))|                            \
/* .......  */ SET(SIM_CLKDIV1_SYSCLKMODE_MASK),                                \
/* FCCFG1   */ SET(SIM_FCFG1_FLASHDOZE_MASK)|CLR(SIM_FCFG1_FLASHDIS_MASK),      \
/* MISC_CTL */ CLR(SIM_MISC_CTL_VREFBUFPD_MASK)|                                \
/* ........ */ CLR(SIM_MISC_CTL_VREFBUFINSEL_MASK)|                             \
/* ........ */ CLR(SIM_MISC_CTL_VREFBUFOUTEN_MASK)|                             \
/* ........ */ CLR(SIM_MISC_CTL_RTCCLKSEL_MASK)|                                \
/* ........ */ CLR(SIM_MISC_CTL_TMR3PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR2PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR1PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR0PCSSEL(0))|                                 \
/* ........ */ SET(SIM_MISC_CTL_TMR3SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR2SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR1SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR0SCSSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_TMR0PLLCLKSEL_MASK)|                            \
/* ........ */ CLR(SIM_MISC_CTL_EWMINSEL_MASK)|                                 \
/* ........ */ CLR(SIM_MISC_CTL_XBARPITOUTSEL(0))|                              \
/* ........ */ CLR(SIM_MISC_CTL_UART3IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART2IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART1IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART0IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UARTMODTYPE_MASK)|                              \
/* ........ */ CLR(SIM_MISC_CTL_AFECLKPADDIR_MASK)|                             \
/* ........ */ SET(SIM_MISC_CTL_AFECLKSEL(0))|                                  \
/* ........ */ SET(SIM_MISC_CTL_DMADONESEL(0))|                                 \
/* ........ */ SET(SIM_MISC_CTL_XBARAFEMODOUTSEL(0)),                           \
}

#define SIM_MODULE_POWER_METERING_VLPR_MODE_CONFIG                               \
(tSIM){                                                                         \
/* SOPT1    */ SET(SIM_SOPT1_OSC32KSEL(0)),                                     \
/* SOPT1_CFG*/ CLR(SIM_SOPT1_CFG_RAMBPEN_MASK)|CLR(SIM_SOPT1_CFG_RAMSBDIS_MASK)|\
/* .........*/ CLR(SIM_SOPT1_CFG_CMPOLPTMR0SEL_MASK)|                           \
/* .........*/ SET(SIM_SOPT1_CFG_LPTMR3SEL(0))|SET(SIM_SOPT1_CFG_LPTMR2SEL(0))| \
/* .........*/ SET(SIM_SOPT1_CFG_LPTMR1SEL(0)),                                 \
/* CTRL_REG */ SET(SIM_CTRL_REG_CLKOUTSEL(0))|                                  \
/* ........ */ SET(SIM_CTRL_REG_SAR_TRG_CLK_SEL(0))|                            \
/* ........ */ CLR(SIM_CTRL_REG_PTC2_HD_EN_MASK)|                               \
/* ........ */ CLR(SIM_CTRL_REG_PLL_VLP_EN_MASK)|SET(SIM_CTRL_REG_NMIDIS_MASK), \
/* SCGC4    */ CLR(SIM_SCGC4_SPI1_MASK)|CLR(SIM_SCGC4_SPI0_MASK)|               \
               SET(SIM_SCGC4_CMP1_MASK)|SET(SIM_SCGC4_CMP0_MASK)|               \
/* .....    */ SET(SIM_SCGC4_VREF_MASK)|CLR(SIM_SCGC4_UART3_MASK)|              \
/* .....    */ SET(SIM_SCGC4_UART2_MASK)|CLR(SIM_SCGC4_UART1_MASK)|             \
/* .....    */ CLR(SIM_SCGC4_UART0_MASK)|CLR(SIM_SCGC4_I2C1_MASK)|              \
/* .....    */ CLR(SIM_SCGC4_I2C0_MASK)|CLR(SIM_SCGC4_OSC_MASK)|                \
/* .....    */ SET(SIM_SCGC4_MCG_MASK)|CLR(SIM_SCGC4_EWM_MASK),                 \
/* SCGC5    */ SET(SIM_SCGC5_TMR3_MASK)|SET(SIM_SCGC5_TMR2_MASK)|               \
               SET(SIM_SCGC5_TMR1_MASK)|SET(SIM_SCGC5_TMR0_MASK)|               \
/* .....    */ SET(SIM_SCGC5_XBAR_MASK)|SET(SIM_SCGC5_WDOG_MASK)|               \
/* .....    */ CLR(SIM_SCGC5_IRTCREGFILE_MASK)|SET(SIM_SCGC5_IRTC_MASK)|        \
/* .....    */ CLR(SIM_SCGC5_PORTI_MASK)|SET(SIM_SCGC5_PORTH_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTG_MASK)|SET(SIM_SCGC5_PORTF_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTE_MASK)|SET(SIM_SCGC5_PORTD_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTC_MASK)|SET(SIM_SCGC5_PORTB_MASK)|             \
/* .....    */ SET(SIM_SCGC5_PORTA_MASK)|SET(SIM_SCGC5_SLCD_MASK),              \
/* SCGC6    */ SET(SIM_SCGC6_SIM_LP_MASK)|                                      \
/* .....    */ SET(SIM_SCGC6_LPTMR_MASK)|CLR(SIM_SCGC6_CRC_MASK)|               \
/* .....    */ SET(SIM_SCGC6_AFE_MASK)|CLR(SIM_SCGC6_PIT1_MASK)|                \
               CLR(SIM_SCGC6_PIT0_MASK)|SET(SIM_SCGC6_ADC_MASK)|                \
/* .....    */ CLR(SIM_SCGC6_RNGA_MASK)|CLR(SIM_SCGC6_DMAMUX3_MASK)|            \
/* .....    */ CLR(SIM_SCGC6_DMAMUX2_MASK)|CLR(SIM_SCGC6_DMAMUX1_MASK)|         \
/* .....    */ CLR(SIM_SCGC6_DMAMUX0_MASK)|SET(SIM_SCGC6_FTFA_MASK),            \
/* SCGC7    */ CLR(SIM_SCGC7_DMA_MASK)|CLR(SIM_SCGC7_MPU_MASK),                 \
/* CLKDIV1  */ SET(SIM_CLKDIV1_SYSDIV(SYSCLK_DIV1))|                            \
/* .......  */ SET(SIM_CLKDIV1_SYSCLKMODE_MASK),                                \
/* FCCFG1   */ SET(SIM_FCFG1_FLASHDOZE_MASK)|CLR(SIM_FCFG1_FLASHDIS_MASK),      \
/* MISC_CTL */ CLR(SIM_MISC_CTL_VREFBUFPD_MASK)|                                \
/* ........ */ CLR(SIM_MISC_CTL_VREFBUFINSEL_MASK)|                             \
/* ........ */ CLR(SIM_MISC_CTL_VREFBUFOUTEN_MASK)|                             \
/* ........ */ CLR(SIM_MISC_CTL_RTCCLKSEL_MASK)|                                \
/* ........ */ CLR(SIM_MISC_CTL_TMR3PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR2PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR1PCSSEL(0))|                                 \
/* ........ */ CLR(SIM_MISC_CTL_TMR0PCSSEL(0))|                                 \
/* ........ */ SET(SIM_MISC_CTL_TMR3SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR2SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR1SCSSEL_MASK)|                               \
/* ........ */ SET(SIM_MISC_CTL_TMR0SCSSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_TMR0PLLCLKSEL_MASK)|                            \
/* ........ */ CLR(SIM_MISC_CTL_EWMINSEL_MASK)|                                 \
/* ........ */ CLR(SIM_MISC_CTL_XBARPITOUTSEL(0))|                              \
/* ........ */ CLR(SIM_MISC_CTL_UART3IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART2IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART1IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UART0IRSEL_MASK)|                               \
/* ........ */ CLR(SIM_MISC_CTL_UARTMODTYPE_MASK)|                              \
/* ........ */ CLR(SIM_MISC_CTL_AFECLKPADDIR_MASK)|                             \
/* ........ */ SET(SIM_MISC_CTL_AFECLKSEL(0))|                                  \
/* ........ */ SET(SIM_MISC_CTL_DMADONESEL(0))|                                 \
/* ........ */ SET(SIM_MISC_CTL_XBARAFEMODOUTSEL(0)),                           \
}
/***************************************************************************//*!
 * @brief   Define interrupt vectors.
 * @param   Supported devices - MCU_MKM34Z5
 * @example 
 *          extern void UART2_UART3_isr (void);
 *          #define VECTOR_036 UART2_UART3_isr
 ******************************************************************************/

/******************************************************************************/
#endif /* __APPCONFIG_ */
