/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************/
#include "common.h"
#include "drivers.h"
#include "metering_modules.h"
#include <math.h>

#define PI              (3.1415926536)
#define RAD_TO_DEG(rad) (rad * 180 / PI)
#define DEG_TO_RAD(deg) (deg * PI / 180)

/*
    This file will precess the calibration data.
*/

static int32_t calib_data_check_calib_data(tMETERING_Calib_Param_1PH *calib_param)
{
    if ((calib_param->irms_msr >= calib_param->irms_cal) &&
        (calib_param->irms_msr <= calib_param->irms_cal * 1.1) &&
        (calib_param->urms_msr >= calib_param->urms_cal) &&
        (calib_param->urms_msr <= calib_param->urms_cal * 1.1))
        return 0;
    else
        return -1;
}

static int32_t calib_data_get_delay(double *q, double *p,  double *angle, uint8_t *flag)
{
    int16_t delay;
    double delta;

    *angle = atan2(*q, *p);
    *angle -= POWER_METERING_CALI_INIT_ANGLE;
    delta = *angle;
#if (POWER_METERING_USE_AFE_TRIGGER_ADC == 1) && (POWER_METERING_USE_SOFT_PHASE_SHIFT == 0)
    delay = (int16_t)((delta / (2 * PI * POWER_METERING_CALIB_GRID_FREQ))
                      * (POWER_METERING_AFE_PLL_CLK / (1 << POWER_METERING_AFE_DIV)));
#else
    /* The delay is for PDB, PDB uses bus clock, so use Bus clock */
    /* 1s is 2 * pi * 50 cycle */
    delay = (int16_t)((delta / (2 * PI * POWER_METERING_CALIB_GRID_FREQ))
                      * POWER_METERING_BUS_CLK_FREQ);
#endif

    return delay;
}

static int32_t calib_data_get_angel(double *q, double *p)
{
    double angle_rad;

    angle_rad = atan2 (*q, *p);
    angle_rad -= POWER_METERING_CALI_INIT_ANGLE;
    return (int32)(1000 * RAD_TO_DEG(angle_rad));
}

static frac32 calib_data_get_gain(double *msr, double *cal)
{
    return FRAC32(-(*cal) / (*msr));
}

static frac32 calib_data_get_offset(frac32 max, frac32 min)
{
    return (max + min) >> 1;
}

static int32_t calib_data_cal_params_1ph(tMETERING_Calib_Data_1PH *calib, tMETERING_Calib_Param_1PH *calib_param)
{
    /* In Calibration, we need to make Q to 0.0 */
    calib->u_gain = calib_data_get_gain(&(calib_param->urms_msr),
                                            &(calib_param->urms_cal));
    calib->u_offset = calib_data_get_offset(calib_param->u_msrmax,
                                            calib_param->u_msrmin);
    calib->i_gain = calib_data_get_gain(&(calib_param->irms_msr),
                                            &(calib_param->irms_cal));
    calib->i_offset = calib_data_get_offset(calib_param->i_msrmax,
                                            calib_param->i_msrmin);
    calib->angle = calib_data_get_angel(&calib_param->Q_msr,
                                            &calib_param->P_msr);
#if (POWER_METERING_USE_AFE_TRIGGER_ADC == 1) && (POWER_METERING_USE_SOFT_PHASE_SHIFT == 0)
    calib->delay -= calib_data_get_delay(&calib_param->Q_msr,
                                            &calib_param->P_msr,
                                            &calib_param->angle_msr,
                                            &calib->CT_flagPh);
#else
    calib->delay = calib_data_get_delay(&calib_param->Q_msr,
                                            &calib_param->P_msr,
                                            &calib_param->angle_msr,
                                            &calib->CT_flagPh);
#endif
    return 0;
}

static int32_t calib_data_cal_params(tMETERING_CONFIG_FLASH_DATA *calib_data)
{	
    tMETERING_Calib_Data_1PH *tmp_calib = NULL;
    tMETERING_Calib_Param_1PH *tmp_calib_param = NULL;

    /* ph1, 2, 3, 4 */
    tmp_calib = &(calib_data->calib_data1);
    tmp_calib_param = &(calib_data->calib_param1);
    calib_data_cal_params_1ph(tmp_calib, tmp_calib_param);

    /* ph1, 2, 3, 4 */
    tmp_calib = &(calib_data->calib_data2);
    tmp_calib_param = &(calib_data->calib_param2);
    calib_data_cal_params_1ph(tmp_calib, tmp_calib_param);

    /* ph1, 2, 3, 4 */
    tmp_calib = &(calib_data->calib_data3);
    tmp_calib_param = &(calib_data->calib_param3);
    calib_data_cal_params_1ph(tmp_calib, tmp_calib_param);

    return 0;
}

static int32_t calib_data_calc_calib_data(tMETERING_CONFIG_FLASH_DATA *calib_data)
{
    tMETERING_Calib_Param_1PH *tmp_calib_param = NULL;

    /* Calculates calibration data if pre-processing completed sucessfully      */
    /* Check calibration conditions to eliminate pre-heating states           */
    tmp_calib_param = &(calib_data->calib_param1);
    if (calib_data_check_calib_data(tmp_calib_param))
        return 1;

    tmp_calib_param = &(calib_data->calib_param2);
    if (calib_data_check_calib_data(tmp_calib_param))
        return 1;

    tmp_calib_param = &(calib_data->calib_param3);
    if (calib_data_check_calib_data(tmp_calib_param))
        return 1;

#if 0
    tmp_calib_param = &(calib_data->calib_param4);
    if (calib_data_check_calib_data(tmp_calib_param))
        return 1;
#endif

    calib_data_cal_params(calib_data);

    return 0;
}

#if POWER_METERING_CALI_ENABLE_ITERATION
static int32_t calib_data_itera_params_1ph(tMETERING_Calib_Data_1PH *calib, tMETERING_Calib_Param_1PH *calib_param)
{
    int16_t delay = 0;
    int32_t angle = 0;
    uint8_t ct_flag = 0;

    /* In Calibration, we need to make Q to 0.0 */
    if (((int32_t)(calib_param->Q_msr - POWER_METERING_CALI_Q_TARGET) != 0))
    {
        angle = calib_data_get_angel(&calib_param->Q_msr,
                                     &calib_param->P_msr);
        delay = calib_data_get_delay(&calib_param->Q_msr,
                                     &calib_param->P_msr,
                                     &calib_param->angle_msr,
                                     &ct_flag);
        /* Iteration */
        calib->angle -= angle;
        calib->delay -= delay;
    }

    return 0;
}

static int32_t calib_data_itera_calib_data(tMETERING_CONFIG_FLASH_DATA *calib_data)
{
    /* ph1, 2, 3, 4 */
    calib_data_itera_params_1ph(&(calib_data->calib_data1),
                                &(calib_data->calib_param1));
    /* ph1, 2, 3, 4  */
    calib_data_itera_params_1ph(&(calib_data->calib_data2),
                                &(calib_data->calib_param2));
    /* ph1, 2, 3, 4 */
    calib_data_itera_params_1ph(&(calib_data->calib_data3),
                                &(calib_data->calib_param3));

    return 0;
}
#endif

static int32_t calib_data_update_max_value_1ph(tMETERING_Calib_Param_1PH *calib_param,
                                           frac32 u, frac32 i)
{
    /* find voltage max. value  */
    if (calib_param->u_msrmax < u)
        calib_param->u_msrmax = u;
    /* find voltage min. value  */
    if (calib_param->u_msrmin > u)
        calib_param->u_msrmin = u;
    /* find current max. value  */
    if (calib_param->i_msrmax < i)
        calib_param->i_msrmax = i;
    /* find current min. value  */
    if (calib_param->i_msrmin > i)
        calib_param->i_msrmin = i;

    return 0;
}

/***************************************************************************//*!
 * @brief   Updates offset of the phase voltage and current measurements 
 *          conditionally.
 * @param   ptr   - pointer to tCONFIG_FLASH_DATA
 * @param   u     - phase voltage sample
 * @param   i     - phase current sample
 * @note    Implemented as a function call.
 ******************************************************************************/
int32_t calib_data_update_offsets(tMETERING_CONFIG_FLASH_DATA *calib_data,
		frac32 u1, frac32 i1, frac32 i2, frac32 i3, frac32 i4)
{
    /* update offsets if pre-processing active         */
    calib_data_update_max_value_1ph(&(calib_data->calib_param1), u1, i1);

    calib_data_update_max_value_1ph(&(calib_data->calib_param2), u1, i2);

    calib_data_update_max_value_1ph(&(calib_data->calib_param3), u1, i3);
    
    return 0;
}

static int32_t calib_data_fill_data_1ph(tMETERING_Calib_Param_1PH *calib_param,
                                        double urms, double irms,
                                        double w, double var)
{
    calib_param->urms_msr    = urms;
    calib_param->irms_msr    = irms;
    calib_param->P_msr       = w;
    calib_param->Q_msr       = var;
    return 0;
}

static int32_t calib_data_fill_data(tMETERING_CONFIG_FLASH_DATA *calib_data,
                              double urms1, double irms1,
                              double urms2, double irms2,
                              double urms3, double irms3,
                              double urms4, double irms4,
                              double w1, double var1,
                              double w2, double var2,
                              double w3, double var3,
                              double w4, double var4)
{
    calib_data_fill_data_1ph(&(calib_data->calib_param1), urms1, irms1, w1, var1);
    
    calib_data_fill_data_1ph(&(calib_data->calib_param2), urms2, irms2, w2, var2);
    
    calib_data_fill_data_1ph(&(calib_data->calib_param3), urms3, irms3, w3, var3);
    
    return 0;
}

static int32_t calib_data_save_config(tMETERING_CONFIG_FLASH_DATA *calib_data)
{
    DisableInterrupts();
    nvm_config_write((const void *)calib_data, sizeof(tMETERING_CONFIG_FLASH_DATA));

#if (POWER_METERING_USE_AFE_TRIGGER_ADC == 1) && (POWER_METERING_USE_SOFT_PHASE_SHIFT == 0)
    AFE_DlyUpdate(calib_data->calib_data1.delay, calib_data->calib_data2.delay, calib_data->calib_data3.delay, 0);
#endif

#if POWER_METERING_USE_QTMR_TRIGGER_ADC
    /* Update the QT delay values */
    TMR_SetCmp1Val(CH1, calib_data->calib_data1.delay);
#endif
    EnableInterrupts();

    return 0;
}

#define TIMEOUT_IN_SEC(x) (x * 4)
int32_t calib_data_processing(tMETERING_CONFIG_FLASH_DATA *calib_data,
                              double urms1, double irms1,
                              double urms2, double irms2,
                              double urms3, double irms3,
                              double urms4, double irms4,
                              double w1, double var1,
                              double w2, double var2,
                              double w3, double var3,
                              double w4, double var4)
{
    static int32_t skip_timeout = 0;
    int32_t calib_result = 0;

    switch (calib_data->calib_flag)
    {
    case CALI_FLAG_INIT:
        /* timeout check - when timeout expires then finish pre-processing state  */
        /* by setting state at which calibration data are calculated after reset  */
        if (skip_timeout++ > TIMEOUT_IN_SEC(POWER_METERING_CALI_SKIP_TIMEOUT)) 
        {
            calib_data->calib_flag = CALI_FLAG_DATA_READY;
        }
        break;
    case CALI_FLAG_DATA_READY:
        skip_timeout = 0;
        calib_data_fill_data(calib_data, urms1, irms1, urms2, irms2, urms3, irms3,
                             urms4, irms4, w1, var1, w2, var2, w3, var3, w4, var4);

        /*
         If calibration data were collected then calibration parameters are
         calculated and saved to flash
        */
        DisableInterrupts();
        calib_result = calib_data_calc_calib_data((tMETERING_CONFIG_FLASH_DATA *)calib_data);
        if (!calib_result)
        {
#if POWER_METERING_CALI_ENABLE_ITERATION
            calib_data->calib_flag = CALI_FLAG_ITERA_INIT;
#else
            calib_data->calib_flag = CALI_FLAG_FIN;
#endif
            calib_data_save_config(calib_data);
#if POWER_METERING_CALI_ENABLE_ITERATION
            /* System reset to do interation procedure */
            SystemReset();
#endif
        }
        else
        {
            calib_data->calib_flag = CALI_FLAG_INIT;
        }
        EnableInterrupts();
        break;
#if POWER_METERING_CALI_ENABLE_ITERATION
    case CALI_FLAG_ITERA_INIT:
        /* timeout check - when timeout expires then finish pre-processing state  */
        /* by setting state at which calibration data are calculated after reset  */
        if (skip_timeout++ > TIMEOUT_IN_SEC(POWER_METERING_CALI_SKIP_TIMEOUT)) 
        {
            calib_data->calib_flag = CALI_FLAG_ITERA;
        }
        break;
    case CALI_FLAG_ITERA:
#if POWER_METERING_CALI_ITERATION_NO_LIMIT
        if (((int32_t)(var1 - POWER_METERING_CALI_Q_TARGET) == 0) &&
            ((int32_t)(var2 - POWER_METERING_CALI_Q_TARGET) == 0) &&
            ((int32_t)(var3 - POWER_METERING_CALI_Q_TARGET) == 0))
        {
            calib_data->calib_flag = CALI_FLAG_FIN;
            calib_data_save_config(calib_data);
#if POWER_METERING_SLCD_SUPPORT
            slcd_disp_show_cali_fin();
#endif
        }
        else
#endif
        {
            calib_data_fill_data(calib_data, urms1, irms1, urms2, irms2, urms3, irms3,
                                urms4, irms4, w1, var1, w2, var2, w3, var3, w4, var4);
            DisableInterrupts();
            calib_result = calib_data_itera_calib_data((tMETERING_CONFIG_FLASH_DATA *)calib_data);
            EnableInterrupts();
            if (!calib_result)
            {
#if POWER_METERING_CALI_ITERATION_NO_LIMIT
                calib_data->calib_flag = CALI_FLAG_ITERA_INIT;
#endif
#if POWER_METERING_CALI_ITERATION_ONCE
                calib_data->calib_flag = CALI_FLAG_FIN;
#endif
                /* Don't need to adjust PDB count, for PDB count has already been added in basic calibration */
                calib_data_save_config(calib_data);
                /* System reset to do interation procedure */
                SystemReset();
            }
            else
            {
                calib_data->calib_flag = CALI_FLAG_INIT;
            }
        }
        break;
#endif
    default:
        break;
    }
    
    return calib_data->calib_flag;
}

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/
