/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************/
#include "common.h"
#include "drivers.h"
#include "metering_modules.h"
#include <stdint.h>

/*
    This file implements the frequency detection
*/
static uint8_t cur_freq_det_ch = POWER_METERING_FREQ_DET_CH3;
static const uint8_t freq_det_chs[POWER_METERING_PHRASE_NUM] =
{
    POWER_METERING_FREQ_DET_CH1,
    POWER_METERING_FREQ_DET_CH2,
    POWER_METERING_FREQ_DET_CH3
};
#if POWER_METERING_ENABLE_FREQ_AVERAGE
/* For performance, average freq will be calculated in integer, the divisor is 1000 */
static uint32_t average_target = 1;
static uint8_t average_cnt = 0;
static uint32_t freq_sum = 0.0;
static uint32_t freq_avg = 0.0;
#endif

#if POWER_METERING_FFT_MODE_SYNC
static void freq_detect_zcd_callback(CMP_CALLBACK_SRC module, int8_t status)
{
    if (CMP2_CALLBACK == module)
    {
        if (status & (CMP_SCR_CFF_MASK))
            metering_func_do_zcd();
    }
}
#endif

static void freq_detect_change_channel(void)
{
    cur_freq_det_ch = freq_det_chs[(cur_freq_det_ch + 1) % POWER_METERING_PHRASE_NUM];
    XBAR_Path(XBAR_CMP0OUT,XBAR_TMR0SEC);
    CMP_MuxCtrl(CMP0, cur_freq_det_ch, IN7_DAC);
}

int32_t freq_detect_init(void)
{
    /* Frequency measurement, used CH2 to measure frequency
       PTE6 set to alt0: CMP0P2
     */
#ifndef POWER_METERING_DEBUG
    PORT_Init (PORTE, PORT_MODULE_ALT0_MODE, PIN6);
#endif

    SIM_SelTmrScs(CH0, 1);
    XBAR_Path(XBAR_CMP0OUT, XBAR_TMR0SEC);
    CMP_Init(CMP0, CMP_FREQ_DETECT_MODE_CONFIG_EN_DAC(32), cur_freq_det_ch, IN7_DAC);
#if POWER_METERING_FFT_MODE_SYNC
    /* In FFT Sync mode, we need CMP to detect zero-cross point, that is, zero-cross-detected */
    //CMP_InstallCallback(PRI_LVL1, (CMP_CALLBACK)freq_detect_zcd_callback);
#endif
    TMR_Init(CH0,
             TMR_FREQ_DETECT_MODE(POWER_METERING_FREQ_CALC_TMR_DIV, SEC_CNTR0_INP, LOAD_POSEDGE),
             0, 0, 0, 0, 0, PRI_LVL1, (TMR_CH_CALLBACK)NULL);
    TMR_Enable(CH0);

    return 0;
}

int32_t freq_detect_deinit(void)
{
    TMR_Disable(CH0);

    return 0;
}

int32_t freq_detect_get_freq(double *freq)
{
    uint16_t tmp16 = 0;
    double tmp_freq = 0.0;

    if (TMR_ReadClrCaptReg(CH0, &tmp16) == TRUE)
        tmp_freq = POWER_METERING_TMR2FREQ(tmp16);
    
    /* We have 3 channels for frequency detection, if one fails, change to another */
    if ((tmp_freq < POWER_METERING_GRID_FREQ_MIN) || (tmp_freq > POWER_METERING_GRID_FREQ_MAX))
    {
        freq_detect_change_channel();
        return -1;
    }

    *freq = tmp_freq;

#if POWER_METERING_ENABLE_FREQ_AVERAGE
    freq_sum += (uint32_t)(tmp_freq * 1000);
    if (++average_cnt == average_target)
    {
        freq_avg = freq_sum / average_target;
        freq_sum = 0;
    }
#endif

    return 0;
}

#if POWER_METERING_ENABLE_FREQ_AVERAGE
int32_t freq_detect_set_freq_average_cnt(uint32_t avg_cnt)
{
    average_target = avg_cnt;

    return 0;
}

int32_t freq_detect_get_average_freq(uint32_t *freq)
{
    if ((freq_avg < (POWER_METERING_GRID_FREQ_MIN * 1000)) || (freq_avg > (POWER_METERING_GRID_FREQ_MAX * 1000)))
        return -1;
    *freq = freq_avg;

    return 0;
}
#endif

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/
