/******************************************************************************
 * (c) Copyright 2010-2015, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************/
#include "common.h"
#include "metering_modules.h"
#include "appconfig.h"
#include "drivers.h"
#include <string.h>
#include <stdio.h>

/*
    This file implements irda functions
*/
static uint8_t data_received_flag = 0;
static uint8_t data_sent_flag = 0;
static uint8_t data_received = 0;

static void irda_lpuart_callback(LPUART_CALLBACK_SRC module, LPUART_CALLBACK_TYPE type,
                             int32_t status)
{
    if (module == LPUART0_CALLBACK)
    {
        if (type == LPUART_RX_CALLBACK)
        {
            if (data_received_flag)
            {
                uint8_t dummy_ch = 0;

                dummy_ch = LPUART_GetChar(LPUART0);
            }
            else
            {
                data_received_flag = 1;
                data_received = LPUART_GetChar(LPUART0);
            }
        }
        if (type == LPUART_TX_CALLBACK)
        {
            data_sent_flag = 1;
        }
    }
}

static uint32_t irda_modclk_init(uint32_t freq)
{
    uint16_t mod_val;

    /* CLK Pin, XBAR_OUT0, We use QTMR0_Tmr0 to output clock */
    mod_val = (POWER_METERING_BUS_CLK_FREQ / freq) / 2 - 1;
    TMR_Init(CH1,
            TMR_CH_FIXED_FREQ_PWM_MODE_CONFIG(BUS_CLK_DIV1),
            mod_val,
            0, 0, mod_val, 0, PRI_LVL2, (TMR_CH_CALLBACK)NULL);
    TMR_Enable(CH1);
    
    /* Use QTmr0 to output 38K modulator PWM */
    /* TMR0 output to PXBAR_OUT4 as UART Tx Modulation Output */
    XBAR_Path(XBAR_TMR1, XBAR_OUT4);

    return (POWER_METERING_BUS_CLK_FREQ / ((mod_val + 1) << 1));
}

int32_t irda_send_b(uint8_t *buf, uint32_t len)
{
    LPUART_Wr(LPUART0, buf, len);
 
    return len;
}

int32_t irda_data_sent_status(void)
{
    return data_sent_flag;
}

uint32_t irda_send_char_nb(uint8_t ch)
{
    LPUART_PutData(LPUART0, ch);

    data_sent_flag = 0;

    return 0;
}

/* Blocking receive */
int32_t irda_receive_b(uint8_t *buf, uint32_t len)
{
    LPUART_Rd(LPUART0, buf, len);

    return len;
}

uint8_t irda_data_receive_status(void)
{
    return data_received_flag;
}

uint8_t irda_receive_char_nb(void)
{
    data_received_flag = 0;

    return data_received;
}

int32_t irda_deinit(void)
{
    TMR_Disable(CH0);

    return 0;
}

int32_t irda_init(uint32_t rx_mode, uint32_t tx_mode)
{
    IRDA_TX_PIN_INIT();
    IRDA_RX_PIN_INIT();
    IRDA_CLK_PIN_INIT();

    /* BUS clock selected as the LPUART clock source */
    SIM_SelLpuartClk(SIM_LPUARTCLK_SRC4);

    irda_modclk_init(POWER_METERING_IRDA_MOD_PWM_FREQ);

    /* LPUART init 9600bd */
    LPUART_Init(LPUART0,
                LPUART_MODULE_INTRMODE_CONFIG(POWER_METERING_IRDA_BAUDRATE,
                                                   POWER_METERING_BUS_CLK_FREQ));
    if (rx_mode == IRDA_XFER_NON_BLOCKING)
    {
        LPUART_RxIsrEnable(LPUART0);
        LPUART_InstallCallback(PRI_LVL1, irda_lpuart_callback);
    }
    if (tx_mode == IRDA_XFER_NON_BLOCKING)
    {
        LPUART_TxIsrEnable(LPUART0);
        LPUART_InstallCallback(PRI_LVL1, irda_lpuart_callback);
    }

    return 0;
}

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/
