/******************************************************************************
*
* (c) Copyright 2015, Freescale Semiconductor Inc.
*
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      rs485_parse.c
*
* @author    Freescale
*
* @version   1.0.0.0
*
* @date      Mar-13-2015
*
* @brief     Header file containing RS485 protocol
*
******************************************************************************/
#include "common.h"
#include "drivers.h"
#include <string.h>

#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include "metering_modules.h"
#include "debug_console.h"

/*
 * Print data buffer in hex to the terminal.
 */
#define MAX_LINE_LENGTH_BYTES (32)
#define DEFAULT_LINE_LENGTH_BYTES (16)

static int32_t rs485_parse_print_buffer(uint32_t addr, const uint8_t *data, uint32_t width, uint32_t count,
		 uint32_t linelen)
{
    int32_t i;

    if (linelen * width > MAX_LINE_LENGTH_BYTES)
        linelen = MAX_LINE_LENGTH_BYTES / width;
    if (linelen < 1)
        linelen = DEFAULT_LINE_LENGTH_BYTES / width;

    while (count)
    {
        uint32_t thislinelen = linelen;
        printf("%02lx:", addr);

        /* check for overflow condition */
        if (count < thislinelen)
            thislinelen = count;

        /* Copy from memory into linebuf and print hex values */
        for (i = 0; i < thislinelen; i++) {
            uint32_t x;
            if (width == 4) {
                x = *(volatile uint32_t *)data;
                printf(" %08x", x);
            } else if (width == 2) {
                x = *(volatile uint16_t *)data;
                printf(" %04x", x);
            } else {
                x = *(volatile uint8_t *)data;
                printf(" %02x", x);
            }
            data += width;
        }

        printf("\r\n");
        /* update references */
        addr += thislinelen * width;
        count -= thislinelen;
    }

    return 0;
}

static int32_t rs485_parse_double_to_int(double real, uint32_t *sign_val, uint32_t *int_val, uint32_t *frag_val)
{
    if (real < 0)
    {
        *sign_val = 1;
        real = fabs(real);
    }

    *int_val = (uint32_t)real;
    real -= *int_val;
    real *= 10000;
    *frag_val = (uint32_t)real;

    return 0;
}

static int32_t rs485_parse_metering_info(int32_t argc, char *argv[])
{
    double tmp0 = 0.0, tmp1 = 0.0, tmp2 = 0.0;
    uint32_t int_val = 0, frac_val = 0, sign_val = 0;
    int32_t i = 0;

    if (argc != 1) {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    for (i = 0; i < POWER_METERING_PHRASE_NUM; ++i)
    {
        printf("Phrase %d:\r\n", i + 1);
        metering_func_get_cur_v_i_ph(&tmp0, &tmp1, i);
        rs485_parse_double_to_int(tmp0, &sign_val, &int_val, &frac_val);
        printf("U: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
        rs485_parse_double_to_int(tmp1, &sign_val, &int_val, &frac_val);
        printf("I: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
        metering_func_get_cur_p_q_s_ph(&tmp0, &tmp1, &tmp2, i);
        rs485_parse_double_to_int(tmp0, &sign_val, &int_val, &frac_val);
        printf("P: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
        rs485_parse_double_to_int(tmp1, &sign_val, &int_val, &frac_val);
        printf("Q: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
        rs485_parse_double_to_int(tmp2, &sign_val, &int_val, &frac_val);
        printf("S: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
    }

    printf("\r\nTotal:\r\n");
    metering_func_get_cur_p_q_total(&tmp0, &tmp1);
    rs485_parse_double_to_int(tmp0, &sign_val, &int_val, &frac_val);
    printf("P: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);
    rs485_parse_double_to_int(tmp1, &sign_val, &int_val, &frac_val);
    printf("Q: %c%d.%d\r\n", (sign_val) ? '-' : ' ', int_val, frac_val);

    printf("\r\nGrid frequency:\r\n");
    freq_detect_get_freq(&tmp0);
    rs485_parse_double_to_int(tmp0, &sign_val, &int_val, &frac_val);
    printf("Freq: %d.%d\r\n", int_val, frac_val);

    return 0;
}

static int32_t rs485_parse_temp(int32_t argc, char *argv[])
{
    uint16_t adc_val = 0;
    int32_t cur_temp = 0;

    if (argc != 1)
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    adc_val = temp_sense_get_average_adc_val();

    cur_temp = temp_sense_calc_cur_temp() / 2;

    printf("ADC Val: %d, Temperature: %dC\r\n", adc_val, cur_temp);

    return 0;
}

static int32_t rs485_parse_comp(int32_t argc, char *argv[])
{
    uint32_t real, imagin;
    char *endp;

    if ((argc != 3) && (argc != 1))
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    if (1 == argc)
    {
        uint16_t tmp_reg = RTC_COMPEN;

        real = tmp_reg >> 12;
        imagin = tmp_reg & 0x7f;
    }
    else
    {
        real = (uint16_t)strtoul(argv[1], &endp, 16);
        if (*argv[1] == 0 || *endp != 0)
            return 1;
        
        if (real > 0xf)
        {
            printf("Real value should be less then 16!\r\n");
            return 1;
        }
        
        imagin = (uint16_t)strtoul(argv[2], &endp, 16);
        if (*argv[1] == 0 || *endp != 0)
            return 1;
        
        if (imagin > 0x7f)
        {
            printf("Imaginary value should be less then 128!\r\n");
            return 1;
        }
        
        IRTC_UpdateFineCompVal(real, imagin);
    }

    printf("Comp Val: Real 0x%x, Fraction: 0x%x\r\n", real, imagin);

    return 0;
}

static int32_t rs485_parse_comp_day_err_sec(int32_t argc, char *argv[])
{
    char *endp;
    uint8_t frac = 0, inte = 0;
    double value;

    if (argc != 2)
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    value = strtod(argv[1], &endp);
    if (*argv[1] == 0 || *endp != 0)
        return 1;

    rtc_comp_cal_rtc_ext(value, (int8_t *)&inte, &frac);

    //rs485_parse_comp_corr_itera(&inte, &frac);
    //inte &= 0xf;

    IRTC_UpdateFineCompVal(inte, frac);

    printf("Comp Val: Real 0x%x, Fraction: 0x%x\r\n", inte, frac);

    return 0;
}

static int32_t rs485_parse_comp_ppm(int32_t argc, char *argv[])
{
    char *endp;
    uint8_t frac = 0, inte = 0;
    double value;

    if (argc != 2)
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    value = strtod(argv[1], &endp);
    if (*argv[1] == 0 || *endp != 0)
        return 1;

    rtc_comp_cal_rtc((int32_t)value, (int8_t *)&inte, &frac);

    //rs485_parse_comp_corr_itera(&inte, &frac);
    //inte &= 0xf;

    IRTC_UpdateFineCompVal(inte, frac);

    printf("Comp Val: Real 0x%x, Fraction: 0x%x\r\n", inte, frac);

    return 0;
}

#if POWER_METERING_IRDA_SUPPORT
static int32_t rs485_parse_irda(int32_t argc, char *argv[])
{
    const char *cmd_op;
    static uint8_t irda_init_flag = 0;

    /* need at least two arguments */
    if ((argc < 2) || (argc > 3))
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    cmd_op = argv[1];

    if (strcmp(cmd_op, "init") == 0)
    {
        if (irda_init(IRDA_XFER_NON_BLOCKING, IRDA_XFER_BLOCKING))
            printf("Failed to initilize IrDA!\r\n");
        else
        {
            printf("IrDA is initilized!\r\n");
            irda_init_flag = 1;
        }
    }
    else
    {
        char *data_str = NULL;
        uint32_t data_len = 0;

        if (!irda_init_flag)
        {
            printf("Please do 'irda init' first!\r\n");
            return 1;
        }

        if (strcmp(cmd_op, "send") == 0)
        {
            if (3 != argc)
            {
                printf("No string to send for IrDA!\r\n");
                return 1;
            }
            data_str = argv[2];
            data_len = strlen(data_str);
            if (data_len != irda_send_b((uint8_t *)data_str, data_len))
            {
                printf("Not all data are sent!\r\n");
                return 1;
            }
            else
                printf("IrDA send data successfully!\r\n");
        }
        else if (strcmp(cmd_op, "receive") == 0)
        {
            printf("IrDA receiving ... will quit when Z is received!\r\n");
            while (1)
            {
                if (irda_data_receive_status())
                {
                    uint8_t in_ch = irda_receive_char_nb();

                    if ('Z' == in_ch)
                    {
                        putchar('\r');
                        putchar('\n');
                        break;
                    }
                    else
                        putchar(in_ch);
                }
            }
        }
        else
        {
            printf("Invalid command for IrDA!\r\n");
            return 1;
        }
    }

    return 0;
}
#endif

static int32_t rs485_parse_cmd_get_data_size(char* arg, int32_t default_size)
{
    /* Check for a size specification .b, .w or .l.
     */
    size_t len = strlen(arg);

    if (len > 2 && arg[len - 2] == '.') {
        switch (arg[len - 1]) {
        case 'b':
                return 1;
        case 'w':
                return 2;
        case 'l':
                return 4;
        case 's':
                return -2;
        default:
                return -1;
        }
    }
    return default_size;
}

static int32_t rs485_parse_md(int32_t argc, char *argv[])
{
    uint32_t addr, length;
    int32_t  size;
    int32_t  rc = 0;
    uint8_t *buf;

    if (argc < 2)
    {
        printf("Invalid parameters!\r\n");
        return 1;
    }
    
    /* New command specified.  Check for a size specification.
     * Defaults to long if no or incorrect specification.
     */
    if ((size = rs485_parse_cmd_get_data_size(argv[0], 1)) < 0)
        return 1;

    /* Address is specified since argc > 1
     */
    addr = strtoul(argv[1], NULL, 16);
    buf = (uint8_t *)addr;

    /* If another parameter, it is the length to display.
     * Length is the number of objects, not number of bytes.
     */
    length = 0x10;
    if (argc > 2)
        length = strtoul(argv[2], NULL, 16);

    /* Print the lines. */
    rs485_parse_print_buffer(addr, buf, size, length,
                             POWER_METERING_RS485_SHELL_DISP_LINE_LEN / size);
        
    return rc;
}

static int32_t rs485_parse_mw(int32_t argc, char *argv[])
{
    uint32_t addr, writeval, count;
    int32_t  size;

    if ((argc < 3) || (argc > 4)) {
        printf("Invalid parameters!\r\n");
        return 1;
    }

    /* Check for size specification.
    */
    if ((size = rs485_parse_cmd_get_data_size(argv[0], 1)) < 1)
        return 1;

    /* Address is specified since argc > 1
    */
    addr = strtoul(argv[1], NULL, 16);

    /* Get the value to write.
    */
    writeval = strtoul(argv[2], NULL, 16);

    /* Count ? */
    if (argc == 4) {
        count = strtoul(argv[3], NULL, 16);
    } else {
        count = 1;
    }

    while (count-- > 0) {
        if (size == 4)
            *((uint32_t *)addr) = (uint32_t)writeval;
        else if (size == 2)
            *((uint16_t *)addr) = (uint16_t)writeval;
        else
            *((uint8_t *)addr) = (uint8_t)writeval;
        addr += size;
    }

    return 0;
}

static int32_t rs485_parse_print_usage(int32_t argc, char *argv[])
{
    printf("RS485 commands:\r\n"
    "info\r\n"
    "        - Show metering info, Voltage, Current, P, Q, S etc.\r\n"
    "comp <real> <imag>\r\n"
    "        - Show or set correction factors, real and imaginary parts.\r\n"
    "          Input without arguments to show correction factors.\r\n"
    "          Real Value should be less than 16.\r\n"
    "          Imaginary value should be less than 128.\r\n"
    "dayerrsec\r\n"
    "        - Input decimal value which is one day error seconds\r\n"
    "ppm <val>\r\n"
    "        - input ppm value\r\n"
    "temp \r\n"
    "        - Show current ADC temperature value and average temperature\r\n"
    "irda init\r\n"
    "        - Initilize IrDA module\r\n"
    "irda send <str>\r\n"
    "        - Send data via IrDA\r\n"
    "irda receive\r\n"
    "        - Receive data from IrDA\r\n"
    "md <addr> [count]\r\n"
    "        - memory display, [.b, .w, .l] address [# of objects]\r\n"
    "mw <addr> <val> [count]\r\n"
    "        - memory write (fill), [.b, .w, .l] address value [count]\r\n"
    "help\r\n"
    "        - print usage\r\n"
    );

    return 0;
}

static shell_cmd_tbl_t rs485_parse_cmd_tbl[] = {
    { "info", 1, rs485_parse_metering_info },
    { "comp", 3, rs485_parse_comp },
    { "dayerrsec", 2, rs485_parse_comp_day_err_sec },
    { "ppm", 2, rs485_parse_comp_ppm },
    { "md", 3, rs485_parse_md },
    { "mw", 4, rs485_parse_mw },
    { "temp", 1, rs485_parse_temp },
#if POWER_METERING_IRDA_SUPPORT
    { "irda", 3, rs485_parse_irda },
#endif
    { "help",  1, rs485_parse_print_usage },
};

int32_t rs485_parse_init(void)
{
    RS485_PARSE_CTRL_PIN_INIT();
  
    RS485_PARSE_IO_PIN_INIT();

    /* UART0 using Busclk as clock source */
    dbg_console_init(POWER_METERING_RS485_UART_BAUD_RATE,
                     POWER_METERING_BUS_CLK_FREQ);

    rs485_shell_init(rs485_parse_cmd_tbl, POWER_METERING_RS485_PARSE_CMD_NUM,
                     POWER_METERING_RS485_SHELL_PROMPT);

    return 0;
}

int32_t rs485_parse_deinit(void)
{
    return 0;
}
