/******************************************************************************
* (c) Copyright 2010-2015, Freescale Semiconductor Inc.
* ALL RIGHTS RESERVED.
***************************************************************************/
#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include <stdlib.h>

/*
 This file implements the RTC function for 1Hz compensation and tamper detection
*/
static int32_t last_temp = -100;
static int32_t rtc_comp_interval = 0;

int32_t rtc_comp_init(void)
{
#if POWER_METERING_ENABLE_RTC_COMP
    /* Suppress initial battery removal "tamper" detect   */
    IRTC_ClrIsrFlag(IRTC_TAMPER0_MASK);
 
    IRTC_Init(IRTC_MODULE_COMP_FINE_CONFIG(LFSR_CLK_8HZ, 4, 0, 0),
            IRTC_TAMPER_PIN_PASSIVE_POL_HIGH_CONFIG(TAMPER_FILT_CLK_8HZ, 10),
            IRTC_TAMPER_PIN_DI_CONFIG,
            IRTC_TAMPER_PIN_DI_CONFIG,
#if POWER_METERING_ENABLE_ESD_TEST_MODE
            0
#else
            IRTC_TAMPER0_MASK
#endif
            );

    rtc_evt_init();

    /* FINE */
    IRTC_SelRtcClkout(RTCCLKOUT_SRC1);
  
    /* Used for IRTC compensation */
    FLL_SelIRCSCLK(FASTCLK_SRC);
    FLL_SetFastClkDiv(0); /* 4MHz clock */
    FLL_CtrlIRCLKEN(TRUE);

    /* Workaround for 1Hz output */
    PORT_Init(PORTD, PORT_MODULE_ALT1_MODE, PIN5);
    GPIO_Init(FGPIOD, GPIO_INP_MODE, PIN5);

    PORT_Init(PORTA,PORT_MODULE_ALT2_MODE, PIN7);
    /* set RTC CLK source to OSC_32K */
    SIM_SelRtcClk(SIM_RTCCLK_SRC1);
    /* Redirect the RTC CLKOUT through XBAR to output */
    XBAR_Path(XBAR_RTCCLKOUT, XBAR_OUT0);

#ifdef POWER_METERING_DEBUG
    PORT_Init(PORTE, PORT_MODULE_ALT5_MODE, PIN0);
    SIM_SelClkout(CLKOUT_SRC2);
#endif

    /* SAR start triger */
    XBAR_Path(XBAR_PIT, XBAR_ADCTRGCHD);

    /* Init PIT0 CH1 for RTC comp, frequency is 1Hz */
    PIT_Init(PIT0, CH1, PIT_CH_TMR_EN_CONFIG, (POWER_METERING_BUS_CLK_FREQ));
#else
    /* Suppress initial battery removal "tamper" detect                       */
    IRTC_ClrIsrFlag(IRTC_TAMPER0_MASK);
    IRTC_Init(IRTC_MODULE_COMP_OFF_CONFIG(0, 0),
                IRTC_TAMPER_PIN_POL_HIGH_CONFIG(TAMPER_FILT_CLK_8HZ, 10),
                IRTC_TAMPER_PIN_DI_CONFIG,
                IRTC_TAMPER_PIN_DI_CONFIG,
                0);
#endif
 
    return 0;
}

int32_t rtc_comp_deinit(void)
{
    PIT_Disable(PIT0, CH1);

    return 0;
}

#if POWER_METERING_ENABLE_RTC_COMP
int32_t rtc_comp_do_comp(void)
{
    int32_t new_temp;
    uint16_t val;
    uint16_t last_comp;
    int32_t real_temp;
    uint16_t near_comp;

    /* 60s to invoke an RTC compensation */
    if (rtc_comp_interval++ < 60)
    {
        return 0;
    }
    rtc_comp_interval = 0;
 
    new_temp = temp_sense_calc_cur_temp();

    /* time to run RTC compensation 
     * looup the comp_val[].
     */
    real_temp = new_temp / 2;
    if (new_temp & 1)
    {
        /* xx.5 C */
        if (real_temp <= 0)
            near_comp = rtc_comp[real_temp + abs(POWER_METERING_TEMP_MIN_TEMP) - 1];
        else
            near_comp = rtc_comp[real_temp + abs(POWER_METERING_TEMP_MIN_TEMP) + 1];
        val = (rtc_comp[real_temp + abs(POWER_METERING_TEMP_MIN_TEMP)] +  near_comp) / 2;
    }
    else
        val = rtc_comp[real_temp + abs(POWER_METERING_TEMP_MIN_TEMP)];

    last_comp = (RTC_COMPEN & 0xf000) >> 4;
    last_comp |= (RTC_COMPEN & 0x007f);
    if (last_comp != val)
        IRTC_UpdateFineCompVal((val >> 8), (val & 0xff));

    if (new_temp != last_temp)
    {
        last_temp = new_temp;
#if POWER_METERING_ENABLE_VREF_COMP
        vref_comp_update_calib_gain(real_temp);
#endif
    }
 
    return 0;
}
#endif

/*
diff: 1 day delta seconds. Read from meter calibration device.
translate diff to PPM.
*/
int32_t rtc_comp_cal_ppm(double diff)
{
    int32_t val;
  
    val = (int32)(diff * 1000000 / 24 / 3600);
  
    return val;
}

/*
translate the ppm to (int, frac) values.
input: 
ppm
output:
(int_part, frac)
IRTC_UpdateFineCompVal(int_part, frac)
*/
int32_t rtc_comp_cal_rtc(int32_t ppm, int8_t *pinte, uint8_t *pfrac)
{
    int32_t int_part;
    uint8_t frac, int_abs, left;
    uint32_t ppm_abs;
  
    int_part = (ppm << 15) / 1000000;
  
    if (ppm < 0)
    {
        int_part -= 1;
        int_abs = 0 - int_part;
        ppm_abs = 0 - ppm;
    }
    else
    {
        int_part += 1;
        int_abs = int_part;
        ppm_abs = ppm;
    }
  
    left = ((1000000 * int_abs) >> 15) - ppm_abs;
    frac = (left << 22) / 1000000;
    if (ppm > 0)
    {
        int_part -= 1;
        frac = 128 - frac;	
    }
  
    *pfrac = frac;
    if (int_part < 0)
        int_part += 16;

    *pinte = int_part;

    return 0;
}

/*
according to diff, cal the int, frac values.
input: 
diff: 1 day delta seconds. Read from meter calibration device.
output:
(integer, frac)

IRTC_UpdateFineCompVal(integer, frac)
*/
int32_t rtc_comp_cal_rtc_ext(double diff, int8_t *pinte, uint8_t *pfrac)
{	
   int32_t ppm;
  
   ppm = rtc_comp_cal_ppm(diff);
  
   return rtc_comp_cal_rtc(ppm, pinte, pfrac);	
}

/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/
