/*
* Copyright (c) 2014, Freescale Semiconductor, Inc.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include <string.h>
#include "scr_gsm11.h"

/*******************************************************************************
* Defines
******************************************************************************/
/*******************************************************************************
* Prototypes
******************************************************************************/

/*******************************************************************************
* Variables
******************************************************************************/

/*******************************************************************************
* Code
******************************************************************************/

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_fileInfo_print
 * Description   : print GSM file status 
 * This function will report GSM file status(MF/DF/EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_fileInfo_print(uint8_t *buf, uint16_t length)
{
    sc_debug("Byte1-2 RFU = 0x%02x%02x\r\n", buf[0], buf[1]);
    //MF/DF info
    if((length >= 13) && (buf[6] != 4))
    {
        sc_debug("MF/DF non-allocated space = 0x%02X%02X\r\n", buf[2], buf[3]);
        sc_debug("MF/DF identification description = 0x%02X%02X\r\n", buf[4], buf[5]);
        switch(buf[6])
        {
        case 0:
            sc_debug("File type: RFU\r\n");
            break;
        case 1:
            sc_debug("1: MF file selected\r\n");
            break;
        case 2:
            sc_debug("2: DF file selected\r\n");
            break;
        case 4:
            sc_debug("4: EF file selected\r\n");
            break;
        default:
            sc_debug("%d: Wrong value for file type\r\n", buf[6]);
            break;
        }
        sc_debug("Byte 8-12 RFU = 0x%02X%02X%02X%02X%02X\r\n", buf[7], buf[8], buf[9], buf[10], buf[11]);
        if ((length >= 13))
        {
            sc_debug("Optional Data bytes count = %d\r\n", buf[12]);
            sc_debug("File parameter = 0x%02X:\r\n", buf[13]);
            sc_debug("Clock parameter = 0b%x%x%x\r\n", (buf[13] & 0x1), (buf[13] & 0x4) >> 2, (buf[13] & 0x8) >> 3);
            sc_debug("Authentication clock parameter = 0b%x\r\n", (buf[13] & 0x2) >> 1);
            sc_debug("CHV1 enable parameter = 0b%x\r\n", (buf[13] & 0x80) >> 7);
            sc_debug("Current directory DF count = %d\r\n", buf[14]);
            sc_debug("Current directory EF count = %d\r\n", buf[15]);
            sc_debug("CHV+PUK+ADM count = %d\r\n", buf[16]);
            sc_debug("Byte18 RFU = 0x%02X\r\n", buf[17]);
            sc_debug("CHV1 status = 0x%02X:\r\n", buf[18]);
            sc_debug("CHV1 remain times = %d\r\n", buf[18] & 0xF);
            sc_debug("CHV1 secure code init = 0b%x\r\n", (buf[18] & 0x80) >> 7);
            sc_debug("UCHV1 status = 0x%02X:\r\n", buf[19]);
            sc_debug("UCHV1 remain times = %d\r\n", buf[19] & 0xF);
            sc_debug("UCHV1 secure code init = 0b%x\r\n", (buf[19] & 0x80) >> 7);
            sc_debug("CHV2 status = 0x%02X:\r\n", buf[20]);
            sc_debug("CHV2 remain times = %d\r\n", buf[20] & 0xF);
            sc_debug("CHV2 secure code init = 0b%x\r\n", (buf[20] & 0x80) >> 7);
            sc_debug("UCHV2 status = 0x%02X:\r\n", buf[21]);
            sc_debug("UCHV2 remain times = %d\r\n", buf[21] & 0xF);
            sc_debug("UCHV2 secure code init = 0b%x\r\n", (buf[21] & 0x80) >> 7);
            sc_debug("Byte23 RFU = 0x%02X\r\n", buf[22]);
        }
    }
    else
    {
        sc_debug("File size =0x%02x%02x\r\n", buf[2], buf[3]);
        sc_debug("EF identification description = 0x%02x%02x\r\n", buf[4], buf[5]);
        switch (buf[6])
        {
        case 0:
            sc_debug("File type: RFU\r\n");
            break;
        case 1:
            sc_debug("1: MF file selected\r\n");
            break;
        case 2:
            sc_debug("2: DF file selected\r\n");
            break;
        case 4:
            sc_debug("4: EF file selected\r\n");
            break;
        default:
            sc_debug("%d: Wrong value for file type\r\n", buf[6]);
            break;
        }
        sc_debug("Byte 8 RFU = 0x%02x\r\n", buf[7]);

        sc_debug("Byte 9-11 Access right = 0x%02x%02x%02x\r\n", buf[8], buf[9], buf[10]);
        sc_debug("File status = 0x%02x\r\n", buf[11]);
        if (length >= 13)
        {
            sc_debug("Optional Data bytes count = %d\r\n", buf[12]);
            sc_debug("File structure =0x%02x\r\n", buf[13]);
            sc_debug("Record length = 0x%02x\r\n", buf[14]);
        }
    }
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_dataInfo_print
 * Description   : print GSM file data information 
 * This function will report GSM file data(MF/DF/EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_dataInfo_print(uint8_t *buf, uint16_t length)
{
    uint16_t i;

    for (i = 0; i < length; i++)
    {
        sc_debug("%02X ", buf[i]);
    }
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_selectFile
 * Description   : Select GSM file via file ID 
 * This function will select GSM file(MF/DF/EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_selectFile(uint16_t fileID)
{
    uint16_t length = 0;

    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_SELECT_FILE;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = 2;
    sc_adpu_command.adpuBody.Data[0] = (uint8_t)(fileID>>8);
    sc_adpu_command.adpuBody.Data[1] = (uint8_t)(fileID&0xFF);
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = 2;

    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);

#ifdef SC_GSM11_PRINT_FILE_INFO
    //only print file select report if command terminate successfully
    if (sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("File select command end successfully\r\n");
        sc_gsm_fileInfo_print(&gRxBuf[1], length-3);
    }
#endif
    
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_readBinary
 * Description   : Read GSM binary file data 
 * This function will read GSM binary file data(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_readBinary(uint16_t offset, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_READ_BINARY;
    sc_adpu_command.adpuHeader.P1 = (uint8_t)(offset >> 8);;
    sc_adpu_command.adpuHeader.P2 = (uint8_t)(offset & 0xFF);;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = length;
    sc_adpu_command.adpuBody.LC = 0;

        
    length = sc_send_command_without_data(&sc_adpu_command, &sc_adpu_response);

#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if (sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("Command end successfully\r\n");
        sc_gsm_dataInfo_print(&gRxBuf[1], length - 3);
    }
#endif
    
    return 0;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_writeBinary
 * Description   : Write GSM binary file data 
 * This function will write data into GSM binary file data(EF)(Not support by GSM)
 *
 *END**************************************************************************/
int32_t sc_gsm_writeBinary(uint16_t offset, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_WRITE_BINARY;
    sc_adpu_command.adpuHeader.P1 = (uint8_t)(offset >> 8);;
    sc_adpu_command.adpuHeader.P2 = (uint8_t)(offset & 0xFF);;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = length;

        
    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("Command end successfully\r\n");
        sc_gsm_dataInfo_print(&gRxBuf[1], length - 3);
    }
#endif
    
    return 0;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_updateBinary
 * Description   : Update GSM binary file data 
 * This function will update data into GSM binary file data(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_updateBinary(uint16_t offset, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_UPDATE_BINARY;
    sc_adpu_command.adpuHeader.P1 = (uint8_t)(offset >> 8);;
    sc_adpu_command.adpuHeader.P2 = (uint8_t)(offset & 0xFF);;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = length;
      
    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);
    
    return length;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_readRecord
 * Description   : Read data from GSM record file 
 * This function will read data from GSM record file data(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_readRecord(uint8_t recordNumber,uint8_t recordMode, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_READ_RECORD;
    sc_adpu_command.adpuHeader.P1 = recordNumber;
    sc_adpu_command.adpuHeader.P2 = recordMode;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = length;
    sc_adpu_command.adpuBody.LC = 0;

        
    length = sc_send_command_without_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("Command end successfully\r\n");
        sc_gsm_dataInfo_print(&gRxBuf[1], length - 3);
    }
#endif
    
    return 0;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_writeRecord
 * Description   : Write GSM record file data 
 * This function will write data into GSM record file data(EF)(Not support by GSM)
 *
 *END**************************************************************************/
int32_t sc_gsm_writeRecord(uint8_t recordNumber,uint8_t recordMode, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_WRITE_RECORD;
    sc_adpu_command.adpuHeader.P1 = recordNumber;
    sc_adpu_command.adpuHeader.P2 = recordMode;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = length;

        
    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("Command end successfully\r\n");
        sc_gsm_dataInfo_print(&gRxBuf[1], length - 3);
    }
#endif
    
    return 0;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_updateRecord
 * Description   : Update GSM record file data 
 * This function will update data into GSM record file data(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_updateRecord(uint8_t recordNumber, uint8_t recordMode, uint16_t length)
{
        
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_UPDATE_RECORD;
    sc_adpu_command.adpuHeader.P1 = recordNumber;
    sc_adpu_command.adpuHeader.P2 = recordMode;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = length;

        
    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);

    return 0;

}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_seek
 * Description   : Search pattern in GSM file 
 * This function will search data patten in GSM file(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_seek(uint16_t length)
{

    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_SEEK;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = length;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = length;

        
    length = sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);

#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("Pattern SEEK command end successfully\r\n");
    }else
    {
        sc_debug("Pattern SEEK command end fail\r\n");
    }
#endif
    
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_increase
 * Description   : Increase data record in GSM file 
 * This function will increase data record in GSM file(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_increase(uint16_t length)
{
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_INCREASE;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = 3;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = 3;

        
    sc_send_command_with_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("INCREASE command end successfully\r\n");
    }
    else
    {
        sc_debug("INCREASE command end fail\r\n");
    }
#endif
    
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_invalidate
 * Description   : Invalidate GSM file 
 * This function will invalidate GSM file(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_invalidate(void)
{
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_INVALIDATE;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = 0;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = 0;

        
    sc_send_command_without_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("INCREASE command end successfully\r\n");
    }else
    {
        sc_debug("INCREASE command end fail\r\n");
    }
#endif
    
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_rehabilitate
 * Description   : Rehabilitate GSM file 
 * This function will rehabilitate GSM file(EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_rehabilitate(void)
{   
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_REHABILITATE;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = 0;
    sc_adpu_command.adpuBody.LE = 0;
    sc_adpu_command.adpuBody.LC = 0;

        
    sc_send_command_without_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_DATA_INFO
    //only print file select report if command terminate successfully
    if(sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("INCREASE command end successfully\r\n");
    }
    else
    {
        sc_debug("INCREASE command end fail\r\n");
    }
#endif
    
    return 0;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : sc_gsm_status
 * Description   : Get GSM file status 
 * This function will get GSM file status(MF/DF/EF)
 *
 *END**************************************************************************/
int32_t sc_gsm_status(uint16_t length)
{
    sc_adpu_command.adpuHeader.CLA = SC_CLA_GSM11;
    sc_adpu_command.adpuHeader.INS = SC_INS_STATUS;
    sc_adpu_command.adpuHeader.P1 = 0;
    sc_adpu_command.adpuHeader.P2 = 0;
    sc_adpu_command.adpuHeader.P3 = length;
       
    length = sc_send_command_without_data(&sc_adpu_command, &sc_adpu_response);


#ifdef SC_GSM11_PRINT_FILE_INFO
    //only print file select report if command terminate successfully
    if (sc_adpu_response.SW1 == 0x90)
    {
        sc_debug("File select command end successfully\r\n");
        sc_gsm_fileInfo_print(&gRxBuf[1], length - 3);
    }
#endif
    
    return 0;
}

/*******************************************************************************
 * EOF
 ******************************************************************************/

