/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __SCR_GSM11_H__
#define __SCR_GSM11_H__

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include "scr_ttl.h"
#include "scr_interface.h"

#define SC_GSM11_PRINT_FILE_INFO
#define SC_GSM11_PRINT_DATA_INFO


#define SC_GSM11_MF_ID           0x3F00

/*------------------------ GSM11 File Structure---------------------*/
#define SC_GSM11_DF_TEL_ID    0x7F10
#define SC_GSM11_EF_ADN_ID    0x6F3A
#define SC_GSM11_EF_FDN_ID    0x6F3B
#define SC_GSM11_EF_SMS_ID    0x6F3C
#define SC_GSM11_EF_CCP_ID    0x6F3D
#define SC_GSM11_EF_MSI_ID    0x6F40
#define SC_GSM11_EF_SMSP_ID   0x6F42
#define SC_GSM11_EF_SMSS_ID   0x6F43
#define SC_GSM11_EF_LND_ID    0x6F44
#define SC_GSM11_EF_SDN_ID    0x6F49
#define SC_GSM11_EF_EXT1_ID   0x6F4A
#define SC_GSM11_EF_EXT2_ID   0x6F4B
#define SC_GSM11_EF_EXT3_ID   0x6F4C

#define SC_GSM11_DF_GSM_ID    0x7F20
#define SC_GSM11_EF_LP_ID     0x6F05
#define SC_GSM11_EF_IMSI_ID   0x6F07
#define SC_GSM11_EF_KC_ID     0x6F20
#define SC_GSM11_EF_PLMN_ID   0x6F30
#define SC_GSM11_EF_HPLMN_ID  0x6F31
#define SC_GSM11_EF_ACMM_ID   0x6F37
#define SC_GSM11_EF_SST_ID    0x6F38
#define SC_GSM11_EF_ACM_ID    0x6F39
#define SC_GSM11_EF_GID1_ID   0x6F3E
#define SC_GSM11_EF_GID2_ID   0x6F3F
#define SC_GSM11_EF_PUCT_ID   0x6F41
#define SC_GSM11_EF_CBMI_ID   0x6F45
#define SC_GSM11_EF_SPN_ID    0x6F46
#define SC_GSM11_EF_CBMID_ID  0x6F48
#define SC_GSM11_EF_BCCH_ID   0x6F74
#define SC_GSM11_EF_ACC_ID    0x6F78
#define SC_GSM11_EF_FPLMN_ID  0x6F7B
#define SC_GSM11_EF_LOCI_ID   0x6F7E
#define SC_GSM11_EF_AD_ID     0x6FAD
#define SC_GSM11_EF_PHASE_ID  0x6FAE
#define SC_GSM11_EF_VGCS_ID   0x6FB1
#define SC_GSM11_EF_VGCSS_ID  0x6FB2
#define SC_GSM11_EF_VBS_ID    0x6FB3
#define SC_GSM11_EF_VBSS_ID   0x6FB4
#define SC_GSM11_EF_EMLPP_ID  0x6FB5
#define SC_GSM11_EF_AAEM_ID   0x6FB6
#define SC_GSM11_EF_ECC_ID    0x6FB7
#define SC_GSM11_EF_CBMIR_ID  0x6F50

#define SC_GSM11_EF_ICCID_ID  0x2FE2


      
#define SC_GSM11_ICCID_LEN        10
#define SC_GSM11_MF_STATUS_LEN    23
#define SC_GSM11_DF_STATUS_LEN    23
#define SC_GSM11_EF_STATUS_LEN    15

/* SC Tree Structure -----------------------------------------------------------
                              MasterFile
                           ________|___________
                          |        |           |
                        System   UserData     Note
------------------------------------------------------------------------------*/
#define SC_DEFAULT_FI      372
#define SC_DEFAULT_DI      1

/* SC ADPU Command: Operation Code -------------------------------------------*/
#define SC_CLA_GSM11       0xA0

/*------------------------ Data Area Management Commands ---------------------*/
#define SC_INS_SELECT_FILE     0xA4
#define SC_INS_GET_RESPONCE    0xC0
#define SC_INS_STATUS          0xF2

#define SC_GET_DATA        0xCA
#define SC_PUT_DATA        0xDA

#define SC_INS_READ_BINARY     0xB0
#define SC_INS_WRITE_BINARY    0xD0
#define SC_INS_UPDATE_BINARY   0xD6
#define SC_INS_ERASE_BINARY    0x0E

#define SC_INS_READ_RECORD     0xB2
#define SC_INS_UPDATE_RECORD   0xDC
#define SC_INS_WRITE_RECORD    0xD2
#define SC_INS_SEEK            0xA2
#define SC_INS_INCREASE        0x32

#define SC_INS_INVALIDATE      0x04
#define SC_INS_REHABILITATE    0x44
                              
                            
/*-------------------------- Administrative Commands -------------------------*/ 
#define SC_INS_CREATE_FILE     0xE0

/*-------------------------- Safety Management Commands ----------------------*/
#define SC_INS_VERIFY          0x20
#define SC_INS_CHANGE          0x24
#define SC_INS_DISABLE         0x26
#define SC_INS_ENABLE          0x28
#define SC_INS_UNBLOCK         0x2C
#define SC_INS_EXTERNAL_AUTH   0x82
#define SC_INS_GET_CHALLENGE   0x84


#if defined(__cplusplus)
extern "C" {
#endif

/* Exported functions ------------------------------------------------------- */

/*!
 * @name GSM card application functions
 * @{
 */

/*!
 * @brief Select GSM card file
 *
 * @param fileID  GSM file ID
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_selectFile(uint16_t fileID);

/*!
 * @brief Get GSM selected file status
 *
 * @param length  length of GSM file status
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_status(uint16_t length);

/*!
 * @brief Get GSM selected file binary data
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_readBinary(uint16_t offset, uint16_t length);

/*!
 * @brief Write data to GSM selected binary file
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_writeBinary(uint16_t offset, uint16_t length);

/*!
 * @brief Update data to GSM selected binary file
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_updateBinary(uint16_t offset, uint16_t length);

/*!
 * @brief Get GSM selected file record data
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_readRecord(uint8_t recordNumber,uint8_t recordMode, uint16_t length);

/*!
 * @brief Write data to GSM selected record file
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_writeRecord(uint8_t recordNumber,uint8_t recordMode, uint16_t length);

/*!
 * @brief Update data to GSM selected record file
 *
 * @param offset  offset of GSM file  
 * @param length  length of data to read from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_updateRecord(uint8_t recordNumber,uint8_t recordMode, uint16_t length);

/*!
 * @brief Search data pattern in GSM selected file
 *
 * @param length  length of data to search from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_seek(uint16_t length);

/*!
 * @brief Increase data in GSM selected file
 *
 * @param length  length of data to increase from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_increase(uint16_t length);

/*!
 * @brief Invalidate GSM selected file
 *
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_invalidate(void);

/*!
 * @brief Rehabilitate data in GSM selected file
 *
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_rehabilitate(void );

/*!
 * @brief Report GSM selected file information
 *
 * @param buf     pointor of data for GSM file status
 * @param length  length of data received from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_fileInfo_print(uint8_t *buf, uint16_t length);

/*!
 * @brief Report data received from GSM selected file
 *
 * @param buf     pointor of data from GSM file
 * @param length  length of data received from GSM file   
 * @return Error or success status returned by API.
 */
int32_t sc_gsm_dataInfo_print(uint8_t *buf, uint16_t length);

#if defined(__cplusplus)
}
#endif

#endif  /* __FSL_SCR_GSM11_H__*/

/*******************************************************************************
 * EOF
 ******************************************************************************/
