/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __SCR_INTERFACE_H__
#define __SCR_INTERFACE_H__

#include "common.h"
#include "metering_modules.h"
#include "drivers.h"
#include "scr_ttl.h"

/*******************************************************************************
 * Defination
 ******************************************************************************/
//If use TDA8020 or not. define 0 to use TDA8020
#define CARD_DIRECT_INTERFACE 0


/*******************************************************************************
 * Defination
 ******************************************************************************/

enum
{
    kColdReset,                   /*!< Cold reset indicator */
    kWarmReset                    /*!< Warm reset indicator */
};

/* Exported macro ------------------------------------------------------------*/
/* Exported functions ------------------------------------------------------- */
/* APPLICATION LAYER ---------------------------------------------------------*/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*!
 * @brief Smart card reader initialization function.
 *
 * @param sCReaderParamsPtr Pointer to smart card paramerter structure
 * @return   1
 */
int8_t scr_init(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Wait and receive an ATR from the card.
 *
 * @param  sCReaderParamsPtr  Pointer to smart card paramerter structure
 * @param  buff               Pointer to a buffer memory
 * @return  Size of ATR received if passes, -1 if fails.
 */
int16_t scr_receive_atr(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buff);

/*!
 * @brief Parse the ATR data as per ISO7816/EMV specifications.
 *
 * @param sCReaderParamsPtr  Pointer to smart card paramerter structure
 * @param buff               Pointer to a buffer memory
 * @param length             length of buffer memory
 * @return  0 if passes, -1 if fails.
 */
int16_t scr_parse_atr(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buff, int16_t length);

/*!
 * @brief Wait and receive an ATR from the card ,parses it and takes actions 
 * like warm reset or deactivation as required.
 *
 * @param sCReaderParamsPtr  Pointer to smart card paramerter structure
 * @param buff               Pointer to a buffer memory
 * @return  Size of ATR received if passes, -1 if fails.
 */
int16_t scr_receive_atr_connect(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *buff);

/*!
 * @brief Rejects a card session.
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_reject_card(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Send PPS command
 *
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @return NA
 */
void scr_pps(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Sends a PPS to the ICC and gets response.
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @param ppsRequest          Pointer to buffer containing PPS request
 * @param ppsReqsponse        Pointer to buffer containing PPS response
 * @return 1 if passes, 0 if fails.
 */
int16_t scr_send_pps_data(smartcardreader_params_t *sCReaderParamsPtr, uint8_t *ppsRequest, uint8_t *ppsResponse, uint16_t length);

/*!
 * @brief Function to setup the driver in T=0 mode
 *
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_setup_t0(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Function to setup the driver in T=1 mode
 * 
 * @param sCReaderParamsPtr  Pointer to smart card paramerter structure
 */
void scr_setup_t1(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Function to reset and activate the card
 *
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @param cardSlotNo          Card slot no.
 * @param resetType           0 = Cold Reset, 1 = Warm Reset
 */
void scr_interface_activate(smartcardreader_params_t *sCReaderParamsPtr, uint8_t vcc, uint8_t resetType);

/*!
 * @brief Function to de-activate the card
 *
 * @param cardSlotNo          Card slot no.
 */
void scr_interface_deactivate(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Initialize the smart card reader interface module(internal or external).
 *
 * @param cardSlotNo   smart card port number
 */
void scr_interface_init(void);
 
/*!
 * @brief Initialize the smart card clock input(TPM channels).
 *
 * @param cardSlotNo   smart card port number
 */
uint32_t scr_clock_init(void);

/*!
 * @brief Active the smart card clock input(active TPM channels).
 *
 * @param cardSlotNo   smart card port number
 */
void scr_clock_active(void);

/*!
 * @brief Active the smart card TDA8020 interface.
 *
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @param cardSlotNo          smart card port number
 * @param vcc                 card voltage
 * @param resetType           Reset type, cold or warm
 */
void scr_interface_activate(smartcardreader_params_t *sCReaderParamsPtr, 
                            uint8_t vcc, uint8_t resetType);

/*!
 * @brief smart card status report.
 *
 * @param cardSlotNo   smart card port number
 */
void scr_interface_report_status(void);

/*!
 * @brief Initialize the uart module for ISO7816/EMV mode.
 * 
 * @param sCReaderParamsPtr  Pointer to smart card paramerter structure
 */
void scr_emv_init(smartcardreader_params_t *sCReaderParamsPtr);
void scr_emv_deinit(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Configures the baudrate of the uart module.
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_configure_baudrate(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Receives characters from the card in polling mode, use UART interrupt 
 *  for error handling.
 * 
 * @param sCReaderParamsPtr  Pointer to smart card paramerter structure
 * @param buf                memory buffer pointer to receive data
 * @param length             length of receive data expected
 * @return  -1 if fails, number of received characters if passes.
 */
int16_t scr_receive_chars_polling(smartcardreader_params_t *sCReaderParamsPtr, uint8_t* buf, uint16_t length);

/*!
 * @brief Receives characters from the card.
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @param buf                 memory buffer pointer to receive data
 * @param length              length of receive data
 * @return   -1 if fails, number of received characters if passes.
 */
int16_t scr_receive_chars_atr(smartcardreader_params_t *sCReaderParamsPtr, uint8_t* buf, uint16_t length);


/*!
 * @brief Sends characters to the card.
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @param buf                 memory buffer pointer to send data
 * @param length              length of transmit data
 */
void scr_transmit_chars_polling(smartcardreader_params_t *sCReaderParamsPtr, uint8_t* buf, int length);

/*!
 * @brief Resets WT timer
 * 
 * @param UART_MemMapPtr ptr - ponter to UART block registers
 */
void scr_reset_wwt_timer(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Resets CWT timer
 *
 * @param UART_MemMapPtr ptr - ponter to UART block registers
 */
void scr_reset_cwt_timer(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Resets BWT timer
 * 
 * @param UART_MemMapPtr ptr - ponter to UART block registers
 */
void scr_reset_bwt_timer(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Resets GT timer
 * 
 * @param UART_MemMapPtr ptr - ponter to UART block registers
 */
void scr_reset_gt_timer(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Sends a cold reset to the card
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @return 1 if passes, 0 if fails.
 */
void scr_cold_reset(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Sends a warm reset to the card
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 * @return 1 if passes, 0 if fails.
 */
void scr_warm_reset(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 *@brief Enable Initial character detection.
 * 
 *@param sCReaderParamsPtr   Pointer to smart card paramerter structure
 *
 */
void scr_enable_init_detect(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 *@brief Setup for ATR reception.
 * 
 *@param sCReaderParamsPtr    Pointer to smart card paramerter structure
 *
 *All settings specific to ATR are done here
 */
void scr_setup_atr(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Enable WT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_enable_wwt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief disable WT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_disable_wwt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Enable ADT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_enable_adt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Disable WT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_disable_adt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Enable CWT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_enable_cwt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Disable CWT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_disable_cwt(UART_MemMapPtr ptr);

/*!
 * @brief Enable BWT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_enable_bwt(smartcardreader_params_t *sCReaderParamsPtr);

/*!
 * @brief Disable WT timer
 * 
 * @param sCReaderParamsPtr   Pointer to smart card paramerter structure
 */
void scr_disable_bwt(smartcardreader_params_t *sCReaderParamsPtr);

#endif /* __SMARTCARDREADER_H__ */

/*******************************************************************************
 * EOF
 ******************************************************************************/
