/***********************************************************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006-2009 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
*********************************************************************************************************************//*!
*
* @file   CTS_Sensor.c
*
* @brief  Contains functions to Perform the sensing ot the electrodes and set the status for each electrode
*
* @version 1.0.28.0
* 
* @date Jun-4-2010
* 
*
***********************************************************************************************************************/

#include "CTS_Sensor.h"

#if TSS_USE_CTS_LOW_LEVEL

  #include "CTS_LowTypes.h"

  /* Macros */
  #define TSS_PORT(port,pin)            &CTS_##port##pin##HS
  #define TSS2CTS_CONV(port, pin)		    TSS_PORT(port, pin)
  #define CTS_LOW_ROUTINE(X)            TSS2CTS_CONV(TSS_##X##_P,TSS_##X##_B)

  /* Variables & Arrays */
  INT8 cts_i8Turbo[TSS_N_ELECTRODES];     /* Array that contains Turbo values  */

  #if TSS_USE_NOISE_WINDOWING
    INT8 cts_ai8WindowBase[TSS_N_ELECTRODES];
    extern const INT8 tss_cai8WindowDeltaLimit[TSS_N_ELECTRODES];
  #endif
  
  /* 
  *  Array that contains pointers to the CTS sensing functions.  
  *  The array is defined depending on the number of electrodes stablished by the user
  */ 
  INT8 (* const faCTS_SampleElectrode[TSS_N_ELECTRODES]) (INT8) =
  {
      CTS_LOW_ROUTINE(E0),                   
      #if TSS_N_ELECTRODES > 1
          CTS_LOW_ROUTINE(E1),               
      #endif 
      #if TSS_N_ELECTRODES > 2
          CTS_LOW_ROUTINE(E2),               
      #endif
      #if TSS_N_ELECTRODES > 3
          CTS_LOW_ROUTINE(E3),               
      #endif
      #if TSS_N_ELECTRODES > 4
          CTS_LOW_ROUTINE(E4),
      #endif
      #if TSS_N_ELECTRODES > 5
          CTS_LOW_ROUTINE(E5),               
      #endif 
      #if TSS_N_ELECTRODES > 6
          CTS_LOW_ROUTINE(E6),               
      #endif
      #if TSS_N_ELECTRODES > 7
          CTS_LOW_ROUTINE(E7),               
      #endif
      #if TSS_N_ELECTRODES > 8
          CTS_LOW_ROUTINE(E8),
      #endif
      #if TSS_N_ELECTRODES > 9
          CTS_LOW_ROUTINE(E9),               
      #endif 
      #if TSS_N_ELECTRODES > 10
          CTS_LOW_ROUTINE(E10),               
      #endif
      #if TSS_N_ELECTRODES > 11
          CTS_LOW_ROUTINE(E11),               
      #endif
      #if TSS_N_ELECTRODES > 12
          CTS_LOW_ROUTINE(E12),
      #endif
      #if TSS_N_ELECTRODES > 13
          CTS_LOW_ROUTINE(E13),               
      #endif
      #if TSS_N_ELECTRODES > 14
          CTS_LOW_ROUTINE(E14),
      #endif
      #if TSS_N_ELECTRODES > 15
          CTS_LOW_ROUTINE(E15),               
      #endif 
      #if TSS_N_ELECTRODES > 16
          CTS_LOW_ROUTINE(E16),               
      #endif
      #if TSS_N_ELECTRODES > 17
          CTS_LOW_ROUTINE(E17),               
      #endif
      #if TSS_N_ELECTRODES > 18
          CTS_LOW_ROUTINE(E18),
      #endif
      #if TSS_N_ELECTRODES > 19
          CTS_LOW_ROUTINE(E19),               
      #endif 
      #if TSS_N_ELECTRODES > 20
          CTS_LOW_ROUTINE(E20),               
      #endif
      #if TSS_N_ELECTRODES > 21
          CTS_LOW_ROUTINE(E21),        
      #endif 
      #if TSS_N_ELECTRODES > 22
          CTS_LOW_ROUTINE(E22),        
      #endif
      #if TSS_N_ELECTRODES > 23
          CTS_LOW_ROUTINE(E23),        
      #endif
      #if TSS_N_ELECTRODES > 24
          CTS_LOW_ROUTINE(E24),
      #endif
      #if TSS_N_ELECTRODES > 25
          CTS_LOW_ROUTINE(E25),        
      #endif 
      #if TSS_N_ELECTRODES > 26
          CTS_LOW_ROUTINE(E26),        
      #endif
      #if TSS_N_ELECTRODES > 27
          CTS_LOW_ROUTINE(E27),        
      #endif
      #if TSS_N_ELECTRODES > 28
          CTS_LOW_ROUTINE(E28),
      #endif
      #if TSS_N_ELECTRODES > 29
          CTS_LOW_ROUTINE(E29),        
      #endif 
      #if TSS_N_ELECTRODES > 30
          CTS_LOW_ROUTINE(E30),       
      #endif
      #if TSS_N_ELECTRODES > 31
          CTS_LOW_ROUTINE(E31),        
      #endif 
      #if TSS_N_ELECTRODES > 32
          CTS_LOW_ROUTINE(E32),        
      #endif
      #if TSS_N_ELECTRODES > 33
          CTS_LOW_ROUTINE(E33),        
      #endif
      #if TSS_N_ELECTRODES > 34
          CTS_LOW_ROUTINE(E34),
      #endif
      #if TSS_N_ELECTRODES > 35
          CTS_LOW_ROUTINE(E35),        
      #endif 
      #if TSS_N_ELECTRODES > 36
          CTS_LOW_ROUTINE(E36),        
      #endif
      #if TSS_N_ELECTRODES > 37
          CTS_LOW_ROUTINE(E37),        
      #endif
      #if TSS_N_ELECTRODES > 38
          CTS_LOW_ROUTINE(E38),
      #endif
      #if TSS_N_ELECTRODES > 39
          CTS_LOW_ROUTINE(E39),        
      #endif 
      #if TSS_N_ELECTRODES > 40
          CTS_LOW_ROUTINE(E40),        
      #endif 
      #if TSS_N_ELECTRODES > 41
          CTS_LOW_ROUTINE(E41),        
      #endif 
      #if TSS_N_ELECTRODES > 42
          CTS_LOW_ROUTINE(E42),        
      #endif
      #if TSS_N_ELECTRODES > 43
          CTS_LOW_ROUTINE(E43),        
      #endif
      #if TSS_N_ELECTRODES > 44
          CTS_LOW_ROUTINE(E44),
      #endif
      #if TSS_N_ELECTRODES > 45
          CTS_LOW_ROUTINE(E45),        
      #endif 
      #if TSS_N_ELECTRODES > 46
          CTS_LOW_ROUTINE(E46),        
      #endif
      #if TSS_N_ELECTRODES > 47
          CTS_LOW_ROUTINE(E47),        
      #endif
      #if TSS_N_ELECTRODES > 48
          CTS_LOW_ROUTINE(E48),
      #endif
      #if TSS_N_ELECTRODES > 49
          CTS_LOW_ROUTINE(E49),        
      #endif 
      #if TSS_N_ELECTRODES > 50
          CTS_LOW_ROUTINE(E50),       
      #endif
      #if TSS_N_ELECTRODES > 51
          CTS_LOW_ROUTINE(E51),        
      #endif 
      #if TSS_N_ELECTRODES > 52
          CTS_LOW_ROUTINE(E52),        
      #endif
      #if TSS_N_ELECTRODES > 53
          CTS_LOW_ROUTINE(E53),        
      #endif
      #if TSS_N_ELECTRODES > 54
          CTS_LOW_ROUTINE(E54),
      #endif
      #if TSS_N_ELECTRODES > 55
          CTS_LOW_ROUTINE(E55),        
      #endif 
      #if TSS_N_ELECTRODES > 56
          CTS_LOW_ROUTINE(E56),        
      #endif
      #if TSS_N_ELECTRODES > 57
          CTS_LOW_ROUTINE(E57),        
      #endif
      #if TSS_N_ELECTRODES > 58
          CTS_LOW_ROUTINE(E58),
      #endif
      #if TSS_N_ELECTRODES > 59
          CTS_LOW_ROUTINE(E59),        
      #endif 
      #if TSS_N_ELECTRODES > 60
          CTS_LOW_ROUTINE(E60),       
      #endif
      #if TSS_N_ELECTRODES > 61
          CTS_LOW_ROUTINE(E61),        
      #endif 
      #if TSS_N_ELECTRODES > 62
          CTS_LOW_ROUTINE(E62),        
      #endif
      #if TSS_N_ELECTRODES > 63
          CTS_LOW_ROUTINE(E63),        
      #endif
  };


  /***************************************************************************//*!
  *
  * @brief  Initializes the variables related to CTS Sensor   
  *
  * @param  Nothing
  *
  * @return Status Code
  *
  * @remarks  Set cts_i8Turbo array to default value and initiliaze Noise Windowing
  *
  ****************************************************************************/ 
  UINT8 CTS_SensorInit(void)
  {
      UINT8 u8counter = 0;

      for (u8counter = 0; u8counter < TSS_N_ELECTRODES; u8counter++) {
        cts_i8Turbo[u8counter] = -127;       /* Set default value */
        
        #if TSS_USE_NOISE_WINDOWING
         /* Initializes the Noise Window variables */
          cts_ai8WindowBase[u8counter] = 0;  /* Set default value */
        #endif  
      }
      
      return TSS_STATUS_OK;
  }
  
  /***************************************************************************//*!
  *
  * @brief  Performs a determination of Turbo values for certain electrode  
  *
  * @param  u8ElecNum Required Electrode number
  *
  * @return Status Code
  *
  * @remarks
  *
  ****************************************************************************/ 
  UINT8 CTS_GetTurbo(UINT8 u8ElecNum) 
  {
     UINT8 u8FaultCnt = CTS_FAULT_TIMEOUT;
     UINT8 u8Result = TSS_STATUS_OK;
     INT8 i8SampleTest = 0;
     INT8 i8TurboTemp = 0;
     
     /* Determine first Turbo value */
     u8FaultCnt = CTS_FAULT_TIMEOUT;
     do {
        i8TurboTemp = faCTS_SampleElectrode[u8ElecNum](cCTS_ARG_TURBO);
        u8FaultCnt--;
     } while ((i8TurboTemp < 0) && (u8FaultCnt > 0));
       
     if (u8FaultCnt > 0) { /* Success */
     
       /* Try maximize turbo speed-up */
       u8FaultCnt = CTS_FAULT_TIMEOUT;
       do {
         do {
            i8SampleTest = faCTS_SampleElectrode[u8ElecNum](++i8TurboTemp);
         } while (i8SampleTest > 0);     

         if (i8SampleTest == cCTS_error_ExISR) { /* Solving of situation when Interrupt occured */
           i8TurboTemp--;   
         }

         u8FaultCnt--; 
       } while ((i8SampleTest == cCTS_error_ExISR) && (u8FaultCnt > 0));
       
       /* Timeout evaluation after measurement */
       if (u8FaultCnt > 0) {
           
           /* Align turbo speed-up to first one appropriate */
           u8FaultCnt = CTS_FAULT_TIMEOUT;
           do {
             do {
                i8SampleTest = faCTS_SampleElectrode[u8ElecNum](--i8TurboTemp);
             } while ((i8SampleTest == 0) && (i8TurboTemp > 0));
             
             if (i8SampleTest == cCTS_error_ExISR) {  /* Solving of situation when Interrupt occured */
               i8TurboTemp++;   
             }
             
             u8FaultCnt--; 
           } while ((i8SampleTest == cCTS_error_ExISR) && (u8FaultCnt > 0));
           
           if (u8FaultCnt > 0) {
              u8Result = TSS_STATUS_OK;
  
              /* Change Turbo value just when difference is more than 1, for prevention of oscillations*/
              if (cts_i8Turbo[u8ElecNum] != i8TurboTemp) {
                if ( ((cts_i8Turbo[u8ElecNum] - i8TurboTemp) > 1) || ((cts_i8Turbo[u8ElecNum] - i8TurboTemp) < (-1)) ) {
                  u8Result = TSS_CALIBRATION_CHANGED;
                  cts_i8Turbo[u8ElecNum] = i8TurboTemp;
                }
              }
           
           } else {
              u8Result = TSS_ERROR_CHARGE_TIMEOUT; /* If a fault has occurred sets the return status code error */
           } 
             
       } else {
          u8Result = TSS_ERROR_CHARGE_TIMEOUT; /* If a fault has occurred sets the return status code error */
       }
     
     } else {
       u8Result = TSS_ERROR_CHARGE_TIMEOUT; /* If a fault has occurred sets the return status code error */
     }
     
     return u8Result;
  }

  /***************************************************************************//*!
  *
  * @brief  Performs a valid reading of one Electrode stores the timer values 
  *         and returns a status code  
  *
  * @param  u8ElecNum Required Electrode number
  *
  * @return Status Code
  *
  * @remarks
  *
  ****************************************************************************/
  UINT8 CTS_SampleElectrode(UINT8 u8ElecNum)
  {
      UINT8 u8NSamp;
      UINT8 u8ElectrodeStatus = TSS_STATUS_OK;  
      UINT8 u8FaultCnt = CTS_FAULT_TIMEOUT;
      UINT8 u8RecalibCounter = 0u;
      INT8 i8Sample = 0;

      #if TSS_USE_NOISE_WINDOWING
        INT16 i16WindowDeltaTemp;
        INT8 i8WindowBaseTemp;
        INT8 i8WindowDeltaLimitTemp;
      #endif

      /* Intitialization of variables */
      tss_u16CapSample = CTS_SIGNAL_OFFSET;
      u8NSamp = tss_CSSys.NSamples;    
      
      #if TSS_USE_NOISE_WINDOWING
        i8WindowBaseTemp = cts_ai8WindowBase[u8ElecNum];
        
        if (i8WindowBaseTemp == 0) {
          i8WindowDeltaLimitTemp = 127;
        } else {
          i8WindowDeltaLimitTemp = tss_cai8WindowDeltaLimit[u8ElecNum];
        }
      #endif

      /* First Turbo Determination */
      if (cts_i8Turbo[u8ElecNum] == -127) {
        u8ElectrodeStatus = CTS_GetTurbo(u8ElecNum);
        
        if ( u8ElectrodeStatus == TSS_CALIBRATION_CHANGED) {
          u8ElectrodeStatus = TSS_STATUS_OK;  /* New Turbo value was set so status is OK */
        }
        if ( u8ElectrodeStatus == TSS_ERROR_CHARGE_TIMEOUT) {
          u8NSamp = 0u;   /* Turbo value wasn't determined so don't measure and disable actual electrode */
        }
      }

      /* Main measurement loop */
      while(u8NSamp > 0) 
      {                  

        i8Sample = faCTS_SampleElectrode[u8ElecNum](cts_i8Turbo[u8ElecNum]);   /* Perform measurement */
      
        if (i8Sample > cCTS_error_LoCAP)  {         /* Capacitance Signal Ok */

          #if TSS_USE_NOISE_WINDOWING
            i16WindowDeltaTemp = (INT16) (i8Sample - i8WindowBaseTemp);
            if(i16WindowDeltaTemp > 127)
              i16WindowDeltaTemp = 127;
            else if(i16WindowDeltaTemp < -127){
              i16WindowDeltaTemp = -127;
            }            
            
            if(i16WindowDeltaTemp > i8WindowDeltaLimitTemp) {
              i8WindowBaseTemp++;
              u8FaultCnt--;
            } else if (i16WindowDeltaTemp < -i8WindowDeltaLimitTemp) {
              i8WindowBaseTemp--;
              u8FaultCnt--;                    
            } else { 
              tss_u16CapSample += i8Sample;
              u8NSamp--;
              u8FaultCnt = CTS_FAULT_TIMEOUT;
            }
          #else
            tss_u16CapSample += i8Sample;
            u8NSamp--;
            u8FaultCnt = CTS_FAULT_TIMEOUT;
          #endif  
          
          if (i8Sample > cCTS_error_HiLimit) {      /* Cap Signal Very High */
            if (u8RecalibCounter < CTS_RECALIB_LIMIT) {
              u8RecalibCounter++;
            } else {
              u8ElectrodeStatus = TSS_RECALIBRATION_REQUEST_HICAP; /* If a recalibration is needed */
            }
          }
        } else if (i8Sample == cCTS_error_LoCAP) {  /* Dynamic Recalibration in Keydetector */

          u8FaultCnt = CTS_FAULT_TIMEOUT;
          if (u8RecalibCounter < CTS_RECALIB_LIMIT) {
            u8RecalibCounter++;
          } else {
            u8NSamp = 0u;
            
            if (cts_i8Turbo[u8ElecNum] < 2)  {
              u8ElectrodeStatus = TSS_ERROR_KEYDETECT_SMALL_CAP; /* If not possible decrease of Turbo value, Disable Electrode */             
            } else {
              u8ElectrodeStatus = TSS_RECALIBRATION_REQUEST_LOCAP; /* If a recalibration is possible, request */
            }
            
          }

        } else if (i8Sample == cCTS_error_ExISR) {  /* Interrupt Occured */
          
          /* Do nothing because the measurement was corrupted by interrupt, and sample again in next loop */
          u8FaultCnt--;
          
        } else { /* if (-128) then Capacitance is extremely high, variable overflow */
          u8NSamp = 0u;
          u8ElectrodeStatus = TSS_ERROR_CHARGE_TIMEOUT; /* If a fault has occurred sets the return status code error */

        }
        
        if(u8FaultCnt == 0u) {
          u8ElectrodeStatus = TSS_ERROR_CHARGE_TIMEOUT; /* If a fault has occurred sets the return status code error */
          u8NSamp = 0u; 
        }
         
      }
      
      #if TSS_USE_NOISE_WINDOWING
        if (u8ElectrodeStatus == TSS_STATUS_OK) {
          if (cts_ai8WindowBase[u8ElecNum] == 0) {
            cts_ai8WindowBase[u8ElecNum] = (INT8) (tss_u16CapSample / tss_CSSys.NSamples);
          } else {
            cts_ai8WindowBase[u8ElecNum] = i8WindowBaseTemp;
          }
        } else {
          cts_ai8WindowBase[u8ElecNum] = 0; /* Initiate Window Base refresh */
        }
      #endif

      
      return u8ElectrodeStatus;   /* Return status code */
  }

#endif  /*TSS_USE_CTS_LOW_LEVEL*/ 
