/*******************************************************************************
*
*  FREESCALE SEMICONDUCTOR INC. 2006
*  ALL RIGHTS RESERVED
*  COPYRIGHT (c)
*  
********************************************************************************
*
*  DESCRIPTION:
*  CEA709 communication eTPU function.
*
*******************************************************************************
*
*  REVISION HISTORY:
*                                                         
*  REVISION    AUTHOR       DATE        DESCRIPTION OF CHANGE                  
*  --------    -----------  ---------   ------------------------------------                  
*  1.0         M. Princ     08/Jun/07   Initial version of file.
*******************************************************************************/
#ifndef __ETPUC_H
#include <etpuc.h>          /* Defines eTPU hardware */
#endif

#ifndef __ETPUC_COMMON_H
#include <eTPUc_common.h>   /* Standard way to use eTPU */
#endif

#define CEA709_RX_FUNCTION_NUMBER 0
#define CEA709_TX_FUNCTION_NUMBER 1
#define CEA709_TXEN_FUNCTION_NUMBER 2
#define CEA709_DMA_CONTROL_FUNCTION_NUMBER 3
#define CEA709_CD_FUNCTION_NUMBER 4

#pragma ETPU_function CEA709_RX, alternate @ CEA709_RX_FUNCTION_NUMBER;
#pragma ETPU_function CEA709_TX, alternate @ CEA709_TX_FUNCTION_NUMBER;
#pragma ETPU_function CEA709_TXEN, alternate @ CEA709_TXEN_FUNCTION_NUMBER;
#pragma ETPU_function CEA709_DMA_CONTROL, alternate @ CEA709_DMA_CONTROL_FUNCTION_NUMBER;
#pragma ETPU_function CEA709_CD, alternate @ CEA709_CD_FUNCTION_NUMBER;

/************************************************************************
*  Definitions. 
*************************************************************************/
/* CEA709 RX/TX Function Modes */
#define   CEA709_DMA_CTR_ENABLED                   (fm0==1)
#define   CEA709_TXEN_ON                           (fm1==1)
#define   CEA709_DMA_CTR_INCREMENT_COUNTER_YES     (fm0==0)

/* HSR values */
#define   CEA709_RX_HSR_INIT                           7
#define   CEA709_TX_HSR_INIT                           7
#define   CEA709_TX_HSR_RUN                            5
#define   CEA709_TXEN_HSR_INIT                         7
#define   CEA709_DMA_CONTROL_HSR_INIT_RISING_TRIGGER   7
#define   CEA709_DMA_CONTROL_HSR_INIT_FALLING_TRIGGER  5
#define   CEA709_CD_HSR_INIT                           7

/* CEA709 status parameter bits */
#define   CEA709_STATUS_RX_FRAME_END               0x000001 /* End of frame = line code violation */
#define   CEA709_STATUS_RX_ERROR                   0x000002 /* Error occurred when receiving */
#define   CEA709_STATUS_TX_PRI_BUFFER_FULL         0x000004 /* TX priority buffer contains new data to be transmitted */
#define   CEA709_STATUS_TX_NONPRI_BUFFER_FULL      0x000008 /* TX non-priority buffer contains new data to be transmitted */
#define   CEA709_STATUS_TX_TRANSMISSION_BEGINNING  0x000010 /* Transmission begins */
#define   CEA709_STATUS_TX_TRANSMISSION_SUCCESSFUL 0x000020 /* Data transmitted successfully */
#define   CEA709_STATUS_TX_ERROR                   0x000040 /* Error occurred when transmitting */
#define   CEA709_STATUS_BACKLOG_OVERFLOW           0x000080 /* Backlog overflow */
#define   CEA709_STATUS_COLLISION_DETECTED         0x000100 /* Collision detected either during the preamble or at the end of packet */

/* CEA709 state parameter bits */
#define   CEA709_STATE_ACTUAL_INPUT_BIT            0x01
#define   CEA709_STATE_PREAMBLE_PERIOD             0x02
#define   CEA709_STATE_BYTE_SYNC                   0x04
#define   CEA709_STATE_CRC_CALCULATED              0x08
#define   CEA709_STATE_FIRST_RX_BYTE               0x10
#define   CEA709_STATE_COLLISION_DETECTED          0x20
#define   CEA709_STATE_LAST_PACKET_TX              0x40
#define   CEA709_STATE_LCV                         0x80

/* CEA709 channel_access_state parameter bits */
#define   CEA709_CHANNEL_ACCESS_BUSY               0x01
#define   CEA709_CHANNEL_ACCESS_BETA1_IDLE         0x02
#define   CEA709_CHANNEL_ACCESS_PRI_WAIT_TX        0x04
#define   CEA709_CHANNEL_ACCESS_PRI_IDLE           0x08
#define   CEA709_CHANNEL_ACCESS_RANDOM_WAIT_TX     0x10
#define   CEA709_CHANNEL_ACCESS_RANDOM_IDLE        0x20
#define   CEA709_CHANNEL_ACCESS_START_TX           0x40

/* CEA709 buffers parameter bits */
#define   CEA709_BUFFERS_TX_PRI_PACKET             0x01
#define   CEA709_BUFFERS_TX_NONPRI_PACKET          0x02
#define   CEA709_BUFFERS_RX_BUFFER_ACTIVE          0x04

/* CEA709 cd_options parameter bits */
#define   CEA709_CDOPTIONS_TAIL                    0x01
#define   CEA709_CDOPTIONS_PREAMBLE                0x02

/* Transition window ratio definitions */
#define   CEA709_WINDOW_RATIO_1                    0x300000 /* 0.375 */
#define   CEA709_WINDOW_RATIO_2                    0x4F9DB2 /* 0.622 */

#define   CEA709_CRC16_POLY                        0x102100
#define   CEA709_BUFFER_SIZE                       0xF
#define   CEA709_MAX_BACKLOG                       0x3F
#define   CEA709_W_BASE                            0x10

#define   CEA709_DGRAM_INFO_OK                     0x00 /* Datagram valid */
#define   CEA709_DGRAM_INFO_CRC_ERROR              0x01 /* CRC invalid */
/*******************************************************************************
  Global variables declaration
*******************************************************************************/
int24 time_stamp_counter_high24; /* High 24 bits of the time stamp counter*/

/*******************************************************************************
* CEA709_channel_params structure declaration.
* 
*  period                  - Bit period measured during the preamble on rx channel.
*  state                   - Group of inner flags. 
*  transmit_period         - Bit period, equal to 1/(Bit Rate).
*  channel_access_state    - State of channel access algorithm.
*  status                  - Group of flags used for communication with the CPU.
*  buffers                 - Group of flags reflecting the state of rx/tx buffers.
*  last_rx_packet_length   - The length of the last received packet.
*  preamble_length         - Number of bit periods in preamble (length of preamble).
*  last_window_end         - The time of the end of the last receiver jitter tolerance window.
*  node_priority           - Priority slot used by the node when sending priority messages on the channel.
*  beta1_transmit          - Beta 1 time after transmission.
*  backlog                 - The actual backlog value.
*  beta1_receive           - Beta 1 time after reception.
*  delta_backlog           - Channel backlog increment to be generated as a result of delivering the MPDU.
*  beta2                   - Beta 2 time.
*  tmp_data_byte           - Temporary parameter for storing the rx/tx data.
*  p_tx_pri_buffer_start   - Pointer to the beginning of the priority tx buffer.
*  cd_options              - Option bits for collision detection setting.
*  p_tx_nonpri_buffer_start- Pointer to the beginning of the non-priority tx buffer.
*  dma_control_channel     - Channel number used for triggering the DMA transfers.
*  p_rx_buffer_start       - Pointer to the beginnig of the rx buffer.
*  bit_sync_threshold      - Number of sync bits.
*  cd_to_end_packet        - This parameter specifies how close to the end of packet the cd signal is checked (in number of bit periods).
*  channel_priorities      - Number of priority slots on the channel.
*  random_number           - Generated 24bit random number.
*  rx_channel              - RX channel number.
*  tmp_preamble            - Temporary parameter for counting the preamble bit periods.
*  tx_enable_channel       - TX Enable channel number.
*  tmp_pointer             - Temporary parameter for storing the pointer to the beginning of the actual tx buffer.
*  tmp_bit_count           - Temporary parameter for storing the number of bits to be transmitted/written to the buffers.
*  tmp_length              - Temporary parameter for storing the length of the packet.
*  last_rx_packet_datagram_info - The additional datagram info of the last received packet.
*  tmp_offset              - Temporary parameter for storing the actual offset within the buffer.
*  tmp_crc                 - Temporary parameter for storing the actual CRC value.
*  tmp_last_16_data_bits   - Temporary parameter for storing the last 16 bits of the actual CRC value.
*  tmp_crc_2bytes_back     - Temporary parameter for storing the CRC value calculated after receiving the last but one byte.
*  tmp_crc_1byte_back      - Temporary parameter for storing the CRC value calculated after receiving the last byte.
*  tmp_time                - Temporary parameter for storing a time value.
*  packet_cycle            - Packet cycle duration for counting down the backlog.
*  cycle_time              - Packet cycle timing only applies when the Mac Layer is Idle 
                             that is when the node is not tranmitting or receiving
                             or waiting to transmit or counting down beta1 
                             or counting down priority slots or counting down 
                             the extra 16 beta2 slots after a transmission.
*  last_time               - Time of the cycle timer last resumption.
*  priority_node_tics      - Time of nodes priority slot.
*  priority_chan_post_tx_ticks - Duration of channel priority slots time after tx packet.
*  priority_chan_post_rx_ticks - Duration of channel priority slots time after rx packet.
*  minimum_cd_time         - Minimum collision detection pulse width for evaluation of collision occurrence.
*  jitter1                 - This parameter is used for scheduling of rx jitter tolerance windows (0.375 * period).
*  jitter2                 - This parameter is used for scheduling of rx jitter tolerance windows (0.622 * period).
*  last_rx_pckt_time_stamp_high - High 24 bits of the last received packet time stamp.
*  last_rx_pckt_time_stamp_low  - Low 24 bits of the last received packet time stamp.
*******************************************************************************/
struct CEA709_channel_params
{
                int24   period;
                int8    state;
                int24   transmit_period;
                int8    channel_access_state;
                int24   status;
                int8    buffers;
                int24   last_rx_packet_length;
                int8    preamble_length;
                int24   last_window_end;
                int8    node_priority;
                int24   beta1_transmit;
                int8    backlog;
                int24   beta1_receive;
                int8    delta_backlog;
                int24   beta2;
                int8    tmp_data_byte;
                int24   p_tx_pri_buffer_start;
                int8    cd_options;
                int24   p_tx_nonpri_buffer_start;
                int8    dma_control_channel;
                int24   p_rx_buffer_start;
                int8    bit_sync_threshold;
                int24   cd_to_end_packet;
                int8    channel_priorities;
                int24   random_number;
                int8    rx_channel;
                int24   tmp_preamble;
                int8    tx_enable_channel;
                int24   tmp_pointer;
                int8    tmp_bit_count;
                int24   tmp_length;
                int8    last_rx_packet_datagram_info;
                int24   tmp_offset;
                int24   tmp_crc;
                int24   tmp_last_16_data_bits;
                int24   tmp_crc_2bytes_back;
                int24   tmp_crc_1byte_back;
                int24   tmp_time;
                int24   packet_cycle;
                int24   cycle_time;
                int24   last_time;
                int24   priority_node_tics;
                int24   priority_chan_post_tx_ticks;
                int24   priority_chan_post_rx_ticks;
                int24   minimum_cd_time;
                int24   jitter1;
                int24   jitter2;
                int24   last_rx_pckt_time_stamp_high;
                int24   last_rx_pckt_time_stamp_low;
}register_chan_base *this;

/*******************************************************************************
  The following defines are declared in order to enable usage 
  of CEA709_channel_params structure members in assembler code.
*******************************************************************************/
#define  asm_period                       prm01 
#define  asm_state                        prm00 
#define  asm_transmit_period              prm05 
#define  asm_channel_access_state         prm04 
#define  asm_status                       prm09
#define  asm_buffers                      prm08 
#define  asm_last_rx_packet_length        prm13
#define  asm_preamble_length              prm12 
#define  asm_last_window_end              prm17
#define  asm_node_priority                prm16
#define  asm_beta1_transmit               prm21 
#define  asm_backlog                      prm20
#define  asm_beta1_receive                prm25
#define  asm_delta_backlog                prm24 
#define  asm_beta2                        prm29
#define  asm_tmp_data_byte                prm28
#define  asm_p_tx_pri_buffer_start        prm33
#define  asm_cd_options                   prm32
#define  asm_p_tx_nonpri_buffer_start     prm37
#define  asm_dma_control_channel          prm36
#define  asm_p_rx_buffer_start            prm41
#define  asm_bit_sync_threshold           prm40
#define  asm_cd_to_end_packet             prm45
#define  asm_channel_priorities           prm44
#define  asm_random_number                prm49 
#define  asm_rx_channel                   prm48
#define  asm_tmp_preamble                 prm53
#define  asm_tx_enable_channel            prm52
#define  asm_tmp_pointer                  prm57
#define  asm_tmp_bit_count                prm56
#define  asm_tmp_length                   prm61
#define  asm_last_rx_packet_datagram_info prm60
#define  asm_tmp_offset                   prm65
#define  asm_tmp_crc                      prm69
#define  asm_tmp_last_16_data_bits        prm73
#define  asm_tmp_crc_2bytes_back          prm77
#define  asm_tmp_crc_1byte_back           prm81
#define  asm_tmp_time                     prm85
#define  asm_packet_cycle                 prm89
#define  asm_cycle_time                   prm93
#define  asm_last_time                    prm97
#define  asm_priority_node_tics           prm101
#define  asm_priority_chan_post_tx_ticks  prm105
#define  asm_priority_chan_post_rx_ticks  prm109
#define  asm_minimum_cd_time              prm113
#define  asm_jitter1                      prm117
#define  asm_jitter2                      prm121
#define  asm_last_rx_pckt_time_stamp_high prn125
#define  asm_last_rx_pckt_time_stamp_low  prn129

/*******************************************************************************
* NAME: CEA709_WRITE_INTO_BUFFER()
*
* DESCRIPTION: This routine is called by RX channel to write new received
*              data (one byte) to the rx buffer. The DMA request
*              is generated through the "DMA CONTROL CHANNEL" or internally 
*              by generating the DMA request when the bottom of the buffer 
*              is reached.
*******************************************************************************/
void CEA709_WRITE_INTO_BUFFER()
{
    /* based on asm_p_rx_buffer_start and tmp_offset write new data from tmp_data_byte to eTPU DATA RAM */
    #asm(  ram p31_24 = asm_tmp_data_byte.            ) /* p31_24 = tmp_data_byte */
    #asm(  alu b = p31_24; ram diob = asm_p_rx_buffer_start.) /* b = tmp_data_byte; diob = p_rx_buffer_start */
    #asm(  ram p = asm_tmp_offset.                    ) /* p = tmp_offset */
    #asm(  alu diob = diob + p.                       ) /* diob = p_rx_buffer_start + tmp_offset */
    #asm(  ram p31_0 = (diob).                        ) /* p31_0 = *diob */
    #asm(  alu a = diob & 0x03, ccs.                  ) /* a = (p_rx_buffer_start + tmp_offset) & 0x03 */ 
    #asm(  alu_if Z == 1 then p31_24 = b.             ) /* if ((p_rx_buffer_start + tmp_offset) & 0x03)==0 then p31_24 = tmp_data_byte */
    #asm(  alu nil = a ^ 0x01, ccs.                   ) 
    #asm(  alu_if Z == 1 then p_high = b.             ) /* if (((p_rx_buffer_start + tmp_offset) & 0x03) ^ 0x01)==0 then p_high = tmp_data_byte */
    #asm(  alu nil = a ^ 0x02, ccs.                   )
    #asm(  alu_if Z == 1 then p_mid = b.              ) /* if (((p_rx_buffer_start + tmp_offset) & 0x03) ^ 0x02)==0 then p_mid = tmp_data_byte */
    #asm(  alu nil = a ^ 0x03, ccs.                   )
    #asm(  alu_if Z == 1 then p_low = b.              ) /* if (((p_rx_buffer_start + tmp_offset) & 0x03) ^ 0x03)==0 then p_low = tmp_data_byte */
    #asm(  ram (diob) = p31_0.                        ) /* *diob = p31_0 */
    
    this->tmp_offset ++;

    /* when the bottom of the rx buffer reached */
    if(this->tmp_offset == (CEA709_BUFFER_SIZE + 1))
    {
        this->tmp_offset = 0;
        if(CEA709_DMA_CTR_ENABLED)
        {
            /* Switch to DMA CONTROL CHANNEL to change the output pin and thus to trigger the DMA transfer */
            SwitchToChannel(this->dma_control_channel);
            SetPinPerPacB();
            ClearMatchBLatch();
            ertb = tcr1 + (this->period)>>3;
            WriteErtBToMatchBAndEnable();
            SwitchToChannel(this->rx_channel);
        }
        else
        {
            /* Generate DMA request. */
           SetDataTransferInterruptRequest();         
        }
    }
}

/*******************************************************************************
* NAME: CEA709_INCREMENT_BACKLOG()
*
* DESCRIPTION: This routine increments backlog and handles backlog overflow.
*******************************************************************************/
void CEA709_INCREMENT_BACKLOG()
{
    this->backlog += this->delta_backlog;
    if(this->backlog > CEA709_MAX_BACKLOG)
    {
        this->backlog = CEA709_MAX_BACKLOG;
        this->status |= CEA709_STATUS_BACKLOG_OVERFLOW;
        SetChannelInterrupt();
    }
}

/*******************************************************************************
* NAME: CEA709_DECREMENT_BACKLOG()
*
* DESCRIPTION: This routine decrements backlog and handles backlog underflow.
*******************************************************************************/
void CEA709_DECREMENT_BACKLOG()
{
    if(this->backlog <= 1)
        this->backlog = 0;
    else
        this->backlog -= 1;
}

/*******************************************************************************
* NAME: CEA709_TX
*
* DESCRIPTION: Implementation of CEA709 MAC layer on eTPU
* 
* FUNCTION PARAMETERS:
* 
*  this - CEA709_channel_params structure.
*                          
* NOTES: !!!!! TX and RX channels must have the same base address. !!!!!!
*******************************************************************************/
void CEA709_TX( struct CEA709_channel_params this )
{
/*******************************************************************************
* THREAD NAME: TX INIT
* DESCRIPTION: Initialize a CEA709 TX channel.
*              1. Set pin low.
*              2. Enable output buffer.
*              3. Configure channel to use TCR1 clock.
*              4. Configure input pin actions to no_detect.
*              5. Select EitherMatch-NonBlocking-SingleTransition channel mode.
*              6. Set backlog to 1.
*              7. Set time of nodes priority slot.
*              8. Set time of channel priority slots.
*              9. Clear high 24 bits of the time stamp counter.
*******************************************************************************/
   if (hsr==CEA709_TX_HSR_INIT)
   {
       SetPinLow();
       EnableOutputBuffer();
       ActionUnitA( MatchTCR1, CaptureTCR1, GreaterEqual);
       ActionUnitB( MatchTCR1, CaptureTCR1, GreaterEqual);
       OnTransA(NoDetect);
       OnTransB(NoDetect);
       EitherMatchNonBlockingSingleTransition();
       this.random_number = tcr1;
       this.backlog = 1;
       this.priority_node_tics = (this.node_priority -1) * this.beta2;
       if(this.channel_priorities > 0)
       {
           this.priority_chan_post_tx_ticks = (this.channel_priorities + CEA709_W_BASE) * this.beta2;
           this.priority_chan_post_rx_ticks = (this.channel_priorities * this.beta2);
       }
       else
       {
           this.priority_chan_post_tx_ticks = 0;
           this.priority_chan_post_rx_ticks = 0;
       }
       this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_IDLE;
       time_stamp_counter_high24 = 0;
   }

/*******************************************************************************
* THREAD NAME: TX TRANSMIT NEW PACKET REQUEST.
* DESCRIPTION: Each time the CPU has a new data to transmit, it writes the data
*              to the applicable tx buffer, set applicable status flags and 
*              raises this host service request.
*              1. If channel_access_state is RANDOM_IDLE get ready for 
*                 transmitting,
*                 otherwise do nothing.
*******************************************************************************/
   else if ( hsr==CEA709_TX_HSR_RUN )
   {
       if(this.channel_access_state & CEA709_CHANNEL_ACCESS_RANDOM_IDLE)
       {
           this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_WAIT_TX;
           this.cycle_time += tcr1 - this.last_time;
           /* Generate random number */
           this.random_number = (this.random_number * 0x41A7) % 0x7FFFFF;
           /* Schedule the end of W_BASE randomizing slot in order to decrement backlog */
           erta = tcr1 + this.beta2*CEA709_W_BASE;
           /* Schedule the end of random delay (integer number of randomizing slots of duration Beta2 ) */
           ertb = tcr1 + this.beta2 * (this.random_number % ((1+this.backlog)*CEA709_W_BASE));
           EitherMatchNonBlockingSingleTransition();
           flag0 = 1;
           flag1 = 0;
           ClearAllLatches();
           EnableEventHandling();
           WriteErtAToMatchAAndEnable();
           WriteErtBToMatchBAndEnable();
       }
   }
/*******************************************************************************
* THREAD NAME: TX PREAMBLE.
* DESCRIPTION: Generate Manchester encoded "1" until tmp_preamble parameter 
*              is zero. Consecutively, generate one Manchester encoded "0" to
*              determine the end of preamble period. If enabled, check whether 
*              the collision occurred.
*              1. If state is BYTE_SYNC schedule Manchester encoded "0" and 
*                 check collision occurrence,
*                 otherwise schedule the next Manchester encoded "1" until
*                 tmp_preamble parameter is zero.
*              2. Schedule time of next matchA and matchB.
*              3. Clear latches.
*              4. Write the match registers and enable the match recognition.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==0) && (flag1==0) )
   {
       if(this.state & CEA709_STATE_BYTE_SYNC)
       {
           flag1=1;
           this.state &= ~CEA709_STATE_BYTE_SYNC;
           /* Toggle pin on matchA to generate Manchester encoded "0" */
           OnMatchA(PinToggle);
           if((this.cd_options & CEA709_CDOPTIONS_PREAMBLE) &&
              (this.state & CEA709_STATE_COLLISION_DETECTED) )
           {
               this.state &= ~CEA709_STATE_COLLISION_DETECTED;
               this.status |= CEA709_STATUS_COLLISION_DETECTED;
               /* Generate Interrupt. */
               SetChannelInterrupt();                     
               
               /* Update backlog */
               this.delta_backlog = 1;
               CEA709_INCREMENT_BACKLOG();
               goto CEA709_L0;
           }
       }
       else
       {
           /* Do not change the polarity of pin on matchA to generate Manchester encoded "1" */
           OnMatchA(NoChange);
           if(--this.tmp_preamble == 0)
               this.state |= CEA709_STATE_BYTE_SYNC;
       }
       /* Schedule time of next matchA and matchB. */
       erta += this.transmit_period;
       ertb += this.transmit_period;
       
       ClearAllLatches();
       WriteErtAToMatchAAndEnable();
       WriteErtBToMatchBAndEnable();
   }
/*******************************************************************************
* THREAD NAME: TX DATA.
* DESCRIPTION: Transmit the whole packet including CRC bits which are calculated
*              step by step each bit period.
*              1. If the line code violation period expires generate the 
*                 interrupt to inform the CPU that the message has been 
*                 transmitted completely, set last_packet_tx bit in status 
*                 parameter, update backlog, set pin low on TX_ENABLE pin (if 
*                 enabled), select SingleMatch-SingleTransition channel mode.
*              2. If all data transmitted (state is CRC_CALCULATED) get the msb 
*                 of CRC,
*                 otherwise make a next step in CRC calculation and get the msb
*                 of tmp_data_byte.
*              3. Based on the msb in tmp_data_byte or in CRC transmit 
*                 Manchester encoded "0" or "1".
*              4. If the last bit within the tmp_data_byte transmitted then 
*                 a. if CRC bytes were transmitted
*                      - update status to indicate that the CPU can load new 
*                        data to the applicable tx buffer
*                      - set LCV bit in state parameter
*                 b. if all data bytes were transmitted
*                      - tmp_bit_count = 16
*                      - set CRC_CALCULATED bit in state parameter
*                 else
*                      - based on tmp_pointer and tmp_offset read new data 
*                        to tmp_data_byte to be transmitted in the next cycle
*                      - if the bottom of the applicable tx buffer reached then 
*                        set tmp_offset to -1 and generate DMA request, either 
*                        directly or through the DMA CONTROL CHANNEL (change the
*                        output pin and thus to trigger the DMA transfer)
*              5. Schedule time of next matchA and matchB.
*              6. Clear latches.
*              7. Write the match registers and enable the match recognition.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==0) && (flag1==1) )
   {
       if(this.state & CEA709_STATE_LCV)
       {
           this.state &= ~CEA709_STATE_LCV;
           this.status |= CEA709_STATUS_TX_TRANSMISSION_SUCCESSFUL;
           /* Generate the interrupt to inform the CPU that the message has been transmitted completely. */
           SetChannelInterrupt();                     
           this.cycle_time = 0;

           /* Set last_packet_tx bit in status parameter to indicate that the last packet was tx */
           this.state |= CEA709_STATE_LAST_PACKET_TX;
           
           /* Update backlog */
           if(this.delta_backlog <= 0)
               CEA709_DECREMENT_BACKLOG();
           else
               CEA709_INCREMENT_BACKLOG();

CEA709_L0:
           /* Set pin low on TX_ENABLE pin (CP2), if enabled */
           if(CEA709_TXEN_ON)
           {
               SwitchToChannel(this.tx_enable_channel);
               SetPinLow();
               SwitchToChannel(this.rx_channel + 1);
           }

           /* If CD at the end of the packet enabled then schedule the time of CD checking, 
              else schedule the time of the beta1 period end */
           if(this.cd_options & CEA709_CDOPTIONS_TAIL)
           {
               /* erta = ertb + cd_to_end_packet */
               erta = ertb + this.cd_to_end_packet;
           }
           else
           {
               /* erta = ertb + beta1 */
               erta = ertb + this.beta1_transmit; 
           }
           /* Select SingleMatch-SingleTransition channel mode */
           SingleMatchSingleTransition();
           /* Save the time of the end of beta1 period */
           this.tmp_time = ertb + this.beta1_transmit;
           this.channel_access_state = CEA709_CHANNEL_ACCESS_BETA1_IDLE;
           OnMatchA(PinLow);
           OnMatchB(PinLow);
           flag0 = 1;
           goto CEA709_L1;
       }
       /* when CRC bits are transmitted */
       if(this.state & CEA709_STATE_CRC_CALCULATED)
       {
           this.tmp_crc <<=1;
       }
       /* when LPDUs are transmitted */
       else
       {
           /* calculate crc */
           #asm(  ram p31_24 = asm_tmp_data_byte.                  ) /* p31_24 = tmp_data_byte */
           #asm(  alu a = p31_24; ram p23_0 = asm_tmp_crc.         ) /* a = tmp_data_byte; p = tmp_crc */
           #asm(  alu a =<<16 a.                                   ) /* a = tmp_data_byte<<16 */
           #asm(  alu nil = p ^ a, ccs.                            ) /* tmp_crc ^ (tmp_data_byte<<16) */
           #asm(  alu p =<< p + #0x00.                             ) /* p = tmp_crc<<1 */
           #asm(  alu sr = CEA709_CRC16_POLY.                     ) /* sr = CEA709_CRC16_POLY */
           #asm(/*alu_if n == 1 then p = p ^ sr.*/  %hex 3D587F71. ) /* if ((msb(tmp_crc))^(msb(tmp_data_byte<<16)))==1 then p = (tmp_crc<<1) ^ CEA709_CRC16_POLY */
           #asm(  ram p -> asm_tmp_crc.                            ) /* tmp_crc = p */
           #asm(  alu p31_24 = p31_24 <<1, ccs.                    ) 
           #asm(  ram p31_24 -> asm_tmp_data_byte.                 ) /* tmp_data_byte <<=1 */
       }

       /* based on the msb in tmp_data_byte or in tmp_crc transmit Manchester encoded "0" or "1" */
       if(CC.C != 0) 
       {
           /* Do not change the polarity of pin on matchA to generate Manchester encoded "1" */
           OnMatchA(NoChange)
       }
       else 
       {
           /* Toggle pin on matchA to generate Manchester encoded "0" */
           OnMatchA(PinToggle);
       }
       
       /* when the last bit within the tmp_data_byte transmitted */
       if(--this.tmp_bit_count == 0)
       {
           /* when all CRC bytes transmitted */
           if(this.state & CEA709_STATE_CRC_CALCULATED)
           {
               /* Update status to indicate that the CPU can load new data to the applicable tx buffer */
               if(this.buffers & CEA709_BUFFERS_TX_NONPRI_PACKET)
               {
                   this.status  &= ~CEA709_STATUS_TX_NONPRI_BUFFER_FULL;
                   this.buffers &= ~CEA709_BUFFERS_TX_NONPRI_PACKET;
               }
               else if(this.buffers & CEA709_BUFFERS_TX_PRI_PACKET)
               {
                   this.status  &= ~CEA709_STATUS_TX_PRI_BUFFER_FULL;
                   this.buffers &= ~CEA709_BUFFERS_TX_PRI_PACKET;
               }
               this.state &= ~CEA709_STATE_CRC_CALCULATED;
               this.state |= CEA709_STATE_LCV;
               OnMatchB(PinLow);
               /* Schedule time of next matchA and matchB. */
               erta += this.transmit_period;
               ertb += (4*this.transmit_period);
               goto CEA709_L1;
           }
           /* when all data bytes transmitted */
           if(--this.tmp_length == 0)
           {
               /* transmit crc bytes */
               this.tmp_crc = this.tmp_crc ^ 0xffff00;
               this.tmp_bit_count = 16;
               this.state |= CEA709_STATE_CRC_CALCULATED;
           }
           else
           {
               this.tmp_offset ++;
               this.tmp_bit_count = 8;

               /* based on tmp_pointer and tmp_offset read new data to tmp_data_byte */
               #asm(  ram diob = asm_tmp_pointer.        )
               #asm(  ram p = asm_tmp_offset.            )
               #asm(  alu diob = diob + p.               ) 
               #asm(  ram p31_0 = (diob).                )
               #asm(  alu a = diob & 0x03.               ) 
               #asm(  alu nil = a ^ 0x01, ccs.           ) 
               #asm(  alu_if Z == 1 then p31_24 = p_high.) 
               #asm(  alu nil = a ^ 0x02, ccs.           ) 
               #asm(  alu_if Z == 1 then p31_24 = p_mid. ) 
               #asm(  alu nil = a ^ 0x03, ccs.           ) 
               #asm(  alu_if Z == 1 then p31_24 = p_low. ) 
               #asm(  ram p31_24 -> asm_tmp_data_byte.   )
        
               /* when the bottom of the priority or non-priority buffer reached */
               if(this.tmp_offset == CEA709_BUFFER_SIZE)
               {
                   this.tmp_offset = -1;
                   if(CEA709_DMA_CTR_ENABLED)
                   {
                       /* Switch to DMA CONTROL CHANNEL to change the output pin and thus to trigger the DMA transfer */
                       SwitchToChannel(this.dma_control_channel);
                       SetPinPerPacB();
                       ClearMatchBLatch();
                       ertb = tcr1 + (this.transmit_period)>>3;
                       WriteErtBToMatchBAndEnable();
                       SwitchToChannel(this.rx_channel+1);
                   }
                   else
                   {
                       /* Generate DMA request. */
                       SetDataTransferInterruptRequest();         
                   }
               }
           }
       }
       /* Schedule time of next matchA and matchB. */
       erta += this.transmit_period;
       ertb += this.transmit_period;
       
CEA709_L1:
       ClearAllLatches();
       WriteErtAToMatchAAndEnable();
       WriteErtBToMatchBAndEnable();
   }
/*******************************************************************************
* THREAD NAME: TX BETA1 EXPIRED.
* DESCRIPTION: If priority access conditions satisfied set the 
*              channel_access_state to PRI_WAIT_TX and calculate the time 
*              of the next matchB,
*              otherwise set the channel_access_state to PRI_IDLE and based on
*              whether the last packet was tx or rx calculate the time of the
*              next matchB.
*              1. Clear latches.
*              2. Enable transitions detection on rx channel.
*              3. If collision detection checking scheduled then handle CD.
*              4. Select EitherMatch-NonBlocking-SingleTransition channel mode.
*              5. If priority access conditions satisfied set the 
*                 channel_access_state to PRI_WAIT_TX and calculate the time 
*                 of the next matchB,
*                 otherwise set the channel_access_state to PRI_IDLE and based
*                 on whether the last packet was tx or rx calculate the time
*                 of the next matchB.
*              6. Write the matchB register and enable the matchB recognition.
*              7. Configure output pin actions to no_change.
*******************************************************************************/
   else if( IsMatchAOrTransitionBEvent() && (flag0==1) && (flag1==1) )
   {
       ClearAllLatches();
       
       /* Enable transitions detection on rx channel */
       chan--;
       ClearAllLatches();
       EnableEventHandling();
       chan++;
       
       if(this.channel_access_state & CEA709_CHANNEL_ACCESS_BETA1_IDLE)
       {
           /* If collision detection checking scheduled */
           if(this.tmp_time != erta)
           {
               /* Handle collision detection */
               if(this.cd_options & CEA709_CDOPTIONS_TAIL)
               {
                   if(this.state & CEA709_STATE_COLLISION_DETECTED)
                   {
                       this.state &= ~CEA709_STATE_COLLISION_DETECTED;
                       this.status |= CEA709_STATUS_COLLISION_DETECTED;
                       /* Generate Interrupt. */
                       SetChannelInterrupt();                     
                      
                       /* Update backlog */
                       this.delta_backlog = 1;
                       CEA709_INCREMENT_BACKLOG();
                   }
                  
                   erta = this.tmp_time;
                   WriteErtAToMatchAAndEnable();
                   goto CEA709_L2;
               }
           }
           /* Select EitherMatch-NonBlocking-SingleTransition channel mode */
           EitherMatchNonBlockingSingleTransition();
           
           /* If the priority packet is waiting to be transmitted and the node has a priority slot */
           if((this.status & CEA709_STATUS_TX_PRI_BUFFER_FULL) && (this.node_priority > 0))
           {
               this.channel_access_state = CEA709_CHANNEL_ACCESS_PRI_WAIT_TX;
               ertb = erta + this.priority_node_tics;
               flag1 = 0;
           }
           else
           {
               this.channel_access_state = CEA709_CHANNEL_ACCESS_PRI_IDLE;
               if(this.state & CEA709_STATE_LAST_PACKET_TX)
               {
                   /* last packet was TX */
                   ertb = erta + this.priority_chan_post_tx_ticks;
               }
               else
               {
                   /* last packet was RX */
                   ertb = erta + this.priority_chan_post_rx_ticks;
               }
           }
           WriteErtBToMatchBAndEnable();
CEA709_L2:
           OnMatchA(NoChange);
           OnMatchB(NoChange);
       }
   }
       
/*******************************************************************************
* THREAD NAME: TX PRIORITY_IDLE 
* DESCRIPTION: If tx priority or non-priority buffer is full then set the 
*              channel_access_state to RANDOM_WAIT_TX and get ready for 
*              transmitting,
*              otherwise set the channel_access_state to RANDOM_IDLE.
*              1. If tx priority or non-priority buffer is full then set the 
*                 channel_access_state to RANDOM_WAIT_TX and get ready for 
*                 transmitting,
*                 otherwise set the channel_access_state to RANDOM_IDLE.
*              2. Clear flag1.
*              3. Configure output pin actions to no_change.
*              4. Clear latches.
*              5. Write the match registers and enable the match recognition.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==1) && (flag1==1) )
   {
       if(this.channel_access_state & CEA709_CHANNEL_ACCESS_PRI_IDLE)
       {
           if((this.status & CEA709_STATUS_TX_PRI_BUFFER_FULL)||
              (this.status & CEA709_STATUS_TX_NONPRI_BUFFER_FULL))
           {
               this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_WAIT_TX;
               /* Generate random number */
               this.random_number = (this.random_number * 0x41A7) % 0x7FFFFF;
               /* Schedule the end of W_BASE randomizing slot in order to decrement backlog */
               erta = ertb + this.beta2*CEA709_W_BASE;
               /* Schedule the end of random delay (integer number of randomizing slots of duration Beta2) */
               ertb += this.beta2 * (this.random_number % ((1+this.backlog)*CEA709_W_BASE));
           }
           else
           {
               this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_IDLE;
               erta = ertb + this.packet_cycle - this.cycle_time;
               this.last_time = tcr1;
               SingleMatchSingleTransition();
           }
           flag1 = 0;
           OnMatchA(NoChange);
           OnMatchB(NoChange);
           WriteErtAToMatchAAndEnable();
           WriteErtBToMatchBAndEnable();
       }
       ClearAllLatches();
   }
/*******************************************************************************
* THREAD NAME: TX RANDOM WAIT / RANDOM IDLE
* DESCRIPTION: The actual channel_access_state is either RANDOM_WAIT_TX and the 
*              end of W_BASE randomizing slot reached or the actual 
*              channel_access_state is RANDOM_IDLE and the packet cycle time 
*              expires.
*              1. Clear matchA latch.
*              2. If channel_access_state is RANDOM_WAIT_TX then schedule the 
*                 end of W_BASE randomizing slot in order to decrement backlog
*                 and update backlog.
*              3. If channel_access_state is RANDOM_IDLE then schedule the time 
*                 of the next packet cycle and update backlog.
*              4. Write the match registers and enable the match recognition.
*******************************************************************************/
   else if( IsMatchAOrTransitionBEvent() && (flag0==1) && (flag1==0) )
   {
       ClearMatchALatch();
       if(this.channel_access_state & CEA709_CHANNEL_ACCESS_RANDOM_WAIT_TX)
       {
           /* Schedule the end of W_BASE randomizing slot in order to decrement backlog */
           erta += this.beta2*CEA709_W_BASE;
           /* Update backlog */
           CEA709_DECREMENT_BACKLOG();
           WriteErtAToMatchAAndEnable(); //this instruction must be placed here!
       }
       else if(this.channel_access_state & CEA709_CHANNEL_ACCESS_RANDOM_IDLE)
       {
           /* Schedule the time of te next packet cycle */
           erta += this.packet_cycle;
           this.last_time = tcr1;
           this.cycle_time = 0;
           /* Update backlog */
           CEA709_DECREMENT_BACKLOG();
           WriteErtAToMatchAAndEnable();
       }
   }
/*******************************************************************************
* THREAD NAME: TX TRANSMIT START
* DESCRIPTION: The actual channel_access_state is either PRIORITY_WAIT_TX or 
*              RANDOM_WAIT_TX and the transmit timer expires (TX is allowed to 
*              transmit).
*              1. Clear matchB latch.
*              2. If channel_access_state is PRIORITY_WAIT_TX or RANDOM_WAIT_TX
*                 then
*                   if one of tx buffers is full then
*                     - set the tmp_pointer
*                     - set pin high on TX_ENABLE pin (CP2), if enabled
*                     - generate the interrupt to the CPU to configure the DMA 
*                       controller for sending direction
*                     - disable transitions detection on rx channel
*                     - read length of next LPDU (the first 2 bytes) + one data 
*                       byte (the third byte)
*                     - set pin low and keep it low for one transmit_period
*                     - clear latches
*                     - select Match2-Ordered-Double-Transition channel mode
*                     - clear flags
*                     - write the match registers and enable the match 
*                       recognition
*                   else (transmit packet not ready, buffers are empty)
*                     - channel_access_state = RANDOM_IDLE
*                     - select Single Match Single Transition channel mode
*                 else
*                   - clear latches
*                   - write the match registers and enable the match recognition
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==1) && (flag1==0) )
   {
       ClearMatchBEvent();
       if((this.channel_access_state = CEA709_CHANNEL_ACCESS_PRI_WAIT_TX)||
          (this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_WAIT_TX))
       {
           /* Check which of tx buffers is full and based on this set the tmp_pointer. */
           if((this.status & CEA709_STATUS_TX_PRI_BUFFER_FULL)||
              (this.status & CEA709_STATUS_TX_NONPRI_BUFFER_FULL))
           {
               if(this.status & CEA709_STATUS_TX_PRI_BUFFER_FULL)
               {
                   /* this.tmp_pointer = this.p_tx_pri_buffer_start; */
                   #asm(  ram p <- asm_p_tx_pri_buffer_start. )
                   #asm(  ram p -> asm_tmp_pointer. )
                   this.buffers |= CEA709_BUFFERS_TX_PRI_PACKET;
               }
               else if(this.status & CEA709_STATUS_TX_NONPRI_BUFFER_FULL)
               {
                   /* this.tmp_pointer = this.p_tx_nonpri_buffer_start; */
                   #asm(  ram p <- asm_p_tx_nonpri_buffer_start. )
                   #asm(  ram p -> asm_tmp_pointer. )
                   this.buffers |= CEA709_BUFFERS_TX_NONPRI_PACKET;
               }
               this.channel_access_state = CEA709_CHANNEL_ACCESS_BUSY;
               
               /* Set pin high on TX_ENABLE pin (CP2), if enabled */
               if(CEA709_TXEN_ON)
               {
                   SwitchToChannel(this.tx_enable_channel);
                   SetPinHigh();
                   SwitchToChannel(this.rx_channel + 1);
               }
               
               this.status |= CEA709_STATUS_TX_TRANSMISSION_BEGINNING;
               this.state &= ~CEA709_STATE_COLLISION_DETECTED;
               /* Generate the interrupt to the CPU to configure the DMA controller for sending direction */
               SetChannelInterrupt();
               
               /* Disable transitions detection on rx channel */
               chan--;
               DisableEventHandling();
               ClearAllLatches();
               chan++;

               /* read length of next LPDU (the first 2 bytes) + one data byte (the third byte) */
               #asm(  ram diob = asm_tmp_pointer.        )
               #asm(  ram p31_0 = (diob).                )
               #asm(  alu a = p31_16.                    )
               #asm(  alu p31_24 = p_mid.                )
               #asm(  ram p31_24 -> asm_tmp_data_byte.   )
               #asm(  alu p31_24 = p31_24 & 0x3F.        )
               #asm(  ram p31_24 -> asm_delta_backlog.   )
               #asm(  alu p = a; ram p -> asm_tmp_length.)
   
               this.tmp_offset = 2;
               this.tmp_bit_count = 8;
               this.tmp_crc = 0xFFFF00;
               
               /* Set pin low and keep it low for one transmit_period */
               SetPinLow();
               erta = tcr1 + (this.transmit_period>>1);
               ertb = tcr1 + this.transmit_period;
               OnMatchA(NoChange);
               OnMatchB(PinToggle);
               ClearAllLatches();
               
               /* Select Match2-Ordered-Double-Transition channel mode. */
               Match2OrderedDoubleTransition();
               
               flag0=0;
               this.tmp_preamble = this.preamble_length - 1;
               
               flag1=0;
               EnableEventHandling();
               WriteErtAToMatchAAndEnable();
               WriteErtBToMatchBAndEnable();
           }
           else
           {
               this.channel_access_state = CEA709_CHANNEL_ACCESS_RANDOM_IDLE;
               erta = ertb + this.packet_cycle - this.cycle_time;
               this.last_time = tcr1;
               SingleMatchSingleTransition();
           }
       }
       ClearAllLatches();
       WriteErtAToMatchAAndEnable();
       WriteErtBToMatchBAndEnable();
   }
/*******************************************************************************
* THREAD NAME: UNHANDLED_EVENTS
*******************************************************************************/
   else 
   {
#ifdef GLOBAL_ERROR_FUNC
		Global_Error_Func();
#else
		ClearAllLatches();
#endif
   }  
}
/************************************************************************
* NAME: CEA709_RX
*
* DESCRIPTION: Implementation of CEA709 MAC layer on eTPU
* 
* FUNCTION PARAMETERS:
*                          
*  this - CEA709_channel_params structure.
*  
* CHANNEL FLAG USAGE: 
*   Flag0 is used for indication of the position within the bit period:
*     0 = halfperiod (the first transition in the bit period)
*     1 = endperiod (the second transition in the bit period)
*   Flag1 usage:
*     0 = the first BitSync period detected
*     1 = wait for the first synchronization transitions
*
* NOTES: !!!!! TX and RX channels must have the same base address. !!!!!!
*************************************************************************/
void CEA709_RX( struct CEA709_channel_params this )
{
/*******************************************************************************
* THREAD NAME: RX INIT.
* DESCRIPTION: Initialize a CEA709 RX channel.
*              1. Disable matches in thread in case there are any previous
*                 pending matches. This could happen if the channel was
*                 running another function before this one.
*              2. Disable output buffer.
*              3. Configure channel to use TCR1 clock.
*              4. Select SingleMatch-DoubleTransition channel mode.
*              5. Disable detection of matches.
*              6. Enable match event handling.
*              7. Clear flag 0.
*              8. Set flag 1.
*              9. Configure pin actions to detect rising edges on Action Unit A.
*              10.Configure pin actions to detect falling edges on Action Unit B.
*              11.Clear latches.
*******************************************************************************/
   if (hsr==CEA709_RX_HSR_INIT)
   {
       DisableMatchesInThread();
       DisableOutputBuffer();

       ActionUnitA( MatchTCR1, CaptureTCR1, GreaterEqual);
       ActionUnitB( MatchTCR1, CaptureTCR1, GreaterEqual);
       SingleMatchDoubleTransition();
       DisableMatchDetection();
       EnableEventHandling();
       flag0=0;
       flag1=1;
       OnTransA(LowHigh);
       OnTransB(HighLow);
       ClearAllLatches();
   }
/*******************************************************************************
* THREAD NAME: RX SYNCHRONIZATION.
* DESCRIPTION: Detect the first BitSync transitions to calculate the length
*              of the bit period. Setup the window for the next halfperiod 
*              transition.
*              1. If channel access state is RANDOM_IDLE then update cycle_time.
*              2. Set channel access state to BUSY.
*              3. Calculate bit period from two consecutively detected 
*                 transitions.
*              4. Select Match2-SingleTransition channel mode.
*              5. Calculate the time of the next jitter tolerance window 
*                 beginning.
*              6. Calculate the time of the next jitter tolerance window end.
*              7. Configure pin actions to detect any edges on Action Units A,B.
*              8. Clear flag0 and flag1.
*              9. Initialize necessary temp parameters.
*              10.Clear latches.
*              11.Write the match registers and enable the match recognition.
*******************************************************************************/
   else if( IsMatchAOrTransitionBEvent() && (flag1==1) )
   {
       if(this.channel_access_state & CEA709_CHANNEL_ACCESS_RANDOM_IDLE)
       {
           this.cycle_time += tcr1 - this.last_time;
       }
       this.channel_access_state = CEA709_CHANNEL_ACCESS_BUSY;
       this.period = ertb - erta;
       Match2SingleTransition();
       
       this.last_window_end = ertb;
       
       /* erta = ertb + (CEA709_WINDOW_RATIO_1 * period); */
       #asm(  ram p23_0 = asm_period.                 )
       #asm(  alu a = CEA709_WINDOW_RATIO_1.         )
       #asm(  mdu a mults p ,ccs.                     )
       #asm(  alu sr = CEA709_WINDOW_RATIO_2.        )
       #asm(  nop.                                    )
       #asm(  nop.                                    )
       #asm(  alu a =<< mach + #0.                    )
       #asm(  alu erta = ertb + a ,ccs.               )

       /* ertb += (CEA709_WINDOW_RATIO_2 * period); */
       #asm(  mdu p mults sr ,ccs.                    )
CEA709_L3:
       #asm(  if mb == 1 then goto CEA709_L3, flush. )
       #asm(  alu p =<< mach + #0.                    )
       #asm(  alu ertb = ertb + p ,ccs.               )
       
       /* jitter2 = CEA709_WINDOW_RATIO_2 * period */
       #asm(  ram p -> asm_jitter2.                   )
       /* jitter1 = CEA709_WINDOW_RATIO_1 * period */
       #asm(  alu p = a; ram p -> asm_jitter1.        )
       
       OnTransA(AnyTrans);
       OnTransB(AnyTrans);

       flag0=0;
       flag1=0;
       this.tmp_data_byte = 0;
       this.tmp_bit_count = 8;
       this.tmp_offset = 0;
       this.tmp_length = 0;
       this.tmp_crc = 0xFFFF00;
       this.tmp_preamble = 1;

       this.buffers |= CEA709_BUFFERS_RX_BUFFER_ACTIVE;

       this.state |= CEA709_STATE_PREAMBLE_PERIOD;
       ClearAllLatches();
       WriteErtAToMatchAAndEnable();
       WriteErtBToMatchBAndEnable();
   }
/*******************************************************************************
* THREAD NAME: RX HALFPERIOD - A.
* DESCRIPTION: Detect the first transitions within the bit period.
*              1. Check whether the input pin has not changed since 
*                 the transition time. If it has not changed then continue 
*                 in part b,
*                 otherwise clear transition latch and thus enable next 
*                 transition detection.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==0) && (flag1==0) && (pin==0))
   {
       if(IsLatchedTransitionA() && (CurrentInputPin==1))
       {
           /* noise transition came => enable next transition detection */
           Clear(TransLatch);
       }
       else goto CEA709_L4;
   }
/*******************************************************************************
* THREAD NAME: RX HALFPERIOD - B.
* DESCRIPTION: Detect the first transitions within the bit period.
*              1. Check whether the input pin has not changed since 
*                 the transition time. If it has not changed then continue, 
*                 otherwise clear transition latch and thus enable next 
*                 transition detection.
*              2. If the transition came, Manchester encoded "0" was detected.
*              3. If the matchB latched, Manchester encoded "1" was detected.
*              4. Calculate the time of the next jitter tolerance window 
*                 beginning and end.
*              5. Based on current pin state configure pin actions to detect 
*                 either rising or falling edge on Action Units A.
*              6. Clear latches.
*              7. Write the match registers and enable the match recognition.
*              8. Set flag0.
*              9. If the data is received (preamble period passed), store the
*                 new received bit and update the CRC.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==0) && (flag1==0) && (pin==1))
   {
       if(IsLatchedTransitionA() && (CurrentInputPin==0))
       {
           /* noise transition came => enable next transition detection */
           Clear(TransLatch);
           return;
       }
CEA709_L4:
       if(IsLatchedTransitionA())
       {
           /* correct transition came => clear actual input bit in state parameter */
           this.state &= ~CEA709_STATE_ACTUAL_INPUT_BIT;
       }
       else if(IsLatchedMatchB())
       {
           /* transition did not come => set actual input bit in state parameter */
           this.state |= CEA709_STATE_ACTUAL_INPUT_BIT;
       }
       /* Calculate the the time of the next jitter tolerance window beginning and end */
       erta = this.last_window_end + (this.period>>1) + this.jitter1;
       ertb = this.last_window_end + (this.period>>1) + this.jitter2;

       /* Based on current pin state configure pin actions to detect either rising or falling edge on Action Units A */
       if(CurrentInputPin==1)
       {
           OnTransA(HighLow);
       }
       else
       {
           OnTransA(LowHigh);
       }
       
       ClearAllLatches();
       WriteErtAToMatchAAndEnable();
       WriteErtBToMatchBAndEnable();
       
       flag0=1;
       if((this.state & CEA709_STATE_PREAMBLE_PERIOD)==0)
       {
           this.tmp_data_byte  <<= 1;
           this.tmp_data_byte += (this.state & CEA709_STATE_ACTUAL_INPUT_BIT);
           this.tmp_last_16_data_bits <<= 1;
           this.tmp_last_16_data_bits += (this.state & CEA709_STATE_ACTUAL_INPUT_BIT);

           /* calculate crc */
           #asm(  ram p31_24 = asm_tmp_data_byte.                  ) /* p31_24 = tmp_data_byte */
           #asm(  alu p_high =<<4 p31_24.                          ) /* p_high = tmp_data_byte<<4 */
           #asm(  alu p_high =<<2 p_high.                          ) /* p_high = tmp_data_byte<<6 */
           #asm(  alu p_high =<< p_high + #0x00.                   ) /* p_high = tmp_data_byte<<7 */
           #asm(  alu a = p; ram p23_0 = asm_tmp_crc.              ) /* a = tmp_data_byte<<7, p = tmp_crc */
           #asm(  alu nil = p ^ a, ccs.                            ) /* tmp_crc ^ (tmp_data_byte<<7) */
           #asm(  alu p =<< p + #0x00.                             ) /* p = tmp_crc<<1 */
           #asm(  alu sr = CEA709_CRC16_POLY.                     ) /* sr = CEA709_CRC16_POLY */
           #asm(/*alu_if n == 1 then p = p ^ sr.*/  %hex 3D587F71. ) /* if ((msb(tmp_crc))^(msb(tmp_data_byte<<7)))==1 then p = (tmp_crc<<1) ^ CEA709_CRC16_POLY */
           #asm(  ram p -> asm_tmp_crc.                            ) /* save tmp_crc */
           
       }
   }
/*******************************************************************************
* THREAD NAME: RX ENDPERIOD - A.
* DESCRIPTION: Detect the second transitions within the bit period.
*              1. Check whether the input pin has not changed since 
*                 the transition time. If it has not changed then continue 
*                 in part b,
*                 otherwise clear transition latch and thus enable next 
*                 transition detection.
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==1) && (flag1==0) && (pin==0))
   {
       if(IsLatchedTransitionA() && (CurrentInputPin==1))
       {
           /* noise transition came => enable next transition detections */
           Clear(TransLatch);
       }
       else goto CEA709_L6;
   }
/*******************************************************************************
* THREAD NAME: RX ENDPERIOD - B.
* DESCRIPTION: Detect the second transitions within the bit period.
*              1. Check whether the input pin has not changed since 
*                 the transition time. If it has not changed then continue, 
*                 otherwise clear transition latch and thus enable next 
*                 transition detection.
*              2. If the transition came, the end of one bit period was 
*                 detected. 
*                 - based on current pin state configure pin actions to detect 
*                   either rising or falling edge on Action Units A
*                 - clear latches
*                 - write the match registers and enable the match recognition
*                 - set flag 0
*                 - increment tmp_preamble
*                 - once the whole byte received, save it to the rx buffer
*                 - if the end of preamble detected update the state parameter
*              3. If the matchB latched, code violation was detected, i.e. the
*                 end of the frame. 
*                 - write the last received byte into the buffer
*                 - generate the DMA request either directly or through the
*                   DMA CONTROL channel
*                 - increment the tmp_length and save it 
*                   in last_rx_packet_length parameter
*                 - save the last rx packet time stamp
*                 - save the additional datagram information 
*                   in last_rx_packet_datagram_info parameter
*                 - mark the rx buffers as not active
*                 - inform the CPU that the end of packet reached + generate
*                   interrupt request
*                 - clear last_packet_tx bit in status parameter to indicate 
*                   that the last packet was rx
*                 - update backlog
*******************************************************************************/
   else if( IsMatchBOrTransitionAEvent() && (flag0==1) && (flag1==0) && (pin==1))
   {
       if(IsLatchedTransitionA() && (CurrentInputPin==0))
       {
           /* noise transition came => enable next transition detections */
           Clear(TransLatch);
           return;
       }
CEA709_L6:
       /* Transition came => the end of one period */
       if(IsLatchedTransitionA())
       {
           this.last_window_end = erta;
           ertb = erta + this.jitter2;
           erta += this.jitter1;

           /* Based on current pin state configure pin actions to detect either rising or falling edge on Action Units A */
           if(CurrentInputPin==1)
           {
               OnTransA(HighLow);
           }
           else
           {
               OnTransA(LowHigh);
           }

           ClearAllLatches();
           WriteErtAToMatchAAndEnable();
           WriteErtBToMatchBAndEnable();

           flag0=0;
           this.tmp_preamble++;
           
           /* Data is received */
           if((this.state & CEA709_STATE_PREAMBLE_PERIOD)==0)
           {
               if(--this.tmp_bit_count == 0)
               {
                   if(this.state & CEA709_STATE_FIRST_RX_BYTE)
                   {
                       this.delta_backlog = this.tmp_data_byte & 0x3F;
                       this.state &= ~CEA709_STATE_FIRST_RX_BYTE;
                   }
                   CEA709_WRITE_INTO_BUFFER();
                   this.tmp_length ++;
                   this.tmp_bit_count = 8;
                   this.tmp_data_byte = 0;
                   this.tmp_crc_2bytes_back = this.tmp_crc_1byte_back;
                   this.tmp_crc_1byte_back = this.tmp_crc;
               }
           }
           /* End of preamble - Manchester encoded "0" was detected (ByteSync) */
           if((this.state & CEA709_STATE_PREAMBLE_PERIOD) && 
             ((this.state & CEA709_STATE_ACTUAL_INPUT_BIT)==0))
           {
               if(this.tmp_preamble > this.bit_sync_threshold)
               {
                   this.state &= ~CEA709_STATE_PREAMBLE_PERIOD;
                   this.state |=  CEA709_STATE_FIRST_RX_BYTE;
               }
               else
               {
                   /* ignore the packet */
                   this.tmp_time = ertb;
                   goto CEA709_L7;
               }
           }
       }
       /* Transition did not come => code violation */
       else if(IsLatchedMatchB())
       {
           flag1=1;
           this.tmp_time = ertb;

           if(this.tmp_length < 8)
           {
               /* packet is not of valid length */
               this.status |= CEA709_STATUS_RX_ERROR;
               SetChannelInterrupt();
               goto CEA709_L7;
           }

           /* write the last received byte into the buffer */
           CEA709_WRITE_INTO_BUFFER();

           /* Generate the DMA request */
           if(this.tmp_offset != 0)
           {
               if(CEA709_DMA_CTR_ENABLED)
               {
                   /* Switch to DMA CONTROL CHANNEL to change the output pin and thus to trigger the DMA transfer */
                   SwitchToChannel(this.dma_control_channel);
                   SetPinPerPacB();
                   ClearMatchBLatch();
                   ertb = tcr1 + (this.period)>>3;
                   WriteErtBToMatchBAndEnable();
                   SwitchToChannel(this.rx_channel);
               }
               else
               {
                   /* Generate DMA request. */
                  SetDataTransferInterruptRequest();         
               }
           }

           /* increment the tmp_length and save it in last_rx_packet_length parameter */
           this.tmp_length ++;
           this.last_rx_packet_length = this.tmp_length;

           /* save the timestamp */
           this.last_rx_pckt_time_stamp_low = tcr1;
           this.last_rx_pckt_time_stamp_high = time_stamp_counter_high24;

           /* save the additional datagram information in last_rx_packet_datagram_info parameter */
           if((this.tmp_last_16_data_bits<<8) == ((this.tmp_crc_2bytes_back)^0xffff00))
           {
               this.last_rx_packet_datagram_info = CEA709_DGRAM_INFO_OK;
               this.cycle_time = 0;
           }
           else
           {
               this.last_rx_packet_datagram_info = CEA709_DGRAM_INFO_CRC_ERROR;
           }
           
           /* mark the rx buffers as not active */
           this.buffers &= ~CEA709_BUFFERS_RX_BUFFER_ACTIVE;

           /* Inform the CPU that the end of packet reached */
           this.status |= CEA709_STATUS_RX_FRAME_END;
           /* Generate Interrupt. */
           SetChannelInterrupt();
           
           /* Clear last_packet_tx bit in status parameter to indicate that the last packet was rx */
           this.state &= ~CEA709_STATE_LAST_PACKET_TX;
           
           /* Update backlog */
           if(this.delta_backlog <= 0)
               CEA709_DECREMENT_BACKLOG();
           else
               CEA709_INCREMENT_BACKLOG();

CEA709_L7:
           SingleMatchDoubleTransition();  
           DisableMatchDetection();
           DisableEventHandling();
           flag0=0;
           flag1=1;
           OnTransA(LowHigh);              
           OnTransB(HighLow);              
           ClearAllLatches();

           /* Switch to TX channel to enter the channel access algorithm */
           chan++;
               erta = this.tmp_time + (2*this.period) + this.beta1_receive;
               /* Save the time of the end of beta1 period */
               this.tmp_time = erta;
               this.channel_access_state = CEA709_CHANNEL_ACCESS_BETA1_IDLE;
               OnMatchA(PinLow);
               OnMatchB(PinLow);
               flag0 = 1;
               flag1 = 1;
               SingleMatchSingleTransition();
           ClearAllLatches();
           EnableEventHandling();
           WriteErtAToMatchAAndEnable();
           WriteErtBToMatchBAndEnable();
           chan--;
       }
   }

/*******************************************************************************
* THREAD NAME: UNHANDLED_EVENTS
*******************************************************************************/
   else 
   {
#ifdef GLOBAL_ERROR_FUNC
		Global_Error_Func();
#else
		ClearAllLatches();
#endif
   }  
}

void CEA709_CD( struct CEA709_channel_params this )
{
/*******************************************************************************
* THREAD NAME: CD INIT.
* DESCRIPTION: Initialize a CEA709 Collision Detection channel.
*              1. Disable matches in thread in case there are any previous
*                 pending matches. This could happen if the channel was
*                 running another function before this one.
*              2. Disable output buffer.
*              3. Configure channel to use TCR1 clock.
*              4. Select SingleMatch-DoubleTransition channel mode.
*              5. Disable detection of matches.
*              6. Enable match event handling.
*              7. Configure pin actions to detect falling edges on Action Unit A.
*              8. Configure pin actions to detect rising edges on Action Unit B.
*              9. Clear latches.
*              10.Set the minimum collision detection time as (bit period)/64.
*******************************************************************************/
   if (hsr==CEA709_CD_HSR_INIT)
   {
       DisableMatchesInThread();
       DisableOutputBuffer();
       ActionUnitA( MatchTCR1, CaptureTCR1, GreaterEqual);  
       ActionUnitB( MatchTCR1, CaptureTCR1, GreaterEqual);  
       SingleMatchDoubleTransition();                       
       DisableMatchDetection();
       EnableEventHandling();
       OnTransA(HighLow);                                   
       OnTransB(LowHigh);                                   
       ClearAllLatches();  
       this.minimum_cd_time = this.transmit_period>>6;
   }
/*******************************************************************************
* THREAD NAME: COLLISION DETECTED.
* DESCRIPTION: Active-low collision detection signal occurred.
*              1. When the time between two consecutively detected transitions
*                 is less than minimum_cd_time the collision is considered.
*              2. Clear latches.
*******************************************************************************/
   else if( IsMatchAOrTransitionBEvent() )
   {
       if( (ertb - erta) > this.minimum_cd_time )
       {
           this.state |= CEA709_STATE_COLLISION_DETECTED;
       }
       ClearAllLatches();
   }
/*******************************************************************************
* THREAD NAME: UNHANDLED_EVENTS
*******************************************************************************/
   else 
   {
#ifdef GLOBAL_ERROR_FUNC
		Global_Error_Func();
#else
		ClearAllLatches();
#endif
   }  
}

void CEA709_TXEN( struct CEA709_channel_params this )
{
/*******************************************************************************
* THREAD NAME: TXEN INIT.
* DESCRIPTION: Initialize a CEA709 Transmit Enable channel.
*              1. Configure output pin action to set the pin low when the matchA
*                 happens.
*              2. Enable output buffer.
*              3. Set output pin to low.
*              4. Clear latches.
*              5. Configure channel to use TCR1 clock.
*              6. Configure input pin action to no_detect.
*              7. Select SingleMatch-SingleTransition channel mode.
*******************************************************************************/
   if (hsr==CEA709_TXEN_HSR_INIT)
   {
      OnMatchA(PinLow);
      EnableOutputBuffer();
      SetPinLow();
      ClearAllLatches();
      ActionUnitA( MatchTCR1, CaptureTCR1, GreaterEqual);
      OnTransA(NoDetect);
      SingleMatchSingleTransition();
   }
/*******************************************************************************
* THREAD NAME: UNHANDLED_EVENTS
*******************************************************************************/
   else 
   {
#ifdef GLOBAL_ERROR_FUNC
		Global_Error_Func();
#else
		ClearAllLatches();
#endif
   }  
}

void CEA709_DMA_CONTROL( struct CEA709_channel_params this )
{
/*******************************************************************************
* THREAD NAME: DMA CONTROL INIT - RISING EDGE TRIGGERS.
* DESCRIPTION: Initialization of the DMA control channel number necessary for 
*              external DMA request generation. Rising edge triggers the DMA 
*              transfer.
*              1. Configure output pin action to no_change on matchA.
*              2. Configure output pin action to toggle the pin when the 
*                 matchB happens.
*              3. Enable output buffer.
*              4. Set output pin to low.
*              5. Clear latches.
*              6. Configure channel to use TCR1.
*              7. Configure input pin action to no_detect.
*              8. Select EitherMatch-NonBlocking-SingleTransition channel mode.
*              9. Set erta to 0xFFFFFF.
*             10. Enable match event handling.
*             11. if fm0==0 then write the matchA register and enable the match 
*                 recognition (i.e. match A will be latched each time the TCR1 
*                 clock overflows, which enables to implement 48bit counter for 
*                 timestamps).
*******************************************************************************/
   if (hsr==CEA709_DMA_CONTROL_HSR_INIT_RISING_TRIGGER)
   {
      OnMatchA(NoChange);
      OnMatchB(PinToggle);
      EnableOutputBuffer();
      SetPinLow();
      ClearAllLatches();

      goto CEA709_DMA_CONTROL_INIT_COMMON;
   }

/*******************************************************************************
* THREAD NAME: DMA CONTROL INIT - FALLING EDGE TRIGGERS.
* DESCRIPTION: Initialization of the DMA control channel number necessary for 
*              external DMA request generation. Falling edge triggers the DMA 
*              transfer.
*              1. Configure output pin action to no_change on matchA.
*              2. Configure output pin action to toggle the pin when the 
*                 matchB happens.
*              3. Enable output buffer.
*              4. Set output pin to high.
*              5. Clear latches.
*              6. Configure channel to use TCR1.
*              7. Configure input pin action to no_detect.
*              8. Select EitherMatch-NonBlocking-SingleTransition channel mode.
*              9. Set erta to 0xFFFFFF.
*             10. Enable match event handling.
*             11. if fm0==0 then write the matchA register and enable the match 
*                 recognition (i.e. match A will be latched each time the TCR1 
*                 clock overflows, which enables to implement 48bit counter for 
*                 timestamps).
*******************************************************************************/
   else if (hsr==CEA709_DMA_CONTROL_HSR_INIT_FALLING_TRIGGER)
   {
      OnMatchA(NoChange);
      OnMatchB(PinToggle);
      EnableOutputBuffer();
      SetPinHigh();
      ClearAllLatches();

CEA709_DMA_CONTROL_INIT_COMMON:
      ActionUnitA( MatchTCR1, CaptureTCR1, EqualOnly);
      ActionUnitB( MatchTCR1, CaptureTCR1, GreaterEqual);
      OnTransA(NoDetect);
      EitherMatchNonBlockingSingleTransition();
      erta = 0xFFFFFF;
      EnableEventHandling();
      if(CEA709_DMA_CTR_INCREMENT_COUNTER_YES)
        WriteErtAToMatchAAndEnable();
   }

/*******************************************************************************
* THREAD NAME: DMA CONTROL MATCH A.
* DESCRIPTION: Match A is latched each time the TCR1 clock overflows. This 
*              enables to implement 48bit counter for timestamps.
*              1. Clear match A latch.
*              2. Increment time_stamp_counter_high24 variable 
*                 (high 24 bits of the 48bit counter)
*              3. Set erta to 0xFFFFFF.
*              4. Write the match A register and enable the match recognition.
*******************************************************************************/
   else if ( IsMatchAOrTransitionBEvent() )
   {
      ClearMatchALatch();
      time_stamp_counter_high24++;
      erta = 0xFFFFFF;
      WriteErtAToMatchAAndEnable();
   }

/*******************************************************************************
* THREAD NAME: DMA CONTROL MATCH B.
* DESCRIPTION: Match B is latched each time the DMA control pulse finishes. 
*              Output pin toggles on this match. Nothing has to be done, just
*              to clear match B latch.
*******************************************************************************/
   else if ( IsMatchBOrTransitionAEvent() )
   {
      ClearMatchBLatch();
   }

/*******************************************************************************
* THREAD NAME: UNHANDLED_EVENTS
*******************************************************************************/
   else 
   {
#ifdef GLOBAL_ERROR_FUNC
		Global_Error_Func();
#else
		ClearAllLatches();
#endif
   }  
}

/************************************************************************
*  Information exported to Host CPU program.
*************************************************************************/
#pragma write h, (::ETPUfilename (etpu_CEA709_auto.h));
#pragma write h, (/**************************************************************** );
#pragma write h, (* WARNING this file is automatically generated DO NOT EDIT IT!    );
#pragma write h, (*                                                                 );
#pragma write h, (* This file provides an interface between eTPU code and CPU       );
#pragma write h, (* code. All references to the CEA709_TX and CEA709_RX functions );
#pragma write h, (* should be made with information in this file. This allows only  );
#pragma write h, (* symbolic information to be referenced which allows the eTPU     );
#pragma write h, (* code to be optimized without effecting the CPU code.            );
#pragma write h, (*****************************************************************/);
#pragma write h, (#ifndef _FS_ETPU_CEA709_AUTO_H_);
#pragma write h, (#define _FS_ETPU_CEA709_AUTO_H_);
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, (* Function Configuration Information. );
#pragma write h, (****************************************************************/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_RX_FUNCTION_NUMBER         ) CEA709_RX_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_RX_TABLE_SELECT            ) ::ETPUentrytype(CEA709_RX) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_RX_NUM_PARMS               ) ::ETPUram(CEA709_RX) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_FUNCTION_NUMBER         ) CEA709_TX_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_TABLE_SELECT            ) ::ETPUentrytype(CEA709_TX) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_NUM_PARMS               ) ::ETPUram(CEA709_TX) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TXEN_FUNCTION_NUMBER       ) CEA709_TXEN_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TXEN_TABLE_SELECT          ) ::ETPUentrytype(CEA709_TXEN) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TXEN_NUM_PARMS             ) ::ETPUram(CEA709_TXEN) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_FUNCTION_NUMBER) CEA709_DMA_CONTROL_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_TABLE_SELECT   ) ::ETPUentrytype(CEA709_DMA_CONTROL) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_NUM_PARMS      ) ::ETPUram(CEA709_DMA_CONTROL) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_FUNCTION_NUMBER         ) CEA709_CD_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_TABLE_SELECT            ) ::ETPUentrytype(CEA709_CD) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_NUM_PARMS               ) ::ETPUram(CEA709_CD) );
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, (* Host Service Request Definitions. );
#pragma write h, (****************************************************************/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_RX_INIT                         ) CEA709_RX_HSR_INIT );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_INIT                         ) CEA709_TX_HSR_INIT );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_RUN                          ) CEA709_TX_HSR_RUN );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TXEN_INIT                       ) CEA709_TXEN_HSR_INIT );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_INIT_RISING_TRIGGER ) CEA709_DMA_CONTROL_HSR_INIT_RISING_TRIGGER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_INIT_FALLING_TRIGGER) CEA709_DMA_CONTROL_HSR_INIT_FALLING_TRIGGER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_INIT                         ) CEA709_CD_HSR_INIT );
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, (* Global Parameters Definition. );
#pragma write h, (****************************************************************/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TIME_STAMP_COUNTER_HIGH24_OFFSET   ) ::ETPUlocation (time_stamp_counter_high24) );
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, (* Parameter Definitions. );
#pragma write h, (****************************************************************/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_PERIOD_OFFSET                      ) ::ETPUlocation (CEA709_TX, this.period) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_STATE_OFFSET                       ) ::ETPUlocation (CEA709_TX, this.state) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TRANSMIT_PERIOD_OFFSET             ) ::ETPUlocation (CEA709_TX, this.transmit_period) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CHANNEL_ACCESS_STATE_OFFSET        ) ::ETPUlocation (CEA709_TX, this.channel_access_state) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_STATUS_OFFSET                      ) ::ETPUlocation (CEA709_TX, this.status) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BUFFERS_OFFSET                     ) ::ETPUlocation (CEA709_TX, this.buffers) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_PREAMBLE_LENGTH_OFFSET             ) ::ETPUlocation (CEA709_TX, this.preamble_length) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_NODE_PRIORITY_OFFSET               ) ::ETPUlocation (CEA709_TX, this.node_priority) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BETA1_TRANSMIT_OFFSET              ) ::ETPUlocation (CEA709_TX, this.beta1_transmit) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BETA1_RECEIVE_OFFSET               ) ::ETPUlocation (CEA709_TX, this.beta1_receive) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BETA2_OFFSET                       ) ::ETPUlocation (CEA709_TX, this.beta2) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BACKLOG_OFFSET                     ) ::ETPUlocation (CEA709_TX, this.backlog) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DELTA_BACKLOG_OFFSET               ) ::ETPUlocation (CEA709_TX, this.delta_backlog) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_OPTIONS_OFFSET                  ) ::ETPUlocation (CEA709_TX, this.cd_options) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CONTROL_CHAN_OFFSET            ) ::ETPUlocation (CEA709_TX, this.dma_control_channel) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_BIT_SYNC_THRESHOLD_OFFSET          ) ::ETPUlocation (CEA709_TX, this.bit_sync_threshold) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CHANNEL_PRIORITIES_OFFSET          ) ::ETPUlocation (CEA709_TX, this.channel_priorities) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_RX_CHAN_OFFSET                     ) ::ETPUlocation (CEA709_TX, this.rx_channel) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_TX_ENABLE_CHAN_OFFSET              ) ::ETPUlocation (CEA709_TX, this.tx_enable_channel) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_CD_TO_END_PACKET_OFFSET            ) ::ETPUlocation (CEA709_TX, this.cd_to_end_packet) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_PACKET_CYCLE_OFFSET                ) ::ETPUlocation (CEA709_TX, this.packet_cycle) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_LENGTH_OFFSET       ) ::ETPUlocation (CEA709_TX, this.last_rx_packet_length) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET          ) ::ETPUlocation (CEA709_TX, this.p_tx_pri_buffer_start) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET       ) ::ETPUlocation (CEA709_TX, this.p_tx_nonpri_buffer_start) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_RX_BUF_START_OFFSET              ) ::ETPUlocation (CEA709_TX, this.p_rx_buffer_start) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET) ::ETPUlocation (CEA709_TX, this.last_rx_packet_datagram_info) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET) ::ETPUlocation (CEA709_TX, this.last_rx_pckt_time_stamp_high) );
//#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET ) ::ETPUlocation (CEA709_TX, this.last_rx_pckt_time_stamp_low) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_TX_PRI_BUF_START_OFFSET            ) ( 0x0021 ) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_TX_NONPRI_BUF_START_OFFSET         ) ( 0x0025 ) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_P_RX_BUF_START_OFFSET                ) ( 0x0029 ) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_DATAGRAM_INFO_OFFSET  ) ( 0x003C ) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_HIGH_OFFSET) ( 0x007D ) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_LAST_RX_PACKET_TIME_STAMP_LOW_OFFSET ) ( 0x0081 ) );
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, (* Value Definitions. );
#pragma write h, (****************************************************************/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_FM0_DMA_CTR_OFF                   ) 0 );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_FM0_DMA_CTR_ON                    ) 1 );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_FM1_TXEN_OFF                      ) 0 );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_FM1_TXEN_ON                       ) 2 );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CTR_FM0_INCREMENT_COUNTER_YES ) 0 );
#pragma write h, (::ETPUliteral(#define FS_ETPU_CEA709_DMA_CTR_FM0_INCREMENT_COUNTER_NO  ) 1 );
#pragma write h, ( );
#pragma write h, (#endif);

/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/
