/*! *********************************************************************************
 * \addtogroup Accelerometer Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 * \file accelerometer_interface.h
 * This file is the interface file for the Accelerometer Service
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _ACCELEROMETER_INTERFACE_H_
#define _ACCELEROMETER_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
/*! Available scales for the accelerometer */
typedef enum acsScales{
  kScale2g,              /*!< +/- 2g */
  kScale4g,              /*!< +/- 4g */
  kScale8g,              /*!< +/- 8g */
  kScaleNotSupported     /*!< Scale not supported */
}acs_scales_t;

/*! Accelerometer Service - Configuration */
typedef struct acsConfig_tag
{
    uint16_t    serviceHandle;                 /*!<Service handle */
    uint8_t     scale;                         /*!<Accelerometer scale */
    int16_t     xAxisValue;                    /*!<X Axis measurement value */
    int16_t     yAxisValue;                    /*!<Y Axis measurement value */
    int16_t     zAxisValue;                    /*!<Z Axis measurement value */
} acsConfig_t;

/*! Accelerometer Readings packer structure */
typedef struct acsReadingsPacker{
  int16_t       xAxisReading;   /*!< X Axis reading */             
  int16_t       yAxisReading;   /*!< Y Axis reading */
  int16_t       zAxisReading;   /*!< Z Axis reading */
}acs_readings_packer_t;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*!**********************************************************************************
* \brief        Starts Accelerometer Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_Start(acsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Stops Accelerometer Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_Stop(acsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Subscribes a GATT client to the Accelerometer service
*
* \param[in]    pClient  Client Id in Device DB.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_Subscribe(deviceId_t clientDeviceId);

/*!**********************************************************************************
* \brief        Unsubscribes a GATT client from the Accelerometer service
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_Unsubscribe();

/*!**********************************************************************************
* \brief        Updates the accelerometer value and sends a notification if active
*
* \param[in]    serviceHandle           Service handle.
* \param[in]    newXAxisValue           New value for the X Axis
* \param[in]    newYAxisValue           New value for the Y Axis
* \param[in]    newZAxisValue           New value for the Z Axis
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_RecordAccelerometerMeasurement (uint16_t serviceHandle, int16_t newXAxisValue, 
                                                int16_t newYAxisValue, int16_t newZAxisValue);

/*!**********************************************************************************
* \brief        Updates the accelerometer scale
*
* \param[in]    serviceHandle           Service handle.
* \param[in]    scale                   New scale value
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Acs_RecordAccelerometerScale (uint16_t serviceHandle, uint8_t scale);

#ifdef __cplusplus
}
#endif

#endif /* _LED_CONTROL_INTERFACE_H_ */

/*! **********************************************************************************
 * @}
 ************************************************************************************/
