/*! *********************************************************************************
 * \addtogroup Compass Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file compass_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "compass_interface.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
/*! Compass Service - Subscribed Client*/
static deviceId_t mCps_SubscribedClientId;

//Characteristic declarations
const bleUuid_t compassHeadingCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x07, 0x57, 0xFF, 0x02}
};

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/

static void Cps_SendCompassHeadingMeasurementNotification (uint16_t handle);

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Cps_Start (cpsConfig_t *pServiceConfig)
{    
    mCps_SubscribedClientId = gInvalidDeviceId_c;
    
    return Cps_RecordCompassHeadingMeasurement (pServiceConfig->serviceHandle, 
                                                pServiceConfig->compassHeadingValue);
}

bleResult_t Cps_Stop (cpsConfig_t *pServiceConfig)
{
    return Cps_Unsubscribe();
}

bleResult_t Cps_Subscribe(deviceId_t deviceId)
{
    mCps_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Cps_Unsubscribe()
{
    mCps_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Cps_RecordCompassHeadingMeasurement (uint16_t serviceHandle, uint16_t newCompassHeadingValue)
{
    uint16_t  handle;
    bleResult_t result;

    /* Get handle of Compass characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&compassHeadingCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    //Utils_RevertByteArray((uint8_t*)&newCompassHeadingValue, sizeof(uint16_t))//Change value to little endian
    result = GattDb_WriteAttribute(handle, sizeof(uint16_t), (uint8_t*)&newCompassHeadingValue);

    if (result != gBleSuccess_c)
        return result;

    Cps_SendCompassHeadingMeasurementNotification(handle);

    return gBleSuccess_c;
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/
static void Cps_SendCompassHeadingMeasurementNotification
(
  uint16_t handle
)
{
    uint16_t  hCccd;
    bool_t isNotificationActive;

    /* Get handle of CCCD */
    if (GattDb_FindCccdHandleForCharValueHandle(handle, &hCccd) != gBleSuccess_c)
        return;

    if (gBleSuccess_c == Gap_CheckNotificationStatus
        (mCps_SubscribedClientId, hCccd, &isNotificationActive) &&
        TRUE == isNotificationActive)
    {
        GattServer_SendNotification(mCps_SubscribedClientId, handle);
    }
}
/*! *********************************************************************************
 * @}
 ********************************************************************************** */
