/*! *********************************************************************************
 * \addtogroup Remote Controller Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 * \file remoteController_control_interface.h
 * This file is the interface file for the Remote Controller Service
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _REMOTE_CONTROLLER_INTERFACE_H_
#define _REMOTE_CONTROLLER_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
/* Available remote controller commands */
typedef enum rcsAvailableCommands{
  kRcsPower,                    /*!< Power On/Off */
  kRcsMute,                     /*!< Mute */
  kRcsVolumeP,                  /*!< Volume + */
  kRcsVolumeM,                  /*!< Volume - */
  kRcsChannelP,                 /*!< Channel + */
  kRcsChannelM,                 /*!< Channel - */
  kRcsInvalidCommand            /*!< Invalid command */
}rcs_available_commands_t;

/* Available remote controller brand configurations */
typedef enum rcsAvailableControllerBrands{
  kRcsBrandLG,                  /*!< LG TVs */
  kRcsBrandSamsung,             /*!< Samsung TVs */
  kRcsBrandSony,                /*!< Sony TVs */
  kRcsBrandInvalid              /*!< Invalid brand */
}rcs_available_controller_brands_t;

/*! Remote Controller Service - Configuration */
typedef struct rcsConfig_tag
{
    uint16_t    serviceHandle;                  /*!<Service handle*/
    uint8_t     controllerCommand;              /*!<Controller command */
    uint8_t     controllerConfiguration;        /*!<Controller brand configuration */
} rcsConfig_t;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*!**********************************************************************************
* \brief        Starts Remote Controller Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_Start(rcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Stops Remote Controller Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_Stop(rcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Subscribes a GATT client to the Remote Controller service
*
* \param[in]    pClient  Client Id in Device DB.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_Subscribe(deviceId_t clientDeviceId);

/*!**********************************************************************************
* \brief        Unsubscribes a GATT client from the Remote Controller service
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_Unsubscribe();

/*!**********************************************************************************
* \brief        Sets a command sent by the client
*
* \param[in]    serviceHandle   Service handle
* \param[in]    commandValue    Command value sent by the client
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_SetCommand (uint16_t serviceHandle, uint8_t commandValue);

/*!**********************************************************************************
* \brief        Sets a configuration sent by the client
*
* \param[in]    serviceHandle           Service handle
* \param[in]    configurationValue      Configuration value sent by the client
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Rcs_SetConfiguration (uint16_t serviceHandle, uint8_t configurationValue);

#ifdef __cplusplus
}
#endif

#endif /* _REMOTE_CONTROLLER_INTERFACE_H_ */

/*! **********************************************************************************
 * @}
 ************************************************************************************/
